/*
 * Decompiled with CFR 0.152.
 */
package uk.gov.dstl.baleen.services;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import uk.gov.dstl.baleen.data.EmptyTemplate;
import uk.gov.dstl.baleen.data.PipelineTemplate;

@Service
public class TemplateService {
    @Value(value="${baleen.templates}")
    private File templatesFolder;
    @Autowired
    private ObjectMapper objectMapper;
    private WatchService watchService = null;
    private final Map<String, PipelineTemplate> templateCache = new HashMap<String, PipelineTemplate>();
    private static final PipelineTemplate EMPTY_TEMPLATE = new EmptyTemplate();
    private static final Logger LOGGER = LoggerFactory.getLogger(TemplateService.class);

    protected TemplateService(File templatesFolder) {
        this();
        this.templatesFolder = templatesFolder;
    }

    public TemplateService() {
        try {
            this.watchService = FileSystems.getDefault().newWatchService();
        }
        catch (IOException e) {
            LOGGER.warn("Unable to create WatchService - files added/removed to the templates folder will not be detected", (Throwable)e);
        }
    }

    @PostConstruct
    public void findInitialTemplates() {
        File[] files;
        if (!this.templatesFolder.exists()) {
            LOGGER.info("Creating templates folder {}", (Object)this.templatesFolder);
            if (!this.templatesFolder.mkdirs()) {
                LOGGER.warn("Unable to create persistence folder {}", (Object)this.templatesFolder);
                return;
            }
        }
        if (!this.templatesFolder.isDirectory()) {
            LOGGER.warn("Templates folder {} is not a directory", (Object)this.templatesFolder);
            return;
        }
        if (!this.templatesFolder.canRead()) {
            LOGGER.warn("Can not read from templates folder {}", (Object)this.templatesFolder);
            return;
        }
        if (this.watchService != null) {
            try {
                this.templatesFolder.toPath().register(this.watchService, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_DELETE);
            }
            catch (IOException e) {
                LOGGER.warn("Unable to create WatchKey - files added/removed to the templates folder will not be detected", (Throwable)e);
            }
        }
        if ((files = this.templatesFolder.listFiles((dir, name) -> name.toLowerCase().endsWith(".json"))) == null) {
            LOGGER.warn("Can not retrieve JSON files from templates folder {}", (Object)this.templatesFolder);
            return;
        }
        for (File f : files) {
            this.cacheTemplateFromFile(f);
        }
    }

    @Scheduled(fixedDelay=5000L)
    private void detectChanges() {
        WatchKey key;
        if (this.watchService == null) {
            return;
        }
        LOGGER.debug("Checking templates folder for changes");
        while ((key = this.watchService.poll()) != null) {
            for (WatchEvent<?> event : key.pollEvents()) {
                PipelineTemplate o;
                Path path = (Path)event.context();
                if (!path.toString().toLowerCase().endsWith(".json")) continue;
                LOGGER.info("{} event detected on path {}", event.kind(), (Object)path);
                File f = new File(this.templatesFolder, path.getFileName().toString());
                if (StandardWatchEventKinds.ENTRY_MODIFY.equals(event.kind())) {
                    this.cacheTemplateFromFile(f);
                    continue;
                }
                if (!StandardWatchEventKinds.ENTRY_DELETE.equals(event.kind()) || (o = this.templateCache.remove(f.getName())) == null) continue;
                LOGGER.info("Removing template {} from cache", (Object)f.getName());
            }
            key.reset();
        }
    }

    private boolean cacheTemplateFromFile(File f) {
        LOGGER.info("Adding template {} to cache", (Object)f.getName());
        try (FileReader reader = new FileReader(f);){
            this.templateCache.put(f.getName(), (PipelineTemplate)this.objectMapper.readValue((Reader)reader, PipelineTemplate.class));
        }
        catch (Exception e) {
            LOGGER.error("Unable to parse file {}", (Object)f, (Object)e);
            return false;
        }
        return true;
    }

    public List<PipelineTemplate> getTemplates() {
        ArrayList<PipelineTemplate> templates = new ArrayList<PipelineTemplate>(this.templateCache.values());
        templates.sort(Comparator.comparing(PipelineTemplate::getName));
        templates.add(0, EMPTY_TEMPLATE);
        return templates;
    }
}

