/*
 * Copyright (C) 2014 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.sdk.service;

import retrofit.http.GET;
import retrofit.http.Path;
import retrofit.http.Query;
import travel.izi.sdk.model.entity.CompactMtgObject;
import travel.izi.sdk.model.entity.FullMtgObject;

import java.util.List;

/**
 * Endpoints for Publisher.
 */
@SuppressWarnings("unused")
public interface PublisherService {

    /**
     * Get full Publisher object by UUID.
     *
     * @param uuid      UUID of the Publisher.
     * @param languages Array of languages which should have Publisher.
     * @param includes  Array of sections which should be included to Publisher.
     * @param except    Array of sections which should NOT be included to Publisher.
     * @return Publisher or 404 HTTP code if there is no Publisher with provided UUID.
     */
    @GET("/mtg/publishers/{uuid}")
    FullMtgObject getPublisher(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except
    );

    /**
     * Get Publisher children in compact form.
     *
     * @param uuid      UUID of the Publisher.
     * @param languages Array of requested languages, i.e. Publisher's children should have one of the requested languages.
     * @param includes  Array of sections which should be included.
     * @param except    Array of sections which should NOT be included.
     * @param limit     Limit for pagination, defaults to 20.
     * @param offset    Offset for pagination, defaults to 0.
     * @return Tours and Museums which belongs to the Publisher. Returns 404 HTTP code if there is no Publisher with provided UUID.
     * @see #getFullChildren(String, String[], String[], String[], Long, Long, Boolean)
     */
    @GET("/mtg/publishers/{uuid}/children?form=compact")
    List<CompactMtgObject> getCompactChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Get Publisher children in full form.
     *
     * @see #getCompactChildren(String, String[], String[], String[], Long, Long)
     */
    @GET("/mtg/publishers/{uuid}/children?form=full")
    List<FullMtgObject> getFullChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("children_count_in_full_form") Boolean children_count
    );

    /**
     * Get the number of Publisher’s children (museums, tours)
     *
     * @param uuid      UUID of the Publisher.
     * @param languages Array of requested languages. Publisher’s children should have one of the requested languages.
     * @return Number of children (tours and museums).
     */
    @GET("/mtg/publishers/{uuid}/children/count")
    Integer getChildrenCount(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages
    );

    /**
     * Get Publisher languages by UUID, i.e. returns list of unique languages on which Publisher have content.
     *
     * @param uuid      UUID of the Publisher.
     * @param languages Array of requested languages. The resulting list of languages to be contained in this array.
     * @return List of languages.
     */
    @GET("/mtg/publishers/{uuid}/children/languages")
    List<String> getLanguages(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages
    );

}