package threads.webrtc;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AlertDialog;
import androidx.fragment.app.DialogFragment;

import java.util.concurrent.atomic.AtomicBoolean;

import threads.core.Preferences;

import static androidx.core.util.Preconditions.checkNotNull;

public class RTCCallingDialogFragment extends DialogFragment {

    public static final String TAG = RTCCallingDialogFragment.class.getSimpleName();
    private static final String TOPIC = "TOPIC";
    private static final String VIDEO = "VIDEO";
    private final AtomicBoolean triggerTimeoutCall = new AtomicBoolean(true);
    private RTCCallingDialogFragment.ActionListener mListener;
    private Context mContext;
    private RTCSoundPool soundPoolManager;

    static RTCCallingDialogFragment newInstance(@NonNull String topic, boolean video) {
        checkNotNull(topic);
        Bundle bundle = new Bundle();
        bundle.putString(TOPIC, topic);
        bundle.putBoolean(VIDEO, video);
        RTCCallingDialogFragment fragment = new RTCCallingDialogFragment();
        fragment.setArguments(bundle);
        return fragment;
    }

    @Override
    public void onDetach() {
        super.onDetach();
        mContext = null;
        soundPoolManager.release();
        triggerTimeoutCall.set(false);
    }

    @Override
    public void onAttach(@NonNull Context context) {
        super.onAttach(context);
        mContext = context;
        try {
            mListener = (RTCCallingDialogFragment.ActionListener) getActivity();
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }
        soundPoolManager = RTCSoundPool.create(mContext, R.raw.incoming, true);
    }

    @Override
    @NonNull
    public Dialog onCreateDialog(Bundle savedInstanceState) {

        Activity activity = getActivity();
        checkNotNull(activity);


        Bundle args = getArguments();
        checkNotNull(args);
        final String topic = args.getString(TOPIC);
        checkNotNull(topic);
        AlertDialog.Builder builder = new AlertDialog.Builder(mContext);

        boolean video = args.getBoolean(VIDEO);
        if (video) {
            builder.setIcon(R.drawable.text_video_call);
        } else {
            builder.setIcon(R.drawable.text_phone);
        }
        builder.setTitle(getString(R.string.incoming_call));
        builder.setPositiveButton(getString(R.string.accept), (dialog, which) -> {

            soundPoolManager.stop();
            triggerTimeoutCall.set(false);
            mListener.acceptCall(topic);
            dialog.dismiss();

        });
        builder.setNegativeButton(getString(R.string.reject), (dialog, which) -> {

            soundPoolManager.stop();
            triggerTimeoutCall.set(false);
            mListener.rejectCall(topic);
            dialog.dismiss();

        });

        AlertDialog dialog = builder.create();
        dialog.setCancelable(false);
        dialog.setCanceledOnTouchOutside(false);
        dialog.requestWindowFeature(Window.FEATURE_NO_TITLE);


        // hide the status bar and enter full screen mode
        checkNotNull(dialog.getWindow());
        dialog.getWindow().setFlags(
                WindowManager.LayoutParams.FLAG_FULLSCREEN,
                WindowManager.LayoutParams.FLAG_FULLSCREEN
        );

        // dismiss the keyguard
        dialog.getWindow().addFlags(WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD);
        dialog.getWindow().addFlags(WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED);
        dialog.getWindow().addFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN);


        // Set the content to appear under the system bars so that the content
        // doesn't resize when the system bars hide and show.
        dialog.getWindow().getDecorView().getRootView().setSystemUiVisibility(
                View.SYSTEM_UI_FLAG_IMMERSIVE
                        // Set the content to appear under the system bars so that the
                        // content doesn't resize when the system bars hide and show.
                        | View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                        | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                        | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                        // Hide the nav bar and status bar
                        | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                        | View.SYSTEM_UI_FLAG_FULLSCREEN);

        // immersive hack 1: set the dialog to not focusable (makes navigation ignore us adding the window)
        dialog.getWindow().addFlags(WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE);

        dialog.setOnShowListener((dialogInterface) -> {

            Button positive = dialog.getButton(AlertDialog.BUTTON_POSITIVE);
            Button negative = dialog.getButton(AlertDialog.BUTTON_NEGATIVE);

            positive.setCompoundDrawablesRelativeWithIntrinsicBounds(
                    R.drawable.phone_in_talk, 0, 0, 0);
            positive.setCompoundDrawablePadding(16);

            negative.setCompoundDrawablesRelativeWithIntrinsicBounds(
                    R.drawable.phone_reject, 0, 0, 0);
            negative.setCompoundDrawablePadding(16);


        });
        final int timeout = Preferences.getConnectionTimeout(mContext) * 1000;
        new Handler().postDelayed(() -> {
            try {
                if (triggerTimeoutCall.get()) {
                    mListener.timeoutCall(topic);
                }
                dialog.dismiss();
            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
            }

        }, timeout);
        soundPoolManager.play();

        return dialog;
    }

    @Override
    public void onDismiss(@NonNull DialogInterface dialog) {
        super.onDismiss(dialog);
    }


    public interface ActionListener {

        void acceptCall(@NonNull String topic);

        void rejectCall(@NonNull String topic);

        void timeoutCall(@NonNull String topic);

    }
}
