package threads.iota;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.common.collect.Iterables;

import org.apache.commons.lang3.StringUtils;
import org.iota.jota.IotaAPI;
import org.iota.jota.IotaLocalPoW;
import org.iota.jota.dto.response.FindTransactionResponse;
import org.iota.jota.dto.response.GetNodeInfoResponse;
import org.iota.jota.model.Transaction;
import org.iota.jota.model.Transfer;
import org.iota.jota.utils.Checksum;
import org.iota.jota.utils.Constants;
import org.iota.jota.utils.InputValidator;
import org.iota.jota.utils.SeedRandomGenerator;
import org.iota.jota.utils.TrytesConverter;

import java.io.ByteArrayOutputStream;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.List;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;


public class IOTA {
    private static final int DEPTH = 1;
    private static final int ZERO_VALUE = 0;
    private static final String POW_SUPPORT = "RemotePOW";
    private static final String LOAD_BALANCER = "loadBalancer";
    private static final int MIN_WEIGHT_MAGNITUDE = 14;
    private final String protocol;
    private final String host;
    private final int port;
    private final int timeout;
    private IotaAPI iotaAPI;
    private Object lock = new Object();

    private IOTA(final Builder builder) {
        checkNotNull(builder);
        this.protocol = builder.protocol;
        this.host = builder.host;
        this.port = builder.port;
        this.timeout = builder.timeout;
        postConstruct();
    }

    public static boolean remotePoW(@NonNull GetNodeInfoResponse nodeInfoResponse) {
        checkNotNull(nodeInfoResponse);
        String[] features = nodeInfoResponse.getFeatures();
        if (features != null) {
            for (String feature : features) {
                if (POW_SUPPORT.equals(feature)) {
                    return true;
                }
            }
        }
        return false;
    }

    public static boolean loadBalancer(@NonNull GetNodeInfoResponse nodeInfoResponse) {
        checkNotNull(nodeInfoResponse);
        String[] features = nodeInfoResponse.getFeatures();
        if (features != null) {
            for (String feature : features) {
                if (LOAD_BALANCER.equals(feature)) {
                    return true;
                }
            }
        }
        return false;
    }

    private static String adaptTrytes(@NonNull String stringAsTrytes) {

        stringAsTrytes = StringUtils.stripEnd(stringAsTrytes, "9");

        // special case take one 9 to much
        int length = stringAsTrytes.length();
        if (length % 2 != 0) {
            stringAsTrytes = stringAsTrytes.concat("9");

        }
        return stringAsTrytes;
    }

    private static boolean isTrytes(String trytes, int length) {
        return trytes.matches("^[A-Z9]{" + (length == 0 ? "0," : length) + "}$");
    }

    public static String generateAddress() {
        String address = SeedRandomGenerator.generateNewSeed();
        return addChecksum(address);
    }

    public static String addChecksum(@NonNull String address) {
        return Checksum.addChecksum(address);
    }

    public static String toTrytes(byte[] bytes) {
        StringBuilder trytes = new StringBuilder();

        for (byte asciiValue : bytes) {
            int value = asciiValue & 0xff;
            int firstValue = value % 27;
            int secondValue = (value - firstValue) / 27;
            String trytesValue = "9ABCDEFGHIJKLMNOPQRSTUVWXYZ".charAt(firstValue) + String.valueOf("9ABCDEFGHIJKLMNOPQRSTUVWXYZ".charAt(secondValue));
            trytes.append(trytesValue);
        }

        return trytes.toString();
    }

    public static byte[] toBytes(String inputTrytes) {
        if (inputTrytes.length() % 2 != 0) {
            return null;
        } else {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();

            for (int i = 0; i < inputTrytes.length(); i += 2) {
                int firstValue = "9ABCDEFGHIJKLMNOPQRSTUVWXYZ".indexOf(inputTrytes.charAt(i));
                int secondValue = "9ABCDEFGHIJKLMNOPQRSTUVWXYZ".indexOf(inputTrytes.charAt(i + 1));
                int decimalValue = firstValue + secondValue * 27;
                bos.write(decimalValue);

            }

            return bos.toByteArray();
        }
    }

    public static boolean isValidAddress(@NonNull String address) {
        return InputValidator.isAddress(address);
    }

    public static String randomTag() {
        char[] chars = Constants.TRYTE_ALPHABET.toCharArray();
        StringBuilder builder = new StringBuilder();
        SecureRandom random = new SecureRandom();
        for (int i = 0; i < Constants.TAG_LENGTH; i++) {
            char c = chars[random.nextInt(chars.length)];
            builder.append(c);
        }
        return builder.toString();
    }

    public static String createTag(byte[] basic) {
        checkNotNull(basic);
        checkArgument(basic.length >= 20);
        String trytes = IOTA.toTrytes(basic);
        return StringUtils.substring(trytes, 0, Constants.TAG_LENGTH);
    }

    @Nullable
    private static byte[] loadBundle(@NonNull Entity bundle) {
        checkNotNull(bundle);
        String bundleContent = bundle.getContent();
        if (!bundleContent.isEmpty()) {
            return IOTA.toBytes(bundleContent);
        }
        return null;
    }

    @NonNull
    public String getProtocol() {
        return protocol;
    }

    @NonNull
    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    private void postConstruct() {

        IotaAPI.Builder builder = new IotaAPI.Builder();
        builder = builder.protocol(protocol);
        builder = builder.host(host);
        builder = builder.port(port);
        builder = builder.timeout(timeout);
        iotaAPI = builder.build();

    }

    @Nullable
    public IotaLocalPoW getLocalPoW() {
        return iotaAPI.getLocalPoW();
    }

    public void setLocalPoW(@Nullable IotaLocalPoW localPoW) {
        iotaAPI.setLocalPoW(localPoW);
    }

    /**
     * @param filter Filter for accepting transactions
     * @param hashes Hashes of transactions
     */
    public void loadHashTransactions(@NonNull Filter filter,
                                     @NonNull String... hashes) {
        checkNotNull(filter);
        checkNotNull(hashes);
        loadHashes(filter, hashes);
    }

    /**
     * @param filter    Filter for accepting transactions
     * @param addresses Addresses of transactions
     */
    public void loadAddressTransactions(@NonNull Filter filter,
                                        @NonNull String... addresses) {
        checkNotNull(filter);
        checkNotNull(addresses);
        loadTransactions(filter, addresses);
    }

    /**
     * @param address Address of the transaction
     * @param data    Data as string
     * @return the entity bundle
     */
    @NonNull
    public Entity insertTransaction(@NonNull String address, @NonNull String data) {
        checkNotNull(address);
        checkNotNull(data);
        return insert(address, TrytesConverter.asciiToTrytes(data), true);
    }

    /**
     * Returns the tangle node info
     *
     * @return the tangle node info
     */
    @NonNull
    public GetNodeInfoResponse getNodeInfo() {
        return iotaAPI.getNodeInfo();
    }

    /**
     * @param filter    Filter for accepting transactions
     * @param addresses Addresses of the transactions
     */
    public void loadTransactions(@NonNull Filter filter, @NonNull String... addresses) {

        checkNotNull(addresses);
        checkNotNull(filter);

        FindTransactionResponse ftr = iotaAPI.findTransactions(
                addresses, null, null, null);


        List<String> findHashes = new ArrayList<>();
        if (ftr != null) {
            String[] hashes = ftr.getHashes();
            for (String hash : hashes) {
                if (filter.acceptHash(hash)) {
                    findHashes.add(hash);
                }
            }
        }

        loadHashes(filter, Iterables.toArray(findHashes, String.class));

    }

    @Nullable
    private Entity getEntity(@NonNull Transaction transaction, boolean trytesToAscii) {

        if (transaction.getLastIndex() == 0) { // only accept bundle with one transaction
            // now we can build the bundle
            String content = IOTA.adaptTrytes(transaction.getSignatureFragments());

            if (trytesToAscii) {
                content = TrytesConverter.trytesToAscii(content);
            }

            return new Entity(
                    transaction.getHash(),
                    transaction.getAddress(),
                    transaction.getTag(),
                    content);

        }

        return null;
    }


    @NonNull
    public Entity setContent(@NonNull String address, @NonNull byte[] bytes) {
        checkNotNull(address);

        checkNotNull(bytes);

        String data = IOTA.toTrytes(bytes);
        return insert(address, data, false);

    }

    @NonNull
    private Entity insert(@NonNull String address, @NonNull String dataTrits, boolean trytesToAscii) {
        checkNotNull(address);
        checkNotNull(dataTrits);

        synchronized (lock) {

            List<Transfer> transfers = new ArrayList<>();

            transfers.add(new Transfer(address, IOTA.ZERO_VALUE, dataTrits, IOTA.randomTag()));

            List<String> trytes = iotaAPI.prepareTransfers(
                    Constants.MIN_SECURITY_LEVEL, transfers, null, null, null,
                    false);
            checkArgument(trytes.size() == 1, "Trytes has not supported size");
            List<Transaction> transactions = iotaAPI.sendTrytes(
                    Iterables.toArray(trytes, String.class),
                    IOTA.DEPTH, MIN_WEIGHT_MAGNITUDE, null);
            checkArgument(transactions.size() == 1, "Transactions not valid");
            Transaction transaction = transactions.get(0);
            Entity entity = getEntity(transaction, trytesToAscii);
            checkNotNull(entity);
            return entity;
        }

    }

    /**
     * Returns the content of the first bundle with the given address.
     *
     * @param hash Hash of the transaction
     * @return the content as bytes when found
     */
    @Nullable
    public byte[] getContent(@NonNull String hash) {
        checkNotNull(hash);

        List<Transaction> transactions = iotaAPI.findTransactionsObjectsByHashes(hash);
        if (transactions != null && transactions.size() == 1) {
            Transaction transaction = transactions.get(0);
            Entity bundle = getEntity(transaction, false);
            if (bundle != null) {
                return IOTA.loadBundle(bundle);
            }
        }

        return null;
    }

    private void loadHashes(@NonNull Filter filter, @NonNull String... hashes) {


        List<Transaction> transactions =
                iotaAPI.findTransactionsObjectsByHashes(hashes);
        for (Transaction transaction : transactions) {
            Entity entity = getEntity(transaction, true);
            if (entity != null) {
                if (filter.acceptEntity(entity)) {
                    loadEntity(entity, filter);
                }
            } else {
                filter.invalidHash(transaction.getHash());
            }
        }

    }

    private void loadEntity(@NonNull Entity entity, @NonNull Filter filter) {
        try {
            String bundleContentTrytes = entity.getContent();
            if (!bundleContentTrytes.isEmpty()) {
                filter.loadEntity(entity);
            } else {
                filter.invalidEntity(entity);
            }
        } catch (Throwable e) {
            filter.error(entity, e);
        }
    }

    public interface Filter {


        /**
         * This function will be invoked when a transaction is loaded.
         * Then the hash of the transaction is returned before loading it,
         * to prevent it from reading it again.
         *
         * @param hash hash of the transaction to be loaded
         * @return true, when the transaction hash should be accepted
         */
        boolean acceptHash(@NonNull String hash);

        /**
         * Invalid hash detected (Probably more then one transaction in bundle)
         *
         * @param hash Transaction hash object
         */
        void invalidHash(@NonNull String hash);

        /**
         * This function will be invoked when the entity object
         * is not fully created. Here additional checks can be implemented
         * whether to load the entity.
         *
         * @param entity The entity is not fully constructed yet.
         * @return true, when the entity should be accepted
         */
        boolean acceptEntity(@NonNull Entity entity);

        /**
         * This function will be invoked when the entity bundle object
         * is fully created.
         *
         * @param entity The entity is fully created. It contains the content and also additional
         *               information about the bundle and transactions.
         */
        void loadEntity(@NonNull Entity entity);

        /**
         * Invalid entity detected
         *
         * @param entity Entity object
         */
        void invalidEntity(@NonNull Entity entity);

        /**
         * Exception occeur during loading of entity.
         *
         * @param entity Entity object
         * @param e      exception which occeurs during loading of entity.
         */
        void error(@NonNull Entity entity, @NonNull Throwable e);


    }


    public static class Builder {
        private String protocol;
        private String host;
        private int port;
        private int timeout = 150;

        public IOTA build() {
            checkNotNull(protocol);
            checkNotNull(host);
            checkNotNull(port);
            checkArgument(timeout >= 0);
            return new IOTA(this);
        }


        public Builder timeout(int timeout) {
            this.timeout = timeout;
            return this;
        }

        public Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        public Builder host(String host) {
            this.host = host;
            return this;
        }

        public Builder port(int port) {
            this.port = port;
            return this;
        }


    }


}
