package threads.iota;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.room.Room;

import java.util.ArrayList;
import java.util.List;

import static androidx.core.util.Preconditions.checkNotNull;

public class EntityService {
    private static final String TAG = EntityService.class.getSimpleName();
    private static EntityService SINGLETON = null;
    @NonNull
    private final HashDatabase hashDatabase;

    private EntityService(@NonNull Context context) {
        checkNotNull(context);

        hashDatabase = Room.databaseBuilder(context,
                HashDatabase.class,
                HashDatabase.class.getSimpleName()).fallbackToDestructiveMigration().build();
    }

    @NonNull
    public static EntityService getInstance(@NonNull Context context) {
        checkNotNull(context);
        if (SINGLETON == null) {
            SINGLETON = new EntityService(context);
        }
        return SINGLETON;
    }

    @NonNull
    public HashDatabase getHashDatabase() {
        return hashDatabase;
    }

    @NonNull
    public List<Entity> loadEntities(@NonNull String address,
                                     @NonNull String protocol,
                                     @NonNull String host,
                                     int port, int timeout) {

        IOTA iota = new IOTA.Builder().
                protocol(protocol).
                timeout(timeout).
                host(host).
                port(port).build();

        return loadEntities(iota, address);
    }

    @NonNull
    public List<Entity> loadEntities(@NonNull IOTA iota, @NonNull String address) {
        checkNotNull(iota);
        checkNotNull(address);

        List<Entity> entities = new ArrayList<>();


        iota.loadAddressTransactions(new IOTA.Filter() {
            @Override
            public boolean acceptHash(@NonNull String hash) {
                return !hasHash(hash);
            }

            @Override
            public void invalidHash(@NonNull String hash) {
                Hash transactionHash = createHash(hash);
                insertHash(transactionHash);
            }

            @Override
            public boolean acceptEntity(@NonNull Entity entity) {
                boolean result = true;
                String hash = entity.getHash();
                if (hasHash(hash)) {
                    result = false;
                } else {
                    Hash transactionHash = createHash(hash);
                    insertHash(transactionHash);
                }
                return result;

            }

            @Override
            public void loadEntity(@NonNull Entity entity) {
                entities.add(entity);
            }

            @Override
            public void invalidEntity(@NonNull Entity entity) {
                String hash = entity.getHash();
                Hash transactionHash = createHash(hash);
                insertHash(transactionHash);
            }

            @Override
            public void error(@NonNull Entity entity, @NonNull Throwable e) {
                String hash = entity.getHash();
                Hash transactionHash = createHash(hash);
                insertHash(transactionHash);
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }, address);

        return entities;
    }


    private boolean hasHash(@NonNull String hash) {
        return getHashDatabase().hashDao().hasHash(hash) > 0;
    }


    private Hash createHash(@NonNull String hash) {
        checkNotNull(hash);
        return new Hash(hash, System.currentTimeMillis());
    }

    private void insertHash(@NonNull Hash hash) {
        checkNotNull(hash);
        getHashDatabase().hashDao().insertHash(hash);
    }
}
