package threads.core;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.apache.commons.lang3.RandomStringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import threads.core.api.Additional;
import threads.core.api.Additionals;
import threads.core.api.AddressType;
import threads.core.api.Peer;
import threads.core.api.PeerInfoEncoder;
import threads.core.api.User;
import threads.iota.Entity;
import threads.iota.EntityService;
import threads.iota.IOTA;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;
import threads.ipfs.api.PeerInfo;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class IdentityService {

    public static final String TAG = IdentityService.class.getSimpleName();

    @Nullable
    public static ResultInfo publishIdentity(
            @NonNull Context context, @NonNull String aesKey, @NonNull Map<String, String> params,
            boolean addIdentity, int numRelays) {

        checkNotNull(context);
        checkNotNull(aesKey);
        checkNotNull(params);

        checkArgument(numRelays >= 0);

        if (!Network.isConnected(context)) {
            return null;
        }

        int timeout = 3;
        // first load stored relays
        GatewayService.connectStoredRelays(context, numRelays, timeout);


        final THREADS threads = Singleton.getInstance(context).getThreads();

        try {

            PID host = Preferences.getPID(context);
            if (host != null) {
                String tag = RandomStringUtils.randomAlphabetic(10);

                PeerInfo info = null;

                if (addIdentity) {
                    IPFS ipfs = Singleton.getInstance(context).getIpfs();
                    if (ipfs != null) {
                        info = ipfs.id();
                    }
                }

                threads.core.api.PeerInfo peer = threads.getPeerInfoByPID(host);
                if (peer != null) {
                    return updatePeerInfo(context, peer, info, aesKey, params, tag,
                            timeout, numRelays);

                } else {
                    return createPeerInfo(context, info, host, aesKey, params, tag,
                            timeout, numRelays);
                }
            }

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }

        return null;
    }

    private static List<String> getMultiAddresses(@NonNull PeerInfo info) {
        checkNotNull(info);
        List<String> addresses = new ArrayList<>();
        for (String address : info.getMultiAddresses()) {
            if (!address.startsWith("/ip6/::1/") && !address.startsWith("/ip4/127.0.0.1/")) {
                addresses.add(address);
            }
        }
        return addresses;
    }

    private static ResultInfo createPeerInfo(
            @NonNull Context context,
            @Nullable PeerInfo info,
            @NonNull PID user,
            @NonNull String aesKey,
            @NonNull Map<String, String> params,
            @NonNull String tag,
            int timeout,
            int numPeers) {

        final THREADS threads = Singleton.getInstance(context).getThreads();

        threads.core.api.PeerInfo peerInfo = threads.createPeerInfo(user);


        List<Peer> relayPeers = GatewayService.getRelayPeers(
                context, tag, numPeers, timeout);
        for (Peer relay : relayPeers) {
            peerInfo.addAddress(relay.getPid(), relay.getMultiAddress());
        }

        for (Map.Entry<String, String> entry : params.entrySet()) {
            peerInfo.addAdditional(entry.getKey(), entry.getValue(), false);
        }

        if (info != null) {
            List<String> multiAddresses = getMultiAddresses(info);
            for (String address : multiAddresses) {
                peerInfo.addMultiAddresses(address);
            }
        }
        threads.storePeerInfo(peerInfo);

        return new ResultInfo(peerInfo, tag, insertPeer(context, peerInfo, aesKey));


    }

    private static boolean insertPeer(@NonNull Context context,
                                      @NonNull threads.core.api.PeerInfo peer,
                                      @NonNull String aesKey) {

        final THREADS threads = Singleton.getInstance(context).getThreads();
        threads.setHash(peer, null);

        long start = System.currentTimeMillis();

        boolean success = insertPeerInfo(context, threads, peer, aesKey);

        long time = (System.currentTimeMillis() - start) / 1000;
        if (success) {
            Singleton.getInstance(context).getConsoleListener().info(
                    "Success store peer discovery information: " + time + " [s]");
        } else {
            Singleton.getInstance(context).getConsoleListener().error(
                    "Failed store peer discovery information: " + time + " [s]");
        }
        return success;
    }

    public static boolean insertPeerInfo(@NonNull Context context,
                                         @NonNull THREADS threads,
                                         @NonNull threads.core.api.PeerInfo peer,
                                         @NonNull String aesKey) {
        checkNotNull(context);
        checkNotNull(threads);
        checkNotNull(peer);
        checkNotNull(aesKey);
        try {
            String address = AddressType.getAddress(peer.getPID(), AddressType.PEER);

            String data = PeerInfoEncoder.convert(peer, aesKey);

            EntityService entityService = EntityService.getInstance(context);
            checkNotNull(entityService);
            Entity entity = entityService.insertData(context, address, data);
            checkNotNull(entity);

            String hash = entity.getHash();
            threads.setHash(peer, hash);
            return true;
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
            return false;
        }
    }

    public static threads.core.api.PeerInfo getPeerInfo(@NonNull Context context,
                                                        @NonNull PID pid,
                                                        @NonNull String aesKey,
                                                        boolean updateUser) {
        checkNotNull(context);
        checkNotNull(pid);
        checkNotNull(aesKey);
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        threads.core.api.PeerInfo peerInfo = threads.getPeer(iota, pid, aesKey);
        if (peerInfo != null && updateUser) {
            boolean update = false;
            User user = threads.getUserByPID(pid);
            if (user != null) {
                Additionals additionals = peerInfo.getAdditionals();
                for (String key : additionals.keySet()) {
                    Additional additional = additionals.get(key);
                    String value = additional.getValue();
                    user.addAdditional(key, value, true);
                    update = true;
                }
            }
            if (update) {
                threads.updateUser(user);
            }
        }
        return peerInfo;
    }


    private static ResultInfo updatePeerInfo(
            @NonNull Context context,
            @NonNull threads.core.api.PeerInfo peerInfo,
            @Nullable PeerInfo info,
            @NonNull String aesKey,
            @NonNull Map<String, String> params,
            @NonNull String tag,
            int timeout,
            int numPeers) {
        checkArgument(timeout > 0);
        final THREADS threads = Singleton.getInstance(context).getThreads();

        peerInfo.removeAddresses();
        List<Peer> peers = GatewayService.getRelayPeers(context, tag, numPeers, timeout);
        for (Peer relay : peers) {
            peerInfo.addAddress(relay.getPid(), relay.getMultiAddress());
        }


        peerInfo.removeAdditionals();
        for (Map.Entry<String, String> entry : params.entrySet()) {
            peerInfo.addAdditional(entry.getKey(), entry.getValue(), false);
        }

        peerInfo.removeMultiAddresses();
        if (info != null) {
            List<String> multiAddresses = getMultiAddresses(info);
            for (String address : multiAddresses) {
                peerInfo.addMultiAddresses(address);
            }
        }

        threads.updatePeerInfo(peerInfo);
        return new ResultInfo(peerInfo, tag, insertPeer(context, peerInfo, aesKey));
    }


    public static class ResultInfo {

        @NonNull
        private final String tag;
        private final boolean insert;
        @NonNull
        private final threads.core.api.PeerInfo peerInfo;

        ResultInfo(@NonNull threads.core.api.PeerInfo peerInfo,
                   @NonNull String tag, boolean insert) {

            this.peerInfo = peerInfo;
            this.tag = tag;
            this.insert = insert;
        }

        @NonNull
        public String getTag() {
            return tag;
        }

        @NonNull
        public threads.core.api.PeerInfo getPeerInfo() {
            return peerInfo;
        }

        public boolean isInsert() {
            return insert;
        }


    }

}
