package threads.core;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import threads.core.api.Additional;
import threads.core.api.Additionals;
import threads.core.api.Peer;
import threads.core.api.User;
import threads.iota.IOTA;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;
import threads.ipfs.api.PeerInfo;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class IdentityService {


    public static boolean publishIdentity(@NonNull Context context,
                                          @NonNull String aesKey,
                                          @NonNull Map<String, String> params,
                                          boolean addIdentity,
                                          int timeout,
                                          int numPeers,
                                          boolean protectPeers) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkNotNull(params);

        checkArgument(numPeers >= 0);
        if (!Network.isConnected(context)) {
            return false;
        }


        final THREADS threads = Singleton.getInstance(context).getThreads();

        try {

            PID host = Preferences.getPID(context);
            if (host != null) {


                PeerInfo info = null;

                if (addIdentity) {
                    IPFS ipfs = Singleton.getInstance(context).getIpfs();
                    if (ipfs != null) {
                        info = ipfs.id();
                    }
                }

                threads.core.api.PeerInfo peer = threads.getPeerInfoByPID(host);
                if (peer != null) {
                    return updatePeerInfo(context, peer, info, aesKey, params,
                            timeout, numPeers, protectPeers);

                } else {
                    return createPeerInfo(context, info, host, aesKey, params,
                            timeout, numPeers, protectPeers);
                }
            }

        } catch (Throwable e) {
            Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
        }

        return false;
    }

    private static List<String> getMultiAddresses(@NonNull PeerInfo info) {
        checkNotNull(info);
        List<String> addresses = new ArrayList<>();
        for (String address : info.getMultiAddresses()) {
            if (!address.startsWith("/ip6/::1/") && !address.startsWith("/ip4/127.0.0.1/")) {
                addresses.add(address);
            }
        }
        return addresses;
    }

    private static boolean createPeerInfo(@NonNull Context context,
                                          @Nullable PeerInfo info,
                                          @NonNull PID user,
                                          @NonNull String aesKey,
                                          @NonNull Map<String, String> params,
                                          int timeout,
                                          int numPeers,
                                          boolean protectRelays) {

        final THREADS threads = Singleton.getInstance(context).getThreads();

        threads.core.api.PeerInfo peerInfo = threads.createPeerInfo(user);


        List<Peer> relayPeers = GatewayService.getRelayPeers(
                context, numPeers, timeout, protectRelays);
        for (Peer relay : relayPeers) {
            peerInfo.addAddress(relay.getPid(), relay.getMultiAddress());
        }

        for (Map.Entry<String, String> entry : params.entrySet()) {
            peerInfo.addAdditional(entry.getKey(), entry.getValue(), false);
        }

        if (info != null) {
            List<String> multiAddresses = getMultiAddresses(info);
            for (String address : multiAddresses) {
                peerInfo.addMultiAddresses(address);
            }
        }
        threads.storePeerInfo(peerInfo);

        return insertPeer(context, peerInfo, aesKey);


    }

    private static boolean insertPeer(@NonNull Context context,
                                      @NonNull threads.core.api.PeerInfo peer,
                                      @NonNull String aesKey) {

        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        checkNotNull(iota);
        threads.setHash(peer, null);

        long start = System.currentTimeMillis();

        boolean success = threads.insertPeerInfo(iota, peer, aesKey);

        long time = (System.currentTimeMillis() - start) / 1000;
        if (success) {
            Singleton.getInstance(context).getConsoleListener().info(
                    "Success store peer discovery information: " + time + " [s]");
        } else {
            Singleton.getInstance(context).getConsoleListener().error(
                    "Failed store peer discovery information: " + time + " [s]");
        }
        return success;
    }

    public static threads.core.api.PeerInfo getPeerInfo(@NonNull Context context,
                                                        @NonNull PID pid,
                                                        @NonNull String aesKey,
                                                        boolean updateUser) {
        checkNotNull(context);
        checkNotNull(pid);
        checkNotNull(aesKey);
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        threads.core.api.PeerInfo peerInfo = threads.getPeer(iota, pid, aesKey);
        if (peerInfo != null && updateUser) {
            boolean update = false;
            User user = threads.getUserByPID(pid);
            if (user != null) {
                Additionals additionals = peerInfo.getAdditionals();
                for (String key : additionals.keySet()) {
                    Additional additional = additionals.get(key);
                    String value = additional.getValue();
                    user.addAdditional(key, value, true);
                    update = true;
                }
            }
            if (update) {
                threads.updateUser(user);
            }
        }
        return peerInfo;
    }


    private static boolean updatePeerInfo(
            @NonNull Context context,
            @NonNull threads.core.api.PeerInfo peerInfo,
            @Nullable PeerInfo info,
            @NonNull String aesKey,
            @NonNull Map<String, String> params,
            int timeout,
            int numPeers,
            boolean protectPeers) {
        checkArgument(timeout > 0);
        final THREADS threads = Singleton.getInstance(context).getThreads();

        peerInfo.removeAddresses();
        List<Peer> peers = GatewayService.getRelayPeers(context, numPeers, timeout, protectPeers);
        for (Peer relay : peers) {
            peerInfo.addAddress(relay.getPid(), relay.getMultiAddress());
        }


        peerInfo.removeAdditionals();
        for (Map.Entry<String, String> entry : params.entrySet()) {
            peerInfo.addAdditional(entry.getKey(), entry.getValue(), false);
        }

        peerInfo.removeMultiAddresses();
        if (info != null) {
            List<String> multiAddresses = getMultiAddresses(info);
            for (String address : multiAddresses) {
                peerInfo.addMultiAddresses(address);
            }
        }

        threads.updatePeerInfo(peerInfo);
        return insertPeer(context, peerInfo, aesKey);
    }


}
