package threads.core.api;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.gson.Gson;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import threads.core.MimeType;
import threads.core.Preferences;
import threads.core.THREADS;
import threads.iota.Entity;
import threads.iota.EntityService;
import threads.iota.Hash;
import threads.iota.IOTA;
import threads.iota.IOTA.Filter;
import threads.ipfs.IPFS;
import threads.ipfs.api.CID;
import threads.ipfs.api.Multihash;
import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;


public class ThreadsAPI {
    private final static String TAG = ThreadsAPI.class.getSimpleName();
    private final Gson gson = new Gson();
    private final ThreadsDatabase threadsDatabase;
    private final EventsDatabase eventsDatabase;
    private final PeersInfoDatabase peersInfoDatabase;
    private final PeersDatabase peersDatabase;
    private final EntityService entityService;

    public ThreadsAPI(@NonNull ThreadsDatabase threadsDatabase,
                      @NonNull EventsDatabase eventsDatabase,
                      @NonNull PeersInfoDatabase peersInfoDatabase,
                      @NonNull PeersDatabase peersDatabase,
                      @NonNull EntityService entityService) {
        checkNotNull(threadsDatabase);
        checkNotNull(eventsDatabase);
        checkNotNull(peersInfoDatabase);
        checkNotNull(peersDatabase);
        checkNotNull(entityService);
        this.threadsDatabase = threadsDatabase;
        this.eventsDatabase = eventsDatabase;
        this.peersInfoDatabase = peersInfoDatabase;
        this.peersDatabase = peersDatabase;
        this.entityService = entityService;
    }

    @NonNull
    private PeersDatabase getPeersDatabase() {
        return peersDatabase;
    }

    @NonNull
    private PeersInfoDatabase getPeersInfoDatabase() {
        return peersInfoDatabase;
    }

    @NonNull
    private EventsDatabase getEventsDatabase() {
        return eventsDatabase;
    }

    @NonNull
    private ThreadsDatabase getThreadsDatabase() {
        return threadsDatabase;
    }


    /**
     * Generates a random address
     *
     * @return a random address for a specific address usage
     */
    @NonNull
    public String getRandomAddress() {
        return IOTA.generateAddress();
    }


    public int getUnreadNotes() {
        return getThreadsDatabase().threadDao().getUnreadNotes();
    }

    public void setThreadStatus(long idx, @NonNull ThreadStatus status) {
        checkNotNull(status);
        getThreadsDatabase().threadDao().setThreadStatus(idx, status);
    }

    public void setThreadsStatus(@NonNull ThreadStatus status, long... idxs) {
        checkNotNull(status);
        getThreadsDatabase().threadDao().setThreadsStatus(status, idxs);
    }

    public void setThreadStatus(@NonNull ThreadStatus oldStatus, @NonNull ThreadStatus newStatus) {
        checkNotNull(oldStatus);
        getThreadsDatabase().threadDao().setThreadStatus(oldStatus, newStatus);
    }

    public void setUserStatus(@NonNull UserStatus oldStatus, @NonNull UserStatus newStatus) {
        checkNotNull(oldStatus);
        getThreadsDatabase().userDao().setUserStatus(oldStatus, newStatus);
    }

    public void setNoteStatus(@NonNull NoteStatus oldStatus, @NonNull NoteStatus newStatus) {
        checkNotNull(oldStatus);
        getThreadsDatabase().noteDao().setNoteStatus(oldStatus, newStatus);
    }

    public void setNoteStatus(long idx, @NonNull NoteStatus status) {
        checkNotNull(status);
        getThreadsDatabase().noteDao().setNoteStatus(idx, status);
    }

    public void setNotesStatus(@NonNull NoteStatus status, long... idxs) {
        checkNotNull(status);
        getThreadsDatabase().noteDao().setNotesStatus(status, idxs);
    }


    public void setImage(@NonNull Thread thread, @NonNull CID image) {
        checkNotNull(thread);
        checkNotNull(image);
        getThreadsDatabase().threadDao().setImage(thread.getIdx(), image);
    }


    public void setImage(@NonNull User user, @NonNull CID image) {
        checkNotNull(user);
        checkNotNull(image);
        getThreadsDatabase().userDao().setImage(user.getPid(), image);
    }

    public void setImage(@NonNull Note note, @NonNull CID image) {
        checkNotNull(note);
        checkNotNull(image);
        getThreadsDatabase().noteDao().setImage(note.getIdx(), image);
    }

    @NonNull
    public String getMimeType(@NonNull Thread thread) {
        checkNotNull(thread);
        return getThreadsDatabase().threadDao().getMimeType(thread.getIdx());
    }

    @NonNull
    public String getMimeType(@NonNull Note note) {
        checkNotNull(note);
        return getThreadsDatabase().noteDao().getMimeType(note.getIdx());
    }


    @Nullable
    public String getThreadMimeType(long idx) {
        return getThreadsDatabase().threadDao().getMimeType(idx);
    }

    @Nullable
    public String getNoteMimeType(long idx) {
        return getThreadsDatabase().noteDao().getMimeType(idx);
    }

    public void setThreadSenderAlias(@NonNull PID pid, @NonNull String alias) {
        checkNotNull(pid);
        checkNotNull(alias);
        getThreadsDatabase().threadDao().setSenderAlias(pid, alias);
    }

    public void setNoteSenderAlias(@NonNull PID pid, @NonNull String alias) {
        checkNotNull(pid);
        checkNotNull(alias);
        getThreadsDatabase().noteDao().setSenderAlias(pid, alias);
    }

    public void setMimeType(@NonNull Thread thread, @NonNull String mimeType) {
        checkNotNull(thread);
        checkNotNull(mimeType);
        getThreadsDatabase().threadDao().setMimeType(thread.getIdx(), mimeType);
    }

    public void setMimeType(@NonNull Note note, @NonNull String mimeType) {
        checkNotNull(note);
        checkNotNull(mimeType);
        getThreadsDatabase().noteDao().setMimeType(note.getIdx(), mimeType);
    }


    @NonNull
    public List<Note> getNotes(@NonNull Thread thread) {
        checkNotNull(thread);
        return getNotesByThread(thread.getCid());
    }

    @NonNull
    private List<Note> getNotesByThread(@Nullable CID thread) {
        return getThreadsDatabase().noteDao().getNotesByThread(thread);
    }

    @NonNull
    public List<Note> loadNotes(@NonNull IOTA iota,
                                @NonNull User user,
                                @NonNull ThreadStatus threadStatus) {
        checkNotNull(iota);
        checkNotNull(user);
        checkNotNull(threadStatus);

        List<Note> notes = new ArrayList<>();

        List<Thread> threads = getThreadsByThreadStatus(threadStatus);

        for (Thread thread : threads) {
            notes.addAll(loadNotes(iota, thread));
        }
        return notes;
    }


    public boolean isUserBlocked(@NonNull PID user) {
        checkNotNull(user);
        return isUserBlocked(user.getPid());
    }

    public boolean isUserBlocked(@NonNull String pid) {
        checkNotNull(pid);
        return getThreadsDatabase().userDao().isBlocked(pid);
    }

    @NonNull

    public List<Thread> loadPublishedThreads(@NonNull IOTA iota,
                                             @NonNull String aesKey,
                                             @NonNull Integer maxNumberOfLoadedThreads,
                                             @NonNull String... addresses) {
        checkNotNull(iota);
        checkNotNull(addresses);
        checkNotNull(aesKey);
        checkNotNull(maxNumberOfLoadedThreads);
        checkArgument(maxNumberOfLoadedThreads > 0);

        AtomicBoolean shouldContinue = new AtomicBoolean(true);
        AtomicInteger maxNum = new AtomicInteger(maxNumberOfLoadedThreads);

        AtomicInteger counter = new AtomicInteger(0);

        List<Thread> threads = new ArrayList<>();
        try {
            iota.loadTransactions(new Filter() {

                @Override
                public boolean acceptHash(@NonNull String hash) {
                    if (!shouldContinue.get()) {
                        return false;
                    }
                    return !hasHash(hash);
                }

                @Override
                public void invalidHash(@NonNull String hash) {
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);
                }

                @Override
                public boolean acceptEntity(@NonNull Entity entity) {

                    boolean result = true;
                    String hash = entity.getHash();
                    if (hasHash(hash)) {
                        result = false;
                    } else {
                        Hash transactionHash =
                                createHash(hash);
                        insertHash(transactionHash);
                    }


                    return result;
                }

                @Override
                public void invalidEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void error(@NonNull Entity entity, @NonNull Throwable e) {
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void loadEntity(@NonNull Entity entity) {
                    if (!shouldContinue.get()) {
                        return;
                    }
                    Thread thread = NotePublishDecoder.convert(entity, aesKey);
                    if (thread != null) {
                        if (isUserBlocked(thread.getSenderPid())) {
                            return;
                        }

                        if (existsSameThread(thread)) { // maybe not yet necessary
                            return;
                        }

                        Date today = THREADS.getToday();
                        if (thread.getExpireDate().after(today)) {
                            threads.add(thread);
                            if (counter.incrementAndGet() > maxNum.get()) {
                                shouldContinue.set(false); // max num is reached
                            }
                        }
                    }

                }

            }, addresses);
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }

        return threads;

    }


    public boolean publishThreadRequestNote(@NonNull IPFS ipfs,
                                            @NonNull User user,
                                            @NonNull Note note) {

        checkNotNull(ipfs);
        checkNotNull(user);
        checkNotNull(note);


        checkArgument(note.getKind() == Kind.OUT);
        checkArgument(note.getNoteType() == NoteType.THREAD_REQUEST);
        checkArgument(!user.getPid().isEmpty());

        try {
            Content content = NoteRequestEncoder.convert(
                    user.getPublicKey(), note);
            checkNotNull(content);
            // NOT ENCRYPTED
            NoteType type = note.getNoteType();
            content.put(Content.EST, String.valueOf(type.getCode()));


            String hash = getHash(note);
            if (hash != null) {
                content.put(Content.HASH, hash);
            }
            sendNotification(ipfs, user.getPID(), content);

            return true;
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
            return false;
        }
    }


    public boolean insertThreadRequestNote(@NonNull IOTA iota,
                                           @NonNull User user,
                                           @NonNull Note note) {
        checkNotNull(iota);
        checkNotNull(user);
        checkNotNull(note);

        checkArgument(note.getKind() == Kind.OUT);
        checkArgument(note.getNoteType() == NoteType.THREAD_REQUEST);
        checkArgument(!user.getPid().isEmpty());

        try {

            Content content = NoteRequestEncoder.convert(user.getPublicKey(), note);
            checkNotNull(content);
            String address = AddressType.getAddress(user.getPID(), AddressType.INBOX);
            checkNotNull(address);
            Entity entity = iota.insertTransaction(address, gson.toJson(content));
            checkNotNull(entity);
            String hash = entity.getHash();

            Hash transactionHash = createHash(hash);
            insertHash(transactionHash);

            setHash(note, hash);

            return true;
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
            return false;
        }
    }


    public boolean insertThreadPublishNote(@NonNull IOTA iota,
                                           @NonNull Note note,
                                           @NonNull String address,
                                           @NonNull String aesKey) {
        checkNotNull(iota);
        checkNotNull(note);
        checkNotNull(address);
        checkNotNull(aesKey);


        checkArgument(note.getKind() == Kind.OUT);
        checkArgument(note.getNoteType() == NoteType.THREAD_PUBLISH);

        try {
            Thread thread = getThread(note);
            checkNotNull(thread);

            String dataTransaction = NotePublishEncoder.convert(thread, aesKey);

            Entity entity = iota.insertTransaction(address, dataTransaction);
            checkNotNull(entity);
            String hash = entity.getHash();

            Hash transactionHash =
                    createHash(hash);
            insertHash(transactionHash);

            setHash(note, hash);
            return true;
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
            return false;
        }
    }


    public boolean insertNote(@NonNull IOTA iota,
                              @NonNull Note note) {
        checkNotNull(iota);
        checkNotNull(note);

        checkArgument(note.getKind() == Kind.OUT);
        checkArgument(note.getNoteType() != NoteType.THREAD_REQUEST);
        checkArgument(note.getNoteType() != NoteType.THREAD_PUBLISH);
        checkArgument(note.getNoteType() != NoteType.HTML);
        checkArgument(note.getNoteType() != NoteType.INFO);

        try {

            Thread thread = getThread(note);
            checkNotNull(thread);
            CID cid = thread.getCid();
            checkNotNull(cid);
            String address = THREADS.getAddress(cid);

            Content content = NoteEncoder.convert(note);

            Entity entity = iota.insertTransaction(address, gson.toJson(content));
            checkNotNull(entity);
            String hash = entity.getHash();

            Hash transactionHash = createHash(hash);
            insertHash(transactionHash);

            setHash(note, hash);

            return true;
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
            return false;
        }
    }

    public boolean publishNote(@NonNull IPFS ipfs, @NonNull String topic, @NonNull Note note) {
        checkNotNull(ipfs);
        checkNotNull(topic);
        checkNotNull(note);

        checkArgument(note.getKind() == Kind.OUT);
        checkArgument(note.getNoteType() != NoteType.THREAD_REQUEST);
        checkArgument(note.getNoteType() != NoteType.THREAD_PUBLISH);
        checkArgument(note.getNoteType() != NoteType.HTML);
        checkArgument(note.getNoteType() != NoteType.INFO);

        try {
            Content content = NoteEncoder.convert(note);
            String message = gson.toJson(content);
            ipfs.pubsubPub(topic, message, 50);
            return true;
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
            return false;
        }
    }

    public boolean publishNote(@NonNull IPFS ipfs, @NonNull PID user, @NonNull Note note) {
        checkNotNull(ipfs);
        checkNotNull(note);
        checkNotNull(user);

        checkArgument(note.getKind() == Kind.OUT);
        checkArgument(note.getNoteType() != NoteType.THREAD_REQUEST);
        checkArgument(note.getNoteType() != NoteType.THREAD_PUBLISH);
        checkArgument(note.getNoteType() != NoteType.HTML);
        checkArgument(note.getNoteType() != NoteType.INFO);


        try {
            Content content = NoteEncoder.convert(note);

            String hash = getHash(note);
            if (hash != null) {
                content.put(Content.HASH, hash);
            }
            sendNotification(ipfs, user, content);
            return true;
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
            return false;
        }
    }


    @NonNull
    public Thread createThread(@NonNull User user,
                               @NonNull ThreadStatus status,
                               @NonNull Kind kind,
                               @NonNull String sesKey,
                               @Nullable CID cid,
                               long thread) {

        checkNotNull(user);
        checkNotNull(status);
        checkNotNull(kind);
        checkNotNull(sesKey);
        checkArgument(thread >= 0);
        Thread result = createThread(
                status,
                kind,
                user.getPID(),
                user.getAlias(),
                user.getPublicKey(),
                sesKey,
                new Date(),
                thread);

        result.setCid(cid);
        return result;
    }


    /**
     * Thread request note to the given sender
     *
     * @param sender Sender user which creates the thread
     * @param thread Thread the message belongs too
     * @return a note to the sender which is not yet added to the database and
     * not insert into the tangle
     */
    @NonNull
    public Note createThreadRequestNote(@NonNull User sender, @NonNull Thread thread) {
        checkNotNull(sender);
        checkNotNull(thread);

        Note note = createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.THREAD_REQUEST,
                thread.getCid(),
                MimeType.PLAIN_MIME_TYPE,
                new Date());

        HashMap<String, String> externals = thread.getExternalAdditions();
        for (String key : externals.keySet()) {
            String value = externals.get(key);
            if (value != null) {
                note.addAdditional(key, value, false);
            }

        }
        return note;
    }


    @NonNull
    public Note createThreadRejectNote(@NonNull User sender,
                                       @NonNull Thread thread) {
        checkNotNull(sender);
        checkNotNull(thread);

        return createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.THREAD_REJECT,
                null,
                MimeType.PLAIN_MIME_TYPE,
                new Date());

    }


    @NonNull
    public Note createThreadJoinNote(@NonNull User sender,
                                     @NonNull Thread thread) {
        checkNotNull(sender);
        checkNotNull(thread);

        return createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.THREAD_JOIN,
                null,
                MimeType.PLAIN_MIME_TYPE,
                new Date());

    }


    @NonNull
    public Note createThreadPublishNote(@NonNull User sender, @NonNull Thread thread) {
        checkNotNull(sender);
        checkNotNull(thread);

        return createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.THREAD_PUBLISH,
                thread.getCid(),
                MimeType.PLAIN_MIME_TYPE,
                new Date());

    }


    @NonNull
    public Note createThreadLeaveNote(@NonNull User sender,
                                      @NonNull Thread thread) {
        checkNotNull(sender);
        checkNotNull(thread);


        return createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.THREAD_LEAVE,
                null,
                MimeType.PLAIN_MIME_TYPE,
                new Date());

    }


    @NonNull
    public Note createHtmlNote(@NonNull User sender,
                               @NonNull Thread thread,
                               @Nullable CID cid) {
        checkNotNull(sender);
        checkNotNull(thread);

        return createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.HTML,
                cid,
                MimeType.PLAIN_MIME_TYPE,
                new Date());

    }


    @NonNull
    public Note createMessageNote(@NonNull User sender,
                                  @NonNull Thread thread,
                                  @NonNull CID cid) {
        checkNotNull(sender);
        checkNotNull(thread);


        return createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.MESSAGE,
                cid,
                MimeType.PLAIN_MIME_TYPE,
                new Date());

    }

    @NonNull
    public Note createVideoCallNote(@NonNull User sender, @NonNull Thread thread) {
        checkNotNull(sender);
        checkNotNull(thread);


        return createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.VIDEO_CALL,
                null,
                MimeType.PLAIN_MIME_TYPE,
                new Date());

    }

    @NonNull
    public Note createCallNote(@NonNull User sender, @NonNull Thread thread) {
        checkNotNull(sender);
        checkNotNull(thread);


        return createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.CALL,
                null,
                MimeType.PLAIN_MIME_TYPE,
                new Date());

    }

    @NonNull
    public Note createLocationNote(@NonNull User sender,
                                   @NonNull Thread thread,
                                   double latitude,
                                   double longitude,
                                   double zoom) {
        checkNotNull(sender);
        checkNotNull(thread);

        Note note = createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.LOCATION,
                null,
                MimeType.GEO_MIME_TYPE,
                new Date());

        note.addAdditional(Preferences.LATITUDE, String.valueOf(latitude), false);
        note.addAdditional(Preferences.LONGITUDE, String.valueOf(longitude), false);
        note.addAdditional(Preferences.ZOOM, String.valueOf(zoom), false);

        return note;
    }

    @NonNull
    public Note createLinkNote(@NonNull User sender,
                               @NonNull Thread thread,
                               @NonNull LinkType linkType,
                               @Nullable CID cid,
                               @NonNull String fileName,
                               @NonNull String mimeType,
                               long fileSize) {
        checkNotNull(sender);
        checkNotNull(thread);
        checkNotNull(linkType);
        checkNotNull(fileName);
        checkNotNull(mimeType);

        Note note = createNote(
                thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.LINK,
                cid,
                mimeType,
                new Date());
        note.addAdditional(LinkType.class.getSimpleName(), linkType.name(), false);
        note.addAdditional(Content.FILENAME, fileName, false);
        note.addAdditional(Content.FILESIZE, String.valueOf(fileSize), false);

        return note;
    }


    @NonNull
    public Note createDataNote(@NonNull User sender,
                               @NonNull Thread thread,
                               @NonNull String mimeType,
                               @Nullable CID cid) {
        checkNotNull(sender);
        checkNotNull(thread);
        checkNotNull(mimeType);

        return createNote(thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.DATA,
                cid,
                mimeType,
                new Date());

    }


    @NonNull
    public Note createAudioNote(@NonNull User sender,
                                @NonNull Thread thread,
                                @NonNull String mimeType,
                                @Nullable CID cid) {
        checkNotNull(sender);
        checkNotNull(thread);
        checkNotNull(mimeType);

        return createNote(thread.getCid(),
                sender.getPID(),
                sender.getAlias(),
                sender.getPublicKey(),
                thread.getSesKey(),
                NoteType.AUDIO,
                cid,
                mimeType,
                new Date());

    }


    @NonNull
    public Note createInfoNote(@NonNull Thread thread,
                               @NonNull String info,
                               @NonNull Date date) {
        checkNotNull(thread);

        Note note = createNote(
                thread.getCid(),
                thread.getSenderPid(),
                thread.getSenderAlias(),
                thread.getSenderKey(),
                thread.getSesKey(),
                NoteType.INFO,
                null,
                MimeType.PLAIN_MIME_TYPE,
                date);
        note.addAdditional(Content.TEXT, info, true);
        return note;
    }


    @Nullable
    public PeerInfo getPeerInfoByHash(@NonNull String hash) {
        checkNotNull(hash);
        return getPeersInfoDatabase().peersInfoDao().getPeerInfoByHash(hash);
    }

    public void storeUser(@NonNull User user) {
        checkNotNull(user);
        getThreadsDatabase().userDao().insertUsers((User) user);
    }


    @NonNull
    public PeerInfo createPeerInfo(@NonNull PID owner) {
        checkNotNull(owner);

        return PeerInfo.createPeerInfo(owner);
    }

    @NonNull
    public Peer createPeer(@NonNull PID pid, @NonNull String multiAddress) {
        checkNotNull(pid);
        checkNotNull(multiAddress);

        return Peer.createPeer(pid, multiAddress);
    }

    @Nullable
    public Peer getPeerByPID(@NonNull PID pid) {
        checkNotNull(pid);
        return getPeersDatabase().peersDao().getPeerByPid(pid.getPid());
    }

    @NonNull
    public List<Peer> getRelayPeers() {
        return getPeersDatabase().peersDao().getRelayPeers();
    }

    @NonNull
    public List<Peer> getAutonatPeers() {
        return getPeersDatabase().peersDao().getAutonatPeers();
    }

    @NonNull
    public List<Peer> getPubsubPeers() {
        return getPeersDatabase().peersDao().getPubsubPeers();
    }


    @Nullable
    public PeerInfo getPeerInfoByPID(@NonNull PID pid) {
        checkNotNull(pid);
        return getPeersInfoDatabase().peersInfoDao().getPeerInfoByPid(pid.getPid());
    }

    public void storePeer(@NonNull Peer peer) {
        checkNotNull(peer);
        getPeersDatabase().peersDao().insertPeer(peer);
    }

    public void storePeerInfo(@NonNull PeerInfo peer) {
        checkNotNull(peer);
        getPeersInfoDatabase().peersInfoDao().insertPeerInfo(peer);
    }

    public void updatePeerInfo(@NonNull PeerInfo peer) {
        checkNotNull(peer);
        getPeersInfoDatabase().peersInfoDao().updatePeerInfo(peer);
    }

    public void updatePeer(@NonNull Peer peer) {
        checkNotNull(peer);
        getPeersDatabase().peersDao().updatePeer(peer);
    }

    public boolean insertPeerInfo(@NonNull IOTA iota,
                                  @NonNull PeerInfo peer,
                                  @NonNull String aesKey) {
        checkNotNull(iota);
        checkNotNull(peer);
        checkNotNull(aesKey);
        try {
            String address = AddressType.getAddress(peer.getPID(), AddressType.PEER);

            String data = PeerInfoEncoder.convert(peer, aesKey);

            Entity entity = iota.insertTransaction(address, data);
            checkNotNull(entity);


            String hash = entity.getHash();

            Hash transactionHash = createHash(hash);
            insertHash(transactionHash);

            setHash(peer, hash);
            return true;
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
            return false;
        }
    }


    @NonNull
    public User createUser(@NonNull PID pid,
                           @NonNull String publicKey,
                           @NonNull String name,
                           @NonNull UserType type,
                           @Nullable CID image) {
        checkNotNull(pid);
        checkNotNull(publicKey);
        checkNotNull(name);
        checkNotNull(type);
        checkArgument(!pid.getPid().isEmpty());

        return User.createUser(type, UserStatus.OFFLINE,
                name, publicKey, pid, image);
    }


    public void handleExpiredThreads(@NonNull IPFS ipfs) {
        Date today = THREADS.getToday();
        List<Thread> threads = getExpiredThreads(today);
        for (Thread thread : threads) {
            if (thread.getKind() == Kind.IN) {
                removeThread(ipfs, thread);
            } else {
                setStatus(thread, ThreadStatus.EXPIRED);
            }

        }
    }


    public void setAdditional(@NonNull Server server,
                              @NonNull String key,
                              @NonNull String value,
                              boolean internal) {
        checkNotNull(server);
        checkNotNull(key);
        checkNotNull(value);
        Server update = getServerByIdx(server.getIdx());
        checkNotNull(update);
        update.addAdditional(key, value, internal);
        updateServer(update);
    }

    public void setAdditional(@NonNull User user,
                              @NonNull String key,
                              @NonNull String value,
                              boolean internal) {
        checkNotNull(user);
        checkNotNull(key);
        checkNotNull(value);
        User update = getUserByPID(user.getPID());
        checkNotNull(update);
        update.addAdditional(key, value, internal);
        updateUser(update);
    }

    public void setAdditional(@NonNull Note note,
                              @NonNull String key,
                              @NonNull String value,
                              boolean internal) {
        checkNotNull(note);
        checkNotNull(key);
        checkNotNull(value);
        Note update = getNoteByIdx(note.getIdx());
        checkNotNull(update);
        update.addAdditional(key, value, internal);
        updateNote(update);
    }

    public void setAdditional(@NonNull Thread thread,
                              @NonNull String key,
                              @NonNull String value,
                              boolean internal) {
        checkNotNull(thread);
        checkNotNull(key);
        checkNotNull(value);
        Thread update = getThreadByIdx(thread.getIdx());
        checkNotNull(update);
        update.addAdditional(key, value, internal);
        updateThread(update);
    }


    public void removeThread(@NonNull IPFS ipfs, @NonNull Thread thread) {
        checkNotNull(ipfs);
        checkNotNull(thread);


        // delete thread notes children
        removeThreadNotes(ipfs, thread);

        removeThread(thread);


        CID cid = thread.getCid();
        if (cid != null) {
            try {
                List<Thread> list = getThreadsByCID(cid);
                if (list.isEmpty()) {
                    pin_rm(ipfs, cid);
                }
            } catch (Throwable e) {
                // for now ignore exception
            }
        }

        CID image = thread.getImage();
        if (image != null) {
            try {
                List<Thread> list = getThreadsByCID(image);
                if (list.isEmpty()) {
                    pin_rm(ipfs, image);
                }
            } catch (Throwable e) {
                // for now ignore exception
            }
        }

        // delete all children
        List<Thread> entries = getThreadsByThread(thread.getIdx());
        for (Thread entry : entries) {
            removeThread(ipfs, entry);
        }
    }


    /**
     * Utility function to remove a note from DB and from local RELAY
     *
     * @param ipfs RELAY client
     * @param note Note object
     */
    public void removeNote(@NonNull IPFS ipfs, @NonNull Note note) {
        checkNotNull(ipfs);
        checkNotNull(note);

        CID cid = note.getCid();

        removeNote(note);

        if (cid != null) {
            try {
                List<Note> list = getNotesByCID(cid);
                if (list.isEmpty()) {
                    pin_rm(ipfs, cid);
                }
            } catch (Throwable e) {
                // for now ignore exception
            }
        }

        CID image = note.getImage();
        if (image != null) {
            try {
                List<Note> list = getNotesByCID(image);
                if (list.isEmpty()) {
                    pin_rm(ipfs, image);
                }
            } catch (Throwable e) {
                // for now ignore exception
            }

        }
    }

    public void insertHash(@NonNull Hash hash) {
        checkNotNull(hash);
        entityService.getHashDatabase().hashDao().insertHash(hash);
    }

    private List<Thread> getExpiredThreads(@NonNull Date date) {
        checkNotNull(date);
        return getThreadsDatabase().threadDao().getExpiredThreads(date);
    }

    @NonNull
    public List<Thread> loadThreadRequests(@NonNull IOTA iota,
                                           @NonNull User user,
                                           @NonNull String privateKey) {

        checkNotNull(iota);
        checkNotNull(user);
        checkNotNull(privateKey);

        List<Thread> threads = new ArrayList<>();
        String address = AddressType.getAddress(user.getPID(), AddressType.INBOX);
        checkNotNull(address);
        try {
            iota.loadTransactions(new Filter() {
                @Override
                public boolean acceptHash(@NonNull String hash) {
                    return !hasHash(hash);
                }

                @Override
                public void invalidHash(@NonNull String hash) {
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);
                }

                @Override
                public boolean acceptEntity(@NonNull Entity entity) {

                    boolean result = true;
                    String hash = entity.getHash();
                    if (hasHash(hash)) {
                        result = false;
                    } else {
                        Hash transactionHash =
                                createHash(hash);
                        insertHash(transactionHash);
                    }


                    return result;
                }

                @Override
                public void invalidEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void error(@NonNull Entity entity, @NonNull Throwable e) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void loadEntity(@NonNull Entity entity) {
                    Thread thread = NoteRequestDecoder.convert(entity, privateKey);
                    if (thread != null) {
                        if (isUserBlocked(thread.getSenderPid())) {
                            return;
                        }

                        if (existsSameThread(thread)) {
                            return;
                        }
                        threads.add(thread); // handle later
                    }
                }

            }, address);
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }

        return threads;
    }

    @Nullable
    public PeerInfo getPeer(@NonNull IOTA iota,
                            @NonNull PID pid,
                            @NonNull String aesKey) {
        checkNotNull(iota);
        checkNotNull(pid);
        checkNotNull(aesKey);

        PeerInfo peer = loadPeer(iota, pid, aesKey);
        if (peer != null) {
            PeerInfo storePeer = getPeerInfoByPID(pid);
            if (storePeer != null) {
                if (storePeer.getTimestamp() > peer.getTimestamp()) {
                    // store peer is newer
                    return storePeer;
                } else {
                    storePeerInfo(peer);
                    return peer;
                }
            } else {
                storePeerInfo(peer);
                return peer;
            }
        }
        return getPeerInfoByPID(pid);
    }

    @Nullable
    public PeerInfo loadPeer(@NonNull IOTA iota,
                             @NonNull PID pid,
                             @NonNull String aesKey) {
        checkNotNull(iota);
        checkNotNull(pid);
        checkNotNull(aesKey);

        String address = AddressType.getAddress(pid, AddressType.PEER);

        AtomicReference<PeerInfo> reference = new AtomicReference<>(null);

        try {
            iota.loadTransactions(new Filter() {
                @Override
                public boolean acceptHash(@NonNull String hash) {
                    return !hasHash(hash);
                }

                @Override
                public void invalidHash(@NonNull String hash) {
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);
                }

                @Override
                public boolean acceptEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    boolean result = true;
                    String hash = entity.getHash();
                    if (hasHash(hash)) {
                        result = false;
                    } else {
                        Hash transactionHash =
                                createHash(hash);
                        insertHash(transactionHash);
                    }


                    return result;
                }

                @Override
                public void invalidEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void error(@NonNull Entity entity, @NonNull Throwable e) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }


                @Override
                public void loadEntity(@NonNull Entity entity) {
                    checkNotNull(entity);

                    PeerInfo peer = PeerInfoDecoder.convert(pid, entity, aesKey);
                    if (peer != null) {
                        PeerInfo inserted = reference.get();
                        if (inserted != null) {

                            if (inserted.getTimestamp() < peer.getTimestamp()) {
                                // replace items (means peer is newer)
                                reference.set(peer);
                            }
                        } else {
                            reference.set(peer);
                        }
                    }

                }

            }, address);
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }

        return reference.get();
    }

    @NonNull
    public List<Server> loadServers(@NonNull IOTA iota,
                                    @NonNull User owner,
                                    @NonNull String address) {

        checkNotNull(iota);
        checkNotNull(owner);
        checkNotNull(address);

        List<Server> servers = new ArrayList<>();
        try {
            iota.loadTransactions(new Filter() {
                @Override
                public boolean acceptHash(@NonNull String hash) {
                    return !hasHash(hash);
                }

                @Override
                public void invalidHash(@NonNull String hash) {
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);
                }

                @Override
                public boolean acceptEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    boolean result = true;
                    String hash = entity.getHash();
                    if (hasHash(hash)) {
                        result = false;
                    } else {
                        Hash transactionHash =
                                createHash(hash);
                        insertHash(transactionHash);
                    }


                    return result;
                }

                @Override
                public void invalidEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void error(@NonNull Entity entity, @NonNull Throwable e) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }


                @Override
                public void loadEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    Server server = ServerDecoder.convert(entity, owner);
                    if (server != null) {
                        servers.add(server);
                    }
                }


            }, address);
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }
        return servers;
    }


    @NonNull
    public List<Note> loadNotes(@NonNull IOTA iota, @NonNull Thread thread) {

        checkNotNull(iota);
        checkNotNull(thread);

        List<Note> notes = new ArrayList<>();
        CID cid = thread.getCid();
        checkNotNull(cid);
        String address = THREADS.getAddress(cid);

        try {
            iota.loadTransactions(new Filter() {
                @Override
                public boolean acceptHash(@NonNull String hash) {
                    return !hasHash(hash);
                }

                @Override
                public void invalidHash(@NonNull String hash) {
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);
                }

                @Override
                public boolean acceptEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    boolean result = true;
                    String hash = entity.getHash();
                    if (hasHash(hash)) {
                        result = false;
                    } else {
                        Hash transactionHash =
                                createHash(hash);
                        insertHash(transactionHash);
                    }


                    return result;
                }

                @Override
                public void invalidEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void error(@NonNull Entity entity, @NonNull Throwable e) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }


                @Override
                public void loadEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    Note note = NoteDecoder.convert(thread, entity);
                    if (note != null) {
                        if (isUserBlocked(note.getSenderPid())) {
                            return;
                        }

                        if (existsSameNote(note)) {
                            return;
                        }
                        notes.add(note); // handle later
                    }
                }


            }, address);
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }
        return notes;
    }


    /**
     * Removes only the thread from DB, not the CID content
     *
     * @param thread Thread which should be removed
     */
    public void removeThread(@NonNull Thread thread) {
        checkNotNull(thread);
        getThreadsDatabase().threadDao().removeThreads(thread);
    }


    public void removeUser(@NonNull User user) {
        checkNotNull(user);
        getThreadsDatabase().userDao().removeUsers(user);
    }


    public void removeUserByPID(@NonNull PID pid) {
        checkNotNull(pid);
        getThreadsDatabase().userDao().removeUserByPid(pid.getPid());
    }

    public void removeUsersByPID(@NonNull String... pids) {
        checkNotNull(pids);
        getThreadsDatabase().userDao().removeUsersByPid(pids);
    }

    public boolean existsUser(@NonNull PID user) {
        checkNotNull(user);
        return existsUser(user.getPid());
    }

    public boolean existsUser(@NonNull String pid) {
        checkNotNull(pid);
        return getThreadsDatabase().userDao().hasUser(pid) > 0;
    }

    public void removeThreads(@NonNull IPFS ipfs, long... idxs) {
        checkNotNull(ipfs);
        List<Thread> threads = getThreadByIdxs(idxs);
        for (Thread thread : threads) {
            removeThread(ipfs, thread);

        }
    }

    public void pin_rm(@NonNull IPFS ipfs, @NonNull CID cid) {
        checkNotNull(ipfs);
        checkNotNull(cid);
        checkNotNull(cid);
        ipfs.rm(cid);
    }


    public void pin_add(@NonNull IPFS ipfs, @NonNull CID cid, int timeout, boolean offline) {
        checkNotNull(ipfs);
        checkNotNull(cid);
        checkNotNull(timeout > 0);
        ipfs.pin_add(cid, timeout, offline);
    }

    public void removeThreads(@NonNull IPFS ipfs, @NonNull List<Thread> threads) {
        checkNotNull(ipfs);
        checkNotNull(threads);

        for (Thread thread : threads) {
            removeThreadNotes(ipfs, thread);
        }
        getThreadsDatabase().threadDao().removeThreads(
                Iterables.toArray(threads, Thread.class));
    }

    @NonNull
    public Event createEvent(@NonNull String identifier, @NonNull String content) {
        checkNotNull(identifier);
        checkNotNull(content);
        return Event.createEvent(identifier, content);
    }

    public void removeEvent(@NonNull Event event) {
        checkNotNull(event);
        getEventsDatabase().eventDao().deleteEvent(event);
    }

    public void removeEvent(@NonNull String identifier) {
        checkNotNull(identifier);
        getEventsDatabase().eventDao().deleteEvent(identifier);
    }


    public void invokeEvent(@NonNull String identifier, @NonNull String content) {
        checkNotNull(identifier);
        checkNotNull(content);
        storeEvent(createEvent(identifier, content));
    }

    public void storeEvent(@NonNull Event event) {
        checkNotNull(event);
        getEventsDatabase().eventDao().insertEvent(event);
    }


    public Message createMessage(@NonNull MessageKind messageKind, @NonNull String message, long timestamp) {
        checkNotNull(messageKind);
        checkNotNull(message);
        return Message.createMessage(messageKind, message, timestamp);
    }


    public void removeMessage(@NonNull Message message) {
        checkNotNull(message);
        getEventsDatabase().messageDao().deleteMessage(message);
    }


    public void storeMessage(@NonNull Message message) {
        checkNotNull(message);
        getEventsDatabase().messageDao().insertMessages(message);
    }


    public void clearMessages() {
        getEventsDatabase().messageDao().clear();
    }


    public List<PID> getMembers(@NonNull Thread thread) {
        checkNotNull(thread);
        return Lists.newArrayList(thread.getMembers());
    }


    public List<Thread> getThreads() {
        return getThreadsDatabase().threadDao().getThreads();
    }


    public void incrementUnreadNotesNumber(@NonNull Thread thread) {
        checkNotNull(thread);
        incrementThreadsUnreadNotesNumber(thread.getIdx());
    }

    public void incrementThreadsUnreadNotesNumber(long... idxs) {
        getThreadsDatabase().threadDao().incrementUnreadNotesNumber(idxs);
    }

    public void refreshNotesNumber(@NonNull Thread thread) {
        checkNotNull(thread);
        int number = thread.getUnreadNotes();
        getThreadsDatabase().threadDao().setUnreadNotesNumber(thread.getIdx(), number);
    }


    public void resetThreadsUnreadNotesNumber(long... idxs) {
        getThreadsDatabase().threadDao().resetUnreadNotes(idxs);
    }

    public void resetThreadUnreadNotesNumber(long thread) {
        getThreadsDatabase().threadDao().resetThreadUnreadNotes(thread);
    }

    @Nullable
    public User getUserByPID(@NonNull PID pid) {
        checkNotNull(pid);
        checkArgument(!pid.getPid().isEmpty());
        return getThreadsDatabase().userDao().getUserByPid(pid.getPid());
    }


    @NonNull
    public List<User> getUsersByPID(@NonNull String... pids) {
        checkNotNull(pids);
        return getThreadsDatabase().userDao().getUsersByPid(pids);
    }

    @NonNull
    public List<User> getUsers() {
        return getThreadsDatabase().userDao().getUsers();
    }


    @NonNull
    public List<User> getUsers(@NonNull Thread thread) {
        checkNotNull(thread);

        List<User> users = new ArrayList<>();
        for (PID pid : thread.getMembers()) {
            User user = getUserByPID(pid);
            if (user != null) {
                users.add(user);
            }
        }
        return users;
    }

    @NonNull
    public String getFileName(@NonNull Note note) {
        checkNotNull(note);
        return note.getAdditional(Content.FILENAME);
    }


    public long getFileSize(@NonNull Note note) {
        checkNotNull(note);
        return Long.valueOf(note.getAdditional(Content.FILESIZE));
    }

    @NonNull
    public LinkType getLinkNoteLinkType(@NonNull Note note) {
        checkNotNull(note);
        checkArgument(note.getNoteType() == NoteType.LINK);
        return LinkType.valueOf(note.getAdditional(LinkType.class.getSimpleName()));
    }


    private void removeThreadNotes(@NonNull IPFS ipfs, @NonNull Thread thread) {
        checkNotNull(thread);
        checkNotNull(ipfs);

        List<Note> notes = getNotesByThread(thread.getCid());
        for (Note note : notes) {
            removeNote(ipfs, note);
        }
    }


    /**
     * Only removes the note, not the CID content
     *
     * @param note note wich should be removed
     */
    public void removeNote(@NonNull Note note) {
        checkNotNull(note);
        getThreadsDatabase().noteDao().removeNote(note);
    }

    @NonNull
    public List<Note> getNotesByNoteType(@NonNull NoteType type) {
        checkNotNull(type);
        return getThreadsDatabase().noteDao().getNotesByType(type);
    }


    @NonNull
    public List<Note> getNotesByKindAndStatus(@NonNull Kind kind, @NonNull NoteStatus status) {
        checkNotNull(kind);
        checkNotNull(status);
        return getThreadsDatabase().noteDao().getNotesByKindAndStatus(kind, status);
    }


    public boolean insertUser(@NonNull IOTA iota,
                              @NonNull User user,
                              @NonNull String address,
                              @NonNull String aesKey) {
        checkNotNull(iota);
        checkNotNull(user);
        checkNotNull(aesKey);

        try {
            String dataTransaction = UserEncoder.convert(user, aesKey);


            Entity entity = iota.insertTransaction(address, dataTransaction);
            checkNotNull(entity);

            String hash = entity.getHash();
            Hash transactionHash = createHash(hash);
            checkNotNull(transactionHash);
            insertHash(transactionHash);

            setHash(user, hash);

            return true;
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
            return false;
        }
    }

    public void setHash(@NonNull Server server, @Nullable String hash) {
        checkNotNull(server);
        long idx = server.getIdx();
        server.setHash(hash);
        getThreadsDatabase().serverDao().setHash(idx, hash);

    }

    @Nullable
    public String getPeerInfoHash(@NonNull PID pid) {
        checkNotNull(pid);
        return getPeersInfoDatabase().peersInfoDao().getPeerInfoHash(pid.getPid());
    }

    public void setHash(@NonNull PeerInfo peer, @Nullable String hash) {
        checkNotNull(peer);
        peer.setHash(hash);
        getPeersInfoDatabase().peersInfoDao().setHash(peer.getPid(), hash);

    }

    public void setHash(@NonNull User user, @Nullable String hash) {
        checkNotNull(user);
        user.setHash(hash);
        getThreadsDatabase().userDao().setHash(user.getPid(), hash);
    }

    @Nullable
    public String getHash(@NonNull Note note) {
        checkNotNull(note);
        return getThreadsDatabase().noteDao().getHash(note.getIdx());
    }


    public void setHash(@NonNull Note note, @Nullable String hash) {
        checkNotNull(note);
        note.setHash(hash);
        getThreadsDatabase().noteDao().setHash(note.getIdx(), hash);
    }

    public void updateSettings(@NonNull Settings settings) {
        checkNotNull(settings);
        getThreadsDatabase().settingsDao().updateSettings(settings);
    }

    public boolean hasHash(@NonNull String hash) {
        return entityService.getHashDatabase().hashDao().hasHash(hash) > 0;
    }


    public void removeHash(@NonNull String hash) {
        entityService.getHashDatabase().hashDao().removeHash(hash);
    }

    private void updateUser(@NonNull User user) {
        getThreadsDatabase().userDao().updateUser(user);
    }


    /**
     * @param iota Tangle client which handles the communication with the server
     * @param pid  PID of the user
     * @param hash PeerInfo hash
     * @return the peer with the given hash, when success (Note: the peer will be
     * store automatically in the database, only when there is no newer peer
     * which is already stored in the database)
     */
    @Nullable
    public PeerInfo getPeerInfoByHash(@NonNull IOTA iota,
                                      @NonNull PID pid,
                                      @NonNull String hash,
                                      @NonNull String aesKey) {
        checkNotNull(iota);
        checkNotNull(hash);
        checkNotNull(pid);
        checkNotNull(aesKey);
        PeerInfo peer = getPeerInfoByHash(hash);

        if (peer != null) {
            return peer;
        } // already loaded

        peer = loadPeerInfoByHash(iota, pid, hash, aesKey);
        if (peer != null) {

            // now we have to check if this peer is newer
            // then the latest store peer
            boolean store = true;
            PeerInfo storedPeer = getPeerInfoByPID(pid);
            if (storedPeer != null) {
                if (peer.getTimestamp() < storedPeer.getTimestamp()) {
                    store = false;
                }
            }

            if (store) {
                storePeerInfo(peer);
            }
        }
        return peer;
    }


    /**
     * @param iota Tangle client which handles the communication with the server
     * @param pid  PID of the user
     * @param hash PeerInfo hash
     * @return user with the hash will be loaded from the tangle
     * <p>
     * NOTE: the user is not yet added to the database
     * (Use storeUser for storing to the database)
     */
    @Nullable
    public PeerInfo loadPeerInfoByHash(@NonNull IOTA iota,
                                       @NonNull PID pid,
                                       @NonNull String hash,
                                       @NonNull String aesKey) {

        checkNotNull(iota);
        checkNotNull(pid);
        checkNotNull(hash);
        checkNotNull(aesKey);

        List<PeerInfo> peers = new ArrayList<>();

        try {
            iota.loadHashTransactions(new Filter() {
                @Override
                public boolean acceptHash(@NonNull String hash) {
                    return true;
                }

                @Override
                public void invalidHash(@NonNull String hash) {
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);
                }

                @Override
                public boolean acceptEntity(@NonNull Entity entity) {
                    boolean result = true;
                    String hash = entity.getHash();
                    if (hasHash(hash)) {
                        result = false;
                    } else {
                        Hash transactionHash =
                                createHash(hash);
                        insertHash(transactionHash);
                    }


                    return result;
                }

                @Override
                public void invalidEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);
                }


                @Override
                public void error(@NonNull Entity entity, @NonNull Throwable e) {
                    String hash = entity.getHash();
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void loadEntity(@NonNull Entity entity) {
                    PeerInfo peer = PeerInfoDecoder.convert(pid, entity, aesKey);
                    if (peer != null) {
                        peers.add(peer);
                    }
                }

            }, hash);
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }
        if (peers.isEmpty()) {
            return null;
        }
        return peers.get(0);
    }

    /**
     * @param iota   Tangle client which handles the communication with the server
     * @param hash   Account hash
     * @param aesKey Application AesKey
     * @return user with the hash will be loaded from the tangle
     * <p>
     * NOTE: the user is not yet added to the database
     * (Use storeUser for storing to the database)
     */
    @Nullable
    public User loadUserByHash(@NonNull IOTA iota,
                               @NonNull String hash,
                               @NonNull String aesKey) {

        checkNotNull(iota);
        checkNotNull(hash);
        checkNotNull(aesKey);
        List<User> users = new ArrayList<>();

        try {
            iota.loadHashTransactions(new Filter() {
                @Override
                public boolean acceptHash(@NonNull String hash) {
                    return true;
                }

                @Override
                public void invalidHash(@NonNull String hash) {
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);
                }

                @Override
                public boolean acceptEntity(@NonNull Entity entity) {
                    boolean result = true;
                    String hash = entity.getHash();
                    if (hasHash(hash)) {
                        result = false;
                    } else {
                        Hash transactionHash =
                                createHash(hash);
                        insertHash(transactionHash);
                    }


                    return result;
                }

                @Override
                public void invalidEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash = createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void error(@NonNull Entity entity, @NonNull Throwable e) {
                    String hash = entity.getHash();
                    Hash transactionHash = createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void loadEntity(@NonNull Entity entity) {
                    User user = UserDecoder.convert(entity, aesKey);
                    if (user != null) {
                        users.add(user);
                    }
                }

            }, hash);
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }
        if (users.isEmpty()) {
            return null;
        }
        return users.get(0);
    }

    /**
     * @param iota    Tangle client which handles the communication with the server
     * @param address Account Address
     * @param aesKey  Application AesKey
     * @return user with the address will be loaded from the tangle
     * <p>
     * NOTE: the user is not yet added to the database
     * (Use storeUser for storing to the database)
     */
    @Nullable
    public User loadUserByAddress(@NonNull IOTA iota,
                                  @NonNull String address,
                                  @NonNull String aesKey) {
        checkNotNull(iota);
        checkNotNull(address);
        checkNotNull(aesKey);

        List<User> users = new ArrayList<>();

        try {
            iota.loadTransactions(new Filter() {
                @Override
                public boolean acceptHash(@NonNull String hash) {
                    return !hasHash(hash);
                }

                @Override
                public void invalidHash(@NonNull String hash) {
                    Hash transactionHash =
                            createHash(hash);
                    insertHash(transactionHash);
                }

                @Override
                public boolean acceptEntity(@NonNull Entity entity) {

                    boolean result = true;
                    String hash = entity.getHash();
                    if (hasHash(hash)) {
                        result = false;
                    } else {
                        Hash transactionHash = createHash(hash);
                        insertHash(transactionHash);
                    }


                    return result;
                }

                @Override
                public void invalidEntity(@NonNull Entity entity) {
                    checkNotNull(entity);
                    String hash = entity.getHash();
                    Hash transactionHash = createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void error(@NonNull Entity entity, @NonNull Throwable e) {
                    String hash = entity.getHash();
                    Hash transactionHash = createHash(hash);
                    insertHash(transactionHash);

                }

                @Override
                public void loadEntity(@NonNull Entity entity) {
                    User user = UserDecoder.convert(entity, aesKey);
                    if (user != null) {
                        users.add(user);
                    }
                }


            }, address);


            if (users.isEmpty()) {
                return null;
            }
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }
        return users.get(0);

    }

    public void setDate(@NonNull Thread thread, @NonNull Date date) {
        checkNotNull(thread);
        checkNotNull(date);
        getThreadsDatabase().threadDao().setThreadDate(thread.getIdx(), date);
    }

    public void resetUnreadNotes(@NonNull Thread thread) {
        checkNotNull(thread);
        getThreadsDatabase().threadDao().resetUnreadNotes(thread.getIdx());
    }


    public long storeThread(@NonNull Thread thread) {
        checkNotNull(thread);
        return getThreadsDatabase().threadDao().insertThread(thread);
    }


    public long storeNote(@NonNull Note note) {
        checkNotNull(note);
        return getThreadsDatabase().noteDao().insertNote(note);
    }


    public void setStatus(@NonNull Thread thread, @NonNull ThreadStatus status) {
        checkNotNull(thread);
        checkNotNull(status);
        getThreadsDatabase().threadDao().setThreadStatus(thread.getIdx(), status);
    }

    public void setThreadPinned(long idx, boolean pinned) {
        getThreadsDatabase().threadDao().setPinned(idx, pinned);

    }

    public boolean isThreadPinned(long idx) {
        return getThreadsDatabase().threadDao().isPinned(idx);
    }

    public void setCID(@NonNull Thread thread, @NonNull CID cid) {
        checkNotNull(thread);
        checkNotNull(cid);
        Multihash.fromBase58(cid.getCid()); // check if cid  is valid (otherwise exception)
        getThreadsDatabase().threadDao().setCid(thread.getIdx(), cid);

    }

    public void setThreadCID(long idx, @NonNull CID cid) {
        checkNotNull(cid);
        getThreadsDatabase().threadDao().setCid(idx, cid);

    }

    public void setNoteCID(long idx, @NonNull CID cid) {
        checkNotNull(cid);
        getThreadsDatabase().noteDao().setCid(idx, cid);

    }

    public void setCID(@NonNull Note note, @NonNull CID cid) {
        checkNotNull(note);
        checkNotNull(cid);
        getThreadsDatabase().noteDao().setCid(note.getIdx(), cid);

    }


    public void setStatus(@NonNull Note note, @NonNull NoteStatus status) {
        checkNotNull(note);
        checkNotNull(status);
        getThreadsDatabase().noteDao().setNoteStatus(note.getIdx(), status);
    }


    public void setNoteStatus(@NonNull NoteStatus status, long idx) {
        checkNotNull(status);
        getThreadsDatabase().noteDao().setNoteStatus(idx, status);
    }


    public void setStatus(@NonNull User user, @NonNull UserStatus status) {
        checkNotNull(user);
        checkNotNull(status);
        setUserStatus(user.getPID(), status);
    }

    public void setUserStatus(@NonNull PID user, @NonNull UserStatus status) {
        checkNotNull(user);
        checkNotNull(status);
        getThreadsDatabase().userDao().setUserStatus(user.getPid(), status);
    }

    public void setUsersStatus(@NonNull UserStatus status, String... pids) {
        checkNotNull(status);
        getThreadsDatabase().userDao().setUsersStatus(status, pids);
    }

    @NonNull
    public UserStatus getStatus(@NonNull User user) {
        checkNotNull(user);
        return getUserStatus(user.getPid());
    }

    @NonNull
    public UserStatus getUserStatus(@NonNull PID user) {
        checkNotNull(user);
        return getUserStatus(user.getPid());
    }

    @NonNull
    public UserStatus getUserStatus(@NonNull String pid) {
        checkNotNull(pid);
        return getThreadsDatabase().userDao().getUserStatus(pid);
    }

    @NonNull
    public ThreadStatus getStatus(@NonNull Thread thread) {
        checkNotNull(thread);
        return getThreadsDatabase().threadDao().getThreadStatus(thread.getIdx());
    }


    public boolean getThreadMarkedFlag(long idx) {
        return getThreadsDatabase().threadDao().getMarkedFlag(idx);
    }

    public void setThreadMarkedFlag(long idx, boolean flag) {
        getThreadsDatabase().threadDao().setMarkedFlag(idx, flag);
    }

    @NonNull
    public List<Note> getNotesByDate(@NonNull Date date) {
        checkNotNull(date);
        return getThreadsDatabase().noteDao().getNotesByDate(date);
    }

    @NonNull
    public List<Thread> getPinnedThreads() {
        return getThreadsDatabase().threadDao().getThreadsByPinned(true);
    }

    @NonNull
    public List<Thread> getThreadsByDate(@NonNull Date date) {
        checkNotNull(date);
        return getThreadsDatabase().threadDao().getThreadsByDate(date);
    }

    private boolean existsSameNote(@NonNull Note note) {
        checkNotNull(note);
        boolean result = false;
        List<Note> notes = getNotesByDate(note.getDate());
        for (Note cmp : notes) {
            if (note.sameNote(cmp)) {
                result = true;
                break;
            }
        }
        return result;
    }

    private boolean existsSameThread(@NonNull Thread thread) {
        checkNotNull(thread);
        boolean result = false;
        List<Thread> notes = getThreadsByDate(thread.getDate());
        for (Thread cmp : notes) {
            if (thread.sameThread(cmp)) {
                result = true;
                break;
            }
        }
        return result;
    }

    @NonNull
    public NoteStatus getStatus(@NonNull Note note) {
        checkNotNull(note);
        return getThreadsDatabase().noteDao().getNoteStatus(note.getIdx());
    }

    @Nullable
    public NoteStatus getNoteStatus(long idx) {
        return getThreadsDatabase().noteDao().getNoteStatus(idx);
    }

    @Nullable
    public ThreadStatus getThreadStatus(long idx) {
        return getThreadsDatabase().threadDao().getThreadStatus(idx);
    }

    @Nullable
    public NoteType getNoteType(long idx) {
        return getThreadsDatabase().noteDao().getNoteType(idx);
    }

    public void setUserPublicKey(@NonNull User user, @NonNull String publicKey) {
        checkNotNull(user);
        checkNotNull(publicKey);
        setUserPublicKey(user.getPid(), publicKey);
    }

    public void setUserPublicKey(@NonNull PID user, @NonNull String publicKey) {
        checkNotNull(user);
        checkNotNull(publicKey);
        setUserPublicKey(user.getPid(), publicKey);
    }

    public void setUserPublicKey(@NonNull String pid, @NonNull String publicKey) {
        checkNotNull(pid);
        checkNotNull(publicKey);
        getThreadsDatabase().userDao().setPublicKey(pid, publicKey);
    }

    public String getUserPublicKey(@NonNull User user) {
        checkNotNull(user);
        return getUserPublicKey(user.getPid());
    }

    public String getUserPublicKey(@NonNull PID user) {
        checkNotNull(user);
        return getUserPublicKey(user.getPid());
    }

    public String getUserPublicKey(@NonNull String pid) {
        checkNotNull(pid);
        return getThreadsDatabase().userDao().getPublicKey(pid);
    }

    public String getUserAlias(@NonNull User user) {
        checkNotNull(user);
        return getUserAlias(user.getPid());
    }

    public String getUserAlias(@NonNull PID user) {
        checkNotNull(user);
        return getUserAlias(user.getPid());
    }

    public String getUserAlias(@NonNull String pid) {
        checkNotNull(pid);
        return getThreadsDatabase().userDao().getAlias(pid);
    }


    public void setUserAlias(@NonNull User user, @NonNull String alias) {
        checkNotNull(user);
        checkNotNull(alias);
        setUserAlias(user.getPid(), alias);
    }

    public void setUserAlias(@NonNull PID user, @NonNull String alias) {
        checkNotNull(user);
        checkNotNull(alias);
        setUserAlias(user.getPid(), alias);
    }

    public void setUserAlias(@NonNull String pid, @NonNull String alias) {
        checkNotNull(pid);
        checkNotNull(alias);
        getThreadsDatabase().userDao().setAlias(pid, alias);
    }


    public void setUserImage(@NonNull PID user, @NonNull CID image) {
        checkNotNull(user);
        checkNotNull(image);
        setUserImage(user.getPid(), image);
    }

    public void setUserImage(@NonNull String pid, @NonNull CID image) {
        checkNotNull(pid);
        checkNotNull(image);
        getThreadsDatabase().userDao().setImage(pid, image);
    }


    public void setUserType(@NonNull User user, @NonNull UserType type) {
        checkNotNull(user);
        checkNotNull(type);
        setUserType(user.getPid(), type);
    }

    public void setUserType(@NonNull PID user, @NonNull UserType type) {
        checkNotNull(user);
        checkNotNull(type);
        setUserType(user.getPid(), type);
    }

    public void setUserType(@NonNull String pid, @NonNull UserType type) {
        checkNotNull(pid);
        checkNotNull(type);
        getThreadsDatabase().userDao().setUserType(pid, type);
    }


    @NonNull
    public Settings createSettings(@NonNull String id) {
        checkNotNull(id);
        return Settings.createSettings(id);
    }


    public void storeNotes(@NonNull List<Note> notes) {
        checkNotNull(notes);
        getThreadsDatabase().noteDao().insertNotes(Iterables.toArray(notes, Note.class));
    }


    public void storeThreads(@NonNull List<Thread> threads) {
        checkNotNull(threads);
        getThreadsDatabase().threadDao().insertThreads(Iterables.toArray(threads, Thread.class));
    }


    private void updateThread(@NonNull Thread thread) {
        checkNotNull(thread);
        getThreadsDatabase().threadDao().updateThreads((Thread) thread);
    }

    @NonNull
    public List<Thread> getThreadsByThread(long thread) {
        return getThreadsDatabase().threadDao().getThreadsByThread(thread);
    }


    @Nullable
    public Thread getThreadByIdx(long idx) {
        return getThreadsDatabase().threadDao().getThreadByIdx(idx);
    }

    public List<Thread> getThreadByIdxs(long... idx) {
        return getThreadsDatabase().threadDao().getThreadByIdxs(idx);
    }


    @Nullable
    public CID getThreadCID(long idx) {
        return getThreadsDatabase().threadDao().getCid(idx);
    }


    @Nullable
    public Thread getThread(@NonNull Note note) {
        checkNotNull(note);
        CID cid = note.getThread();
        if (cid == null) {
            return null;
        }
        List<Thread> threads = getThreadsByCID(cid);
        if (threads.isEmpty()) {
            return null;
        }
        checkArgument(threads.size() == 1);
        return threads.get(0);
    }

    @Nullable
    public Note getNoteByIdx(long idx) {
        return getThreadsDatabase().noteDao().getNoteByIdx(idx);
    }

    private void updateServer(@NonNull Server server) {
        checkNotNull(server);
        getThreadsDatabase().serverDao().updateServer(server);
    }

    private void updateNote(@NonNull Note note) {
        checkNotNull(note);
        getThreadsDatabase().noteDao().updateNote(note);
    }


    public void blockUser(@NonNull PID user) {
        checkNotNull(user);
        getThreadsDatabase().userDao().setBlocked(user.getPid(), true);
    }


    public void blockUser(@NonNull User user) {
        checkNotNull(user);
        blockUser(user.getPID());
    }


    public void unblockUser(@NonNull User user) {
        checkNotNull(user);
        unblockUser(user.getPID());
    }

    public void unblockUser(@NonNull PID user) {
        checkNotNull(user);
        getThreadsDatabase().userDao().setBlocked(user.getPid(), false);
    }


    @Nullable

    public Settings getSettings(@NonNull String id) {
        checkNotNull(id);
        return getThreadsDatabase().settingsDao().getSettings(id);
    }


    public void storeSettings(@NonNull Settings settings) {
        checkNotNull(settings);
        getThreadsDatabase().settingsDao().insertSettings(settings);
    }


    @NonNull
    public List<Thread> getThreadsByThreadStatus(@NonNull ThreadStatus readStatus) {
        checkNotNull(readStatus);
        return getThreadsDatabase().threadDao().getThreadsByThreadStatus(readStatus);
    }


    @NonNull
    public List<Thread> getThreadsByKindAndThreadStatus(@NonNull Kind kind, @NonNull ThreadStatus status) {
        checkNotNull(kind);
        checkNotNull(status);
        return getThreadsDatabase().threadDao().getThreadsByKindAndThreadStatus(kind, status);
    }

    private void sendNotification(@NonNull IPFS ipfs,
                                  @NonNull PID pid,
                                  @NonNull Content content) {

        checkNotNull(ipfs);
        checkNotNull(pid);
        checkNotNull(content);

        String json_message = gson.toJson(content);

        ipfs.pubsubPub(pid.getPid(), json_message, 50);

    }

    /**
     * Utility function to check if the sender of the note is already
     * add as member to the thread (Note: in case a new member was added
     * to the thread, the thread still has to be updated in the DB)
     * <p>
     * Side effect of this operation is, that the sender of the note
     * will be update in regards of its relay PID.
     *
     * @param thread Thread object
     * @param note   Note object
     * @return user when a new member of the thread was found
     */
    @Nullable
    public User verifyMember(@NonNull Thread thread, @NonNull Note note) {
        checkNotNull(thread);
        checkNotNull(note);
        PID pid = note.getSenderPid();

        if (!thread.getMembers().contains(pid)) {
            User senderUser = getUserByPID(pid);
            if (senderUser == null) {


                senderUser = createUser(pid,
                        note.getSenderKey(),
                        note.getSenderAlias(),
                        UserType.UNKNOWN,
                        null);


                storeUser(senderUser);

                return senderUser;
            }
            addMember(thread, pid);
            return null;

        }
        return null;
    }

    public void addMember(@NonNull Thread thread, @NonNull PID user) {
        checkNotNull(thread);
        checkNotNull(user);
        thread.addMember(user);
        Thread update = getThreadByIdx(thread.getIdx());
        checkNotNull(update);
        update.addMember(user);
        updateThread(update);
    }

    public void removeMember(@NonNull Thread thread, @NonNull PID user) {
        checkNotNull(thread);
        checkNotNull(user);
        thread.removeMember(user);
        Thread update = getThreadByIdx(thread.getIdx());
        checkNotNull(update);
        update.removeMember(user);
        updateThread(update);
    }

    @NonNull
    public Hash createHash(@NonNull String hash) {
        checkNotNull(hash);
        return Hash.create(hash, System.currentTimeMillis());
    }


    private Note createNote(@Nullable CID thread,
                            @NonNull PID senderPid,
                            @NonNull String senderAuthor,
                            @NonNull String senderKey,
                            @NonNull String sesKey,
                            @NonNull NoteType noteType,
                            @Nullable CID cid,
                            @NonNull String mimeType,
                            @NonNull Date date) {
        Note note = Note.createNote(thread, senderPid, senderAuthor, senderKey,
                sesKey, NoteStatus.OFFLINE, Kind.OUT, noteType, mimeType, date);
        note.setCid(cid);

        return note;
    }


    @NonNull
    private Thread createThread(@NonNull ThreadStatus threadStatus,
                                @NonNull Kind kind,
                                @NonNull PID senderPid,
                                @NonNull String senderAlias,
                                @NonNull String senderKey,
                                @NonNull String sesKey,
                                @NonNull Date date,
                                long thread) {
        return Thread.createThread(threadStatus,
                senderPid, senderAlias, senderKey,
                sesKey, kind, date, thread);
    }


    @Nullable
    public Server getServerByIdx(long idx) {
        return getThreadsDatabase().serverDao().getServerByIdx(idx);
    }

    public void storeServer(@NonNull Server server) {
        getThreadsDatabase().serverDao().insertServer(server);
    }


    @NonNull
    public List<Server> getServers() {
        return getThreadsDatabase().serverDao().getServers();
    }


    @NonNull
    public Server createServer(@NonNull String protocol,
                               @NonNull String host,
                               int port,
                               @NonNull String alias) {
        checkNotNull(protocol);
        checkNotNull(host);
        checkArgument(port > 0);
        checkNotNull(alias);
        return Server.createServer(protocol, host, port, alias);
    }


    @NonNull
    public List<Note> getNotesByCID(@NonNull CID cid) {
        checkNotNull(cid);
        return getThreadsDatabase().noteDao().getNotesByCid(cid);
    }

    @NonNull
    public List<PID> getUsersPIDs() {
        List<PID> result = new ArrayList<>();
        List<String> pids = getThreadsDatabase().userDao().getUserPids();
        for (String pid : pids) {
            result.add(PID.create(pid));
        }
        return result;
    }

    @NonNull
    public List<Thread> getThreadsByCIDAndThread(@NonNull CID cid, long thread) {
        checkNotNull(cid);
        return getThreadsDatabase().threadDao().getThreadsByCidAndThread(cid, thread);
    }

    @NonNull
    public List<Thread> getThreadsByCID(@NonNull CID cid) {
        checkNotNull(cid);
        return getThreadsDatabase().threadDao().getThreadsByCid(cid);
    }

    public boolean insertServer(@NonNull IOTA iota, @NonNull Server server, @NonNull String address) {
        checkNotNull(iota);
        checkNotNull(server);
        checkNotNull(address);

        try {
            Content content = ServerEncoder.convert(server);

            Entity entity = iota.insertTransaction(address, gson.toJson(content));
            checkNotNull(entity);
            String hash = entity.getHash();

            Hash transactionHash = createHash(hash);
            insertHash(transactionHash);

            setHash(server, hash);

            return true;
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
            return false;
        }
    }

    public void removePeerInfo(@NonNull PeerInfo peer) {
        checkNotNull(peer);
        getPeersInfoDatabase().peersInfoDao().deletePeerInfo(peer);
    }

    public void removePeer(@NonNull Peer peer) {
        checkNotNull(peer);
        getPeersDatabase().peersDao().deletePeer(peer);
    }


}
