/*
 * Decompiled with CFR 0.152.
 */
package tech.powerjob.server.core.instance;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.LineNumberReader;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Resource;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.time.FastDateFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.CollectionUtils;
import tech.powerjob.common.enums.LogLevel;
import tech.powerjob.common.enums.TimeExpressionType;
import tech.powerjob.common.model.InstanceLogContent;
import tech.powerjob.common.utils.CommonUtils;
import tech.powerjob.common.utils.NetUtils;
import tech.powerjob.common.utils.SegmentLock;
import tech.powerjob.server.common.utils.OmsFileUtils;
import tech.powerjob.server.core.instance.InstanceMetadataService;
import tech.powerjob.server.persistence.StringPage;
import tech.powerjob.server.persistence.local.LocalInstanceLogDO;
import tech.powerjob.server.persistence.local.LocalInstanceLogRepository;
import tech.powerjob.server.persistence.mongodb.GridFsManager;
import tech.powerjob.server.persistence.remote.model.JobInfoDO;
import tech.powerjob.server.remote.server.redirector.DesignateServer;

@Service
public class InstanceLogService {
    private static final Logger log = LoggerFactory.getLogger(InstanceLogService.class);
    @Value(value="${server.port}")
    private int port;
    @Resource
    private InstanceMetadataService instanceMetadataService;
    @Resource
    private GridFsManager gridFsManager;
    @Resource(name="localTransactionTemplate")
    private TransactionTemplate localTransactionTemplate;
    @Resource
    private LocalInstanceLogRepository localInstanceLogRepository;
    private final Map<Long, Long> instanceId2LastReportTime = Maps.newConcurrentMap();
    @Resource(name="PowerJobBackgroundPool")
    private AsyncTaskExecutor powerJobBackgroundPool;
    private final SegmentLock segmentLock = new SegmentLock(8);
    private static final FastDateFormat DATE_FORMAT = FastDateFormat.getInstance((String)"yyyy-MM-dd HH:mm:ss.SSS");
    private static final int MAX_LINE_COUNT = 100;
    private static final long EXPIRE_INTERVAL_MS = 60000L;

    @Async(value="PowerJobLocalDbPool")
    public void submitLogs(String workerAddress, List<InstanceLogContent> logs) {
        List logList = logs.stream().map(x -> {
            this.instanceId2LastReportTime.put(x.getInstanceId(), System.currentTimeMillis());
            LocalInstanceLogDO y = new LocalInstanceLogDO();
            BeanUtils.copyProperties((Object)x, (Object)y);
            y.setWorkerAddress(workerAddress);
            return y;
        }).collect(Collectors.toList());
        try {
            CommonUtils.executeWithRetry0(() -> this.localInstanceLogRepository.saveAll((Iterable)logList));
        }
        catch (Exception e) {
            log.warn("[InstanceLogService] persistent instance logs failed, these logs will be dropped: {}.", logs, (Object)e);
        }
    }

    @DesignateServer
    public StringPage fetchInstanceLog(Long appId, Long instanceId, Long index) {
        try {
            Future<File> fileFuture = this.prepareLogFile(instanceId);
            File logFile = fileFuture.get(5L, TimeUnit.SECONDS);
            long lines = 0L;
            StringBuilder sb = new StringBuilder();
            long left = index * 100L;
            long right = left + 100L;
            try (LineNumberReader lr = new LineNumberReader(new FileReader(logFile));){
                String lineStr;
                while ((lineStr = lr.readLine()) != null) {
                    if (lines >= left && lines < right) {
                        sb.append(lineStr).append(System.lineSeparator());
                    }
                    ++lines;
                }
            }
            catch (Exception e) {
                log.warn("[InstanceLog-{}] read logFile from disk failed for app: {}.", new Object[]{instanceId, appId, e});
                return StringPage.simple((String)("oms-server execution exception, caused by " + ExceptionUtils.getRootCauseMessage((Throwable)e)));
            }
            double totalPage = Math.ceil(1.0 * (double)lines / 100.0);
            return new StringPage(index.longValue(), (long)totalPage, sb.toString());
        }
        catch (TimeoutException te) {
            return StringPage.simple((String)"log file is being prepared, please try again later.");
        }
        catch (Exception e) {
            log.warn("[InstanceLog-{}] fetch instance log failed.", (Object)instanceId, (Object)e);
            return StringPage.simple((String)("oms-server execution exception, caused by " + ExceptionUtils.getRootCauseMessage((Throwable)e)));
        }
    }

    @DesignateServer
    public String fetchDownloadUrl(Long appId, Long instanceId) {
        String url = "http://" + NetUtils.getLocalHost() + ":" + this.port + "/instance/downloadLog?instanceId=" + instanceId;
        log.info("[InstanceLog-{}] downloadURL for appId[{}]: {}", new Object[]{instanceId, appId, url});
        return url;
    }

    public File downloadInstanceLog(long instanceId) throws Exception {
        Future<File> fileFuture = this.prepareLogFile(instanceId);
        return fileFuture.get(1L, TimeUnit.MINUTES);
    }

    private Future<File> prepareLogFile(long instanceId) {
        return this.powerJobBackgroundPool.submit(() -> {
            if (this.instanceId2LastReportTime.containsKey(instanceId)) {
                return this.genTemporaryLogFile(instanceId);
            }
            return this.genStableLogFile(instanceId);
        });
    }

    @Async(value="PowerJobBackgroundPool")
    public void sync(Long instanceId) {
        Stopwatch sw = Stopwatch.createStarted();
        try {
            File stableLogFile = this.genStableLogFile(instanceId);
            if (this.gridFsManager.available()) {
                try {
                    this.gridFsManager.store(stableLogFile, "log", InstanceLogService.genMongoFileName(instanceId));
                    log.info("[InstanceLog-{}] push local instanceLogs to mongoDB succeed, using: {}.", (Object)instanceId, (Object)sw.stop());
                }
                catch (Exception e) {
                    log.warn("[InstanceLog-{}] push local instanceLogs to mongoDB failed.", (Object)instanceId, (Object)e);
                }
            }
        }
        catch (Exception e) {
            log.warn("[InstanceLog-{}] sync local instanceLogs failed.", (Object)instanceId, (Object)e);
        }
        try {
            this.instanceId2LastReportTime.remove(instanceId);
            CommonUtils.executeWithRetry0(() -> this.localInstanceLogRepository.deleteByInstanceId(instanceId));
            log.info("[InstanceLog-{}] delete local instanceLog successfully.", (Object)instanceId);
        }
        catch (Exception e) {
            log.warn("[InstanceLog-{}] delete local instanceLog failed.", (Object)instanceId, (Object)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File genTemporaryLogFile(long instanceId) {
        String path = InstanceLogService.genLogFilePath(instanceId, false);
        int lockId = ("tpFileLock-" + instanceId).hashCode();
        try {
            this.segmentLock.lockInterruptibleSafe(lockId);
            File file = (File)this.localTransactionTemplate.execute(status -> {
                File f = new File(path);
                if (f.exists() && System.currentTimeMillis() - f.lastModified() < 60000L) {
                    return f;
                }
                try {
                    FileUtils.forceMkdirParent((File)f);
                    try (Stream allLogStream = this.localInstanceLogRepository.findByInstanceIdOrderByLogTime(Long.valueOf(instanceId));){
                        this.stream2File(allLogStream, f);
                    }
                    return f;
                }
                catch (Exception e) {
                    CommonUtils.executeIgnoreException(() -> FileUtils.forceDelete((File)f));
                    throw new RuntimeException(e);
                }
            });
            return file;
        }
        finally {
            this.segmentLock.unlock(lockId);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File genStableLogFile(long instanceId) {
        String path = InstanceLogService.genLogFilePath(instanceId, true);
        int lockId = ("stFileLock-" + instanceId).hashCode();
        try {
            this.segmentLock.lockInterruptibleSafe(lockId);
            File file = (File)this.localTransactionTemplate.execute(status -> {
                File f = new File(path);
                if (f.exists()) {
                    return f;
                }
                try {
                    FileUtils.forceMkdirParent((File)f);
                    if (this.instanceId2LastReportTime.containsKey(instanceId)) {
                        try (Stream allLogStream = this.localInstanceLogRepository.findByInstanceIdOrderByLogTime(Long.valueOf(instanceId));){
                            this.stream2File(allLogStream, f);
                        }
                    } else {
                        if (!this.gridFsManager.available()) {
                            OmsFileUtils.string2File((String)"SYSTEM: There is no local log for this task now, you need to use mongoDB to store the past logs.", (File)f);
                            return f;
                        }
                        if (!this.gridFsManager.exists("log", InstanceLogService.genMongoFileName(instanceId))) {
                            OmsFileUtils.string2File((String)"SYSTEM: There is no online log for this job instance.", (File)f);
                            return f;
                        }
                        this.gridFsManager.download(f, "log", InstanceLogService.genMongoFileName(instanceId));
                    }
                    return f;
                }
                catch (Exception e) {
                    CommonUtils.executeIgnoreException(() -> FileUtils.forceDelete((File)f));
                    throw new RuntimeException(e);
                }
            });
            return file;
        }
        finally {
            this.segmentLock.unlock(lockId);
        }
    }

    private void stream2File(Stream<LocalInstanceLogDO> stream, File logFile) {
        try (FileWriter fw = new FileWriter(logFile);
             BufferedWriter bfw = new BufferedWriter(fw);){
            stream.forEach(instanceLog -> {
                try {
                    bfw.write(InstanceLogService.convertLog(instanceLog) + System.lineSeparator());
                }
                catch (Exception exception) {
                    // empty catch block
                }
            });
        }
        catch (IOException ie) {
            ExceptionUtils.rethrow((Throwable)ie);
        }
    }

    private static String convertLog(LocalInstanceLogDO instanceLog) {
        return String.format("%s [%s] %s %s", DATE_FORMAT.format((Object)instanceLog.getLogTime()), instanceLog.getWorkerAddress(), LogLevel.genLogLevelString((Integer)instanceLog.getLogLevel()), instanceLog.getLogContent());
    }

    @Async(value="PowerJobTimingPool")
    @Scheduled(fixedDelay=120000L)
    public void timingCheck() {
        LinkedList frequentInstanceIds = Lists.newLinkedList();
        this.instanceId2LastReportTime.keySet().forEach(instanceId -> {
            try {
                JobInfoDO jobInfo = this.instanceMetadataService.fetchJobInfoByInstanceId((Long)instanceId);
                if (TimeExpressionType.FREQUENT_TYPES.contains(jobInfo.getTimeExpressionType())) {
                    frequentInstanceIds.add(instanceId);
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
        });
        if (!CollectionUtils.isEmpty((Collection)frequentInstanceIds)) {
            long time = System.currentTimeMillis() - 600000L;
            Lists.partition((List)frequentInstanceIds, (int)100).forEach(p -> {
                try {
                    this.localInstanceLogRepository.deleteByInstanceIdInAndLogTimeLessThan(p, Long.valueOf(time));
                }
                catch (Exception e) {
                    log.warn("[InstanceLogService] delete expired logs for instance: {} failed.", p, (Object)e);
                }
            });
        }
    }

    private static String genLogFilePath(long instanceId, boolean stable) {
        if (stable) {
            return OmsFileUtils.genLogDirPath() + String.format("%d-stable.log", instanceId);
        }
        return OmsFileUtils.genLogDirPath() + String.format("%d-temporary.log", instanceId);
    }

    private static String genMongoFileName(long instanceId) {
        return String.format("oms-%d.log", instanceId);
    }
}

