/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.utils;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.regex.Pattern;
import software.amazon.smithy.utils.CodeFormatter;
import software.amazon.smithy.utils.MapUtils;
import software.amazon.smithy.utils.StringUtils;

public class CodeWriter {
    private static final Pattern LINES = Pattern.compile("\\r?\\n");
    private static final Map<Character, BiFunction<Object, String, String>> DEFAULT_FORMATTERS = MapUtils.of(Character.valueOf('L'), (s, i) -> CodeWriter.formatLiteral(s), Character.valueOf('S'), (s, i) -> StringUtils.escapeJavaString(CodeWriter.formatLiteral(s), i));
    private final CodeFormatter formatter = new CodeFormatter();
    private final Deque<State> states = new ArrayDeque<State>();
    private State currentState;
    private boolean trailingNewline = true;
    private int trimBlankLines = -1;
    private boolean needsIndentation = true;

    public CodeWriter() {
        this.states.push(new State());
        this.currentState = this.states.getFirst();
        this.currentState.builder = new StringBuilder();
        DEFAULT_FORMATTERS.forEach(this.formatter::putFormatter);
    }

    public static CodeWriter createDefault() {
        return new CodeWriter().trimTrailingSpaces();
    }

    public static String formatLiteral(Object value) {
        if (value == null) {
            return "";
        }
        if (value instanceof Optional) {
            Optional optional = (Optional)value;
            return optional.isPresent() ? CodeWriter.formatLiteral(optional.get()) : "";
        }
        return String.valueOf(value);
    }

    public final CodeWriter putFormatter(char identifier, BiFunction<Object, String, String> formatter) {
        this.formatter.putFormatter(Character.valueOf(identifier), formatter);
        return this;
    }

    public CodeWriter setExpressionStart(char expressionStart) {
        if (expressionStart == ' ' || expressionStart == '\n') {
            throw new IllegalArgumentException("expressionStart must not be set to " + expressionStart);
        }
        this.currentState.expressionStart = expressionStart;
        return this;
    }

    public char getExpressionStart() {
        return this.currentState.expressionStart;
    }

    public String toString() {
        String result = this.currentState.toString();
        if (this.trimBlankLines > -1) {
            StringBuilder builder = new StringBuilder(result.length());
            String[] lines = LINES.split(result);
            int blankCount = 0;
            for (String line : lines) {
                if (!StringUtils.isBlank(line)) {
                    builder.append(line).append(this.currentState.newline);
                    blankCount = 0;
                    continue;
                }
                if (blankCount++ >= this.trimBlankLines) continue;
                builder.append(line).append(this.currentState.newline);
            }
            result = builder.toString();
        }
        if (result.isEmpty()) {
            return this.trailingNewline ? String.valueOf(this.currentState.newline) : "";
        }
        if (this.currentState.trimTrailingSpaces) {
            result = StringUtils.stripEnd(result, " ");
        }
        if (this.trailingNewline) {
            return result.charAt(result.length() - 1) != this.currentState.newline ? result + this.currentState.newline : result;
        }
        if (result.charAt(result.length() - 1) == this.currentState.newline) {
            return result.substring(0, result.length() - 1);
        }
        return result;
    }

    public final CodeWriter pushState() {
        return this.pushState(null);
    }

    public CodeWriter pushState(String sectionName) {
        this.currentState = new State(this.currentState);
        this.states.push(this.currentState);
        if (sectionName != null) {
            this.currentState.sectionName = sectionName;
            this.currentState.builder = null;
            this.currentState.newlinePrefix = "";
            this.dedent(-1);
        }
        return this;
    }

    public CodeWriter pushFilteredState(Function<String, String> filter) {
        String sectionName = "__filtered_state_" + this.states.size() + 1;
        this.pushState(sectionName);
        this.onSection(sectionName, content -> this.writeWithNoFormatting(filter.apply(content.toString())));
        return this;
    }

    public CodeWriter popState() {
        if (this.states.size() == 1) {
            throw new IllegalStateException("Cannot pop CodeWriter state because at the root state");
        }
        State popped = this.states.pop();
        this.currentState = this.states.getFirst();
        if (popped.sectionName != null) {
            String result = this.getTrimmedPoppedStateContents(popped);
            if (popped.interceptors.containsKey(popped.sectionName)) {
                List interceptors = (List)popped.interceptors.get(popped.sectionName);
                for (Consumer interceptor : interceptors) {
                    result = this.expandSection("__" + popped.sectionName, result, interceptor);
                }
            }
            if (popped.isInline) {
                popped.builder.setLength(0);
                popped.builder.append(result);
            } else {
                this.writeOptional(result.replace("$", "$$"));
            }
        }
        return this;
    }

    private String getTrimmedPoppedStateContents(State state) {
        StringBuilder builder = state.builder;
        String result = "";
        if (builder != null && builder.length() > 0) {
            if (builder.charAt(builder.length() - 1) == this.currentState.newline) {
                builder.delete(builder.length() - 1, builder.length());
            }
            result = builder.toString();
        }
        return result;
    }

    public CodeWriter onSection(String sectionName, Consumer<Object> interceptor) {
        this.currentState.putInterceptor(sectionName, interceptor);
        return this;
    }

    public final CodeWriter onSectionPrepend(String sectionName, Runnable writeBefore) {
        return this.onSection(sectionName, contents -> {
            writeBefore.run();
            this.write(contents, new Object[0]);
        });
    }

    public final CodeWriter onSectionAppend(String sectionName, Runnable writeAfter) {
        return this.onSection(sectionName, contents -> {
            this.write(contents, new Object[0]);
            writeAfter.run();
        });
    }

    public CodeWriter disableNewlines() {
        this.currentState.disableNewline = true;
        return this;
    }

    public CodeWriter enableNewlines() {
        this.currentState.disableNewline = false;
        return this;
    }

    public final CodeWriter setNewline(String newline) {
        if (newline.isEmpty()) {
            return this.disableNewlines();
        }
        if (newline.length() > 1) {
            throw new IllegalArgumentException("newline must be set to an empty string or a single character");
        }
        return this.setNewline(newline.charAt(0));
    }

    public final CodeWriter setNewline(char newline) {
        this.currentState.newline = newline;
        this.enableNewlines();
        return this;
    }

    public final CodeWriter setIndentText(String indentText) {
        this.currentState.indent(0, indentText);
        return this;
    }

    public final CodeWriter trimTrailingSpaces() {
        return this.trimTrailingSpaces(true);
    }

    public final CodeWriter trimTrailingSpaces(boolean trimTrailingSpaces) {
        this.currentState.trimTrailingSpaces = trimTrailingSpaces;
        return this;
    }

    public final CodeWriter trimBlankLines() {
        return this.trimBlankLines(1);
    }

    public final CodeWriter trimBlankLines(int trimBlankLines) {
        this.trimBlankLines = trimBlankLines;
        return this;
    }

    public final CodeWriter insertTrailingNewline() {
        return this.insertTrailingNewline(true);
    }

    public final CodeWriter insertTrailingNewline(boolean trailingNewline) {
        this.trailingNewline = trailingNewline;
        return this;
    }

    public final CodeWriter setNewlinePrefix(String newlinePrefix) {
        this.currentState.newlinePrefix = newlinePrefix;
        return this;
    }

    public final CodeWriter indent() {
        return this.indent(1);
    }

    public final CodeWriter indent(int levels) {
        this.currentState.indent(levels, null);
        return this;
    }

    public final CodeWriter dedent() {
        return this.dedent(1);
    }

    public final CodeWriter dedent(int levels) {
        int adjusted = levels == -1 ? Integer.MIN_VALUE : -1 * levels;
        this.currentState.indent(adjusted, null);
        return this;
    }

    public final CodeWriter openBlock(String textBeforeNewline, Object ... args) {
        return this.write(textBeforeNewline, args).indent();
    }

    public final CodeWriter openBlock(String textBeforeNewline, String textAfterNewline, Runnable f) {
        return this.openBlock(textBeforeNewline, textAfterNewline, new Object[0], f);
    }

    public final CodeWriter openBlock(String textBeforeNewline, String textAfterNewline, Object arg1, Runnable f) {
        return this.openBlock(textBeforeNewline, textAfterNewline, new Object[]{arg1}, f);
    }

    public final CodeWriter openBlock(String textBeforeNewline, String textAfterNewline, Object arg1, Object arg2, Runnable f) {
        return this.openBlock(textBeforeNewline, textAfterNewline, new Object[]{arg1, arg2}, f);
    }

    public final CodeWriter openBlock(String textBeforeNewline, String textAfterNewline, Object arg1, Object arg2, Object arg3, Runnable f) {
        return this.openBlock(textBeforeNewline, textAfterNewline, new Object[]{arg1, arg2, arg3}, f);
    }

    public final CodeWriter openBlock(String textBeforeNewline, String textAfterNewline, Object arg1, Object arg2, Object arg3, Object arg4, Runnable f) {
        return this.openBlock(textBeforeNewline, textAfterNewline, new Object[]{arg1, arg2, arg3, arg4}, f);
    }

    public final CodeWriter openBlock(String textBeforeNewline, String textAfterNewline, Object arg1, Object arg2, Object arg3, Object arg4, Object arg5, Runnable f) {
        return this.openBlock(textBeforeNewline, textAfterNewline, new Object[]{arg1, arg2, arg3, arg4, arg5}, f);
    }

    public final CodeWriter openBlock(String textBeforeNewline, String textAfterNewline, Object[] args, Runnable f) {
        this.write(textBeforeNewline, args).indent();
        f.run();
        this.closeBlock(textAfterNewline, new Object[0]);
        return this;
    }

    public final CodeWriter closeBlock(String textAfterNewline, Object ... args) {
        return this.dedent().write(textAfterNewline, args);
    }

    public final CodeWriter writeWithNoFormatting(Object content) {
        this.currentState.writeLine(content.toString());
        return this;
    }

    public final String format(Object content, Object ... args) {
        return this.formatter.format(this.currentState.expressionStart, content, this.currentState.indentText, this, args);
    }

    public final CodeWriter write(Object content, Object ... args) {
        String value = this.format(content, args);
        this.currentState.writeLine(value);
        return this;
    }

    public final CodeWriter writeInline(Object content, Object ... args) {
        String value = this.format(content, args);
        this.currentState.write(value);
        return this;
    }

    public final CodeWriter writeOptional(Object content) {
        if (content == null) {
            return this;
        }
        if (content instanceof Optional) {
            return this.writeOptional(((Optional)content).orElse(null));
        }
        String value = content.toString();
        return !value.isEmpty() ? this.write(value, new Object[0]) : this;
    }

    public final CodeWriter call(Runnable task) {
        task.run();
        return this;
    }

    public CodeWriter putContext(String key, Object value) {
        this.currentState.putContext(key, value);
        return this;
    }

    public final CodeWriter putContext(Map<String, Object> mappings) {
        mappings.forEach(this.currentState::putContext);
        return this;
    }

    public CodeWriter removeContext(String key) {
        this.currentState.removeContext(key);
        return this;
    }

    public Object getContext(String key) {
        return this.currentState.context.get(key);
    }

    String expandSection(String sectionName, String defaultContent, Consumer<Object> writerConsumer) {
        StringBuilder buffer = new StringBuilder();
        this.pushState(sectionName);
        this.currentState.isInline = true;
        this.currentState.builder = buffer;
        writerConsumer.accept(defaultContent);
        this.popState();
        return buffer.toString();
    }

    private final class State {
        private String indentText = "    ";
        private String leadingIndentString = "";
        private String newlinePrefix = "";
        private int indentation;
        private boolean trimTrailingSpaces;
        private boolean disableNewline;
        private char newline = (char)10;
        private char expressionStart = (char)36;
        private transient String sectionName;
        private transient boolean isInline;
        private StringBuilder builder;
        private Map<String, Object> context = MapUtils.of();
        private transient boolean copiedContext = false;
        private Map<String, List<Consumer<Object>>> interceptors = MapUtils.of();
        private transient boolean copiedInterceptors = false;

        State() {
        }

        private State(State copy) {
            this.newline = copy.newline;
            this.expressionStart = copy.expressionStart;
            this.builder = copy.builder;
            this.context = copy.context;
            this.indentText = copy.indentText;
            this.leadingIndentString = copy.leadingIndentString;
            this.indentation = copy.indentation;
            this.newlinePrefix = copy.newlinePrefix;
            this.trimTrailingSpaces = copy.trimTrailingSpaces;
            this.interceptors = copy.interceptors;
            this.disableNewline = copy.disableNewline;
        }

        public String toString() {
            return this.builder == null ? "" : this.builder.toString();
        }

        private void mutateContext() {
            if (!this.copiedContext) {
                this.context = new HashMap<String, Object>(this.context);
                this.copiedContext = true;
            }
        }

        void putContext(String key, Object value) {
            this.mutateContext();
            this.context.put(key, value);
        }

        void removeContext(String key) {
            if (this.context.containsKey(key)) {
                this.mutateContext();
                this.context.remove(key);
            }
        }

        void putInterceptor(String section, Consumer<Object> interceptor) {
            if (!this.copiedInterceptors) {
                this.interceptors = new HashMap<String, List<Consumer<Object>>>(this.interceptors);
                this.copiedInterceptors = true;
            }
            this.interceptors.computeIfAbsent(section, s -> new ArrayList()).add(interceptor);
        }

        void write(String contents) {
            if (this.builder == null) {
                this.builder = new StringBuilder();
            }
            for (int i = 0; i < contents.length(); ++i) {
                this.append(contents.charAt(i));
            }
        }

        void append(char c) {
            if (CodeWriter.this.needsIndentation) {
                this.builder.append(this.leadingIndentString);
                this.builder.append(this.newlinePrefix);
                CodeWriter.this.needsIndentation = false;
            }
            if (c == this.newline) {
                CodeWriter.this.needsIndentation = true;
                this.trimSpaces();
            }
            this.builder.append(c);
        }

        void writeLine(String line) {
            this.write(line);
            if (!this.disableNewline) {
                this.append(this.newline);
            }
        }

        private void trimSpaces() {
            if (!this.trimTrailingSpaces) {
                return;
            }
            int toRemove = 0;
            for (int i = this.builder.length() - 1; i > 0 && this.builder.charAt(i) == ' '; --i) {
                ++toRemove;
            }
            if (toRemove > 0) {
                this.builder.delete(this.builder.length() - toRemove, this.builder.length());
            }
        }

        private void indent(int levels, String indentText) {
            if (levels == Integer.MIN_VALUE) {
                this.indentation = 0;
            } else {
                if (levels + this.indentation < 0) {
                    throw new IllegalStateException(String.format("Cannot dedent CodeWriter %d levels", levels));
                }
                this.indentation += levels;
            }
            if (indentText != null) {
                this.indentText = indentText;
            }
            this.leadingIndentString = StringUtils.repeat(this.indentText, this.indentation);
        }
    }
}

