/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.utils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import software.amazon.smithy.utils.AbstractCodeWriter;
import software.amazon.smithy.utils.CodeSection;
import software.amazon.smithy.utils.CodeWriterFormatterContainer;
import software.amazon.smithy.utils.SimpleParser;
import software.amazon.smithy.utils.SmithyInternalApi;

@SmithyInternalApi
final class CodeFormatter {
    private CodeFormatter() {
    }

    static void run(Appendable sink, AbstractCodeWriter<?> writer, String template, Object[] args) {
        ColumnTrackingAppendable wrappedSink = new ColumnTrackingAppendable(sink);
        List program = new Parser(writer, template, args).parse();
        try {
            for (Operation op : program) {
                op.apply(wrappedSink, writer, wrappedSink.column);
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Error appending to CodeWriter template: " + e, e);
        }
    }

    private static final class ColumnTrackingAppendable
    extends DecoratedAppendable {
        int column = 0;
        private final Appendable delegate;

        ColumnTrackingAppendable(Appendable delegate) {
            this.delegate = delegate;
        }

        @Override
        public Appendable append(char c) throws IOException {
            this.column = c == '\r' || c == '\n' ? 0 : ++this.column;
            this.delegate.append(c);
            return this;
        }
    }

    private static final class Parser {
        private static final Pattern NAME_PATTERN = Pattern.compile("^[a-z]+[a-zA-Z0-9_.#$]*$");
        private final String template;
        private final SimpleParser parser;
        private final char expressionStart;
        private final AbstractCodeWriter<?> writer;
        private final Object[] arguments;
        private final boolean[] positionals;
        private final List<Operation> operations = new ArrayList<Operation>();
        private int relativeIndex = 0;

        Parser(AbstractCodeWriter<?> writer, String template, Object[] arguments) {
            this.template = template;
            this.writer = writer;
            this.expressionStart = writer.getExpressionStart();
            this.parser = new SimpleParser(template);
            this.arguments = arguments;
            this.positionals = new boolean[arguments.length];
        }

        private void pushOperation(Operation op) {
            this.operations.add(op);
        }

        private RuntimeException error(String message) {
            return this.parser.syntax(message + " (template: " + this.template + ") " + this.writer.getDebugInfo());
        }

        private List<Operation> parse() {
            boolean parsingLiteral = false;
            int literalStartCharacter = 0;
            while (!this.parser.eof()) {
                char c = this.parser.peek();
                this.parser.skip();
                if (c != this.expressionStart) {
                    parsingLiteral = true;
                    continue;
                }
                if (this.parser.peek() == this.expressionStart) {
                    this.pushOperation(Operation.stringSlice(this.template, literalStartCharacter, this.parser.position()));
                    this.parser.expect(this.expressionStart);
                    parsingLiteral = true;
                    literalStartCharacter = this.parser.position();
                    continue;
                }
                if (parsingLiteral) {
                    this.pushOperation(Operation.stringSlice(this.template, literalStartCharacter, this.parser.position() - 1));
                    parsingLiteral = false;
                }
                this.pushOperation(this.parseArgument());
                literalStartCharacter = this.parser.position();
            }
            if (parsingLiteral && literalStartCharacter < this.parser.position() && this.parser.position() > 0) {
                this.pushOperation(Operation.stringSlice(this.template, literalStartCharacter, this.parser.position()));
            }
            if (this.relativeIndex == -1) {
                this.ensureAllPositionalArgumentsWereUsed();
            } else if (this.relativeIndex < this.arguments.length) {
                int unusedCount = this.arguments.length - this.relativeIndex;
                throw this.error(String.format("Found %d unused relative format arguments", unusedCount));
            }
            return this.operations;
        }

        private void ensureAllPositionalArgumentsWereUsed() {
            int unused = 0;
            for (boolean b : this.positionals) {
                if (b) continue;
                ++unused;
            }
            if (unused > 0) {
                throw this.error(String.format("Found %d unused positional format arguments", unused));
            }
        }

        private Operation parseArgument() {
            return this.parser.peek() == '{' ? this.parseBracedArgument() : this.parseNormalArgument();
        }

        private Operation parseBracedArgument() {
            int startPosition = this.parser.position() - 1;
            int startColumn = this.parser.column() - 2;
            this.parser.expect('{');
            Operation operation = this.parseNormalArgument();
            if (this.parser.peek() == '@') {
                this.parser.skip();
                int start = this.parser.position();
                this.parser.consumeUntilNoLongerMatches(c -> c.charValue() != '}' && c.charValue() != '|');
                String sectionName = this.parser.sliceFrom(start);
                this.ensureNameIsValid(sectionName);
                operation = Operation.inlineSection(sectionName, operation);
            }
            if (this.parser.peek() == '|') {
                String staticWhitespace = this.isAllLeadingWhitespaceOnLine(startPosition, startColumn) ? this.template.substring(startPosition - startColumn, startPosition) : null;
                this.parser.expect('|');
                operation = Operation.block(operation, staticWhitespace);
            }
            this.parser.expect('}');
            return operation;
        }

        private boolean isAllLeadingWhitespaceOnLine(int startPosition, int startColumn) {
            for (int i = startPosition - startColumn; i < startPosition; ++i) {
                char ch = this.template.charAt(i);
                if (ch == ' ' || ch == '\t') continue;
                return false;
            }
            return true;
        }

        private Operation parseNormalArgument() {
            char c = this.parser.peek();
            Object value = Character.isLowerCase(c) ? this.parseNamedArgument() : (Character.isDigit(c) ? this.parsePositionalArgument() : this.parseRelativeArgument());
            String formatted = this.consumeAndApplyFormatterIdentifier(value);
            return Operation.staticValue(formatted);
        }

        private Object parseNamedArgument() {
            int start = this.parser.position();
            this.parser.consumeUntilNoLongerMatches(c -> c.charValue() != ':');
            String name = this.parser.sliceFrom(start);
            this.ensureNameIsValid(name);
            this.parser.expect(':');
            if (this.parser.eof()) {
                throw this.error("Expected an identifier after the ':' in a named argument");
            }
            return this.writer.getContext(name);
        }

        private String consumeAndApplyFormatterIdentifier(Object value) {
            char identifier = this.parser.expect(CodeWriterFormatterContainer.VALID_FORMATTER_CHARS);
            String result = this.writer.applyFormatter(identifier, value);
            if (result == null) {
                throw this.error(String.format("Unknown formatter `%c` found in format string", Character.valueOf(identifier)));
            }
            return result;
        }

        private Object parseRelativeArgument() {
            if (this.relativeIndex == -1) {
                throw this.error("Cannot mix positional and relative arguments");
            }
            ++this.relativeIndex;
            return this.getPositionalArgument(this.relativeIndex - 1);
        }

        private Object getPositionalArgument(int index) {
            if (index >= this.arguments.length) {
                throw this.error(String.format("Given %d arguments but attempted to format index %d", this.arguments.length, index));
            }
            this.positionals[index] = true;
            return this.arguments[index];
        }

        private Object parsePositionalArgument() {
            if (this.relativeIndex > 0) {
                throw this.error("Cannot mix positional and relative arguments");
            }
            this.relativeIndex = -1;
            int startPosition = this.parser.position();
            this.parser.consumeUntilNoLongerMatches(Character::isDigit);
            int index = Integer.parseInt(this.parser.sliceFrom(startPosition)) - 1;
            if (index < 0 || index >= this.arguments.length) {
                throw this.error(String.format("Positional argument index %d out of range of provided %d arguments in format string", index, this.arguments.length));
            }
            return this.getPositionalArgument(index);
        }

        private void ensureNameIsValid(String name) {
            if (!NAME_PATTERN.matcher(name).matches()) {
                throw this.error(String.format("Invalid format expression name `%s`", name));
            }
        }
    }

    @FunctionalInterface
    private static interface Operation {
        public void apply(Appendable var1, AbstractCodeWriter<?> var2, int var3) throws IOException;

        public static Operation stringSlice(String source, int start, int end) {
            return (sink, writer, column) -> sink.append(source, start, end);
        }

        public static Operation staticValue(String value) {
            return (sink, writer, column) -> sink.append(value);
        }

        public static Operation inlineSection(String sectionName, Operation delegate) {
            return (sink, writer, column) -> {
                StringBuilder buffer = new StringBuilder();
                delegate.apply(buffer, writer, column);
                String defaultValue = buffer.toString();
                CodeSection section = CodeSection.forName(sectionName);
                String expanded = writer.expandSection(section, defaultValue, writer::writeInlineWithNoFormatting);
                expanded = writer.removeTrailingNewline(expanded);
                sink.append(expanded);
            };
        }

        public static Operation block(Operation delegate, String staticWhitespace) {
            return (sink, writer, column) -> delegate.apply(new BlockAppender(sink, column, staticWhitespace), writer, column);
        }
    }

    private static final class BlockAppender
    extends DecoratedAppendable {
        private final Appendable delegate;
        private final int spaces;
        private final String staticWhitespace;
        private boolean previousIsCarriageReturn;
        private boolean previousIsNewline;

        BlockAppender(Appendable delegate, int spaces, String staticWhitespace) {
            this.delegate = delegate;
            this.spaces = spaces;
            this.staticWhitespace = staticWhitespace;
        }

        @Override
        public Appendable append(char c) throws IOException {
            if (this.previousIsNewline) {
                this.writeSpaces();
                this.previousIsNewline = false;
            }
            if (c == '\n') {
                this.delegate.append('\n');
                this.previousIsNewline = true;
                this.previousIsCarriageReturn = false;
            } else {
                if (this.previousIsCarriageReturn) {
                    this.writeSpaces();
                }
                this.previousIsCarriageReturn = c == '\r';
                this.delegate.append(c);
            }
            return this;
        }

        private void writeSpaces() throws IOException {
            if (this.staticWhitespace != null) {
                this.delegate.append(this.staticWhitespace);
            } else {
                for (int i = 0; i < this.spaces; ++i) {
                    this.delegate.append(' ');
                }
            }
        }
    }

    private static abstract class DecoratedAppendable
    implements Appendable {
        private DecoratedAppendable() {
        }

        @Override
        public Appendable append(CharSequence csq) throws IOException {
            return this.append(csq, 0, csq.length());
        }

        @Override
        public Appendable append(CharSequence csq, int start, int end) throws IOException {
            for (int i = start; i < end; ++i) {
                this.append(csq.charAt(i));
            }
            return this;
        }
    }
}

