/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.pattern;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import software.amazon.smithy.model.pattern.InvalidPatternException;
import software.amazon.smithy.model.shapes.ShapeId;

public class SmithyPattern {
    private final String pattern;
    private final List<Segment> segments;

    protected SmithyPattern(Builder builder) {
        this.pattern = Objects.requireNonNull(builder.pattern);
        this.segments = Objects.requireNonNull(builder.segments);
        this.checkForDuplicateLabels();
        if (builder.allowsGreedyLabels) {
            this.checkForLabelsAfterGreedyLabels();
        } else if (this.segments.stream().anyMatch(Segment::isGreedyLabel)) {
            throw new InvalidPatternException("Pattern must not contain a greedy label. Found " + this.pattern);
        }
    }

    public final List<Segment> getSegments() {
        return Collections.unmodifiableList(this.segments);
    }

    public final List<Segment> getLabels() {
        return Collections.unmodifiableList(this.segments.stream().filter(Segment::isLabel).collect(Collectors.toList()));
    }

    public final Optional<Segment> getLabel(String name) {
        String searchKey = name.toLowerCase(Locale.US);
        return this.segments.stream().filter(Segment::isLabel).filter(label -> label.getContent().toLowerCase(Locale.US).equals(searchKey)).findFirst();
    }

    public final Optional<Segment> getGreedyLabel() {
        return this.segments.stream().filter(Segment::isGreedyLabel).findFirst();
    }

    public String toString() {
        return this.pattern;
    }

    public boolean equals(Object other) {
        return other instanceof SmithyPattern && this.pattern.equals(((SmithyPattern)other).pattern);
    }

    public int hashCode() {
        return this.pattern.hashCode();
    }

    private void checkForDuplicateLabels() {
        HashSet labels = new HashSet();
        this.segments.forEach(segment -> {
            if (segment.isLabel() && !labels.add(segment.getContent().toLowerCase(Locale.US))) {
                throw new InvalidPatternException(String.format("Label `%s` is defined more than once in pattern: %s", segment.getContent(), this.pattern));
            }
        });
    }

    private void checkForLabelsAfterGreedyLabels() {
        for (int i = 0; i < this.segments.size(); ++i) {
            Segment s = this.segments.get(i);
            if (!s.isGreedyLabel()) continue;
            for (int j = i + 1; j < this.segments.size(); ++j) {
                if (this.segments.get(j).isGreedyLabel()) {
                    throw new InvalidPatternException("At most one greedy label segment may exist in a pattern: " + this.pattern);
                }
                if (!this.segments.get(j).isLabel()) continue;
                throw new InvalidPatternException("A greedy label must be the last label in its pattern: " + this.pattern);
            }
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Segment {
        private final String asString;
        private final String content;
        private final Type segmentType;

        public Segment(String content, Type segmentType) {
            this.content = Objects.requireNonNull(content);
            this.segmentType = segmentType;
            this.checkForInvalidContents();
            this.asString = segmentType == Type.GREEDY_LABEL ? "{" + content + "+}" : (segmentType == Type.LABEL ? "{" + content + "}" : content);
        }

        private void checkForInvalidContents() {
            if (this.segmentType == Type.LITERAL) {
                if (this.content.isEmpty()) {
                    throw new InvalidPatternException("Segments must not be empty");
                }
                if (this.content.contains("{") || this.content.contains("}")) {
                    throw new InvalidPatternException("Literal segments must not contain `{` or `}` characters. Found segment `" + this.content + "`");
                }
            } else {
                if (this.content.isEmpty()) {
                    throw new InvalidPatternException("Empty label declaration in pattern.");
                }
                if (!ShapeId.isValidIdentifier(this.content)) {
                    throw new InvalidPatternException("Invalid label name in pattern: '" + this.content + "'. Labels must contain value identifiers.");
                }
            }
        }

        public static Segment parse(String content, int offset) {
            if (content.length() >= 2 && content.charAt(0) == '{' && content.charAt(content.length() - 1) == '}') {
                Type labelType = content.charAt(content.length() - 2) == '+' ? Type.GREEDY_LABEL : Type.LABEL;
                content = labelType == Type.GREEDY_LABEL ? content.substring(1, content.length() - 2) : content.substring(1, content.length() - 1);
                return new Segment(content, labelType);
            }
            return new Segment(content, Type.LITERAL);
        }

        public String getContent() {
            return this.content;
        }

        public boolean isLabel() {
            return this.segmentType != Type.LITERAL;
        }

        public boolean isGreedyLabel() {
            return this.segmentType == Type.GREEDY_LABEL;
        }

        public String toString() {
            return this.asString;
        }

        public boolean equals(Object other) {
            return other instanceof Segment && this.asString.equals(((Segment)other).asString);
        }

        public int hashCode() {
            return this.asString.hashCode();
        }

        public static enum Type {
            LITERAL,
            LABEL,
            GREEDY_LABEL;

        }
    }

    public static final class Builder {
        private boolean allowsGreedyLabels = true;
        private String pattern;
        private List<Segment> segments;

        private Builder() {
        }

        public Builder allowsGreedyLabels(boolean allowsGreedyLabels) {
            this.allowsGreedyLabels = allowsGreedyLabels;
            return this;
        }

        public Builder pattern(String pattern) {
            this.pattern = pattern;
            return this;
        }

        public Builder segments(List<Segment> segments) {
            this.segments = segments;
            return this;
        }

        public SmithyPattern build() {
            return new SmithyPattern(this);
        }
    }
}

