/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.validation.validators;

import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.knowledge.NeighborProviderIndex;
import software.amazon.smithy.model.neighbor.NeighborProvider;
import software.amazon.smithy.model.neighbor.Relationship;
import software.amazon.smithy.model.neighbor.RelationshipDirection;
import software.amazon.smithy.model.neighbor.RelationshipType;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeType;
import software.amazon.smithy.model.traits.StreamingTrait;
import software.amazon.smithy.model.traits.TraitDefinition;
import software.amazon.smithy.model.validation.AbstractValidator;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.FunctionalUtils;
import software.amazon.smithy.utils.OptionalUtils;
import software.amazon.smithy.utils.SetUtils;
import software.amazon.smithy.utils.StringUtils;

public final class TargetValidator
extends AbstractValidator {
    private static final int MAX_EDIT_DISTANCE_FOR_SUGGESTIONS = 2;
    private static final Set<ShapeType> INVALID_MEMBER_TARGETS = SetUtils.of((Object[])new ShapeType[]{ShapeType.SERVICE, ShapeType.RESOURCE, ShapeType.OPERATION, ShapeType.MEMBER});
    private static final Set<ShapeType> VALID_SET_TARGETS = SetUtils.of((Object[])new ShapeType[]{ShapeType.STRING, ShapeType.BYTE, ShapeType.SHORT, ShapeType.INTEGER, ShapeType.LONG, ShapeType.BIG_INTEGER, ShapeType.BIG_DECIMAL, ShapeType.BLOB});

    @Override
    public List<ValidationEvent> validate(Model model) {
        NeighborProvider neighborProvider = NeighborProviderIndex.of(model).getProvider();
        return model.shapes().flatMap(shape -> this.validateShape(model, (Shape)shape, neighborProvider.getNeighbors((Shape)shape))).collect(Collectors.toList());
    }

    private Stream<ValidationEvent> validateShape(Model model, Shape shape, List<Relationship> relationships) {
        return relationships.stream().flatMap(relationship -> {
            if (relationship.getNeighborShape().isPresent()) {
                return OptionalUtils.stream(this.validateTarget(model, shape, relationship.getNeighborShape().get(), (Relationship)relationship));
            }
            return Stream.of(this.unresolvedTarget(model, shape, (Relationship)relationship));
        });
    }

    private Optional<ValidationEvent> validateTarget(Model model, Shape shape, Shape target, Relationship rel) {
        RelationshipType relType = rel.getRelationshipType();
        if (relType.getDirection() == RelationshipDirection.DIRECTED && target.hasTrait(TraitDefinition.class)) {
            return Optional.of(this.error(shape, String.format("Found a %s reference to trait definition `%s`. Trait definitions cannot be targeted by members or referenced by shapes in any other context other than applying them as traits.", new Object[]{relType, rel.getNeighborShapeId()})));
        }
        switch (relType) {
            case MEMBER_TARGET: {
                if (INVALID_MEMBER_TARGETS.contains((Object)target.getType())) {
                    return Optional.of(this.error(shape, String.format("Members cannot target %s shapes, but found %s", new Object[]{target.getType(), target})));
                }
            }
            case MAP_KEY: {
                return target.asMemberShape().flatMap(m -> this.validateMapKey(shape, m.getTarget(), model));
            }
            case SET_MEMBER: {
                return target.asMemberShape().flatMap(m -> this.validateSetMember(shape, m.getTarget(), model));
            }
            case RESOURCE: {
                if (target.getType() != ShapeType.RESOURCE) {
                    return Optional.of(this.badType(shape, target, relType, ShapeType.RESOURCE));
                }
                return Optional.empty();
            }
            case OPERATION: {
                if (target.getType() != ShapeType.OPERATION) {
                    return Optional.of(this.badType(shape, target, relType, ShapeType.OPERATION));
                }
                return Optional.empty();
            }
            case INPUT: 
            case OUTPUT: {
                if (target.getType() != ShapeType.STRUCTURE) {
                    return Optional.of(this.badType(shape, target, relType, ShapeType.STRUCTURE));
                }
                if (target.findTrait("error").isPresent()) {
                    return Optional.of(this.inputOutputWithErrorTrait(shape, target, rel.getRelationshipType()));
                }
                return Optional.empty();
            }
            case ERROR: {
                if (target.getType() != ShapeType.STRUCTURE) {
                    return Optional.of(this.badType(shape, target, relType, ShapeType.STRUCTURE));
                }
                if (!target.findTrait("error").isPresent()) {
                    return Optional.of(this.errorNoTrait(shape, target.getId()));
                }
                return Optional.empty();
            }
            case IDENTIFIER: {
                return this.validateIdentifier(shape, target);
            }
            case CREATE: 
            case READ: 
            case UPDATE: 
            case DELETE: 
            case LIST: {
                if (target.getType() != ShapeType.OPERATION) {
                    return Optional.of(this.error(shape, String.format("Resource %s lifecycle operation must target an operation, but found %s", relType.toString().toLowerCase(Locale.US), target)));
                }
                return Optional.empty();
            }
        }
        return Optional.empty();
    }

    private Optional<ValidationEvent> validateMapKey(Shape shape, ShapeId target, Model model) {
        return model.getShape(target).filter(FunctionalUtils.not(Shape::isStringShape)).map(resolved -> this.error(shape, String.format("Map key member targets %s, but is expected to target a string", resolved)));
    }

    private Optional<ValidationEvent> validateSetMember(Shape shape, ShapeId target, Model model) {
        Shape targetShape = model.getShape(target).orElse(null);
        if (targetShape == null) {
            return Optional.empty();
        }
        if (!VALID_SET_TARGETS.contains((Object)targetShape.getType())) {
            return Optional.of(this.warning(shape, String.format("Set member targets %s, but sets can target only %s. You can model a collection of %s shapes by changing this shape to a list. Modeling a set of values of other types is problematic to support across a wide range of programming languages. This will be upgraded to an ERROR in a future release.", new Object[]{targetShape, VALID_SET_TARGETS.stream().map(ShapeType::toString).sorted().collect(Collectors.joining(", ")), targetShape.getType()})));
        }
        if (targetShape.hasTrait(StreamingTrait.class)) {
            return Optional.of(this.error(shape, String.format("Set member targets %s, a shape marked with the @streaming trait. Sets do not support unbounded values.", targetShape)));
        }
        return Optional.empty();
    }

    private Optional<ValidationEvent> validateIdentifier(Shape shape, Shape target) {
        if (target.getType() != ShapeType.STRING) {
            return Optional.of(this.badType(shape, target, RelationshipType.IDENTIFIER, ShapeType.STRING));
        }
        return Optional.empty();
    }

    private ValidationEvent unresolvedTarget(Model model, Shape shape, Relationship rel) {
        String suggestionText;
        Collection<String> suggestions = this.computeTargetSuggestions(model, rel.getNeighborShapeId());
        String string = suggestionText = !suggestions.isEmpty() ? ". Did you mean " + String.join((CharSequence)", ", suggestions) + "?" : "";
        if (rel.getRelationshipType() == RelationshipType.MEMBER_TARGET) {
            return this.error(shape, String.format("member shape targets an unresolved shape `%s`%s", rel.getNeighborShapeId(), suggestionText));
        }
        String indefiniteArticle = TargetValidator.isUppercaseVowel(rel.getRelationshipType().toString().charAt(0)) ? "an" : "a";
        return this.error(shape, String.format("%s shape has %s `%s` relationship to an unresolved shape `%s`%s", new Object[]{shape.getType(), indefiniteArticle, rel.getRelationshipType().toString().toLowerCase(Locale.US), rel.getNeighborShapeId(), suggestionText}));
    }

    private Collection<String> computeTargetSuggestions(Model model, ShapeId target) {
        String targetString = target.toString();
        int floor = Integer.MAX_VALUE;
        TreeSet<String> candidates = new TreeSet<String>();
        for (Shape shape : model.toSet()) {
            String idString = shape.getId().toString();
            int distance = StringUtils.levenshteinDistance((CharSequence)targetString, (CharSequence)idString, (int)2);
            if (distance == floor) {
                candidates.add(idString);
                continue;
            }
            if (distance <= -1 || distance >= floor) continue;
            floor = distance;
            candidates.clear();
            candidates.add(idString);
        }
        return candidates;
    }

    private static boolean isUppercaseVowel(char c) {
        return c == 'A' || c == 'E' || c == 'I' || c == 'O' || c == 'U';
    }

    private ValidationEvent badType(Shape shape, Shape target, RelationshipType rel, ShapeType valid) {
        return this.error(shape, String.format("%s shape `%s` relationships must target a %s shape, but found %s", new Object[]{shape.getType(), rel.toString().toLowerCase(Locale.US), valid, target}));
    }

    private ValidationEvent inputOutputWithErrorTrait(Shape shape, Shape target, RelationshipType rel) {
        String descriptor = rel == RelationshipType.INPUT ? "input" : "output";
        return this.error(shape, String.format("Operation %s targets an invalid structure `%s` that is marked with the `error` trait.", descriptor, target.getId()));
    }

    private ValidationEvent errorNoTrait(Shape shape, ShapeId target) {
        return this.error(shape, String.format("`%s` cannot be bound as an error because it is not marked with the `error` trait.", target));
    }
}

