/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.loader.FullyResolvedModelFile;
import software.amazon.smithy.model.loader.LoaderUtils;
import software.amazon.smithy.model.loader.ModelFile;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.BigDecimalShape;
import software.amazon.smithy.model.shapes.BigIntegerShape;
import software.amazon.smithy.model.shapes.BlobShape;
import software.amazon.smithy.model.shapes.BooleanShape;
import software.amazon.smithy.model.shapes.ByteShape;
import software.amazon.smithy.model.shapes.CollectionShape;
import software.amazon.smithy.model.shapes.DocumentShape;
import software.amazon.smithy.model.shapes.DoubleShape;
import software.amazon.smithy.model.shapes.FloatShape;
import software.amazon.smithy.model.shapes.IntegerShape;
import software.amazon.smithy.model.shapes.ListShape;
import software.amazon.smithy.model.shapes.LongShape;
import software.amazon.smithy.model.shapes.MapShape;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.ResourceShape;
import software.amazon.smithy.model.shapes.ServiceShape;
import software.amazon.smithy.model.shapes.SetShape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShortShape;
import software.amazon.smithy.model.shapes.StringShape;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.TimestampShape;
import software.amazon.smithy.model.shapes.UnionShape;
import software.amazon.smithy.model.traits.TraitFactory;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.SetUtils;

enum AstModelLoader {
    INSTANCE;

    private static final String METADATA = "metadata";
    private static final String MEMBERS = "members";
    private static final String SHAPES = "shapes";
    private static final String TRAITS = "traits";
    private static final String TYPE = "type";
    private static final String TARGET = "target";
    private static final String ERRORS = "errors";
    private static final List<String> TOP_LEVEL_PROPERTIES;
    private static final List<String> APPLY_PROPERTIES;
    private static final List<String> SIMPLE_PROPERTY_NAMES;
    private static final List<String> STRUCTURE_AND_UNION_PROPERTY_NAMES;
    private static final List<String> COLLECTION_PROPERTY_NAMES;
    private static final List<String> MAP_PROPERTY_NAMES;
    private static final Set<String> MEMBER_PROPERTIES;
    private static final Set<String> REFERENCE_PROPERTIES;
    private static final Set<String> OPERATION_PROPERTY_NAMES;
    private static final Set<String> RESOURCE_PROPERTIES;
    private static final Set<String> SERVICE_PROPERTIES;

    ModelFile load(TraitFactory traitFactory, ObjectNode model) {
        FullyResolvedModelFile modelFile = new FullyResolvedModelFile(traitFactory);
        LoaderUtils.checkForAdditionalProperties(model, null, TOP_LEVEL_PROPERTIES, modelFile.events());
        this.loadMetadata(model, modelFile);
        this.loadShapes(model, modelFile);
        return modelFile;
    }

    private void loadMetadata(ObjectNode model, FullyResolvedModelFile modelFile) {
        try {
            model.getObjectMember(METADATA).ifPresent(metadata -> {
                for (Map.Entry<String, Node> entry : metadata.getStringMap().entrySet()) {
                    modelFile.putMetadata(entry.getKey(), entry.getValue());
                }
            });
        }
        catch (SourceException e) {
            modelFile.events().add(ValidationEvent.fromSourceException(e));
        }
    }

    private void loadShapes(ObjectNode model, FullyResolvedModelFile modelFile) {
        model.getObjectMember(SHAPES).ifPresent(shapes -> {
            for (Map.Entry<StringNode, Node> entry : shapes.getMembers().entrySet()) {
                ShapeId id = entry.getKey().expectShapeId();
                ObjectNode definition = entry.getValue().expectObjectNode();
                String type = definition.expectStringMember(TYPE).getValue();
                try {
                    this.loadShape(id, type, definition, modelFile);
                }
                catch (SourceException e) {
                    ValidationEvent event = ValidationEvent.fromSourceException(e).toBuilder().shapeId(id).build();
                    modelFile.events().add(event);
                }
            }
        });
    }

    private void loadShape(ShapeId id, String type, ObjectNode value, FullyResolvedModelFile modelFile) {
        switch (type) {
            case "blob": {
                this.loadSimpleShape(id, value, BlobShape.builder(), modelFile);
                break;
            }
            case "boolean": {
                this.loadSimpleShape(id, value, BooleanShape.builder(), modelFile);
                break;
            }
            case "byte": {
                this.loadSimpleShape(id, value, ByteShape.builder(), modelFile);
                break;
            }
            case "short": {
                this.loadSimpleShape(id, value, ShortShape.builder(), modelFile);
                break;
            }
            case "integer": {
                this.loadSimpleShape(id, value, IntegerShape.builder(), modelFile);
                break;
            }
            case "long": {
                this.loadSimpleShape(id, value, LongShape.builder(), modelFile);
                break;
            }
            case "float": {
                this.loadSimpleShape(id, value, FloatShape.builder(), modelFile);
                break;
            }
            case "double": {
                this.loadSimpleShape(id, value, DoubleShape.builder(), modelFile);
                break;
            }
            case "document": {
                this.loadSimpleShape(id, value, DocumentShape.builder(), modelFile);
                break;
            }
            case "bigDecimal": {
                this.loadSimpleShape(id, value, BigDecimalShape.builder(), modelFile);
                break;
            }
            case "bigInteger": {
                this.loadSimpleShape(id, value, BigIntegerShape.builder(), modelFile);
                break;
            }
            case "string": {
                this.loadSimpleShape(id, value, StringShape.builder(), modelFile);
                break;
            }
            case "timestamp": {
                this.loadSimpleShape(id, value, TimestampShape.builder(), modelFile);
                break;
            }
            case "list": {
                this.loadCollection(id, value, ListShape.builder(), modelFile);
                break;
            }
            case "set": {
                this.loadCollection(id, value, SetShape.builder(), modelFile);
                break;
            }
            case "map": {
                this.loadMap(id, value, modelFile);
                break;
            }
            case "resource": {
                this.loadResource(id, value, modelFile);
                break;
            }
            case "service": {
                this.loadService(id, value, modelFile);
                break;
            }
            case "structure": {
                this.loadStructure(id, value, modelFile);
                break;
            }
            case "union": {
                this.loadUnion(id, value, modelFile);
                break;
            }
            case "operation": {
                this.loadOperation(id, value, modelFile);
                break;
            }
            case "apply": {
                LoaderUtils.checkForAdditionalProperties(value, id, APPLY_PROPERTIES, modelFile.events());
                this.applyTraits(id, value.expectObjectMember(TRAITS), modelFile);
                break;
            }
            default: {
                throw new SourceException("Invalid shape `type`: " + type, value);
            }
        }
    }

    private void applyTraits(ShapeId id, ObjectNode traits, FullyResolvedModelFile modelFile) {
        for (Map.Entry<StringNode, Node> traitNode : traits.getMembers().entrySet()) {
            ShapeId traitId = traitNode.getKey().expectShapeId();
            modelFile.onTrait(id, traitId, traitNode.getValue());
        }
    }

    private void applyShapeTraits(ShapeId id, ObjectNode node, FullyResolvedModelFile modelFile) {
        node.getObjectMember(TRAITS).ifPresent(traits -> this.applyTraits(id, (ObjectNode)traits, modelFile));
    }

    private void loadMember(FullyResolvedModelFile modelFile, ShapeId id, ObjectNode targetNode) {
        LoaderUtils.checkForAdditionalProperties(targetNode, id, MEMBER_PROPERTIES, modelFile.events());
        MemberShape.Builder builder = (MemberShape.Builder)((MemberShape.Builder)MemberShape.builder().source(targetNode.getSourceLocation())).id(id);
        ShapeId target = targetNode.expectStringMember(TARGET).expectShapeId();
        builder.target(target);
        this.applyShapeTraits(id, targetNode, modelFile);
        modelFile.onShape(builder);
    }

    private void loadCollection(ShapeId id, ObjectNode node, CollectionShape.Builder<?, ?> builder, FullyResolvedModelFile modelFile) {
        LoaderUtils.checkForAdditionalProperties(node, id, COLLECTION_PROPERTY_NAMES, modelFile.events());
        this.applyShapeTraits(id, node, modelFile);
        this.loadMember(modelFile, id.withMember("member"), node.expectObjectMember("member"));
        modelFile.onShape((AbstractShapeBuilder<?, ?>)builder.id(id).source(node.getSourceLocation()));
    }

    private void loadMap(ShapeId id, ObjectNode node, FullyResolvedModelFile modelFile) {
        LoaderUtils.checkForAdditionalProperties(node, id, MAP_PROPERTY_NAMES, modelFile.events());
        this.loadMember(modelFile, id.withMember("key"), node.expectObjectMember("key"));
        this.loadMember(modelFile, id.withMember("value"), node.expectObjectMember("value"));
        this.applyShapeTraits(id, node, modelFile);
        modelFile.onShape((AbstractShapeBuilder<?, ?>)MapShape.builder().id(id).source(node.getSourceLocation()));
    }

    private void loadOperation(ShapeId id, ObjectNode node, FullyResolvedModelFile modelFile) {
        LoaderUtils.checkForAdditionalProperties(node, id, OPERATION_PROPERTY_NAMES, modelFile.events());
        this.applyShapeTraits(id, node, modelFile);
        OperationShape.Builder builder = ((OperationShape.Builder)((OperationShape.Builder)OperationShape.builder().id(id)).source(node.getSourceLocation())).addErrors(this.loadOptionalTargetList(modelFile, id, node, ERRORS));
        this.loadOptionalTarget(modelFile, id, node, "input").ifPresent(builder::input);
        this.loadOptionalTarget(modelFile, id, node, "output").ifPresent(builder::output);
        modelFile.onShape(builder);
    }

    private void loadResource(ShapeId id, ObjectNode node, FullyResolvedModelFile modelFile) {
        LoaderUtils.checkForAdditionalProperties(node, id, RESOURCE_PROPERTIES, modelFile.events());
        this.applyShapeTraits(id, node, modelFile);
        ResourceShape.Builder builder = (ResourceShape.Builder)((ResourceShape.Builder)ResourceShape.builder().id(id)).source(node.getSourceLocation());
        this.loadOptionalTarget(modelFile, id, node, "put").ifPresent(builder::put);
        this.loadOptionalTarget(modelFile, id, node, "create").ifPresent(builder::create);
        this.loadOptionalTarget(modelFile, id, node, "read").ifPresent(builder::read);
        this.loadOptionalTarget(modelFile, id, node, "update").ifPresent(builder::update);
        this.loadOptionalTarget(modelFile, id, node, "delete").ifPresent(builder::delete);
        this.loadOptionalTarget(modelFile, id, node, "list").ifPresent(builder::list);
        builder.operations(this.loadOptionalTargetList(modelFile, id, node, "operations"));
        builder.collectionOperations(this.loadOptionalTargetList(modelFile, id, node, "collectionOperations"));
        builder.resources(this.loadOptionalTargetList(modelFile, id, node, "resources"));
        node.getObjectMember("identifiers").ifPresent(ids -> {
            for (Map.Entry<StringNode, Node> entry : ids.getMembers().entrySet()) {
                String name = entry.getKey().getValue();
                ShapeId target = this.loadReferenceBody(modelFile, id, entry.getValue());
                builder.addIdentifier(name, target);
            }
        });
        modelFile.onShape(builder);
    }

    private void loadService(ShapeId id, ObjectNode node, FullyResolvedModelFile modelFile) {
        LoaderUtils.checkForAdditionalProperties(node, id, SERVICE_PROPERTIES, modelFile.events());
        this.applyShapeTraits(id, node, modelFile);
        ServiceShape.Builder builder = (ServiceShape.Builder)((ServiceShape.Builder)new ServiceShape.Builder().id(id)).source(node.getSourceLocation());
        builder.version(node.expectStringMember("version").getValue());
        builder.operations(this.loadOptionalTargetList(modelFile, id, node, "operations"));
        builder.resources(this.loadOptionalTargetList(modelFile, id, node, "resources"));
        AstModelLoader.loadServiceRenameIntoBuilder(builder, node);
        builder.addErrors(this.loadOptionalTargetList(modelFile, id, node, ERRORS));
        modelFile.onShape(builder);
    }

    static void loadServiceRenameIntoBuilder(ServiceShape.Builder builder, ObjectNode node) {
        node.getObjectMember("rename").ifPresent(rename -> {
            for (Map.Entry<StringNode, Node> entry : rename.getMembers().entrySet()) {
                ShapeId fromId = entry.getKey().expectShapeId();
                String toName = entry.getValue().expectStringNode().getValue();
                builder.putRename(fromId, toName);
            }
        });
    }

    private void loadSimpleShape(ShapeId id, ObjectNode node, AbstractShapeBuilder<?, ?> builder, FullyResolvedModelFile modelFile) {
        LoaderUtils.checkForAdditionalProperties(node, id, SIMPLE_PROPERTY_NAMES, modelFile.events());
        this.applyShapeTraits(id, node, modelFile);
        modelFile.onShape((AbstractShapeBuilder<?, ?>)((AbstractShapeBuilder)builder.id(id)).source(node.getSourceLocation()));
    }

    private void loadStructure(ShapeId id, ObjectNode node, FullyResolvedModelFile modelFile) {
        LoaderUtils.checkForAdditionalProperties(node, id, STRUCTURE_AND_UNION_PROPERTY_NAMES, modelFile.events());
        modelFile.onShape((AbstractShapeBuilder<?, ?>)((StructureShape.Builder)StructureShape.builder().id(id)).source(node.getSourceLocation()));
        this.finishLoadingStructOrUnionMembers(id, node, modelFile);
    }

    private void loadUnion(ShapeId id, ObjectNode node, FullyResolvedModelFile modelFile) {
        LoaderUtils.checkForAdditionalProperties(node, id, STRUCTURE_AND_UNION_PROPERTY_NAMES, modelFile.events());
        modelFile.onShape((AbstractShapeBuilder<?, ?>)((UnionShape.Builder)UnionShape.builder().id(id)).source(node.getSourceLocation()));
        this.finishLoadingStructOrUnionMembers(id, node, modelFile);
    }

    private void finishLoadingStructOrUnionMembers(ShapeId id, ObjectNode node, FullyResolvedModelFile modelFile) {
        this.applyShapeTraits(id, node, modelFile);
        ObjectNode memberObject = node.getObjectMember(MEMBERS).orElse(Node.objectNode());
        for (Map.Entry<String, Node> entry : memberObject.getStringMap().entrySet()) {
            this.loadMember(modelFile, id.withMember(entry.getKey()), entry.getValue().expectObjectNode());
        }
    }

    private Optional<ShapeId> loadOptionalTarget(FullyResolvedModelFile modelFile, ShapeId id, ObjectNode node, String member) {
        return node.getObjectMember(member).map(r -> this.loadReferenceBody(modelFile, id, (Node)r));
    }

    private ShapeId loadReferenceBody(FullyResolvedModelFile modelFile, ShapeId id, Node reference) {
        ObjectNode referenceObject = reference.expectObjectNode();
        LoaderUtils.checkForAdditionalProperties(referenceObject, id, REFERENCE_PROPERTIES, modelFile.events());
        return referenceObject.expectStringMember(TARGET).expectShapeId();
    }

    private List<ShapeId> loadOptionalTargetList(FullyResolvedModelFile modelFile, ShapeId id, ObjectNode node, String member) {
        return node.getArrayMember(member).map(array -> {
            ArrayList<ShapeId> ids = new ArrayList<ShapeId>(array.size());
            for (Node element : array.getElements()) {
                ids.add(this.loadReferenceBody(modelFile, id, element));
            }
            return ids;
        }).orElseGet(Collections::emptyList);
    }

    static {
        TOP_LEVEL_PROPERTIES = ListUtils.of((Object[])new String[]{"smithy", SHAPES, METADATA});
        APPLY_PROPERTIES = ListUtils.of((Object[])new String[]{TYPE, TRAITS});
        SIMPLE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{TYPE, TRAITS});
        STRUCTURE_AND_UNION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{TYPE, MEMBERS, TRAITS});
        COLLECTION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{TYPE, "member", TRAITS});
        MAP_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{TYPE, "key", "value", TRAITS});
        MEMBER_PROPERTIES = SetUtils.of((Object[])new String[]{TARGET, TRAITS});
        REFERENCE_PROPERTIES = SetUtils.of((Object)TARGET);
        OPERATION_PROPERTY_NAMES = SetUtils.of((Object[])new String[]{TYPE, "input", "output", ERRORS, TRAITS});
        RESOURCE_PROPERTIES = SetUtils.of((Object[])new String[]{TYPE, "create", "read", "update", "delete", "list", "put", "identifiers", "resources", "operations", "collectionOperations", TRAITS});
        SERVICE_PROPERTIES = SetUtils.of((Object[])new String[]{TYPE, "version", "operations", "resources", "rename", ERRORS, TRAITS});
    }
}

