/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.StringJoiner;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.IdlNodeParser;
import software.amazon.smithy.model.loader.IdlTraitParser;
import software.amazon.smithy.model.loader.LoaderVisitor;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.loader.ParserUtils;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.NumberNode;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.BigDecimalShape;
import software.amazon.smithy.model.shapes.BigIntegerShape;
import software.amazon.smithy.model.shapes.BlobShape;
import software.amazon.smithy.model.shapes.BooleanShape;
import software.amazon.smithy.model.shapes.ByteShape;
import software.amazon.smithy.model.shapes.CollectionShape;
import software.amazon.smithy.model.shapes.DocumentShape;
import software.amazon.smithy.model.shapes.DoubleShape;
import software.amazon.smithy.model.shapes.FloatShape;
import software.amazon.smithy.model.shapes.IntegerShape;
import software.amazon.smithy.model.shapes.ListShape;
import software.amazon.smithy.model.shapes.LongShape;
import software.amazon.smithy.model.shapes.MapShape;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.ResourceShape;
import software.amazon.smithy.model.shapes.ServiceShape;
import software.amazon.smithy.model.shapes.SetShape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeType;
import software.amazon.smithy.model.shapes.ShortShape;
import software.amazon.smithy.model.shapes.StringShape;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.TimestampShape;
import software.amazon.smithy.model.shapes.ToShapeId;
import software.amazon.smithy.model.shapes.UnionShape;
import software.amazon.smithy.model.traits.DocumentationTrait;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.model.validation.ValidationUtils;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.SetUtils;
import software.amazon.smithy.utils.SimpleParser;
import software.amazon.smithy.utils.StringUtils;

final class IdlModelParser
extends SimpleParser {
    private static final int MAX_NESTING_LEVEL = 250;
    private static final String PUT_KEY = "put";
    private static final String CREATE_KEY = "create";
    private static final String READ_KEY = "read";
    private static final String UPDATE_KEY = "update";
    private static final String DELETE_KEY = "delete";
    private static final String LIST_KEY = "list";
    private static final String RESOURCES_KEY = "resources";
    private static final String OPERATIONS_KEY = "operations";
    private static final String COLLECTION_OPERATIONS_KEY = "collectionOperations";
    private static final String IDENTIFIERS_KEY = "identifiers";
    private static final String VERSION_KEY = "version";
    private static final String TYPE_KEY = "type";
    static final Collection<String> RESOURCE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "create", "read", "update", "delete", "list", "identifiers", "resources", "operations", "put", "collectionOperations"});
    static final List<String> SERVICE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "version", "operations", "resources"});
    private static final Collection<String> OPERATION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"input", "output", "errors"});
    private static final Set<String> SHAPE_TYPES = new HashSet<String>();
    private final String filename;
    private final LoaderVisitor visitor;
    private String namespace;
    private String definedVersion;
    private TraitEntry pendingDocumentationComment;
    private final Map<String, ShapeId> useShapes = new HashMap<String, ShapeId>();

    IdlModelParser(String filename, String model, LoaderVisitor visitor) {
        super(model, 250);
        this.filename = filename;
        this.visitor = visitor;
    }

    void parse() {
        this.ws();
        this.parseControlSection();
        this.parseMetadataSection();
        this.parseShapeSection();
    }

    public void ws() {
        while (!this.eof()) {
            char c = this.peek();
            if (c == '/') {
                if (this.peekDocComment()) {
                    this.parseDocComment();
                    continue;
                }
                this.parseComment();
                continue;
            }
            if (c != ' ' && c != '\t' && c != '\r' && c != '\n') break;
            this.skip();
        }
    }

    public ModelSyntaxException syntax(String message) {
        String formatted = String.format("Parse error at line %d, column %d near `%s`: %s", this.line(), this.column(), this.peekDebugMessage(), message);
        return new ModelSyntaxException(formatted, this.filename, this.line(), this.column());
    }

    private void parseControlSection() {
        while (this.peek() == '$') {
            this.expect('$');
            this.ws();
            String key = IdlNodeParser.parseNodeObjectKey(this);
            this.ws();
            this.expect(':');
            this.ws();
            if (key.equals(VERSION_KEY) && this.definedVersion != null) {
                throw this.syntax("Cannot define multiple versions in the same file");
            }
            Node value = IdlNodeParser.parseNode(this);
            if (key.equals(VERSION_KEY)) {
                this.onVersion(value);
            } else {
                this.visitor.onError(ValidationEvent.builder().id("Model").sourceLocation(value).severity(Severity.WARNING).message(String.format("Unknown control statement `%s` with value `%s", key, Node.printJson(value))).build());
            }
            this.br();
            this.ws();
        }
    }

    private void onVersion(Node value) {
        String parsedVersion;
        if (!value.isStringNode()) {
            value.expectStringNode("The $version control statement must have a string value, but found " + Node.printJson(value));
        }
        if (!this.visitor.isVersionSupported(parsedVersion = value.expectStringNode().getValue())) {
            throw this.syntax("Unsupported Smithy version number: " + parsedVersion);
        }
        this.definedVersion = parsedVersion;
    }

    private void parseMetadataSection() {
        while (this.peek() == 'm') {
            this.expect('m');
            this.expect('e');
            this.expect('t');
            this.expect('a');
            this.expect('d');
            this.expect('a');
            this.expect('t');
            this.expect('a');
            this.ws();
            String key = IdlNodeParser.parseNodeObjectKey(this);
            this.ws();
            this.expect('=');
            this.ws();
            this.visitor.onMetadata(key, IdlNodeParser.parseNode(this));
            this.br();
            this.ws();
        }
    }

    private void parseShapeSection() {
        if (this.peek() == 'n') {
            this.expect('n');
            this.expect('a');
            this.expect('m');
            this.expect('e');
            this.expect('s');
            this.expect('p');
            this.expect('a');
            this.expect('c');
            this.expect('e');
            this.ws();
            int start = this.position();
            ParserUtils.consumeNamespace(this);
            this.namespace = this.sliceFrom(start);
            this.br();
            this.clearPendingDocs();
            this.ws();
            this.parseUseSection();
            this.parseShapeStatements();
        } else if (!this.eof()) {
            if (!ParserUtils.isIdentifierStart(this.peek())) {
                throw this.syntax("Expected a namespace definition, but found unexpected syntax");
            }
            throw this.syntax("A namespace must be defined before a use statement or shapes");
        }
    }

    private void parseUseSection() {
        while (this.peek() == 'u' && this.peek(1) == 's') {
            this.expect('u');
            this.expect('s');
            this.expect('e');
            this.ws();
            int start = this.position();
            ParserUtils.consumeNamespace(this);
            this.expect('#');
            ParserUtils.consumeIdentifier(this);
            String lexeme = this.sliceFrom(start);
            this.br();
            this.clearPendingDocs();
            this.ws();
            ShapeId target = ShapeId.from(lexeme);
            ShapeId previous = this.useShapes.put(target.getName(), target);
            if (previous == null) continue;
            throw this.syntax(String.format("Cannot use name `%s` because it conflicts with `%s`", target, previous));
        }
    }

    private void parseShapeStatements() {
        while (!this.eof()) {
            boolean docsOnly;
            this.ws();
            if (this.peek() == 'a') {
                this.parseApplyStatement();
                continue;
            }
            boolean bl = docsOnly = this.pendingDocumentationComment != null;
            List<TraitEntry> traits = this.parseDocsAndTraits();
            if (!this.parseShapeDefinition(traits, docsOnly)) continue;
            this.parseShape(traits);
        }
    }

    private void clearPendingDocs() {
        this.pendingDocumentationComment = null;
    }

    private boolean parseShapeDefinition(List<TraitEntry> traits, boolean docsOnly) {
        if (this.eof()) {
            return !traits.isEmpty() && !docsOnly;
        }
        return true;
    }

    private List<TraitEntry> parseDocsAndTraits() {
        TraitEntry docComment = this.pendingDocumentationComment;
        this.clearPendingDocs();
        this.ws();
        List<TraitEntry> traits = IdlTraitParser.parseTraits(this);
        if (docComment != null) {
            traits.add(docComment);
        }
        this.ws();
        return traits;
    }

    private void parseShape(List<TraitEntry> traits) {
        SourceLocation location = this.currentLocation();
        String shapeType = ParserUtils.parseIdentifier(this);
        if (!SHAPE_TYPES.contains(shapeType)) {
            switch (shapeType) {
                case "use": {
                    throw this.syntax("A use statement must come before any shape definition");
                }
                case "namespace": {
                    throw this.syntax("Only a single namespace can be declared per/file");
                }
                case "metadata": {
                    throw this.syntax("Metadata statements must appear before a namespace statement");
                }
            }
            throw this.syntax("Unexpected shape type: " + shapeType);
        }
        this.ws();
        ShapeId id = this.parseShapeName();
        switch (shapeType) {
            case "service": {
                this.parseServiceStatement(id, location);
                break;
            }
            case "resource": {
                this.parseResourceStatement(id, location);
                break;
            }
            case "operation": {
                this.parseOperationStatement(id, location);
                break;
            }
            case "structure": {
                this.parseStructuredShape(id, location, StructureShape.builder());
                break;
            }
            case "union": {
                this.parseStructuredShape(id, location, UnionShape.builder());
                break;
            }
            case "list": {
                this.parseCollection(id, location, ListShape.builder());
                break;
            }
            case "set": {
                this.parseCollection(id, location, SetShape.builder());
                break;
            }
            case "map": {
                this.parseMapStatement(id, location);
                break;
            }
            case "boolean": {
                this.parseSimpleShape(id, location, BooleanShape.builder());
                break;
            }
            case "string": {
                this.parseSimpleShape(id, location, StringShape.builder());
                break;
            }
            case "blob": {
                this.parseSimpleShape(id, location, BlobShape.builder());
                break;
            }
            case "byte": {
                this.parseSimpleShape(id, location, ByteShape.builder());
                break;
            }
            case "short": {
                this.parseSimpleShape(id, location, ShortShape.builder());
                break;
            }
            case "integer": {
                this.parseSimpleShape(id, location, IntegerShape.builder());
                break;
            }
            case "long": {
                this.parseSimpleShape(id, location, LongShape.builder());
                break;
            }
            case "float": {
                this.parseSimpleShape(id, location, FloatShape.builder());
                break;
            }
            case "document": {
                this.parseSimpleShape(id, location, DocumentShape.builder());
                break;
            }
            case "double": {
                this.parseSimpleShape(id, location, DoubleShape.builder());
                break;
            }
            case "bigInteger": {
                this.parseSimpleShape(id, location, BigIntegerShape.builder());
                break;
            }
            case "bigDecimal": {
                this.parseSimpleShape(id, location, BigDecimalShape.builder());
                break;
            }
            case "timestamp": {
                this.parseSimpleShape(id, location, TimestampShape.builder());
                break;
            }
            default: {
                throw this.syntax("Unexpected shape type: " + shapeType);
            }
        }
        this.addTraits(id, traits);
        this.clearPendingDocs();
        this.br();
    }

    private ShapeId parseShapeName() {
        String name = ParserUtils.parseIdentifier(this);
        if (this.useShapes.containsKey(name)) {
            throw this.syntax(String.format("shape name `%s` conflicts with imported shape `%s`", name, this.useShapes.get(name)));
        }
        return ShapeId.fromRelative(this.namespace, name);
    }

    private void parseSimpleShape(ShapeId id, SourceLocation location, AbstractShapeBuilder builder) {
        this.visitor.onShape((AbstractShapeBuilder)((AbstractShapeBuilder)builder.source(location)).id(id));
    }

    private void parseCollection(ShapeId id, SourceLocation location, CollectionShape.Builder builder) {
        this.ws();
        ((AbstractShapeBuilder)builder.id(id)).source(location);
        this.parseMembers(id, SetUtils.of((Object)"member"));
        this.visitor.onShape((AbstractShapeBuilder)builder.id(id));
    }

    private void parseMembers(ShapeId id, Set<String> requiredMembers) {
        HashSet<String> definedMembers = new HashSet<String>();
        HashSet<String> remaining = requiredMembers.isEmpty() ? requiredMembers : new HashSet<String>(requiredMembers);
        this.ws();
        this.expect('{');
        this.clearPendingDocs();
        this.ws();
        if (this.peek() != '}') {
            this.parseMember(id, requiredMembers, definedMembers, remaining);
            while (!this.eof()) {
                this.ws();
                if (this.peek() != ',') break;
                this.expect(',');
                this.clearPendingDocs();
                this.ws();
                if (this.peek() == '}') break;
                this.parseMember(id, requiredMembers, definedMembers, remaining);
            }
        }
        if (!remaining.isEmpty()) {
            throw this.syntax("Missing required members of shape `" + id + "`: [" + ValidationUtils.tickedList(remaining) + ']');
        }
        this.expect('}');
    }

    private String parseMember(ShapeId parent, Set<String> required, Set<String> defined, Set<String> remaining) {
        List<TraitEntry> memberTraits = this.parseDocsAndTraits();
        SourceLocation memberLocation = this.currentLocation();
        String memberName = ParserUtils.parseIdentifier(this);
        if (defined.contains(memberName)) {
            throw this.syntax("Duplicate member of " + parent + ": '" + memberName + '\'');
        }
        defined.add(memberName);
        remaining.remove(memberName);
        if (!required.isEmpty() && !required.contains(memberName)) {
            throw this.syntax("Unexpected member of " + parent + ": '" + memberName + '\'');
        }
        this.ws();
        this.expect(':');
        this.ws();
        ShapeId memberId = parent.withMember(memberName);
        MemberShape.Builder memberBuilder = (MemberShape.Builder)((MemberShape.Builder)MemberShape.builder().id(memberId)).source(memberLocation);
        SourceLocation targetLocation = this.currentLocation();
        String target = ParserUtils.parseShapeId(this);
        this.visitor.onShape(memberBuilder);
        this.onShapeTarget(target, targetLocation, memberBuilder::target);
        this.addTraits(memberId, memberTraits);
        return memberName;
    }

    private void parseMapStatement(ShapeId id, SourceLocation location) {
        this.parseMembers(id, SetUtils.of((Object[])new String[]{"key", "value"}));
        this.visitor.onShape((AbstractShapeBuilder)((MapShape.Builder)MapShape.builder().id(id)).source(location));
    }

    private void parseStructuredShape(ShapeId id, SourceLocation location, AbstractShapeBuilder builder) {
        this.visitor.onShape((AbstractShapeBuilder)((AbstractShapeBuilder)builder.id(id)).source(location));
        this.parseMembers(id, Collections.emptySet());
    }

    private void parseOperationStatement(ShapeId id, SourceLocation location) {
        this.ws();
        OperationShape.Builder builder = (OperationShape.Builder)((OperationShape.Builder)OperationShape.builder().id(id)).source(location);
        ObjectNode node = IdlNodeParser.parseObjectNode(this);
        this.visitor.checkForAdditionalProperties(node, id, OPERATION_PROPERTY_NAMES);
        this.visitor.onShape(builder);
        node.getStringMember("input").ifPresent(input -> this.onShapeTarget(input.getValue(), (FromSourceLocation)input, builder::input));
        node.getStringMember("output").ifPresent(output -> this.onShapeTarget(output.getValue(), (FromSourceLocation)output, builder::output));
        node.getArrayMember("errors").ifPresent(errors -> {
            for (StringNode value : errors.getElementsAs(StringNode.class)) {
                this.onShapeTarget(value.getValue(), value, builder::addError);
            }
        });
    }

    private void parseServiceStatement(ShapeId id, SourceLocation location) {
        this.ws();
        ServiceShape.Builder builder = (ServiceShape.Builder)((ServiceShape.Builder)new ServiceShape.Builder().id(id)).source(location);
        ObjectNode shapeNode = IdlNodeParser.parseObjectNode(this);
        this.visitor.checkForAdditionalProperties(shapeNode, id, SERVICE_PROPERTY_NAMES);
        builder.version(shapeNode.expectStringMember(VERSION_KEY).getValue());
        this.visitor.onShape(builder);
        IdlModelParser.optionalIdList(shapeNode, id.getNamespace(), OPERATIONS_KEY).forEach(builder::addOperation);
        IdlModelParser.optionalIdList(shapeNode, id.getNamespace(), RESOURCES_KEY).forEach(builder::addResource);
    }

    private static Optional<ShapeId> optionalId(ObjectNode node, String namespace, String name) {
        return node.getStringMember(name).map(stringNode -> stringNode.expectShapeId(namespace));
    }

    private static List<ShapeId> optionalIdList(ObjectNode node, String namespace, String name) {
        return node.getArrayMember(name).map(array -> array.getElements().stream().map(Node::expectStringNode).map(s -> s.expectShapeId(namespace)).collect(Collectors.toList())).orElseGet(Collections::emptyList);
    }

    private void parseResourceStatement(ShapeId id, SourceLocation location) {
        this.ws();
        ResourceShape.Builder builder = (ResourceShape.Builder)((ResourceShape.Builder)ResourceShape.builder().id(id)).source(location);
        this.visitor.onShape(builder);
        ObjectNode shapeNode = IdlNodeParser.parseObjectNode(this);
        this.visitor.checkForAdditionalProperties(shapeNode, id, RESOURCE_PROPERTY_NAMES);
        IdlModelParser.optionalId(shapeNode, id.getNamespace(), PUT_KEY).ifPresent(builder::put);
        IdlModelParser.optionalId(shapeNode, id.getNamespace(), CREATE_KEY).ifPresent(builder::create);
        IdlModelParser.optionalId(shapeNode, id.getNamespace(), READ_KEY).ifPresent(builder::read);
        IdlModelParser.optionalId(shapeNode, id.getNamespace(), UPDATE_KEY).ifPresent(builder::update);
        IdlModelParser.optionalId(shapeNode, id.getNamespace(), DELETE_KEY).ifPresent(builder::delete);
        IdlModelParser.optionalId(shapeNode, id.getNamespace(), LIST_KEY).ifPresent(builder::list);
        IdlModelParser.optionalIdList(shapeNode, id.getNamespace(), OPERATIONS_KEY).forEach(builder::addOperation);
        IdlModelParser.optionalIdList(shapeNode, id.getNamespace(), RESOURCES_KEY).forEach(builder::addResource);
        IdlModelParser.optionalIdList(shapeNode, id.getNamespace(), COLLECTION_OPERATIONS_KEY).forEach(builder::addCollectionOperation);
        shapeNode.getObjectMember(IDENTIFIERS_KEY).ifPresent(ids -> {
            for (Map.Entry<StringNode, Node> entry : ids.getMembers().entrySet()) {
                String name = entry.getKey().getValue();
                StringNode target = entry.getValue().expectStringNode();
                this.onShapeTarget(target.getValue(), target, targetId -> builder.addIdentifier(name, (ToShapeId)targetId));
            }
        });
    }

    private void parseComment() {
        this.expect('/');
        this.consumeRemainingCharactersOnLine();
    }

    private void parseDocComment() {
        SourceLocation location = this.currentLocation();
        StringJoiner joiner = new StringJoiner("\n");
        do {
            joiner.add(this.parseDocCommentLine());
        } while (this.peekDocComment());
        this.pendingDocumentationComment = new TraitEntry(DocumentationTrait.ID.toString(), new StringNode(joiner.toString(), location), false);
    }

    private boolean peekDocComment() {
        return this.peek() == '/' && this.peek(1) == '/' && this.peek(2) == '/';
    }

    private String parseDocCommentLine() {
        this.expect('/');
        this.expect('/');
        this.expect('/');
        if (this.peek() == ' ') {
            this.skip();
        }
        int start = this.position();
        this.consumeRemainingCharactersOnLine();
        this.br();
        this.sp();
        return StringUtils.stripEnd((String)this.sliceFrom(start), (String)" \t\r\n");
    }

    private void parseApplyStatement() {
        this.expect('a');
        this.expect('p');
        this.expect('p');
        this.expect('l');
        this.expect('y');
        this.ws();
        SourceLocation location = this.currentLocation();
        String name = ParserUtils.parseShapeId(this);
        this.ws();
        TraitEntry traitEntry = IdlTraitParser.parseTraitValue(this);
        this.onShapeTarget(name, location, id -> this.onDeferredTrait((ShapeId)id, traitEntry.traitName, traitEntry.value, traitEntry.isAnnotation));
        this.clearPendingDocs();
        this.br();
        this.ws();
    }

    void onShapeTarget(String target, FromSourceLocation sourceLocation, Consumer<ShapeId> resolver) {
        ShapeId expectedId;
        if (this.useShapes.containsKey(target)) {
            resolver.accept(this.useShapes.get(target));
            return;
        }
        ShapeId shapeId = expectedId = this.namespace == null ? ShapeId.from(target) : ShapeId.fromOptionalNamespace(this.namespace, target);
        if (this.isRealizedShapeId(expectedId, target)) {
            resolver.accept(expectedId);
        } else {
            this.visitor.addForwardReference(expectedId, resolver);
        }
    }

    private boolean isRealizedShapeId(ShapeId expectedId, String target) {
        return Objects.equals(this.namespace, "smithy.api") || this.visitor.hasDefinedShape(expectedId) || target.contains("#");
    }

    private void addTraits(ShapeId id, List<TraitEntry> traits) {
        for (TraitEntry traitEntry : traits) {
            this.onDeferredTrait(id, traitEntry.traitName, traitEntry.value, traitEntry.isAnnotation);
        }
    }

    private void onDeferredTrait(ShapeId target, String traitName, Node traitValue, boolean isAnnotation) {
        this.onShapeTarget(traitName, traitValue.getSourceLocation(), id -> {
            if (isAnnotation) {
                this.visitor.onAnnotationTrait(target, (ShapeId)id, traitValue.expectNullNode());
            } else {
                this.visitor.onTrait(target, (ShapeId)id, traitValue);
            }
        });
    }

    SourceLocation currentLocation() {
        return new SourceLocation(this.filename, this.line(), this.column());
    }

    NumberNode parseNumberNode() {
        SourceLocation location = this.currentLocation();
        String lexeme = ParserUtils.parseNumber(this);
        if (lexeme.contains("e") || lexeme.contains(".")) {
            return new NumberNode(Double.valueOf(lexeme), location);
        }
        return new NumberNode(Long.parseLong(lexeme), location);
    }

    private String peekDebugMessage() {
        StringBuilder result = new StringBuilder(this.expression().length());
        char c = this.peek();
        if (c == ' ' || ParserUtils.isIdentifierStart(c) || ParserUtils.isDigit(c)) {
            int i;
            if (c == ' ') {
                result.append(' ');
            }
            int n = i = c == ' ' ? 1 : 0;
            while (i < 16 && (ParserUtils.isIdentifierStart(c = this.peek(i)) || ParserUtils.isDigit(c))) {
                result.append(c);
                ++i;
            }
            return result.toString();
        }
        for (int i = 0; i < 2; ++i) {
            char peek = this.peek(i);
            if (peek == '\u0000') {
                result.append("[EOF]");
                break;
            }
            result.append(peek);
        }
        return result.toString();
    }

    static {
        for (ShapeType type : ShapeType.values()) {
            if (type == ShapeType.MEMBER) continue;
            SHAPE_TYPES.add(type.toString());
        }
    }

    static final class TraitEntry {
        final String traitName;
        final Node value;
        final boolean isAnnotation;

        TraitEntry(String traitName, Node value, boolean isAnnotation) {
            this.traitName = traitName;
            this.value = value;
            this.isAnnotation = isAnnotation;
        }
    }
}

