/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.cli.dependencies;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import software.amazon.smithy.build.model.MavenRepository;
import software.amazon.smithy.cli.dependencies.DependencyResolver;
import software.amazon.smithy.cli.dependencies.DependencyResolverException;
import software.amazon.smithy.cli.dependencies.ResolvedArtifact;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;

public final class FileCacheResolver
implements DependencyResolver {
    private static final Duration SMITHY_MAVEN_TTL = Duration.parse("P1D");
    private static final Logger LOGGER = Logger.getLogger(FileCacheResolver.class.getName());
    private final DependencyResolver delegate;
    private final File location;
    private final long referenceTimeInMillis;

    public FileCacheResolver(File location, long referenceTimeInMillis, DependencyResolver delegate) {
        this.location = location;
        this.referenceTimeInMillis = referenceTimeInMillis;
        this.delegate = delegate;
    }

    @Override
    public void addRepository(MavenRepository repository) {
        this.delegate.addRepository(repository);
    }

    @Override
    public void addDependency(String coordinates) {
        this.delegate.addDependency(coordinates);
    }

    @Override
    public List<ResolvedArtifact> resolve() {
        List<ResolvedArtifact> cachedResult = this.load();
        if (!cachedResult.isEmpty()) {
            LOGGER.fine(() -> "Classpath found in cache: " + cachedResult);
            return cachedResult;
        }
        List<ResolvedArtifact> result = this.delegate.resolve();
        this.save(result);
        return result;
    }

    private List<ResolvedArtifact> load() {
        ObjectNode node;
        long cacheLastModifiedMillis = this.location.lastModified();
        long currentTimeMillis = new Date().getTime();
        long ttlMillis = SMITHY_MAVEN_TTL.toMillis();
        if (this.location.length() == 0L) {
            return Collections.emptyList();
        }
        if (!this.isCacheValid(cacheLastModifiedMillis)) {
            LOGGER.fine("Invalidating dependency cache: config is newer than the cache");
            this.invalidate();
            return Collections.emptyList();
        }
        if (currentTimeMillis - cacheLastModifiedMillis > ttlMillis) {
            LOGGER.fine(() -> "Invalidating dependency cache: Cache exceeded TTL (TTL: " + ttlMillis + ")");
            this.invalidate();
            return Collections.emptyList();
        }
        try {
            InputStream stream = Files.newInputStream(this.location.toPath(), new OpenOption[0]);
            Object object = null;
            try {
                node = Node.parse((InputStream)stream, (String)this.location.toString()).expectObjectNode();
            }
            catch (Throwable throwable) {
                object = throwable;
                throw throwable;
            }
            finally {
                if (stream != null) {
                    if (object != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable) {
                            ((Throwable)object).addSuppressed(throwable);
                        }
                    } else {
                        stream.close();
                    }
                }
            }
        }
        catch (IOException | ModelSyntaxException e) {
            throw new DependencyResolverException("Error loading dependency cache file from " + this.location, e);
        }
        ArrayList<ResolvedArtifact> result = new ArrayList<ResolvedArtifact>(node.getStringMap().size());
        for (Map.Entry entry : node.getStringMap().entrySet()) {
            Path artifactLocation = Paths.get(((Node)entry.getValue()).expectStringNode().getValue(), new String[0]);
            long lastModifiedOfArtifact = artifactLocation.toFile().lastModified();
            if (lastModifiedOfArtifact == 0L || lastModifiedOfArtifact > cacheLastModifiedMillis) {
                LOGGER.fine(() -> "Invalidating dependency cache: artifact is newer than cache: " + artifactLocation);
                this.invalidate();
                return Collections.emptyList();
            }
            result.add(ResolvedArtifact.fromCoordinates(artifactLocation, (String)entry.getKey()));
        }
        return result;
    }

    private void save(List<ResolvedArtifact> result) {
        Path filePath = this.location.toPath();
        Path parent = filePath.getParent();
        if (parent == null) {
            throw new DependencyResolverException("Invalid classpath cache location: " + this.location);
        }
        try {
            Files.createDirectories(parent, new FileAttribute[0]);
            ObjectNode.Builder builder = Node.objectNodeBuilder();
            for (ResolvedArtifact artifact : result) {
                builder.withMember(artifact.getCoordinates(), artifact.getPath().toString());
            }
            ObjectNode objectNode = builder.build();
            Files.write(filePath, Node.printJson((Node)objectNode).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            throw new DependencyResolverException("Unable to write classpath cache file: " + e.getMessage(), e);
        }
    }

    private boolean isCacheValid(long cacheLastModifiedMillis) {
        return this.referenceTimeInMillis <= cacheLastModifiedMillis;
    }

    private void invalidate() {
        if (this.location.exists() && !this.location.delete()) {
            LOGGER.warning("Unable to invalidate dependency cache file: " + this.location);
        }
    }
}

