/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.build;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.Future;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import software.amazon.smithy.build.FileManifest;
import software.amazon.smithy.build.PluginContext;
import software.amazon.smithy.build.ProjectionResult;
import software.amazon.smithy.build.ProjectionTransformer;
import software.amazon.smithy.build.SmithyBuild;
import software.amazon.smithy.build.SmithyBuildException;
import software.amazon.smithy.build.SmithyBuildPlugin;
import software.amazon.smithy.build.TransformContext;
import software.amazon.smithy.build.UnknownProjectionException;
import software.amazon.smithy.build.UnknownTransformException;
import software.amazon.smithy.build.model.ProjectionConfig;
import software.amazon.smithy.build.model.SmithyBuildConfig;
import software.amazon.smithy.build.model.TransformConfig;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.loader.ModelAssembler;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.transform.ModelTransformer;
import software.amazon.smithy.model.validation.ValidatedResult;
import software.amazon.smithy.utils.Pair;
import software.amazon.smithy.utils.SmithyBuilder;

final class SmithyBuildImpl {
    private static final Logger LOGGER = Logger.getLogger(SmithyBuild.class.getName());
    private static final Pattern PATTERN = Pattern.compile("^[A-Za-z0-9\\-_.]+$");
    private final SmithyBuildConfig config;
    private final Function<Path, FileManifest> fileManifestFactory;
    private final Supplier<ModelAssembler> modelAssemblerSupplier;
    private final Path outputDirectory;
    private final Map<String, List<Pair<ObjectNode, ProjectionTransformer>>> transformers = new HashMap<String, List<Pair<ObjectNode, ProjectionTransformer>>>();
    private final ModelTransformer modelTransformer;
    private final Function<String, Optional<ProjectionTransformer>> transformFactory;
    private final Function<String, Optional<SmithyBuildPlugin>> pluginFactory;
    private final Model model;
    private final ClassLoader pluginClassLoader;
    private final Set<Path> sources;
    private final Predicate<String> projectionFilter;
    private final Predicate<String> pluginFilter;

    SmithyBuildImpl(SmithyBuild builder) {
        this.config = SmithyBuildImpl.prepareConfig((SmithyBuildConfig)SmithyBuilder.requiredState((String)"config", (Object)builder.config));
        this.sources = builder.sources;
        this.fileManifestFactory = builder.fileManifestFactory != null ? builder.fileManifestFactory : FileManifest::create;
        this.modelAssemblerSupplier = builder.modelAssemblerSupplier != null ? builder.modelAssemblerSupplier : Model::assembler;
        this.modelTransformer = builder.modelTransformer != null ? builder.modelTransformer : ModelTransformer.create();
        this.transformFactory = builder.transformFactory != null ? builder.transformFactory : ProjectionTransformer.createServiceFactory(this.getClass().getClassLoader());
        this.pluginFactory = builder.pluginFactory != null ? builder.pluginFactory : SmithyBuildPlugin.createServiceFactory(this.getClass().getClassLoader());
        Model model = this.model = builder.model != null ? builder.model : Model.builder().build();
        this.outputDirectory = builder.outputDirectory != null ? builder.outputDirectory : (this.config.getOutputDirectory().isPresent() ? Paths.get(this.config.getOutputDirectory().get(), new String[0]) : Paths.get(".", new String[0]).toAbsolutePath().normalize().resolve("build").resolve("smithy"));
        this.config.getProjections().forEach((projectionName, projectionConfig) -> this.transformers.put((String)projectionName, this.createTransformers((String)projectionName, (ProjectionConfig)projectionConfig)));
        this.pluginClassLoader = builder.pluginClassLoader;
        this.projectionFilter = builder.projectionFilter;
        this.pluginFilter = builder.pluginFilter;
    }

    private static SmithyBuildConfig prepareConfig(SmithyBuildConfig config) {
        ProjectionConfig sourceProjection;
        if (!config.getProjections().containsKey("source")) {
            HashMap<String, ProjectionConfig> projections = new HashMap<String, ProjectionConfig>(config.getProjections());
            projections.put("source", ProjectionConfig.builder().build());
            config = config.toBuilder().projections(projections).build();
        }
        if (!(sourceProjection = config.getProjections().get("source")).getTransforms().isEmpty()) {
            throw new SmithyBuildException("The source projection cannot contain any transforms");
        }
        config.getPlugins().keySet().forEach(p -> SmithyBuildImpl.validatePluginName("[top-level]", p));
        for (Map.Entry<String, ProjectionConfig> entry : config.getProjections().entrySet()) {
            String projectionName = entry.getKey();
            if (!PATTERN.matcher(projectionName).matches()) {
                throw new SmithyBuildException(String.format("Invalid Smithy build projection name `%s`. Projection names must match the following regex: %s", projectionName, PATTERN));
            }
            entry.getValue().getPlugins().keySet().forEach(p -> SmithyBuildImpl.validatePluginName((String)entry.getKey(), p));
            entry.getValue().getTransforms().forEach(t -> SmithyBuildImpl.validateTransformName((String)entry.getKey(), t.getName()));
        }
        return config;
    }

    private static void validateTransformName(String projection, String transformName) {
        if (!PATTERN.matcher(transformName).matches()) {
            throw new SmithyBuildException(String.format("Invalid transform name `%s` found in the `%s` projection.  Transform names must match the following regex: %s", transformName, projection, PATTERN));
        }
    }

    private static void validatePluginName(String projection, String plugin) {
        if (!PATTERN.matcher(plugin).matches()) {
            throw new SmithyBuildException(String.format("Invalid plugin name `%s` found in the `%s` projection.  Plugin names must match the following regex: %s", plugin, projection, PATTERN));
        }
    }

    void applyAllProjections(Consumer<ProjectionResult> projectionResultConsumer, BiConsumer<String, Throwable> projectionExceptionConsumer) {
        ValidatedResult<Model> resolvedModel = this.createBaseModel();
        ArrayList<Callable<Void>> parallelProjections = new ArrayList<Callable<Void>>();
        ArrayList<String> parallelProjectionNameOrder = new ArrayList<String>();
        for (Map.Entry<String, ProjectionConfig> entry : this.config.getProjections().entrySet()) {
            String name = entry.getKey();
            ProjectionConfig config = entry.getValue();
            if (config.isAbstract() || !this.projectionFilter.test(name)) continue;
            boolean isSerial = this.resolvePlugins(config).keySet().stream().anyMatch(pluginName -> {
                Optional<SmithyBuildPlugin> plugin = this.pluginFactory.apply((String)pluginName);
                return plugin.isPresent() && plugin.get().isSerial();
            });
            if (isSerial) {
                this.executeSerialProjection(resolvedModel, name, config, projectionResultConsumer, projectionExceptionConsumer);
                continue;
            }
            parallelProjectionNameOrder.add(name);
            parallelProjections.add(() -> {
                this.executeSerialProjection(resolvedModel, name, config, projectionResultConsumer, projectionExceptionConsumer);
                return null;
            });
        }
        if (parallelProjections.size() == 1) {
            try {
                ((Callable)parallelProjections.get(0)).call();
            }
            catch (Throwable e) {
                if (e instanceof RuntimeException) {
                    throw (RuntimeException)e;
                }
                throw new RuntimeException(e);
            }
        } else if (!parallelProjections.isEmpty()) {
            this.executeParallelProjections(parallelProjections, parallelProjectionNameOrder, projectionExceptionConsumer);
        }
    }

    private void executeSerialProjection(ValidatedResult<Model> baseModel, String name, ProjectionConfig config, Consumer<ProjectionResult> projectionResultConsumer, BiConsumer<String, Throwable> projectionExceptionConsumer) {
        ProjectionResult result = null;
        try {
            result = this.applyProjection(name, config, baseModel);
        }
        catch (Throwable e) {
            projectionExceptionConsumer.accept(name, e);
        }
        if (result != null) {
            projectionResultConsumer.accept(result);
        }
    }

    private void executeParallelProjections(List<Callable<Void>> parallelProjections, List<String> parallelProjectionNameOrder, BiConsumer<String, Throwable> projectionExceptionConsumer) {
        ForkJoinPool executor = ForkJoinPool.commonPool();
        try {
            List<Future<Void>> futures = executor.invokeAll(parallelProjections);
            for (int i = 0; i < futures.size(); ++i) {
                try {
                    futures.get(i).get();
                    continue;
                }
                catch (ExecutionException e) {
                    Throwable cause = e.getCause() != null ? e.getCause() : e;
                    String failedProjectionName = parallelProjectionNameOrder.get(i);
                    projectionExceptionConsumer.accept(failedProjectionName, cause);
                }
            }
        }
        catch (InterruptedException e) {
            throw new SmithyBuildException(e.getMessage(), e);
        }
    }

    private ValidatedResult<Model> createBaseModel() {
        if (!this.config.getImports().isEmpty()) {
            LOGGER.fine(() -> "Merging the following imports into the loaded model: " + this.config.getImports());
        }
        ModelAssembler assembler = this.modelAssemblerSupplier.get().addModel(this.model);
        this.config.getImports().forEach(arg_0 -> ((ModelAssembler)assembler).addImport(arg_0));
        return assembler.assemble();
    }

    private ProjectionResult applyProjection(String projectionName, ProjectionConfig projection, ValidatedResult<Model> baseModel) {
        Model resolvedModel = (Model)baseModel.unwrap();
        LOGGER.fine(() -> String.format("Creating the `%s` projection", projectionName));
        if (!projection.getImports().isEmpty()) {
            LOGGER.fine(() -> String.format("Merging the following `%s` projection imports into the loaded model: %s", projectionName, projection.getImports()));
            ModelAssembler assembler = this.modelAssemblerSupplier.get().addModel(resolvedModel);
            projection.getImports().forEach(arg_0 -> ((ModelAssembler)assembler).addImport(arg_0));
            baseModel = assembler.assemble();
            if (!baseModel.getResult().isPresent()) {
                LOGGER.severe(String.format("The model could not be merged with the following imports: [%s]", projection.getImports()));
                return ProjectionResult.builder().projectionName(projectionName).events(baseModel.getValidationEvents()).build();
            }
            resolvedModel = (Model)baseModel.unwrap();
        }
        Path baseProjectionDir = this.outputDirectory.resolve(projectionName);
        Model projectedModel = resolvedModel;
        ValidatedResult modelResult = baseModel;
        if (!projection.getTransforms().isEmpty()) {
            projectedModel = this.applyProjectionTransforms(baseModel, resolvedModel, projectionName, Collections.emptySet());
            modelResult = this.modelAssemblerSupplier.get().addModel(projectedModel).assemble();
        }
        ProjectionResult.Builder resultBuilder = ProjectionResult.builder().projectionName(projectionName).model(projectedModel).events(modelResult.getValidationEvents());
        for (Map.Entry<String, ObjectNode> entry : this.resolvePlugins(projection).entrySet()) {
            if (!this.pluginFilter.test(entry.getKey())) continue;
            this.applyPlugin(projectionName, projection, baseProjectionDir, entry.getKey(), entry.getValue(), projectedModel, resolvedModel, (ValidatedResult<Model>)modelResult, resultBuilder);
        }
        return resultBuilder.build();
    }

    private Model applyProjectionTransforms(ValidatedResult<Model> baseModel, Model currentModel, String projectionName, Set<String> visited) {
        Model originalModel = (Model)baseModel.unwrap();
        for (Pair<ObjectNode, ProjectionTransformer> transformerBinding : this.transformers.get(projectionName)) {
            TransformContext context = TransformContext.builder().model(currentModel).originalModel(originalModel).originalModelValidationEvents(baseModel.getValidationEvents()).transformer(this.modelTransformer).projectionName(projectionName).sources(this.sources).settings((ObjectNode)transformerBinding.left).build();
            currentModel = ((ProjectionTransformer)transformerBinding.right).transform(context);
            currentModel = this.applyQueuedProjections(context, currentModel, visited);
        }
        return currentModel;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void applyPlugin(String projectionName, ProjectionConfig projection, Path baseProjectionDir, String pluginName, ObjectNode pluginSettings, Model projectedModel, Model resolvedModel, ValidatedResult<Model> modelResult, ProjectionResult.Builder resultBuilder) {
        Path pluginBaseDir = baseProjectionDir.resolve(pluginName);
        FileManifest manifest = this.fileManifestFactory.apply(pluginBaseDir);
        SmithyBuildPlugin resolved = this.pluginFactory.apply(pluginName).orElse(null);
        if (resolved == null) {
            String message = "Unable to find a plugin named `" + pluginName + "` in the `" + projectionName + "` projection. Is this the correct spelling? Are you missing a dependency? Is your classpath configured correctly?";
            if (!this.config.isIgnoreMissingPlugins()) throw new SmithyBuildException(message);
            LOGGER.severe(message);
            return;
        } else if (resolved.requiresValidModel() && modelResult.isBroken()) {
            LOGGER.fine(() -> String.format("Skipping `%s` plugin for `%s` projection because the model is broken", pluginName, projectionName));
            return;
        } else {
            LOGGER.info(() -> String.format("Applying `%s` plugin to `%s` projection", pluginName, projectionName));
            resolved.execute(PluginContext.builder().model(projectedModel).originalModel(resolvedModel).projection(projectionName, projection).events(modelResult.getValidationEvents()).settings(pluginSettings).fileManifest(manifest).pluginClassLoader(this.pluginClassLoader).sources(this.sources).build());
            resultBuilder.addPluginManifest(pluginName, manifest);
        }
    }

    private Map<String, ObjectNode> resolvePlugins(ProjectionConfig projection) {
        TreeMap<String, ObjectNode> result = new TreeMap<String, ObjectNode>(this.config.getPlugins());
        result.putAll(projection.getPlugins());
        return result;
    }

    private List<Pair<ObjectNode, ProjectionTransformer>> createTransformers(String projectionName, ProjectionConfig config) {
        ArrayList<Pair<ObjectNode, ProjectionTransformer>> resolved = new ArrayList<Pair<ObjectNode, ProjectionTransformer>>(config.getTransforms().size());
        for (TransformConfig transformConfig : config.getTransforms()) {
            String name = transformConfig.getName();
            ProjectionTransformer transformer = this.transformFactory.apply(name).orElseThrow(() -> new UnknownTransformException(String.format("Unable to find a transform named `%s` in the `%s` projection. Is this the correct spelling? Are you missing a dependency? Is your classpath configured correctly?", name, projectionName)));
            resolved.add((Pair<ObjectNode, ProjectionTransformer>)Pair.of((Object)transformConfig.getArgs(), (Object)transformer));
        }
        return resolved;
    }

    private Model applyQueuedProjections(TransformContext context, Model currentModel, Set<String> visited) {
        for (String projectionTarget : context.getQueuedProjections()) {
            LinkedHashSet<String> updatedVisited = new LinkedHashSet<String>(visited);
            if (context.getProjectionName().equals(projectionTarget)) {
                throw new SmithyBuildException("Cannot recursively apply the same projection: " + projectionTarget);
            }
            if (!this.transformers.containsKey(projectionTarget)) {
                throw new UnknownProjectionException(String.format("Unable to find projection named `%s` referenced by the `%s` projection", projectionTarget, context.getProjectionName()));
            }
            if (visited.contains(projectionTarget)) {
                updatedVisited.add(projectionTarget);
                throw new SmithyBuildException(String.format("Cycle found in apply transforms: %s -> ...", String.join((CharSequence)" -> ", updatedVisited)));
            }
            updatedVisited.add(projectionTarget);
            currentModel = this.applyProjectionTransforms((ValidatedResult<Model>)new ValidatedResult((Object)context.getOriginalModel().orElse(currentModel), context.getOriginalModelValidationEvents()), currentModel, projectionTarget, updatedVisited);
        }
        return currentModel;
    }
}

