/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.qldb;

import com.amazon.ion.IonSystem;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.services.qldbsession.QldbSessionClient;
import software.amazon.awssdk.services.qldbsession.model.InvalidSessionException;
import software.amazon.awssdk.utils.Validate;
import software.amazon.qldb.ExecutionContext;
import software.amazon.qldb.Executor;
import software.amazon.qldb.ExecutorNoReturn;
import software.amazon.qldb.QldbDriver;
import software.amazon.qldb.QldbSession;
import software.amazon.qldb.Result;
import software.amazon.qldb.RetryPolicy;
import software.amazon.qldb.Session;
import software.amazon.qldb.TableNameIterable;
import software.amazon.qldb.TransactionExecutor;
import software.amazon.qldb.exceptions.Errors;
import software.amazon.qldb.exceptions.QldbDriverException;

@ThreadSafe
class QldbDriverImpl
implements QldbDriver {
    static final String TABLE_NAME_QUERY = "SELECT VALUE name FROM information_schema.user_tables WHERE status = 'ACTIVE'";
    private static final Logger logger = LoggerFactory.getLogger(QldbDriver.class);
    private static final long DEFAULT_TIMEOUT_MS = 1L;
    private final String ledgerName;
    private final Semaphore poolPermits;
    private final BlockingQueue<QldbSession> pool;
    private final int readAhead;
    private final ExecutorService executorService;
    private final QldbSessionClient amazonQldbSession;
    private final RetryPolicy retryPolicy;
    private final IonSystem ionSystem;
    private final AtomicBoolean isClosed;
    private final int transactionRetryLimit;

    protected QldbDriverImpl(String ledgerName, QldbSessionClient qldbSessionClient, RetryPolicy retryPolicy, int readAhead, int maxConcurrentTransactions, IonSystem ionSystem, ExecutorService executorService) {
        this.ledgerName = ledgerName;
        this.amazonQldbSession = qldbSessionClient;
        this.retryPolicy = retryPolicy;
        this.ionSystem = ionSystem;
        this.isClosed = new AtomicBoolean(false);
        this.readAhead = readAhead;
        this.executorService = executorService;
        this.transactionRetryLimit = Math.max(maxConcurrentTransactions + 3, maxConcurrentTransactions);
        this.poolPermits = new Semaphore(maxConcurrentTransactions, true);
        this.pool = new LinkedBlockingQueue<QldbSession>();
    }

    @Override
    public void close() {
        if (!this.isClosed.getAndSet(true)) {
            QldbSession curSession = (QldbSession)this.pool.poll();
            while (curSession != null) {
                curSession.close();
                curSession = (QldbSession)this.pool.poll();
            }
        }
    }

    @Override
    public void execute(ExecutorNoReturn executor) {
        this.execute(executor, this.retryPolicy);
    }

    @Override
    public void execute(ExecutorNoReturn executor, RetryPolicy retryPolicy) {
        this.execute((TransactionExecutor txn) -> {
            executor.execute(txn);
            return Boolean.TRUE;
        }, retryPolicy);
    }

    @Override
    public <T> T execute(Executor<T> executor) {
        return this.execute(executor, this.retryPolicy);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <T> T execute(Executor<T> executor, RetryPolicy retryPolicy) {
        Validate.notNull((Object)retryPolicy, (String)"retryPolicy", (Object[])new Object[0]);
        if (this.isClosed.get()) {
            logger.error(Errors.DRIVER_CLOSED.get());
            throw QldbDriverException.create(Errors.DRIVER_CLOSED.get());
        }
        ExecutionContext executionContext = new ExecutionContext();
        int transactionExecutionAttempt = 0;
        while (true) {
            QldbSession qldbSession = null;
            try {
                ++transactionExecutionAttempt;
                qldbSession = this.getSession();
                T t = qldbSession.execute(executor, retryPolicy, executionContext);
                return t;
            }
            catch (InvalidSessionException ise) {
                if (transactionExecutionAttempt >= this.transactionRetryLimit) {
                    logger.debug("Transaction retry limit reached");
                    throw ise;
                }
                if (ise.getMessage().matches("Transaction .* has expired")) {
                    logger.debug("Encountered Transaction expiry. Error {}", (Object)ise.getMessage());
                    throw ise;
                }
                logger.debug("Retrying with another session. Error {}", (Object)ise.getMessage());
                continue;
            }
            finally {
                if (qldbSession == null) continue;
                this.releaseSession(qldbSession);
                continue;
            }
            break;
        }
    }

    @Override
    public Iterable<String> getTableNames() {
        Result result = this.execute((TransactionExecutor txn) -> txn.execute(TABLE_NAME_QUERY), this.retryPolicy);
        return new TableNameIterable(result);
    }

    private QldbSession getSession() {
        logger.debug("Getting session. There are {} free sessions; currently available permits is: {}.", (Object)this.pool.size(), (Object)this.poolPermits.availablePermits());
        try {
            if (this.poolPermits.tryAcquire(1L, TimeUnit.MILLISECONDS)) {
                try {
                    QldbSession session = (QldbSession)this.pool.poll();
                    if (session == null) {
                        session = this.createNewSession();
                        logger.debug("Creating new pooled session. Session ID: {}.", (Object)session.getSessionId());
                    }
                    return session;
                }
                catch (Exception e) {
                    this.poolPermits.release();
                    throw e;
                }
            }
            throw QldbDriverException.create(String.format(Errors.NO_SESSION_AVAILABLE.get(), 1L));
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            throw QldbDriverException.create(Errors.GET_SESSION_INTERRUPTED.get());
        }
    }

    private QldbSession createNewSession() {
        Session session = Session.startSession(this.ledgerName, this.amazonQldbSession);
        return new QldbSession(session, this.readAhead, this.ionSystem, this.executorService);
    }

    private void releaseSession(QldbSession session) {
        if (!session.isClosed()) {
            this.pool.add(session);
        }
        this.poolPermits.release();
        logger.debug("Session returned to pool; pool size is now: {}.", (Object)this.pool.size());
    }
}

