/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tinkerpop.gremlin.driver;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.tinkerpop.gremlin.driver.ApprovalResult;
import org.apache.tinkerpop.gremlin.driver.ChooseEndpointStrategy;
import org.apache.tinkerpop.gremlin.driver.Connection;
import org.apache.tinkerpop.gremlin.driver.ConnectionMetrics;
import org.apache.tinkerpop.gremlin.driver.ConnectionMetricsCollector;
import org.apache.tinkerpop.gremlin.driver.Endpoint;
import org.apache.tinkerpop.gremlin.driver.EndpointClient;
import org.apache.tinkerpop.gremlin.driver.EndpointCollection;
import org.apache.tinkerpop.gremlin.driver.EndpointConnectionMetrics;
import org.apache.tinkerpop.gremlin.driver.EndpointRequestMetrics;
import org.apache.tinkerpop.gremlin.driver.MetricsHandler;
import org.apache.tinkerpop.gremlin.driver.RequestMetrics;
import org.apache.tinkerpop.gremlin.driver.RequestMetricsCollector;
import org.apache.tinkerpop.gremlin.driver.exception.NoHostAvailableException;
import org.apache.tinkerpop.gremlin.driver.message.RequestMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class EndpointClientCollection
implements Iterable<EndpointClient> {
    private final List<EndpointClient> endpointClients;
    private final EndpointCollection rejectedEndpoints;
    private final boolean collectMetrics;
    private final ConnectionMetricsCollector connectionMetrics;
    private final RequestMetricsCollector requestMetrics;
    private final long startMillis = System.currentTimeMillis();
    private final ExecutorService executorService;
    private static final Logger logger = LoggerFactory.getLogger(EndpointClientCollection.class);

    public static Builder builder() {
        return new Builder();
    }

    EndpointClientCollection(Builder builder) {
        this.rejectedEndpoints = builder.getRejectedEndpoints();
        this.endpointClients = builder.getEndpointClients();
        this.collectMetrics = builder.collectMetrics();
        this.executorService = this.collectMetrics ? Executors.newSingleThreadExecutor() : null;
        this.connectionMetrics = this.collectMetrics ? this.initConnectionMetrics(this.endpointClients) : null;
        this.requestMetrics = this.collectMetrics ? this.initRequestMetrics(this.endpointClients) : null;
    }

    EndpointClientCollection() {
        this(new Builder());
    }

    private RequestMetricsCollector initRequestMetrics(List<EndpointClient> endpointClients) {
        ConcurrentHashMap<String, EndpointRequestMetrics> requestMetrics = new ConcurrentHashMap<String, EndpointRequestMetrics>();
        for (EndpointClient endpointClient : endpointClients) {
            String address = endpointClient.endpoint().getAddress();
            requestMetrics.put(address, new EndpointRequestMetrics(address));
        }
        return new RequestMetricsCollector(requestMetrics);
    }

    private ConnectionMetricsCollector initConnectionMetrics(List<EndpointClient> endpointClients) {
        ConcurrentHashMap<String, EndpointConnectionMetrics> endpointClientMetrics = new ConcurrentHashMap<String, EndpointConnectionMetrics>();
        for (EndpointClient endpointClient : endpointClients) {
            String address = endpointClient.endpoint().getAddress();
            endpointClientMetrics.put(address, new EndpointConnectionMetrics(address));
        }
        return new ConnectionMetricsCollector(endpointClientMetrics);
    }

    List<EndpointClient> getSurvivingEndpointClients(EndpointCollection acceptedEndpoints) {
        ArrayList<EndpointClient> results = new ArrayList<EndpointClient>();
        for (EndpointClient endpointClient : this.endpointClients) {
            Endpoint endpoint = endpointClient.endpoint();
            if (!acceptedEndpoints.containsEndpoint(endpoint)) continue;
            logger.info("Retaining client for {}", (Object)endpoint.getAddress());
            results.add(endpointClient);
        }
        return results;
    }

    Connection chooseConnection(RequestMessage msg, ChooseEndpointStrategy strategy) throws TimeoutException {
        UUID traceId = msg.getRequestId();
        long startMillis = System.currentTimeMillis();
        EndpointClient endpointClient = strategy.choose(this);
        String address = endpointClient.endpoint().getAddress();
        if (!endpointClient.isAvailable()) {
            logger.debug("No connections available for {}", (Object)address);
            this.submitMetrics(() -> this.connectionMetrics.unavailable(address, startMillis));
            return null;
        }
        try {
            Connection connection = endpointClient.client().chooseConnection(msg);
            if (connection.isClosing()) {
                logger.debug("Connection is closing: {}", (Object)address);
                this.submitMetrics(() -> this.connectionMetrics.closing(address, startMillis));
                return null;
            }
            if (connection.isDead()) {
                logger.debug("Connection is dead: {}", (Object)address);
                this.submitMetrics(() -> this.connectionMetrics.dead(address, startMillis));
                return null;
            }
            this.submitMetrics(() -> {
                try {
                    this.connectionMetrics.succeeded(address, startMillis);
                    this.requestMetrics.registerAddressForTraceId(traceId, address);
                }
                catch (Exception e) {
                    logger.error("Error while submitting metrics", (Throwable)e);
                }
            });
            return connection;
        }
        catch (NullPointerException e) {
            logger.debug("NullPointerException: {}", (Object)address, (Object)e);
            this.submitMetrics(() -> this.connectionMetrics.npe(address, startMillis));
            return null;
        }
        catch (NoHostAvailableException e) {
            logger.debug("No connection available: {}", (Object)address, (Object)e);
            this.submitMetrics(() -> this.connectionMetrics.nha(address, startMillis));
            return null;
        }
    }

    EndpointClient get(int index) {
        return this.endpointClients.get(index);
    }

    int size() {
        return this.endpointClients.size();
    }

    boolean isEmpty() {
        return this.endpointClients.isEmpty();
    }

    @Override
    public Iterator<EndpointClient> iterator() {
        return this.endpointClients.iterator();
    }

    Stream<EndpointClient> stream() {
        return this.endpointClients.stream();
    }

    EndpointCollection endpoints() {
        List endpoints = this.endpointClients.stream().map(EndpointClient::endpoint).collect(Collectors.toList());
        return new EndpointCollection(endpoints);
    }

    boolean hasRejectedEndpoints() {
        return !this.rejectedEndpoints.isEmpty();
    }

    Collection<String> rejectionReasons() {
        return this.rejectedEndpoints.stream().map(e -> e.getAnnotations().getOrDefault(ApprovalResult.REJECTED_REASON_ANNOTATION, "unknown")).collect(Collectors.toSet());
    }

    private void submitMetrics(Runnable runnable) {
        if (this.collectMetrics) {
            try {
                this.executorService.submit(runnable);
            }
            catch (RejectedExecutionException rejectedExecutionException) {
                // empty catch block
            }
        }
    }

    void close(MetricsHandler handler) {
        if (this.executorService != null) {
            this.executorService.shutdownNow();
        }
        if (!this.collectMetrics) {
            return;
        }
        if (handler != null) {
            long duration = System.currentTimeMillis() - this.startMillis;
            ConnectionMetrics conMetrics = new ConnectionMetrics(duration, this.connectionMetrics.totalConnectionAttempts(), this.connectionMetrics.metrics());
            RequestMetrics reqMetrics = new RequestMetrics(duration, this.requestMetrics.totalRequests(), this.requestMetrics.failedRequests(), this.requestMetrics.droppedRequests(), this.requestMetrics.skippedResponses(), this.requestMetrics.metrics());
            handler.onMetricsPublished(conMetrics, reqMetrics);
        }
    }

    void registerDurationForTraceId(UUID traceId, long durationMillis, Throwable e) {
        this.submitMetrics(() -> this.requestMetrics.registerDurationForTraceId(traceId, durationMillis, e));
    }

    static class Builder {
        private List<EndpointClient> endpointClients = new ArrayList<EndpointClient>();
        private EndpointCollection rejectedEndpoints = new EndpointCollection();
        private boolean collectMetrics = false;

        private Builder() {
        }

        public Builder withEndpointClients(List<EndpointClient> endpointClients) {
            this.endpointClients = endpointClients;
            return this;
        }

        public Builder withRejectedEndpoints(EndpointCollection rejectedEndpoints) {
            this.rejectedEndpoints = rejectedEndpoints;
            return this;
        }

        public Builder setCollectMetrics(boolean collectMetrics) {
            this.collectMetrics = collectMetrics;
            return this;
        }

        List<EndpointClient> getEndpointClients() {
            return this.endpointClients;
        }

        EndpointCollection getRejectedEndpoints() {
            return this.rejectedEndpoints;
        }

        boolean collectMetrics() {
            return this.collectMetrics;
        }
    }
}

