/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.kinesis.leases.dynamodb;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.dynamodb.DynamoDbAsyncClient;
import software.amazon.awssdk.services.dynamodb.model.AttributeValueUpdate;
import software.amazon.awssdk.services.dynamodb.model.ConditionalCheckFailedException;
import software.amazon.awssdk.services.dynamodb.model.CreateTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteItemRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DynamoDbException;
import software.amazon.awssdk.services.dynamodb.model.GetItemRequest;
import software.amazon.awssdk.services.dynamodb.model.GetItemResponse;
import software.amazon.awssdk.services.dynamodb.model.LimitExceededException;
import software.amazon.awssdk.services.dynamodb.model.ProvisionedThroughput;
import software.amazon.awssdk.services.dynamodb.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.dynamodb.model.PutItemRequest;
import software.amazon.awssdk.services.dynamodb.model.ResourceInUseException;
import software.amazon.awssdk.services.dynamodb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.ScanRequest;
import software.amazon.awssdk.services.dynamodb.model.ScanResponse;
import software.amazon.awssdk.services.dynamodb.model.TableStatus;
import software.amazon.awssdk.services.dynamodb.model.UpdateItemRequest;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.kinesis.annotations.KinesisClientInternalApi;
import software.amazon.kinesis.leases.Lease;
import software.amazon.kinesis.leases.LeaseRefresher;
import software.amazon.kinesis.leases.LeaseSerializer;
import software.amazon.kinesis.leases.exceptions.DependencyException;
import software.amazon.kinesis.leases.exceptions.InvalidStateException;
import software.amazon.kinesis.leases.exceptions.ProvisionedThroughputException;
import software.amazon.kinesis.retrieval.AWSExceptionManager;
import software.amazon.kinesis.retrieval.kpl.ExtendedSequenceNumber;

@KinesisClientInternalApi
public class DynamoDBLeaseRefresher
implements LeaseRefresher {
    private static final Logger log = LoggerFactory.getLogger(DynamoDBLeaseRefresher.class);
    protected final String table;
    protected final DynamoDbAsyncClient dynamoDBClient;
    protected final LeaseSerializer serializer;
    protected final boolean consistentReads;

    @Override
    public boolean createLeaseTableIfNotExists(@NonNull Long readCapacity, @NonNull Long writeCapacity) throws ProvisionedThroughputException, DependencyException {
        if (readCapacity == null) {
            throw new NullPointerException("readCapacity");
        }
        if (writeCapacity == null) {
            throw new NullPointerException("writeCapacity");
        }
        try {
            if (this.tableStatus() != null) {
                return false;
            }
        }
        catch (DependencyException de) {
            log.error("Failed to get table status for {}", (Object)this.table, (Object)de);
        }
        ProvisionedThroughput throughput = (ProvisionedThroughput)ProvisionedThroughput.builder().readCapacityUnits(readCapacity).writeCapacityUnits(writeCapacity).build();
        CreateTableRequest request = (CreateTableRequest)CreateTableRequest.builder().tableName(this.table).keySchema(this.serializer.getKeySchema()).attributeDefinitions(this.serializer.getAttributeDefinitions()).provisionedThroughput(throughput).build();
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        exceptionManager.add(ResourceInUseException.class, t -> t);
        exceptionManager.add(LimitExceededException.class, t -> t);
        try {
            try {
                this.dynamoDBClient.createTable(request).get();
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (ResourceInUseException e) {
            log.info("Table {} already exists.", (Object)this.table);
            return false;
        }
        catch (LimitExceededException e) {
            throw new ProvisionedThroughputException("Capacity exceeded when creating table " + this.table, e);
        }
        catch (DynamoDbException e) {
            throw new DependencyException(e);
        }
        return true;
    }

    @Override
    public boolean leaseTableExists() throws DependencyException {
        return TableStatus.ACTIVE == this.tableStatus();
    }

    private TableStatus tableStatus() throws DependencyException {
        DescribeTableResponse result;
        DescribeTableRequest request = (DescribeTableRequest)DescribeTableRequest.builder().tableName(this.table).build();
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        exceptionManager.add(ResourceNotFoundException.class, t -> t);
        try {
            try {
                result = (DescribeTableResponse)this.dynamoDBClient.describeTable(request).get();
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (ResourceNotFoundException e) {
            log.debug("Got ResourceNotFoundException for table {} in leaseTableExists, returning false.", (Object)this.table);
            return null;
        }
        catch (DynamoDbException e) {
            throw new DependencyException(e);
        }
        TableStatus tableStatus = result.table().tableStatus();
        log.debug("Lease table exists and is in status {}", (Object)tableStatus);
        return tableStatus;
    }

    @Override
    public boolean waitUntilLeaseTableExists(long secondsBetweenPolls, long timeoutSeconds) throws DependencyException {
        long sleepTimeRemaining = TimeUnit.SECONDS.toMillis(timeoutSeconds);
        while (!this.leaseTableExists()) {
            if (sleepTimeRemaining <= 0L) {
                return false;
            }
            long timeToSleepMillis = Math.min(TimeUnit.SECONDS.toMillis(secondsBetweenPolls), sleepTimeRemaining);
            sleepTimeRemaining -= this.sleep(timeToSleepMillis);
        }
        return true;
    }

    long sleep(long timeToSleepMillis) {
        long startTime = System.currentTimeMillis();
        try {
            Thread.sleep(timeToSleepMillis);
        }
        catch (InterruptedException e) {
            log.debug("Interrupted while sleeping");
        }
        return System.currentTimeMillis() - startTime;
    }

    @Override
    public List<Lease> listLeases() throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        return this.list(null);
    }

    @Override
    public boolean isLeaseTableEmpty() throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        return this.list(1).isEmpty();
    }

    List<Lease> list(Integer limit) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        log.debug("Listing leases from table {}", (Object)this.table);
        ScanRequest.Builder scanRequestBuilder = ScanRequest.builder().tableName(this.table);
        if (limit != null) {
            scanRequestBuilder = scanRequestBuilder.limit(limit);
        }
        ScanRequest scanRequest = (ScanRequest)scanRequestBuilder.build();
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        exceptionManager.add(ResourceNotFoundException.class, t -> t);
        exceptionManager.add(ProvisionedThroughputExceededException.class, t -> t);
        try {
            try {
                ScanResponse scanResult = (ScanResponse)this.dynamoDBClient.scan(scanRequest).get();
                ArrayList<Lease> result = new ArrayList<Lease>();
                while (scanResult != null) {
                    for (Map item : scanResult.items()) {
                        log.debug("Got item {} from DynamoDB.", (Object)item.toString());
                        result.add(this.serializer.fromDynamoRecord(item));
                    }
                    Map lastEvaluatedKey = scanResult.lastEvaluatedKey();
                    if (CollectionUtils.isNullOrEmpty((Map)lastEvaluatedKey)) {
                        scanResult = null;
                        log.debug("lastEvaluatedKey was null - scan finished.");
                        continue;
                    }
                    scanRequest = (ScanRequest)scanRequest.toBuilder().exclusiveStartKey(lastEvaluatedKey).build();
                    log.debug("lastEvaluatedKey was {}, continuing scan.", (Object)lastEvaluatedKey);
                    scanResult = (ScanResponse)this.dynamoDBClient.scan(scanRequest).get();
                }
                log.debug("Listed {} leases from table {}", (Object)result.size(), (Object)this.table);
                return result;
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (ResourceNotFoundException e) {
            throw new InvalidStateException("Cannot scan lease table " + this.table + " because it does not exist.", e);
        }
        catch (ProvisionedThroughputExceededException e) {
            throw new ProvisionedThroughputException(e);
        }
        catch (DynamoDbException e) {
            throw new DependencyException(e);
        }
    }

    @Override
    public boolean createLeaseIfNotExists(@NonNull Lease lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        if (lease == null) {
            throw new NullPointerException("lease");
        }
        log.debug("Creating lease {}", (Object)lease);
        PutItemRequest request = (PutItemRequest)PutItemRequest.builder().tableName(this.table).item(this.serializer.toDynamoRecord(lease)).expected(this.serializer.getDynamoNonexistantExpectation()).build();
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        exceptionManager.add(ConditionalCheckFailedException.class, t -> t);
        try {
            try {
                this.dynamoDBClient.putItem(request).get();
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (ConditionalCheckFailedException e) {
            log.debug("Did not create lease {} because it already existed", (Object)lease);
            return false;
        }
        catch (DynamoDbException e) {
            throw this.convertAndRethrowExceptions("create", lease.leaseKey(), (Exception)((Object)e));
        }
        return true;
    }

    @Override
    public Lease getLease(@NonNull String leaseKey) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        if (leaseKey == null) {
            throw new NullPointerException("leaseKey");
        }
        log.debug("Getting lease with key {}", (Object)leaseKey);
        GetItemRequest request = (GetItemRequest)GetItemRequest.builder().tableName(this.table).key(this.serializer.getDynamoHashKey(leaseKey)).consistentRead(Boolean.valueOf(this.consistentReads)).build();
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        try {
            try {
                GetItemResponse result = (GetItemResponse)this.dynamoDBClient.getItem(request).get();
                Map dynamoRecord = result.item();
                if (CollectionUtils.isNullOrEmpty((Map)dynamoRecord)) {
                    log.debug("No lease found with key {}, returning null.", (Object)leaseKey);
                    return null;
                }
                Lease lease = this.serializer.fromDynamoRecord(dynamoRecord);
                log.debug("Got lease {}", (Object)lease);
                return lease;
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (DynamoDbException e) {
            throw this.convertAndRethrowExceptions("get", leaseKey, (Exception)((Object)e));
        }
    }

    @Override
    public boolean renewLease(@NonNull Lease lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        if (lease == null) {
            throw new NullPointerException("lease");
        }
        log.debug("Renewing lease with key {}", (Object)lease.leaseKey());
        UpdateItemRequest request = (UpdateItemRequest)UpdateItemRequest.builder().tableName(this.table).key(this.serializer.getDynamoHashKey(lease)).expected(this.serializer.getDynamoLeaseCounterExpectation(lease)).attributeUpdates(this.serializer.getDynamoLeaseCounterUpdate(lease)).build();
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        exceptionManager.add(ConditionalCheckFailedException.class, t -> t);
        try {
            try {
                this.dynamoDBClient.updateItem(request).get();
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (ConditionalCheckFailedException e) {
            log.debug("Lease renewal failed for lease with key {} because the lease counter was not {}", (Object)lease.leaseKey(), (Object)lease.leaseCounter());
            String expectedOwner = lease.leaseOwner();
            Long expectedCounter = lease.leaseCounter() + 1L;
            Lease updatedLease = this.getLease(lease.leaseKey());
            if (updatedLease == null || !expectedOwner.equals(updatedLease.leaseOwner()) || !expectedCounter.equals(updatedLease.leaseCounter())) {
                return false;
            }
            log.info("Detected spurious renewal failure for lease with key {}, but recovered", (Object)lease.leaseKey());
        }
        catch (DynamoDbException e) {
            throw new DependencyException(e);
        }
        lease.leaseCounter(lease.leaseCounter() + 1L);
        return true;
    }

    @Override
    public boolean takeLease(@NonNull Lease lease, @NonNull String owner) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        if (lease == null) {
            throw new NullPointerException("lease");
        }
        if (owner == null) {
            throw new NullPointerException("owner");
        }
        String oldOwner = lease.leaseOwner();
        log.debug("Taking lease with leaseKey {} from {} to {}", new Object[]{lease.leaseKey(), lease.leaseOwner() == null ? "nobody" : lease.leaseOwner(), owner});
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        exceptionManager.add(ConditionalCheckFailedException.class, t -> t);
        Map<String, AttributeValueUpdate> updates = this.serializer.getDynamoLeaseCounterUpdate(lease);
        updates.putAll(this.serializer.getDynamoTakeLeaseUpdate(lease, owner));
        UpdateItemRequest request = (UpdateItemRequest)UpdateItemRequest.builder().tableName(this.table).key(this.serializer.getDynamoHashKey(lease)).expected(this.serializer.getDynamoLeaseCounterExpectation(lease)).attributeUpdates(updates).build();
        try {
            try {
                this.dynamoDBClient.updateItem(request).get();
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (ConditionalCheckFailedException e) {
            log.debug("Lease renewal failed for lease with key {} because the lease counter was not {}", (Object)lease.leaseKey(), (Object)lease.leaseCounter());
            return false;
        }
        catch (DynamoDbException e) {
            throw this.convertAndRethrowExceptions("take", lease.leaseKey(), (Exception)((Object)e));
        }
        lease.leaseCounter(lease.leaseCounter() + 1L);
        lease.leaseOwner(owner);
        if (oldOwner != null && !oldOwner.equals(owner)) {
            lease.ownerSwitchesSinceCheckpoint(lease.ownerSwitchesSinceCheckpoint() + 1L);
        }
        return true;
    }

    @Override
    public boolean evictLease(@NonNull Lease lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        if (lease == null) {
            throw new NullPointerException("lease");
        }
        log.debug("Evicting lease with leaseKey {} owned by {}", (Object)lease.leaseKey(), (Object)lease.leaseOwner());
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        exceptionManager.add(ConditionalCheckFailedException.class, t -> t);
        Map<String, AttributeValueUpdate> updates = this.serializer.getDynamoLeaseCounterUpdate(lease);
        updates.putAll(this.serializer.getDynamoEvictLeaseUpdate(lease));
        UpdateItemRequest request = (UpdateItemRequest)UpdateItemRequest.builder().tableName(this.table).key(this.serializer.getDynamoHashKey(lease)).expected(this.serializer.getDynamoLeaseOwnerExpectation(lease)).attributeUpdates(updates).build();
        try {
            try {
                this.dynamoDBClient.updateItem(request).get();
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (ConditionalCheckFailedException e) {
            log.debug("Lease eviction failed for lease with key {} because the lease owner was not {}", (Object)lease.leaseKey(), (Object)lease.leaseOwner());
            return false;
        }
        catch (DynamoDbException e) {
            throw this.convertAndRethrowExceptions("evict", lease.leaseKey(), (Exception)((Object)e));
        }
        lease.leaseOwner(null);
        lease.leaseCounter(lease.leaseCounter() + 1L);
        return true;
    }

    @Override
    public void deleteAll() throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        List<Lease> allLeases = this.listLeases();
        log.warn("Deleting {} items from table {}", (Object)allLeases.size(), (Object)this.table);
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        for (Lease lease : allLeases) {
            DeleteItemRequest deleteRequest = (DeleteItemRequest)DeleteItemRequest.builder().tableName(this.table).key(this.serializer.getDynamoHashKey(lease)).build();
            try {
                try {
                    this.dynamoDBClient.deleteItem(deleteRequest).get();
                }
                catch (ExecutionException e) {
                    throw exceptionManager.apply(e.getCause());
                }
                catch (InterruptedException e) {
                    throw new DependencyException(e);
                }
            }
            catch (DynamoDbException e) {
                throw this.convertAndRethrowExceptions("deleteAll", lease.leaseKey(), (Exception)((Object)e));
            }
        }
    }

    @Override
    public void deleteLease(@NonNull Lease lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        if (lease == null) {
            throw new NullPointerException("lease");
        }
        log.debug("Deleting lease with leaseKey {}", (Object)lease.leaseKey());
        DeleteItemRequest deleteRequest = (DeleteItemRequest)DeleteItemRequest.builder().tableName(this.table).key(this.serializer.getDynamoHashKey(lease)).build();
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        try {
            try {
                this.dynamoDBClient.deleteItem(deleteRequest).get();
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (DynamoDbException e) {
            throw this.convertAndRethrowExceptions("delete", lease.leaseKey(), (Exception)((Object)e));
        }
    }

    @Override
    public boolean updateLease(@NonNull Lease lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        if (lease == null) {
            throw new NullPointerException("lease");
        }
        log.debug("Updating lease {}", (Object)lease);
        AWSExceptionManager exceptionManager = this.createExceptionManager();
        exceptionManager.add(ConditionalCheckFailedException.class, t -> t);
        Map<String, AttributeValueUpdate> updates = this.serializer.getDynamoLeaseCounterUpdate(lease);
        updates.putAll(this.serializer.getDynamoUpdateLeaseUpdate(lease));
        UpdateItemRequest request = (UpdateItemRequest)UpdateItemRequest.builder().tableName(this.table).key(this.serializer.getDynamoHashKey(lease)).expected(this.serializer.getDynamoLeaseCounterExpectation(lease)).attributeUpdates(updates).build();
        try {
            try {
                this.dynamoDBClient.updateItem(request).get();
            }
            catch (ExecutionException e) {
                throw exceptionManager.apply(e.getCause());
            }
            catch (InterruptedException e) {
                throw new DependencyException(e);
            }
        }
        catch (ConditionalCheckFailedException e) {
            log.debug("Lease update failed for lease with key {} because the lease counter was not {}", (Object)lease.leaseKey(), (Object)lease.leaseCounter());
            return false;
        }
        catch (DynamoDbException e) {
            throw this.convertAndRethrowExceptions("update", lease.leaseKey(), (Exception)((Object)e));
        }
        lease.leaseCounter(lease.leaseCounter() + 1L);
        return true;
    }

    @Override
    public ExtendedSequenceNumber getCheckpoint(String shardId) throws ProvisionedThroughputException, InvalidStateException, DependencyException {
        ExtendedSequenceNumber checkpoint = null;
        Lease lease = this.getLease(shardId);
        if (lease != null) {
            checkpoint = lease.checkpoint();
        }
        return checkpoint;
    }

    protected DependencyException convertAndRethrowExceptions(String operation, String leaseKey, Exception e) throws ProvisionedThroughputException, InvalidStateException {
        if (e instanceof ProvisionedThroughputExceededException) {
            log.warn("Provisioned Throughput on the lease table has been exceeded. It's recommended that you increase the IOPs on the table. Failure to increase the IOPs may cause the application to not make progress.");
            throw new ProvisionedThroughputException(e);
        }
        if (e instanceof ResourceNotFoundException) {
            throw new InvalidStateException(String.format("Cannot %s lease with key %s because table %s does not exist.", operation, leaseKey, this.table), e);
        }
        return new DependencyException(e);
    }

    private AWSExceptionManager createExceptionManager() {
        AWSExceptionManager exceptionManager = new AWSExceptionManager();
        exceptionManager.add(DynamoDbException.class, t -> t);
        return exceptionManager;
    }

    public DynamoDBLeaseRefresher(String table, DynamoDbAsyncClient dynamoDBClient, LeaseSerializer serializer, boolean consistentReads) {
        this.table = table;
        this.dynamoDBClient = dynamoDBClient;
        this.serializer = serializer;
        this.consistentReads = consistentReads;
    }
}

