/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.jdbc.util.storage;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.logging.Logger;
import org.checkerframework.checker.nullness.qual.Nullable;
import software.amazon.jdbc.util.Messages;
import software.amazon.jdbc.util.storage.CacheItem;
import software.amazon.jdbc.util.storage.ItemDisposalFunc;
import software.amazon.jdbc.util.storage.ShouldDisposeFunc;

public class ExpirationCache<K, V> {
    private static final Logger LOGGER = Logger.getLogger(ExpirationCache.class.getName());
    protected static final long DEFAULT_TIME_TO_LIVE_NANOS = TimeUnit.MINUTES.toNanos(5L);
    protected final Map<K, CacheItem<V>> cache = new ConcurrentHashMap<K, CacheItem<V>>();
    protected final boolean isRenewableExpiration;
    protected final long timeToLiveNanos;
    protected final ShouldDisposeFunc<V> shouldDisposeFunc;
    protected final ItemDisposalFunc<V> itemDisposalFunc;

    public ExpirationCache() {
        this(false, DEFAULT_TIME_TO_LIVE_NANOS, null, null);
    }

    public ExpirationCache(boolean isRenewableExpiration, long timeToLiveNanos, @Nullable ShouldDisposeFunc<V> shouldDisposeFunc, @Nullable ItemDisposalFunc<V> itemDisposalFunc) {
        this.isRenewableExpiration = isRenewableExpiration;
        this.timeToLiveNanos = timeToLiveNanos;
        this.shouldDisposeFunc = shouldDisposeFunc;
        this.itemDisposalFunc = itemDisposalFunc;
    }

    public @Nullable V put(K key, V value) {
        CacheItem<V> cacheItem = this.cache.put(key, new CacheItem<V>(value, System.nanoTime() + this.timeToLiveNanos, this.shouldDisposeFunc));
        if (cacheItem == null) {
            return null;
        }
        if (this.itemDisposalFunc != null) {
            this.itemDisposalFunc.dispose(cacheItem.item);
        }
        return cacheItem.item;
    }

    public @Nullable V computeIfAbsent(K key, Function<? super K, ? extends V> mappingFunction) {
        ArrayList toDisposeList = new ArrayList(1);
        CacheItem cacheItem = this.cache.compute(key, (k, valueItem) -> {
            if (valueItem == null) {
                return new CacheItem(mappingFunction.apply((K)k), System.nanoTime() + this.timeToLiveNanos, this.shouldDisposeFunc);
            }
            if (valueItem.shouldCleanup() && !this.isRenewableExpiration) {
                toDisposeList.add(valueItem.item);
                return new CacheItem(mappingFunction.apply((K)k), System.nanoTime() + this.timeToLiveNanos, this.shouldDisposeFunc);
            }
            if (this.isRenewableExpiration) {
                valueItem.extendExpiration(this.timeToLiveNanos);
            }
            return valueItem;
        });
        if (this.itemDisposalFunc != null && !toDisposeList.isEmpty()) {
            this.itemDisposalFunc.dispose(toDisposeList.get(0));
        }
        return cacheItem.item;
    }

    public @Nullable V get(K key) {
        CacheItem<V> cacheItem = this.cache.get(key);
        if (cacheItem == null) {
            return null;
        }
        if (this.isRenewableExpiration) {
            cacheItem.extendExpiration(this.timeToLiveNanos);
        } else if (cacheItem.shouldCleanup()) {
            return null;
        }
        return cacheItem.item;
    }

    public boolean exists(K key) {
        CacheItem<V> cacheItem = this.cache.get(key);
        return cacheItem != null && !cacheItem.shouldCleanup();
    }

    public @Nullable V remove(K key) {
        return this.removeAndDispose(key);
    }

    protected @Nullable V removeAndDispose(K key) {
        CacheItem<V> cacheItem = this.cache.remove(key);
        if (cacheItem == null) {
            return null;
        }
        if (this.itemDisposalFunc != null) {
            this.itemDisposalFunc.dispose(cacheItem.item);
        }
        return cacheItem.item;
    }

    public void removeExpiredEntries() {
        this.cache.forEach((key, value) -> {
            try {
                this.removeIfExpired(key);
            }
            catch (Exception ex) {
                LOGGER.fine(Messages.get("ExpirationCache.exceptionWhileRemovingEntry", new Object[]{key, value, ex}));
            }
        });
    }

    public void removeIfExpired(K key) {
        ArrayList itemList = new ArrayList(1);
        this.cache.computeIfPresent(key, (k, cacheItem) -> {
            if (cacheItem.shouldCleanup()) {
                itemList.add(cacheItem.item);
                return null;
            }
            return cacheItem;
        });
        if (itemList.isEmpty()) {
            return;
        }
        Object item = itemList.get(0);
        if (item != null && this.itemDisposalFunc != null) {
            this.itemDisposalFunc.dispose(item);
        }
    }

    public void clear() {
        for (K key : this.cache.keySet()) {
            this.removeAndDispose(key);
        }
        this.cache.clear();
    }

    public Map<K, V> getEntries() {
        HashMap entries = new HashMap();
        for (Map.Entry<K, CacheItem<V>> entry : this.cache.entrySet()) {
            entries.put(entry.getKey(), entry.getValue().item);
        }
        return entries;
    }

    public int size() {
        return this.cache.size();
    }
}

