/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.jdbc.util;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.checkerframework.checker.nullness.qual.Nullable;
import software.amazon.jdbc.util.RdsUrlType;
import software.amazon.jdbc.util.StringUtils;

public class RdsUtils {
    private static final Pattern AURORA_DNS_PATTERN = Pattern.compile("^(?<instance>.+)\\.(?<dns>proxy-|cluster-|cluster-ro-|cluster-custom-|shardgrp-)?(?<domain>[a-zA-Z0-9]+\\.(?<region>[a-zA-Z0-9\\-]+)\\.rds\\.amazonaws\\.com)$", 2);
    private static final Pattern AURORA_CLUSTER_PATTERN = Pattern.compile("^(?<instance>.+)\\.(?<dns>cluster-|cluster-ro-)+(?<domain>[a-zA-Z0-9]+\\.(?<region>[a-zA-Z0-9\\-]+)\\.rds\\.amazonaws\\.com)$", 2);
    private static final Pattern AURORA_LIMITLESS_CLUSTER_PATTERN = Pattern.compile("(?<instance>.+)\\.(?<dns>shardgrp-)+(?<domain>[a-zA-Z0-9]+\\.(?<region>[a-zA-Z0-9\\-]+)\\.rds\\.(amazonaws\\.com(\\.cn)?|sc2s\\.sgov\\.gov|c2s\\.ic\\.gov))$", 2);
    private static final Pattern AURORA_CHINA_DNS_PATTERN = Pattern.compile("^(?<instance>.+)\\.(?<dns>proxy-|cluster-|cluster-ro-|cluster-custom-|shardgrp-)?(?<domain>[a-zA-Z0-9]+\\.rds\\.(?<region>[a-zA-Z0-9\\-]+)\\.amazonaws\\.com\\.cn)$", 2);
    private static final Pattern AURORA_CHINA_CLUSTER_PATTERN = Pattern.compile("^(?<instance>.+)\\.(?<dns>cluster-|cluster-ro-)+(?<domain>[a-zA-Z0-9]+\\.rds\\.(?<region>[a-zA-Z0-9\\-]+)\\.amazonaws\\.com\\.cn)$", 2);
    private static final Pattern AURORA_OLD_CHINA_DNS_PATTERN = Pattern.compile("^(?<instance>.+)\\.(?<dns>proxy-|cluster-|cluster-ro-|cluster-custom-|shardgrp-)?(?<domain>[a-zA-Z0-9]+\\.(?<region>[a-zA-Z0-9\\-]+)\\.rds\\.amazonaws\\.com\\.cn)$", 2);
    private static final Pattern AURORA_OLD_CHINA_CLUSTER_PATTERN = Pattern.compile("^(?<instance>.+)\\.(?<dns>cluster-|cluster-ro-)+(?<domain>[a-zA-Z0-9]+\\.(?<region>[a-zA-Z0-9\\-]+)\\.rds\\.amazonaws\\.com\\.cn)$", 2);
    private static final Pattern AURORA_GOV_DNS_PATTERN = Pattern.compile("^(?<instance>.+)\\.(?<dns>proxy-|cluster-|cluster-ro-|cluster-custom-|shardgrp-)?(?<domain>[a-zA-Z0-9]+\\.rds\\.(?<region>[a-zA-Z0-9\\-]+)\\.(amazonaws\\.com|c2s\\.ic\\.gov|sc2s\\.sgov\\.gov))$", 2);
    private static final Pattern AURORA_GOV_CLUSTER_PATTERN = Pattern.compile("^(?<instance>.+)\\.(?<dns>cluster-|cluster-ro-)+(?<domain>[a-zA-Z0-9]+\\.rds\\.(?<region>[a-zA-Z0-9\\-]+)\\.(amazonaws\\.com|c2s\\.ic\\.gov|sc2s\\.sgov\\.gov))$", 2);
    private static final Pattern ELB_PATTERN = Pattern.compile("^(?<instance>.+)\\.elb\\.((?<region>[a-zA-Z0-9\\-]+)\\.amazonaws\\.com)$", 2);
    private static final Pattern IP_V4 = Pattern.compile("^(([1-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\\.){1}(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\\.){2}([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])$");
    private static final Pattern IP_V6 = Pattern.compile("^[0-9a-fA-F]{1,4}(:[0-9a-fA-F]{1,4}){7}$");
    private static final Pattern IP_V6_COMPRESSED = Pattern.compile("^(([0-9A-Fa-f]{1,4}(:[0-9A-Fa-f]{1,4}){0,5})?)::(([0-9A-Fa-f]{1,4}(:[0-9A-Fa-f]{1,4}){0,5})?)$");
    private static final Pattern BG_GREEN_HOST_PATTERN = Pattern.compile(".*(?<prefix>-green-[0-9a-z]{6})\\..*", 2);
    private static final Map<String, Matcher> cachedPatterns = new ConcurrentHashMap<String, Matcher>();
    private static final Map<String, String> cachedDnsPatterns = new ConcurrentHashMap<String, String>();
    private static final String INSTANCE_GROUP = "instance";
    private static final String DNS_GROUP = "dns";
    private static final String DOMAIN_GROUP = "domain";
    private static final String REGION_GROUP = "region";

    public boolean isRdsClusterDns(String host) {
        String dnsGroup = this.getDnsGroup(host);
        return dnsGroup != null && (dnsGroup.equalsIgnoreCase("cluster-") || dnsGroup.equalsIgnoreCase("cluster-ro-"));
    }

    public boolean isRdsCustomClusterDns(String host) {
        String dnsGroup = this.getDnsGroup(host);
        return dnsGroup != null && dnsGroup.startsWith("cluster-custom-");
    }

    public boolean isRdsDns(String host) {
        Matcher matcher = this.cacheMatcher(host, AURORA_DNS_PATTERN, AURORA_CHINA_DNS_PATTERN, AURORA_OLD_CHINA_DNS_PATTERN, AURORA_GOV_DNS_PATTERN);
        String group = this.getRegexGroup(matcher, DNS_GROUP);
        if (group != null) {
            cachedDnsPatterns.put(host, group);
        }
        return matcher != null;
    }

    public boolean isRdsInstance(String host) {
        return this.getDnsGroup(host) == null && this.isRdsDns(host);
    }

    public boolean isRdsProxyDns(String host) {
        String dnsGroup = this.getDnsGroup(host);
        return dnsGroup != null && dnsGroup.startsWith("proxy-");
    }

    public @Nullable String getRdsClusterId(String host) {
        if (StringUtils.isNullOrEmpty(host)) {
            return null;
        }
        Matcher matcher = this.cacheMatcher(host, AURORA_DNS_PATTERN, AURORA_CHINA_DNS_PATTERN, AURORA_OLD_CHINA_DNS_PATTERN, AURORA_GOV_DNS_PATTERN);
        if (this.getRegexGroup(matcher, DNS_GROUP) != null) {
            return this.getRegexGroup(matcher, INSTANCE_GROUP);
        }
        return null;
    }

    public @Nullable String getRdsInstanceId(String host) {
        if (StringUtils.isNullOrEmpty(host)) {
            return null;
        }
        Matcher matcher = this.cacheMatcher(host, AURORA_DNS_PATTERN, AURORA_CHINA_DNS_PATTERN, AURORA_OLD_CHINA_DNS_PATTERN, AURORA_GOV_DNS_PATTERN);
        if (this.getRegexGroup(matcher, DNS_GROUP) == null) {
            return this.getRegexGroup(matcher, INSTANCE_GROUP);
        }
        return null;
    }

    public String getRdsInstanceHostPattern(String host) {
        if (StringUtils.isNullOrEmpty(host)) {
            return "?";
        }
        Matcher matcher = this.cacheMatcher(host, AURORA_DNS_PATTERN, AURORA_CHINA_DNS_PATTERN, AURORA_OLD_CHINA_DNS_PATTERN, AURORA_GOV_DNS_PATTERN);
        String group = this.getRegexGroup(matcher, DOMAIN_GROUP);
        return group == null ? "?" : "?." + group;
    }

    public String getRdsRegion(String host) {
        if (StringUtils.isNullOrEmpty(host)) {
            return null;
        }
        Matcher matcher = this.cacheMatcher(host, AURORA_DNS_PATTERN, AURORA_CHINA_DNS_PATTERN, AURORA_OLD_CHINA_DNS_PATTERN, AURORA_GOV_DNS_PATTERN);
        String group = this.getRegexGroup(matcher, REGION_GROUP);
        if (group != null) {
            return group;
        }
        Matcher elbMatcher = ELB_PATTERN.matcher(host);
        if (elbMatcher.find()) {
            return this.getRegexGroup(elbMatcher, REGION_GROUP);
        }
        return null;
    }

    public boolean isWriterClusterDns(String host) {
        String dnsGroup = this.getDnsGroup(host);
        return dnsGroup != null && dnsGroup.equalsIgnoreCase("cluster-");
    }

    public boolean isReaderClusterDns(String host) {
        String dnsGroup = this.getDnsGroup(host);
        return dnsGroup != null && dnsGroup.equalsIgnoreCase("cluster-ro-");
    }

    public boolean isLimitlessDbShardGroupDns(String host) {
        String dnsGroup = this.getDnsGroup(host);
        return dnsGroup != null && dnsGroup.equalsIgnoreCase("shardgrp-");
    }

    public String getRdsClusterHostUrl(String host) {
        if (StringUtils.isNullOrEmpty(host)) {
            return null;
        }
        Matcher matcher = AURORA_CLUSTER_PATTERN.matcher(host);
        if (matcher.find()) {
            return host.replaceAll(AURORA_CLUSTER_PATTERN.pattern(), "${instance}.cluster-${domain}");
        }
        Matcher chinaMatcher = AURORA_CHINA_CLUSTER_PATTERN.matcher(host);
        if (chinaMatcher.find()) {
            return host.replaceAll(AURORA_CHINA_CLUSTER_PATTERN.pattern(), "${instance}.cluster-${domain}");
        }
        Matcher oldChinaMatcher = AURORA_OLD_CHINA_CLUSTER_PATTERN.matcher(host);
        if (oldChinaMatcher.find()) {
            return host.replaceAll(AURORA_OLD_CHINA_CLUSTER_PATTERN.pattern(), "${instance}.cluster-${domain}");
        }
        Matcher govMatcher = AURORA_GOV_CLUSTER_PATTERN.matcher(host);
        if (govMatcher.find()) {
            return host.replaceAll(AURORA_GOV_CLUSTER_PATTERN.pattern(), "${instance}.cluster-${domain}");
        }
        Matcher limitlessMatcher = AURORA_LIMITLESS_CLUSTER_PATTERN.matcher(host);
        if (limitlessMatcher.find()) {
            return host.replaceAll(AURORA_LIMITLESS_CLUSTER_PATTERN.pattern(), "${instance}.cluster-${domain}");
        }
        return null;
    }

    public boolean isIPv4(String ip) {
        return !StringUtils.isNullOrEmpty(ip) && IP_V4.matcher(ip).matches();
    }

    public boolean isIPv6(String ip) {
        return !StringUtils.isNullOrEmpty(ip) && (IP_V6.matcher(ip).matches() || IP_V6_COMPRESSED.matcher(ip).matches());
    }

    public boolean isDnsPatternValid(String pattern) {
        return pattern.contains("?");
    }

    public RdsUrlType identifyRdsType(String host) {
        if (StringUtils.isNullOrEmpty(host)) {
            return RdsUrlType.OTHER;
        }
        if (this.isIPv4(host) || this.isIPv6(host)) {
            return RdsUrlType.IP_ADDRESS;
        }
        if (this.isWriterClusterDns(host)) {
            return RdsUrlType.RDS_WRITER_CLUSTER;
        }
        if (this.isReaderClusterDns(host)) {
            return RdsUrlType.RDS_READER_CLUSTER;
        }
        if (this.isRdsCustomClusterDns(host)) {
            return RdsUrlType.RDS_CUSTOM_CLUSTER;
        }
        if (this.isLimitlessDbShardGroupDns(host)) {
            return RdsUrlType.RDS_AURORA_LIMITLESS_DB_SHARD_GROUP;
        }
        if (this.isRdsProxyDns(host)) {
            return RdsUrlType.RDS_PROXY;
        }
        if (this.isRdsDns(host)) {
            return RdsUrlType.RDS_INSTANCE;
        }
        return RdsUrlType.OTHER;
    }

    public String removePort(String hostAndPort) {
        if (StringUtils.isNullOrEmpty(hostAndPort)) {
            return hostAndPort;
        }
        int index = hostAndPort.indexOf(":");
        if (index == -1) {
            return hostAndPort;
        }
        return hostAndPort.substring(0, index);
    }

    public boolean isGreenInstance(String host) {
        return !StringUtils.isNullOrEmpty(host) && BG_GREEN_HOST_PATTERN.matcher(host).matches();
    }

    public String removeGreenInstancePrefix(String host) {
        if (StringUtils.isNullOrEmpty(host)) {
            return host;
        }
        Matcher matcher = BG_GREEN_HOST_PATTERN.matcher(host);
        if (!matcher.matches()) {
            return host;
        }
        String prefix = matcher.group("prefix");
        if (StringUtils.isNullOrEmpty(prefix)) {
            return host;
        }
        return host.replace(prefix + ".", ".");
    }

    private Matcher cacheMatcher(String host, Pattern ... patterns) {
        Matcher matcher = null;
        for (Pattern pattern : patterns) {
            matcher = cachedPatterns.get(host);
            if (matcher != null) {
                return matcher;
            }
            matcher = pattern.matcher(host);
            if (!matcher.find()) continue;
            cachedPatterns.put(host, matcher);
            return matcher;
        }
        return null;
    }

    private String getRegexGroup(Matcher matcher, String groupName) {
        if (matcher == null) {
            return null;
        }
        try {
            return matcher.group(groupName);
        }
        catch (IllegalStateException e) {
            return null;
        }
    }

    private String getDnsGroup(String host) {
        if (StringUtils.isNullOrEmpty(host)) {
            return null;
        }
        return cachedDnsPatterns.computeIfAbsent(host, k -> {
            Matcher matcher = this.cacheMatcher((String)k, AURORA_DNS_PATTERN, AURORA_CHINA_DNS_PATTERN, AURORA_OLD_CHINA_DNS_PATTERN, AURORA_GOV_DNS_PATTERN);
            return this.getRegexGroup(matcher, DNS_GROUP);
        });
    }

    public static void clearCache() {
        cachedPatterns.clear();
    }
}

