/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.jdbc.plugin.readwritesplitting;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;
import org.checkerframework.checker.nullness.qual.NonNull;
import software.amazon.jdbc.HostListProviderService;
import software.amazon.jdbc.HostRole;
import software.amazon.jdbc.HostSpec;
import software.amazon.jdbc.JdbcCallable;
import software.amazon.jdbc.NodeChangeOptions;
import software.amazon.jdbc.OldConnectionSuggestedAction;
import software.amazon.jdbc.PluginService;
import software.amazon.jdbc.cleanup.CanReleaseResources;
import software.amazon.jdbc.plugin.AbstractConnectionPlugin;
import software.amazon.jdbc.plugin.failover.FailoverSQLException;
import software.amazon.jdbc.plugin.readwritesplitting.ReadWriteSplittingSQLException;
import software.amazon.jdbc.util.Messages;
import software.amazon.jdbc.util.RdsUrlType;
import software.amazon.jdbc.util.RdsUtils;
import software.amazon.jdbc.util.SqlState;
import software.amazon.jdbc.util.WrapperUtils;

public class ReadWriteSplittingPlugin
extends AbstractConnectionPlugin
implements CanReleaseResources {
    private static final Logger LOGGER = Logger.getLogger(ReadWriteSplittingPlugin.class.getName());
    private static final Set<String> subscribedMethods = Collections.unmodifiableSet(new HashSet<String>(){
        {
            this.add("initHostProvider");
            this.add("connect");
            this.add("notifyConnectionChanged");
            this.add(ReadWriteSplittingPlugin.METHOD_SET_READ_ONLY);
        }
    });
    static final String METHOD_SET_READ_ONLY = "Connection.setReadOnly";
    static final String PG_DRIVER_PROTOCOL = "jdbc:postgresql:";
    static final String PG_GET_INSTANCE_NAME_SQL = "SELECT aurora_db_instance_identifier()";
    static final String PG_INSTANCE_NAME_COL = "aurora_db_instance_identifier";
    static final String MYSQL_DRIVER_PROTOCOL = "jdbc:mysql:";
    static final String MYSQL_GET_INSTANCE_NAME_SQL = "SELECT @@aurora_server_id";
    static final String MYSQL_INSTANCE_NAME_COL = "@@aurora_server_id";
    private final PluginService pluginService;
    private final Properties properties;
    private final RdsUtils rdsUtils = new RdsUtils();
    private final AtomicBoolean inReadWriteSplit = new AtomicBoolean(false);
    private HostListProviderService hostListProviderService;
    private Connection writerConnection;
    private Connection readerConnection;
    private HostSpec readerHostSpec;

    ReadWriteSplittingPlugin(PluginService pluginService, Properties properties) {
        this.pluginService = pluginService;
        this.properties = properties;
    }

    ReadWriteSplittingPlugin(PluginService pluginService, Properties properties, HostListProviderService hostListProviderService, Connection writerConnection, Connection readerConnection) {
        this(pluginService, properties);
        this.hostListProviderService = hostListProviderService;
        this.writerConnection = writerConnection;
        this.readerConnection = readerConnection;
    }

    @Override
    public Set<String> getSubscribedMethods() {
        return subscribedMethods;
    }

    @Override
    public void initHostProvider(String driverProtocol, String initialUrl, Properties props, HostListProviderService hostListProviderService, JdbcCallable<Void, SQLException> initHostProviderFunc) throws SQLException {
        this.hostListProviderService = hostListProviderService;
        initHostProviderFunc.call();
    }

    @Override
    public Connection connect(String driverProtocol, HostSpec hostSpec, Properties props, boolean isInitialConnection, @NonNull JdbcCallable<Connection, SQLException> connectFunc) throws SQLException {
        Connection currentConnection = connectFunc.call();
        if (!isInitialConnection || this.hostListProviderService.isStaticHostListProvider()) {
            return currentConnection;
        }
        RdsUrlType urlType = this.rdsUtils.identifyRdsType(hostSpec.getHost());
        if (RdsUrlType.RDS_WRITER_CLUSTER.equals((Object)urlType) || RdsUrlType.RDS_READER_CLUSTER.equals((Object)urlType)) {
            return currentConnection;
        }
        this.pluginService.refreshHostList(currentConnection);
        HostSpec currentHost = this.pluginService.getCurrentHostSpec();
        HostSpec updatedCurrentHost = RdsUrlType.RDS_INSTANCE.equals((Object)urlType) ? this.getHostSpecFromUrl(currentHost.getUrl()) : this.getHostSpecFromInstanceId(this.getCurrentInstanceId(currentConnection, driverProtocol));
        if (updatedCurrentHost == null) {
            this.logAndThrowException(Messages.get("ReadWriteSplittingPlugin.errorUpdatingHostSpecRole"));
            return null;
        }
        HostSpec updatedRoleHostSpec = new HostSpec(currentHost.getHost(), currentHost.getPort(), updatedCurrentHost.getRole(), currentHost.getAvailability());
        this.hostListProviderService.setInitialConnectionHostSpec(updatedRoleHostSpec);
        return currentConnection;
    }

    private HostSpec getHostSpecFromUrl(String url) {
        if (url == null) {
            return null;
        }
        List<HostSpec> hosts = this.pluginService.getHosts();
        for (HostSpec host : hosts) {
            if (!host.getUrl().equals(url)) continue;
            return host;
        }
        return null;
    }

    private HostSpec getHostSpecFromInstanceId(String instanceId) {
        if (instanceId == null) {
            return null;
        }
        List<HostSpec> hosts = this.pluginService.getHosts();
        for (HostSpec host : hosts) {
            if (!host.getUrl().startsWith(instanceId)) continue;
            return host;
        }
        return null;
    }

    private String getCurrentInstanceId(Connection conn, String driverProtocol) {
        String instanceNameCol;
        String retrieveInstanceQuery;
        if (driverProtocol.startsWith(PG_DRIVER_PROTOCOL)) {
            retrieveInstanceQuery = PG_GET_INSTANCE_NAME_SQL;
            instanceNameCol = PG_INSTANCE_NAME_COL;
        } else if (driverProtocol.startsWith(MYSQL_DRIVER_PROTOCOL)) {
            retrieveInstanceQuery = MYSQL_GET_INSTANCE_NAME_SQL;
            instanceNameCol = MYSQL_INSTANCE_NAME_COL;
        } else {
            throw new UnsupportedOperationException(Messages.get("ReadWriteSplittingPlugin.unsupportedDriverProtocol", new Object[]{driverProtocol}));
        }
        String instanceName = null;
        try (Statement stmt = conn.createStatement();
             ResultSet resultSet = stmt.executeQuery(retrieveInstanceQuery);){
            if (resultSet.next()) {
                instanceName = resultSet.getString(instanceNameCol);
            }
        }
        catch (SQLException e) {
            return null;
        }
        return instanceName;
    }

    @Override
    public OldConnectionSuggestedAction notifyConnectionChanged(EnumSet<NodeChangeOptions> changes) {
        try {
            this.updateInternalConnectionInfo();
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
        if (this.inReadWriteSplit.get()) {
            return OldConnectionSuggestedAction.PRESERVE;
        }
        return OldConnectionSuggestedAction.NO_OPINION;
    }

    @Override
    public <T, E extends Exception> T execute(Class<T> resultClass, Class<E> exceptionClass, Object methodInvokeOn, String methodName, JdbcCallable<T, E> jdbcMethodFunc, Object[] args) throws E {
        Connection conn = WrapperUtils.getConnectionFromSqlObject(methodInvokeOn);
        if (conn != null && conn != this.pluginService.getCurrentConnection()) {
            LOGGER.fine(() -> Messages.get("ReadWriteSplittingPlugin.executingAgainstOldConnection", new Object[]{methodInvokeOn}));
            return jdbcMethodFunc.call();
        }
        if (methodName.equals(METHOD_SET_READ_ONLY) && args != null && args.length > 0) {
            try {
                this.switchConnectionIfRequired((Boolean)args[0]);
            }
            catch (SQLException e) {
                throw WrapperUtils.wrapExceptionIfNeeded(exceptionClass, e);
            }
        }
        try {
            return jdbcMethodFunc.call();
        }
        catch (Exception e) {
            if (e instanceof FailoverSQLException) {
                LOGGER.finer(() -> Messages.get("ReadWriteSplittingPlugin.failoverExceptionWhileExecutingCommand", new Object[]{methodName}));
                this.closeIdleConnections();
            } else {
                LOGGER.finest(() -> Messages.get("ReadWriteSplittingPlugin.exceptionWhileExecutingCommand", new Object[]{methodName}));
            }
            throw e;
        }
    }

    private void updateInternalConnectionInfo() throws SQLException {
        Connection currentConnection = this.pluginService.getCurrentConnection();
        HostSpec currentHost = this.pluginService.getCurrentHostSpec();
        if (currentConnection == null || currentHost == null) {
            return;
        }
        if (this.isWriter(currentHost)) {
            this.setWriterConnection(currentConnection, currentHost);
        } else {
            this.setReaderConnection(currentConnection, currentHost);
        }
    }

    private boolean isWriter(@NonNull HostSpec hostSpec) {
        return HostRole.WRITER.equals((Object)hostSpec.getRole());
    }

    private boolean isReader(@NonNull HostSpec hostSpec) {
        return HostRole.READER.equals((Object)hostSpec.getRole());
    }

    private void getNewWriterConnection(HostSpec writerHostSpec) throws SQLException {
        Connection conn = this.pluginService.connect(writerHostSpec, this.properties);
        this.setWriterConnection(conn, writerHostSpec);
        this.switchCurrentConnectionTo(this.writerConnection, writerHostSpec);
    }

    private void setWriterConnection(Connection writerConnection, HostSpec writerHostSpec) {
        this.writerConnection = writerConnection;
        LOGGER.finest(() -> Messages.get("ReadWriteSplittingPlugin.setWriterConnection", new Object[]{writerHostSpec.getUrl()}));
    }

    private void setReaderConnection(Connection conn, HostSpec host) {
        this.readerConnection = conn;
        this.readerHostSpec = host;
        LOGGER.finest(() -> Messages.get("ReadWriteSplittingPlugin.setReaderConnection", new Object[]{host.getUrl()}));
    }

    void switchConnectionIfRequired(boolean readOnly) throws SQLException {
        List<HostSpec> hosts;
        Connection currentConnection = this.pluginService.getCurrentConnection();
        if (currentConnection != null && currentConnection.isClosed()) {
            this.logAndThrowException(Messages.get("ReadWriteSplittingPlugin.setReadOnlyOnClosedConnection"), SqlState.CONNECTION_NOT_OPEN);
        }
        if (this.isConnectionUsable(currentConnection)) {
            try {
                this.pluginService.refreshHostList();
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
        }
        if ((hosts = this.pluginService.getHosts()) == null || hosts.isEmpty()) {
            this.logAndThrowException(Messages.get("ReadWriteSplittingPlugin.emptyHostList"));
        }
        HostSpec currentHost = this.pluginService.getCurrentHostSpec();
        if (readOnly) {
            if (!this.pluginService.isInTransaction() && !this.isReader(currentHost)) {
                try {
                    this.switchToReaderConnection(hosts);
                }
                catch (SQLException e) {
                    if (!this.isConnectionUsable(currentConnection)) {
                        this.logAndThrowException(Messages.get("ReadWriteSplittingPlugin.errorSwitchingToReader"), SqlState.CONNECTION_UNABLE_TO_CONNECT);
                        return;
                    }
                    LOGGER.warning(() -> Messages.get("ReadWriteSplittingPlugin.fallbackToWriter", new Object[]{this.pluginService.getCurrentHostSpec().getUrl()}));
                }
            }
        } else {
            if (!this.isWriter(currentHost) && this.pluginService.isInTransaction()) {
                this.logAndThrowException(Messages.get("ReadWriteSplittingPlugin.setReadOnlyFalseInTransaction"), SqlState.ACTIVE_SQL_TRANSACTION);
            }
            if (!this.isWriter(currentHost)) {
                try {
                    this.switchToWriterConnection(hosts);
                }
                catch (SQLException e) {
                    this.logAndThrowException(Messages.get("ReadWriteSplittingPlugin.errorSwitchingToWriter"), SqlState.CONNECTION_UNABLE_TO_CONNECT);
                }
            }
        }
    }

    private void logAndThrowException(String logMessage) throws SQLException {
        LOGGER.severe(logMessage);
        throw new ReadWriteSplittingSQLException(logMessage);
    }

    private void logAndThrowException(String logMessage, SqlState sqlState) throws SQLException {
        LOGGER.severe(logMessage);
        throw new ReadWriteSplittingSQLException(logMessage, sqlState.getState());
    }

    private synchronized void switchToWriterConnection(List<HostSpec> hosts) throws SQLException {
        Connection currentConnection = this.pluginService.getCurrentConnection();
        HostSpec currentHost = this.pluginService.getCurrentHostSpec();
        if (this.isWriter(currentHost) && this.isConnectionUsable(currentConnection)) {
            return;
        }
        this.inReadWriteSplit.set(true);
        HostSpec writerHost = this.getWriter(hosts);
        if (!this.isConnectionUsable(this.writerConnection)) {
            this.getNewWriterConnection(writerHost);
        } else {
            this.switchCurrentConnectionTo(this.writerConnection, writerHost);
        }
        LOGGER.finer(() -> Messages.get("ReadWriteSplittingPlugin.switchedFromReaderToWriter", new Object[]{writerHost.getUrl()}));
    }

    private void switchCurrentConnectionTo(Connection newConnection, HostSpec newConnectionHost) throws SQLException {
        Connection currentConnection = this.pluginService.getCurrentConnection();
        if (currentConnection == newConnection) {
            return;
        }
        this.transferSessionStateOnReadWriteSplit(newConnection);
        this.pluginService.setCurrentConnection(newConnection, newConnectionHost);
        LOGGER.finest(() -> Messages.get("ReadWriteSplittingPlugin.settingCurrentConnection", new Object[]{newConnectionHost.getUrl()}));
    }

    protected void transferSessionStateOnReadWriteSplit(Connection to) throws SQLException {
        Connection from = this.pluginService.getCurrentConnection();
        if (from == null || to == null) {
            return;
        }
        to.setAutoCommit(from.getAutoCommit());
        to.setTransactionIsolation(from.getTransactionIsolation());
    }

    private synchronized void switchToReaderConnection(List<HostSpec> hosts) throws SQLException {
        Connection currentConnection = this.pluginService.getCurrentConnection();
        HostSpec currentHost = this.pluginService.getCurrentHostSpec();
        if (this.isReader(currentHost) && this.isConnectionUsable(currentConnection)) {
            return;
        }
        this.inReadWriteSplit.set(true);
        if (!this.isConnectionUsable(this.readerConnection)) {
            this.initializeReaderConnection(hosts);
        } else {
            this.switchCurrentConnectionTo(this.readerConnection, this.readerHostSpec);
            LOGGER.finer(() -> Messages.get("ReadWriteSplittingPlugin.switchedFromWriterToReader", new Object[]{this.readerHostSpec.getUrl()}));
        }
    }

    private void initializeReaderConnection(@NonNull List<HostSpec> hosts) throws SQLException {
        if (hosts.size() == 1) {
            HostSpec writerHost = this.getWriter(hosts);
            if (!this.isConnectionUsable(this.writerConnection)) {
                this.getNewWriterConnection(writerHost);
            }
            LOGGER.warning(() -> Messages.get("ReadWriteSplittingPlugin.noReadersFound", new Object[]{writerHost.getUrl()}));
        } else {
            this.getNewReaderConnection();
            LOGGER.finer(() -> Messages.get("ReadWriteSplittingPlugin.switchedFromWriterToReader", new Object[]{this.readerHostSpec.getUrl()}));
        }
    }

    private HostSpec getWriter(@NonNull List<HostSpec> hosts) throws SQLException {
        HostSpec writerHost = null;
        for (HostSpec hostSpec : hosts) {
            if (!HostRole.WRITER.equals((Object)hostSpec.getRole())) continue;
            writerHost = hostSpec;
            break;
        }
        if (writerHost == null) {
            this.logAndThrowException(Messages.get("ReadWriteSplittingPlugin.noWriterFound"));
        }
        return writerHost;
    }

    private void getNewReaderConnection() throws SQLException {
        ArrayDeque<HostSpec> readerHosts = this.getRandomReaderHosts();
        Connection conn = null;
        HostSpec readerHost = null;
        while (!readerHosts.isEmpty()) {
            HostSpec host = readerHosts.poll();
            try {
                conn = this.pluginService.connect(host, this.properties);
                readerHost = host;
                break;
            }
            catch (SQLException e) {
                LOGGER.config(() -> Messages.get("ReadWriteSplittingPlugin.failedToConnectToReader", new Object[]{host.getUrl()}));
            }
        }
        if (conn == null || readerHost == null) {
            this.logAndThrowException(Messages.get("ReadWriteSplittingPlugin.noReadersAvailable"), SqlState.CONNECTION_UNABLE_TO_CONNECT);
            return;
        }
        HostSpec finalReaderHost = readerHost;
        LOGGER.finest(() -> Messages.get("ReadWriteSplittingPlugin.successfullyConnectedToReader", new Object[]{finalReaderHost.getUrl()}));
        this.setReaderConnection(conn, readerHost);
        this.switchCurrentConnectionTo(this.readerConnection, this.readerHostSpec);
    }

    private ArrayDeque<HostSpec> getRandomReaderHosts() {
        List<HostSpec> hosts = this.pluginService.getHosts();
        ArrayList<HostSpec> readerHosts = new ArrayList<HostSpec>();
        for (HostSpec host : hosts) {
            if (!HostRole.READER.equals((Object)host.getRole()) || this.pluginService.getCurrentHostSpec().getUrl().equals(host.getUrl())) continue;
            readerHosts.add(host);
        }
        Collections.shuffle(readerHosts);
        return new ArrayDeque<HostSpec>(readerHosts);
    }

    private boolean isConnectionUsable(Connection connection) throws SQLException {
        return connection != null && !connection.isClosed();
    }

    @Override
    public void releaseResources() {
        this.closeIdleConnections();
    }

    private void closeIdleConnections() {
        LOGGER.finest(() -> Messages.get("ReadWriteSplittingPlugin.closingInternalConnections"));
        this.closeConnectionIfIdle(this.readerConnection);
        this.closeConnectionIfIdle(this.writerConnection);
    }

    private void closeConnectionIfIdle(Connection internalConnection) {
        Connection currentConnection = this.pluginService.getCurrentConnection();
        try {
            if (internalConnection != null && internalConnection != currentConnection && !internalConnection.isClosed()) {
                internalConnection.close();
                if (internalConnection == this.writerConnection) {
                    this.writerConnection = null;
                }
                if (internalConnection == this.readerConnection) {
                    this.readerConnection = null;
                }
            }
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    Connection getWriterConnection() {
        return this.writerConnection;
    }

    Connection getReaderConnection() {
        return this.readerConnection;
    }
}

