/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.documentdb.jdbc.sshtunnel;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.hash.Hashing;
import com.jcraft.jsch.HostKey;
import com.jcraft.jsch.HostKeyRepository;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.documentdb.jdbc.DocumentDbConnectionProperties;
import software.amazon.documentdb.jdbc.common.utilities.SqlError;
import software.amazon.documentdb.jdbc.common.utilities.SqlState;

public final class DocumentDbSshTunnelServer
implements AutoCloseable {
    public static final String SSH_KNOWN_HOSTS_FILE = "~/.ssh/known_hosts";
    public static final String STRICT_HOST_KEY_CHECKING = "StrictHostKeyChecking";
    public static final String HASH_KNOWN_HOSTS = "HashKnownHosts";
    public static final String SERVER_HOST_KEY = "server_host_key";
    public static final String YES = "yes";
    public static final String NO = "no";
    public static final String LOCALHOST = "localhost";
    public static final int DEFAULT_DOCUMENTDB_PORT = 27017;
    public static final int DEFAULT_SSH_PORT = 22;
    private static final Logger LOGGER = LoggerFactory.getLogger(DocumentDbSshTunnelServer.class);
    public static final int DEFAULT_CLOSE_DELAY_MS = 30000;
    private final Object mutex = new Object();
    private final AtomicLong clientCount = new AtomicLong(0L);
    private final String sshUser;
    private final String sshHostname;
    private final String sshPrivateKeyFile;
    private final String sshPrivateKeyPassphrase;
    private final boolean sshStrictHostKeyChecking;
    private final String sshKnownHostsFile;
    private final String remoteHostname;
    private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);
    private SshPortForwardingSession session = null;
    private ScheduledFuture<?> scheduledFuture = null;
    private long closeDelayMS = 30000L;

    private DocumentDbSshTunnelServer(DocumentDbSshTunnelServerBuilder builder) {
        this.sshUser = builder.sshUser;
        this.sshHostname = builder.sshHostname;
        this.sshPrivateKeyFile = builder.sshPrivateKeyFile;
        this.remoteHostname = builder.sshRemoteHostname;
        this.sshPrivateKeyPassphrase = builder.sshPrivateKeyPassphrase;
        this.sshStrictHostKeyChecking = builder.sshStrictHostKeyChecking;
        this.sshKnownHostsFile = builder.sshKnownHostsFile;
        LOGGER.debug("sshUser='{}' sshHostname='{}' sshPrivateKeyFile='{}' remoteHostname'{} sshPrivateKeyPassphrase='{}' sshStrictHostKeyChecking='{}' sshKnownHostsFile='{}'", new Object[]{this.sshUser, this.sshHostname, this.sshPrivateKeyFile, this.remoteHostname, this.sshPrivateKeyPassphrase, this.sshStrictHostKeyChecking, this.sshKnownHostsFile});
    }

    static String getHashString(String sshUser, String sshHostname, String sshPrivateKeyFile, String remoteHostname) {
        String sshPropertiesString = sshUser + "-" + sshHostname + "-" + sshPrivateKeyFile + remoteHostname;
        return Hashing.sha256().hashString((CharSequence)sshPropertiesString, StandardCharsets.UTF_8).toString();
    }

    public static SshPortForwardingSession createSshTunnel(DocumentDbConnectionProperties connectionProperties) throws SQLException {
        DocumentDbSshTunnelServer.validateSshPrivateKeyFile(connectionProperties);
        LOGGER.debug("Internal SSH tunnel starting.");
        try {
            JSch jSch = new JSch();
            DocumentDbSshTunnelServer.addIdentity(connectionProperties, jSch);
            Session session = DocumentDbSshTunnelServer.createSession(connectionProperties, jSch);
            DocumentDbSshTunnelServer.connectSession(connectionProperties, jSch, session);
            SshPortForwardingSession portForwardingSession = DocumentDbSshTunnelServer.getPortForwardingSession(connectionProperties, session);
            LOGGER.debug("Internal SSH tunnel started on local port '{}'.", (Object)portForwardingSession.getLocalPort());
            LOGGER.debug("Internal SSH tunnel started.");
            return portForwardingSession;
        }
        catch (Exception e) {
            throw DocumentDbSshTunnelServer.logException(e);
        }
    }

    private static SshPortForwardingSession getPortForwardingSession(DocumentDbConnectionProperties connectionProperties, Session session) throws JSchException {
        Pair<String, Integer> clusterHostAndPort = DocumentDbSshTunnelServer.getHostAndPort(connectionProperties.getHostname(), 27017);
        int localPort = session.setPortForwardingL(LOCALHOST, 0, (String)clusterHostAndPort.getLeft(), ((Integer)clusterHostAndPort.getRight()).intValue());
        return new SshPortForwardingSession(session, localPort);
    }

    private static Pair<String, Integer> getHostAndPort(String hostname, int defaultPort) {
        int clusterPort;
        String clusterHost;
        int portSeparatorIndex = hostname.indexOf(58);
        if (portSeparatorIndex >= 0) {
            clusterHost = hostname.substring(0, portSeparatorIndex);
            clusterPort = Integer.parseInt(hostname.substring(portSeparatorIndex + 1));
        } else {
            clusterHost = hostname;
            clusterPort = defaultPort;
        }
        return new ImmutablePair((Object)clusterHost, (Object)clusterPort);
    }

    private static void connectSession(DocumentDbConnectionProperties connectionProperties, JSch jSch, Session session) throws SQLException {
        DocumentDbSshTunnelServer.setSecurityConfig(connectionProperties, jSch, session);
        try {
            session.connect();
        }
        catch (JSchException e) {
            throw DocumentDbSshTunnelServer.logException(e);
        }
    }

    private static void addIdentity(DocumentDbConnectionProperties connectionProperties, JSch jSch) throws JSchException {
        String privateKeyFileName = DocumentDbConnectionProperties.getPath(connectionProperties.getSshPrivateKeyFile(), DocumentDbConnectionProperties.getDocumentDbSearchPaths()).toString();
        LOGGER.debug("SSH private key file resolved to '{}'.", (Object)privateKeyFileName);
        String passPhrase = !DocumentDbConnectionProperties.isNullOrWhitespace(connectionProperties.getSshPrivateKeyPassphrase()) ? connectionProperties.getSshPrivateKeyPassphrase() : null;
        jSch.addIdentity(privateKeyFileName, passPhrase);
    }

    private static Session createSession(DocumentDbConnectionProperties connectionProperties, JSch jSch) throws SQLException {
        String sshUsername = connectionProperties.getSshUser();
        Pair<String, Integer> sshHostAndPort = DocumentDbSshTunnelServer.getHostAndPort(connectionProperties.getSshHostname(), 22);
        DocumentDbSshTunnelServer.setKnownHostsFile(connectionProperties, jSch);
        try {
            return jSch.getSession(sshUsername, (String)sshHostAndPort.getLeft(), ((Integer)sshHostAndPort.getRight()).intValue());
        }
        catch (JSchException e) {
            throw DocumentDbSshTunnelServer.logException(e);
        }
    }

    private static void setSecurityConfig(DocumentDbConnectionProperties connectionProperties, JSch jSch, Session session) {
        if (!connectionProperties.getSshStrictHostKeyChecking()) {
            session.setConfig(STRICT_HOST_KEY_CHECKING, NO);
            return;
        }
        DocumentDbSshTunnelServer.setHostKeyType(connectionProperties, jSch, session);
    }

    private static void setHostKeyType(DocumentDbConnectionProperties connectionProperties, JSch jSch, Session session) {
        String hostKeyType;
        HostKeyRepository keyRepository = jSch.getHostKeyRepository();
        HostKey[] hostKeys = keyRepository.getHostKey();
        Pair<String, Integer> sshHostAndPort = DocumentDbSshTunnelServer.getHostAndPort(connectionProperties.getSshHostname(), 22);
        HostKey hostKey = Arrays.stream(hostKeys).filter(hk -> hk.getHost().equals(sshHostAndPort.getLeft())).findFirst().orElse(null);
        String string = hostKeyType = hostKey != null ? hostKey.getType() : null;
        if (hostKeyType != null) {
            session.setConfig(SERVER_HOST_KEY, session.getConfig(SERVER_HOST_KEY) + "," + hostKeyType);
        }
        session.setConfig(HASH_KNOWN_HOSTS, YES);
    }

    private static void setKnownHostsFile(DocumentDbConnectionProperties connectionProperties, JSch jSch) throws SQLException {
        if (!connectionProperties.getSshStrictHostKeyChecking()) {
            return;
        }
        String knownHostsFilename = DocumentDbSshTunnelServer.getSshKnownHostsFilename(connectionProperties);
        try {
            jSch.setKnownHosts(knownHostsFilename);
        }
        catch (JSchException e) {
            throw DocumentDbSshTunnelServer.logException(e);
        }
    }

    private static <T extends Exception> SQLException logException(T e) {
        LOGGER.error(e.getMessage(), e);
        if (e instanceof SQLException) {
            return (SQLException)e;
        }
        return new SQLException(e.getMessage(), e);
    }

    public int getServiceListeningPort() {
        return this.session != null ? this.session.getLocalPort() : 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        Object object = this.mutex;
        synchronized (object) {
            if (this.session != null) {
                LOGGER.debug("Internal SSH Tunnel is stopping.");
                this.session.getSession().disconnect();
                this.session = null;
                LOGGER.debug("Internal SSH Tunnel is stopped.");
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addClient() throws SQLException {
        Object object = this.mutex;
        synchronized (object) {
            this.cancelScheduledFutureClose();
            this.clientCount.incrementAndGet();
            if (this.session != null && this.session.getLocalPort() != 0) {
                return;
            }
            this.validateLocalSshFilesExists();
            this.session = DocumentDbSshTunnelServer.createSshTunnel(this.getConnectionProperties());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeClient() throws SQLException {
        Object object = this.mutex;
        synchronized (object) {
            if (this.clientCount.get() <= 0L || this.clientCount.decrementAndGet() > 0L) {
                return;
            }
            this.closeSession();
        }
    }

    private void closeSession() throws SQLException {
        this.cancelScheduledFutureClose();
        long delayMS = this.getCloseDelayMS();
        if (delayMS <= 0L) {
            this.close();
        } else {
            LOGGER.debug("Close timer is being scheduled.");
            this.scheduledFuture = this.scheduler.schedule(this.getCloseTimerTask(), delayMS, TimeUnit.MILLISECONDS);
        }
    }

    private Runnable getCloseTimerTask() {
        return () -> {
            try {
                this.close();
            }
            catch (Exception e) {
                LOGGER.warn(e.getMessage(), (Throwable)e);
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cancelScheduledFutureClose() throws SQLException {
        Object object = this.mutex;
        synchronized (object) {
            if (this.scheduledFuture != null) {
                LOGGER.debug("Close timer is being cancelled.");
                while (!this.scheduledFuture.isDone()) {
                    this.scheduledFuture.cancel(false);
                    try {
                        TimeUnit.MILLISECONDS.sleep(10L);
                    }
                    catch (InterruptedException e) {
                        throw new SQLException(e.getMessage(), e);
                    }
                }
            }
            this.scheduledFuture = null;
        }
    }

    @VisibleForTesting
    long getCloseDelayMS() {
        return this.closeDelayMS;
    }

    @VisibleForTesting
    void setCloseDelayMS(long closeDelayMS) {
        this.closeDelayMS = closeDelayMS > 0L ? closeDelayMS : 0L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    long getClientCount() {
        Object object = this.mutex;
        synchronized (object) {
            return this.clientCount.get();
        }
    }

    public boolean isAlive() {
        return this.session != null;
    }

    public static DocumentDbSshTunnelServerBuilder builder(String user, String hostname, String privateKeyFile, String remoteHostname) {
        return new DocumentDbSshTunnelServerBuilder(user, hostname, privateKeyFile, remoteHostname);
    }

    private @NonNull DocumentDbConnectionProperties getConnectionProperties() {
        DocumentDbConnectionProperties connectionProperties = new DocumentDbConnectionProperties();
        connectionProperties.setHostname(this.remoteHostname);
        connectionProperties.setSshUser(this.sshUser);
        connectionProperties.setSshHostname(this.sshHostname);
        connectionProperties.setSshPrivateKeyFile(this.sshPrivateKeyFile);
        connectionProperties.setSshStrictHostKeyChecking(String.valueOf(this.sshStrictHostKeyChecking));
        if (this.sshPrivateKeyPassphrase != null) {
            connectionProperties.setSshPrivateKeyPassphrase(this.sshPrivateKeyPassphrase);
        }
        if (this.sshKnownHostsFile != null) {
            connectionProperties.setSshKnownHostsFile(this.sshKnownHostsFile);
        }
        return connectionProperties;
    }

    private void validateLocalSshFilesExists() throws SQLException {
        DocumentDbConnectionProperties connectionProperties = this.getConnectionProperties();
        DocumentDbSshTunnelServer.validateSshPrivateKeyFile(connectionProperties);
        DocumentDbSshTunnelServer.getSshKnownHostsFilename(connectionProperties);
    }

    static void validateSshPrivateKeyFile(DocumentDbConnectionProperties connectionProperties) throws SQLException {
        if (!connectionProperties.isSshPrivateKeyFileExists()) {
            throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.SSH_PRIVATE_KEY_FILE_NOT_FOUND, connectionProperties.getSshPrivateKeyFile());
        }
    }

    static String getSshKnownHostsFilename(DocumentDbConnectionProperties connectionProperties) throws SQLException {
        String knowHostsFilename;
        if (!DocumentDbConnectionProperties.isNullOrWhitespace(connectionProperties.getSshKnownHostsFile())) {
            Path knownHostsPath = DocumentDbConnectionProperties.getPath(connectionProperties.getSshKnownHostsFile(), new String[0]);
            DocumentDbSshTunnelServer.validateSshKnownHostsFile(connectionProperties, knownHostsPath);
            knowHostsFilename = knownHostsPath.toString();
        } else {
            knowHostsFilename = DocumentDbConnectionProperties.getPath(SSH_KNOWN_HOSTS_FILE, new String[0]).toString();
        }
        return knowHostsFilename;
    }

    private static void validateSshKnownHostsFile(DocumentDbConnectionProperties connectionProperties, Path knownHostsPath) throws SQLException {
        if (!Files.exists(knownHostsPath, new LinkOption[0])) {
            throw SqlError.createSQLException(LOGGER, SqlState.INVALID_PARAMETER_VALUE, SqlError.KNOWN_HOSTS_FILE_NOT_FOUND, connectionProperties.getSshKnownHostsFile());
        }
    }

    static class SshPortForwardingSession {
        private final Session session;
        private final int localPort;

        public Session getSession() {
            return this.session;
        }

        public int getLocalPort() {
            return this.localPort;
        }

        public SshPortForwardingSession(Session session, int localPort) {
            this.session = session;
            this.localPort = localPort;
        }
    }

    public static class DocumentDbSshTunnelServerBuilder {
        private final String sshUser;
        private final String sshHostname;
        private final String sshPrivateKeyFile;
        private final String sshRemoteHostname;
        private String sshPrivateKeyPassphrase = null;
        private boolean sshStrictHostKeyChecking = true;
        private String sshKnownHostsFile = null;
        private static final ConcurrentMap<String, DocumentDbSshTunnelServer> SSH_TUNNEL_MAP = new ConcurrentHashMap<String, DocumentDbSshTunnelServer>();

        DocumentDbSshTunnelServerBuilder(String sshUser, String sshHostname, String sshPrivateKeyFile, String sshRemoteHostname) {
            this.sshUser = sshUser;
            this.sshHostname = sshHostname;
            this.sshPrivateKeyFile = sshPrivateKeyFile;
            this.sshRemoteHostname = sshRemoteHostname;
        }

        public DocumentDbSshTunnelServerBuilder sshPrivateKeyPassphrase(String sshPrivateKeyPassphrase) {
            this.sshPrivateKeyPassphrase = sshPrivateKeyPassphrase;
            return this;
        }

        public DocumentDbSshTunnelServerBuilder sshStrictHostKeyChecking(boolean sshStrictHostKeyChecking) {
            this.sshStrictHostKeyChecking = sshStrictHostKeyChecking;
            return this;
        }

        public DocumentDbSshTunnelServerBuilder sshKnownHostsFile(String sshKnownHostsFile) {
            this.sshKnownHostsFile = sshKnownHostsFile;
            return this;
        }

        public DocumentDbSshTunnelServer build() {
            String hashString = DocumentDbSshTunnelServer.getHashString(this.sshUser, this.sshHostname, this.sshPrivateKeyFile, this.sshRemoteHostname);
            return SSH_TUNNEL_MAP.computeIfAbsent(hashString, key -> new DocumentDbSshTunnelServer(this));
        }
    }
}

