/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.documentdb.jdbc.sshtunnel;

import com.jcraft.jsch.HostKey;
import com.jcraft.jsch.HostKeyRepository;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.channels.Channels;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.sql.SQLException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.documentdb.jdbc.DocumentDbConnectionProperties;
import software.amazon.documentdb.jdbc.sshtunnel.DocumentDbMultiThreadFileChannel;
import software.amazon.documentdb.jdbc.sshtunnel.DocumentDbSshTunnelLock;
import software.amazon.documentdb.jdbc.sshtunnel.DocumentDbSshTunnelServer;

public class DocumentDbSshTunnelService
implements AutoCloseable,
Runnable {
    public static final String SSH_KNOWN_HOSTS_FILE = "~/.ssh/known_hosts";
    public static final String STRICT_HOST_KEY_CHECKING = "StrictHostKeyChecking";
    public static final String HASH_KNOWN_HOSTS = "HashKnownHosts";
    public static final String SERVER_HOST_KEY = "server_host_key";
    public static final String YES = "yes";
    public static final String NO = "no";
    public static final String LOCALHOST = "localhost";
    public static final int DEFAULT_DOCUMENTDB_PORT = 27017;
    public static final int DEFAULT_SSH_PORT = 22;
    private static final Logger LOGGER = LoggerFactory.getLogger(DocumentDbSshTunnelService.class);
    public static final int CLIENT_WATCH_POLL_TIME = 500;
    private final DocumentDbConnectionProperties connectionProperties;
    private final String sshPropertiesHashString;
    private volatile boolean completed = false;
    private volatile boolean interrupted = false;
    private final ConcurrentLinkedDeque<Exception> exceptions = new ConcurrentLinkedDeque();

    public DocumentDbSshTunnelService(String connectionString) throws SQLException {
        this.connectionProperties = DocumentDbConnectionProperties.getPropertiesFromConnectionString(connectionString, DocumentDbConnectionProperties.ValidationType.SSH_TUNNEL);
        this.sshPropertiesHashString = DocumentDbSshTunnelLock.getHashString(this.connectionProperties.getSshUser(), this.connectionProperties.getSshHostname(), this.connectionProperties.getSshPrivateKeyFile(), this.connectionProperties.getHostname());
    }

    @Override
    public void close() {
        this.interrupted = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     */
    @Override
    public void run() {
        SshPortForwardingSession session = null;
        DocumentDbMultiThreadFileChannel serverChannel = null;
        FileLock serverLock = null;
        while (!this.interrupted && !this.completed) {
            LOGGER.debug("SSH Tunnel service starting.");
            session = this.performSshTunnelSessionStartup();
            Map.Entry<DocumentDbMultiThreadFileChannel, FileLock> lock = this.acquireServerLock();
            serverChannel = lock.getKey();
            serverLock = lock.getValue();
            LOGGER.debug("SSH Tunnel service started.");
            this.waitForClients(serverLock);
            try {
                LOGGER.debug("SSH Tunnel service stopping.");
                this.cleanupResourcesInGlobalLock(session, serverChannel, serverLock);
            }
            catch (Exception e) {
                this.exceptions.add(DocumentDbSshTunnelService.logException(e));
            }
            this.completed = true;
            continue;
            catch (InterruptedException e) {
                DocumentDbSshTunnelService.logException(e);
                this.interrupted = true;
                try {
                    LOGGER.debug("SSH Tunnel service stopping.");
                    this.cleanupResourcesInGlobalLock(session, serverChannel, serverLock);
                }
                catch (Exception e2) {
                    this.exceptions.add(DocumentDbSshTunnelService.logException(e2));
                }
                this.completed = true;
                continue;
            }
            catch (Exception e2) {
                this.exceptions.add(DocumentDbSshTunnelService.logException(e2));
                {
                    catch (Throwable throwable) {
                        try {
                            LOGGER.debug("SSH Tunnel service stopping.");
                            this.cleanupResourcesInGlobalLock(session, serverChannel, serverLock);
                        }
                        catch (Exception e3) {
                            this.exceptions.add(DocumentDbSshTunnelService.logException(e3));
                        }
                        this.completed = true;
                        throw throwable;
                    }
                }
                try {
                    LOGGER.debug("SSH Tunnel service stopping.");
                    this.cleanupResourcesInGlobalLock(session, serverChannel, serverLock);
                }
                catch (Exception e4) {
                    this.exceptions.add(DocumentDbSshTunnelService.logException(e4));
                }
                this.completed = true;
            }
        }
        LOGGER.debug("SSH Tunnel service stopped.");
    }

    private void cleanupResourcesInGlobalLock(SshPortForwardingSession session, DocumentDbMultiThreadFileChannel serverChannel, FileLock serverLock) throws Exception {
        DocumentDbSshTunnelLock.runInGlobalLock(this.sshPropertiesHashString, () -> this.closeResources(session, serverChannel, serverLock, this.exceptions));
    }

    private Queue<Exception> closeResources(SshPortForwardingSession session, DocumentDbMultiThreadFileChannel serverChannel, FileLock serverLock, Queue<Exception> exceptions) {
        try {
            if (serverLock != null && serverLock.isValid()) {
                serverLock.close();
            }
            if (serverChannel != null && serverChannel.isOpen()) {
                serverChannel.close();
            }
            if (session != null) {
                session.getSession().disconnect();
            }
            Path portLockPath = DocumentDbSshTunnelLock.getPortLockPath(this.sshPropertiesHashString);
            Path startupLockPath = DocumentDbSshTunnelLock.getStartupLockPath(this.sshPropertiesHashString);
            Files.deleteIfExists(portLockPath);
            Files.deleteIfExists(startupLockPath);
        }
        catch (Exception e) {
            exceptions.add(DocumentDbSshTunnelService.logException(e));
        }
        return exceptions;
    }

    public String getSshPropertiesHashString() {
        return this.sshPropertiesHashString;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void waitForClients(FileLock serverLock) throws InterruptedException {
        ClientWatcher clientWatcher = null;
        try {
            clientWatcher = new ClientWatcher(serverLock, this.sshPropertiesHashString);
            Thread clientWatcherThread = new Thread(clientWatcher);
            clientWatcherThread.setDaemon(true);
            clientWatcherThread.start();
            do {
                clientWatcherThread.join(1000L);
            } while (clientWatcherThread.isAlive() && !this.interrupted);
        }
        finally {
            if (clientWatcher != null) {
                this.exceptions.addAll(clientWatcher.getExceptions());
            }
        }
    }

    private static Exception closeServerLock(FileLock serverLock) {
        SQLException result = null;
        if (serverLock != null && serverLock.isValid()) {
            try {
                serverLock.close();
            }
            catch (IOException e) {
                result = DocumentDbSshTunnelService.logException(e);
            }
        }
        return result;
    }

    public static SshPortForwardingSession createSshTunnel(DocumentDbConnectionProperties connectionProperties) throws SQLException {
        DocumentDbSshTunnelServer.validateSshPrivateKeyFile(connectionProperties);
        LOGGER.debug("Internal SSH tunnel starting.");
        try {
            JSch jSch = new JSch();
            DocumentDbSshTunnelService.addIdentity(connectionProperties, jSch);
            Session session = DocumentDbSshTunnelService.createSession(connectionProperties, jSch);
            DocumentDbSshTunnelService.connectSession(connectionProperties, jSch, session);
            SshPortForwardingSession portForwardingSession = DocumentDbSshTunnelService.getPortForwardingSession(connectionProperties, session);
            LOGGER.debug("Internal SSH tunnel started on local port '{}'.", (Object)portForwardingSession.getLocalPort());
            return portForwardingSession;
        }
        catch (Exception e) {
            throw DocumentDbSshTunnelService.logException(e);
        }
    }

    private static SshPortForwardingSession getPortForwardingSession(DocumentDbConnectionProperties connectionProperties, Session session) throws JSchException {
        Pair<String, Integer> clusterHostAndPort = DocumentDbSshTunnelService.getHostAndPort(connectionProperties.getHostname(), 27017);
        int localPort = session.setPortForwardingL(LOCALHOST, 0, (String)clusterHostAndPort.getLeft(), ((Integer)clusterHostAndPort.getRight()).intValue());
        return new SshPortForwardingSession(session, localPort);
    }

    private static Pair<String, Integer> getHostAndPort(String hostname, int defaultPort) {
        int clusterPort;
        String clusterHost;
        int portSeparatorIndex = hostname.indexOf(58);
        if (portSeparatorIndex >= 0) {
            clusterHost = hostname.substring(0, portSeparatorIndex);
            clusterPort = Integer.parseInt(hostname.substring(portSeparatorIndex + 1));
        } else {
            clusterHost = hostname;
            clusterPort = defaultPort;
        }
        return new ImmutablePair((Object)clusterHost, (Object)clusterPort);
    }

    private static void connectSession(DocumentDbConnectionProperties connectionProperties, JSch jSch, Session session) throws SQLException {
        DocumentDbSshTunnelService.setSecurityConfig(connectionProperties, jSch, session);
        try {
            session.connect();
        }
        catch (JSchException e) {
            throw DocumentDbSshTunnelService.logException(e);
        }
    }

    private static void addIdentity(DocumentDbConnectionProperties connectionProperties, JSch jSch) throws JSchException {
        String privateKeyFileName = DocumentDbConnectionProperties.getPath(connectionProperties.getSshPrivateKeyFile(), DocumentDbConnectionProperties.getDocumentDbSearchPaths()).toString();
        LOGGER.debug("SSH private key file resolved to '{}'.", (Object)privateKeyFileName);
        String passPhrase = !DocumentDbConnectionProperties.isNullOrWhitespace(connectionProperties.getSshPrivateKeyPassphrase()) ? connectionProperties.getSshPrivateKeyPassphrase() : null;
        jSch.addIdentity(privateKeyFileName, passPhrase);
    }

    private static Session createSession(DocumentDbConnectionProperties connectionProperties, JSch jSch) throws SQLException {
        String sshUsername = connectionProperties.getSshUser();
        Pair<String, Integer> sshHostAndPort = DocumentDbSshTunnelService.getHostAndPort(connectionProperties.getSshHostname(), 22);
        DocumentDbSshTunnelService.setKnownHostsFile(connectionProperties, jSch);
        try {
            return jSch.getSession(sshUsername, (String)sshHostAndPort.getLeft(), ((Integer)sshHostAndPort.getRight()).intValue());
        }
        catch (JSchException e) {
            throw DocumentDbSshTunnelService.logException(e);
        }
    }

    private static void setSecurityConfig(DocumentDbConnectionProperties connectionProperties, JSch jSch, Session session) {
        if (!connectionProperties.getSshStrictHostKeyChecking()) {
            session.setConfig(STRICT_HOST_KEY_CHECKING, NO);
            return;
        }
        DocumentDbSshTunnelService.setHostKeyType(connectionProperties, jSch, session);
    }

    private static void setHostKeyType(DocumentDbConnectionProperties connectionProperties, JSch jSch, Session session) {
        String hostKeyType;
        HostKeyRepository keyRepository = jSch.getHostKeyRepository();
        HostKey[] hostKeys = keyRepository.getHostKey();
        Pair<String, Integer> sshHostAndPort = DocumentDbSshTunnelService.getHostAndPort(connectionProperties.getSshHostname(), 22);
        HostKey hostKey = Arrays.stream(hostKeys).filter(hk -> hk.getHost().equals(sshHostAndPort.getLeft())).findFirst().orElse(null);
        String string = hostKeyType = hostKey != null ? hostKey.getType() : null;
        if (hostKeyType != null) {
            session.setConfig(SERVER_HOST_KEY, session.getConfig(SERVER_HOST_KEY) + "," + hostKeyType);
        }
        session.setConfig(HASH_KNOWN_HOSTS, YES);
    }

    private static void setKnownHostsFile(DocumentDbConnectionProperties connectionProperties, JSch jSch) throws SQLException {
        if (!connectionProperties.getSshStrictHostKeyChecking()) {
            return;
        }
        String knowHostsFilename = DocumentDbSshTunnelServer.getSshKnownHostsFilename(connectionProperties);
        try {
            jSch.setKnownHosts(knowHostsFilename);
        }
        catch (JSchException e) {
            throw DocumentDbSshTunnelService.logException(e);
        }
    }

    private Map.Entry<DocumentDbMultiThreadFileChannel, FileLock> acquireServerLock() throws IOException, InterruptedException {
        FileLock serverLock;
        Path serverLockPath = DocumentDbSshTunnelLock.getServerLockPath(this.sshPropertiesHashString);
        Path parentPath = serverLockPath.getParent();
        assert (parentPath != null);
        Files.createDirectories(parentPath, new FileAttribute[0]);
        DocumentDbMultiThreadFileChannel serverChannel = DocumentDbMultiThreadFileChannel.open(serverLockPath, StandardOpenOption.CREATE, StandardOpenOption.WRITE);
        int pollTimeMS = 100;
        while ((serverLock = serverChannel.tryLock()) == null) {
            TimeUnit.MILLISECONDS.sleep(100L);
        }
        return new AbstractMap.SimpleImmutableEntry<DocumentDbMultiThreadFileChannel, FileLock>(serverChannel, serverLock);
    }

    private SshPortForwardingSession performSshTunnelSessionStartup() throws Exception {
        SshPortForwardingSession session;
        if (!this.connectionProperties.enableSshTunnel()) {
            throw new UnsupportedOperationException("Unable to create SSH tunnel session. Invalid properties provided.");
        }
        Path startupLockPath = DocumentDbSshTunnelLock.getStartupLockPath(this.sshPropertiesHashString);
        Path parentPath = startupLockPath.getParent();
        assert (parentPath != null);
        Files.createDirectories(parentPath, new FileAttribute[0]);
        try (DocumentDbMultiThreadFileChannel startupChannel = DocumentDbMultiThreadFileChannel.open(startupLockPath, StandardOpenOption.CREATE, StandardOpenOption.WRITE);
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(Channels.newOutputStream(startupChannel.getFileChannel()), StandardCharsets.UTF_8));
             FileLock ignored = startupChannel.lock();){
            try {
                session = DocumentDbSshTunnelService.createSshTunnel(this.connectionProperties);
            }
            catch (Exception e) {
                DocumentDbSshTunnelService.logException(e);
                writer.write(e.toString());
                throw e;
            }
            this.writeSssTunnelPort(session);
        }
        return session;
    }

    private void writeSssTunnelPort(SshPortForwardingSession session) throws IOException {
        Path portLockPath = DocumentDbSshTunnelLock.getPortLockPath(this.sshPropertiesHashString);
        try (FileOutputStream outputStream = new FileOutputStream(portLockPath.toFile());
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)outputStream, StandardCharsets.UTF_8));
             FileLock ignored = outputStream.getChannel().lock();){
            writer.write(String.format("%d%n", session.getLocalPort()));
        }
    }

    public List<Exception> getExceptions() {
        return Collections.unmodifiableList(new ArrayList<Exception>(this.exceptions));
    }

    private static <T extends Exception> SQLException logException(T e) {
        LOGGER.error(e.getMessage(), e);
        if (e instanceof SQLException) {
            return (SQLException)e;
        }
        return new SQLException(e.getMessage(), e);
    }

    static class SshPortForwardingSession {
        private final Session session;
        private final int localPort;

        public Session getSession() {
            return this.session;
        }

        public int getLocalPort() {
            return this.localPort;
        }

        public SshPortForwardingSession(Session session, int localPort) {
            this.session = session;
            this.localPort = localPort;
        }
    }

    private static class ClientWatcher
    implements Runnable {
        private final ConcurrentLinkedDeque<Exception> exceptions = new ConcurrentLinkedDeque();
        private final FileLock serverLock;
        private final String sshPropertiesHashString;

        public ClientWatcher(FileLock serverLock, String sshPropertiesHashString) {
            this.serverLock = serverLock;
            this.sshPropertiesHashString = sshPropertiesHashString;
        }

        @Override
        public void run() {
            ThreadState state = ThreadState.RUNNING;
            try {
                AtomicInteger clientCount = new AtomicInteger();
                do {
                    clientCount.set(0);
                    DocumentDbSshTunnelLock.runInGlobalLock(this.sshPropertiesHashString, () -> ClientWatcher.checkAndHandleClientLocks(clientCount, this.sshPropertiesHashString, this.serverLock));
                    if (clientCount.get() > 0) {
                        TimeUnit.MILLISECONDS.sleep(500L);
                        continue;
                    }
                    state = ThreadState.EXITING;
                } while (state == ThreadState.RUNNING);
            }
            catch (Exception e) {
                this.exceptions.add(DocumentDbSshTunnelService.logException(e));
            }
            finally {
                try {
                    Exception localException = DocumentDbSshTunnelLock.runInGlobalLock(this.sshPropertiesHashString, () -> DocumentDbSshTunnelService.closeServerLock(this.serverLock));
                    if (localException != null) {
                        this.exceptions.add(localException);
                    }
                }
                catch (Exception e) {
                    this.exceptions.add(DocumentDbSshTunnelService.logException(e));
                }
            }
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        private static Exception checkAndHandleClientLocks(AtomicInteger clientCount, String sshPropertiesHashString, FileLock serverLock) {
            Exception result = null;
            Path clientsFolderPath = DocumentDbSshTunnelLock.getClientsFolderPath(sshPropertiesHashString);
            Files.createDirectories(clientsFolderPath, new FileAttribute[0]);
            try (Stream<Path> files = Files.list(clientsFolderPath);){
                for (Path filePath : files.collect(Collectors.toList())) {
                    Exception exception = ClientWatcher.checkClientLock(clientCount, filePath);
                    if (exception == null) continue;
                    Exception exception2 = exception;
                    return exception2;
                }
            }
            if (clientCount.get() != 0) return result;
            return DocumentDbSshTunnelService.closeServerLock(serverLock);
        }

        private static Exception checkClientLock(AtomicInteger clientCount, Path filePath) {
            SQLException result = null;
            try (DocumentDbMultiThreadFileChannel fileChannel = DocumentDbMultiThreadFileChannel.open(filePath, StandardOpenOption.WRITE);){
                FileLock fileLock = fileChannel.tryLock();
                if (fileLock == null) {
                    clientCount.getAndIncrement();
                } else {
                    fileLock.close();
                    Files.deleteIfExists(filePath);
                }
            }
            catch (Exception e) {
                result = DocumentDbSshTunnelService.logException(e);
            }
            return result;
        }

        public @NonNull ConcurrentLinkedDeque<Exception> getExceptions() {
            return this.exceptions;
        }

        private static enum ThreadState {
            UNKNOWN,
            RUNNING,
            INTERRUPTED,
            EXITING;

        }
    }
}

