/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.xray.model.BatchGetTracesRequest;
import software.amazon.awssdk.services.xray.model.BatchGetTracesResponse;
import software.amazon.awssdk.services.xray.model.CreateGroupRequest;
import software.amazon.awssdk.services.xray.model.CreateGroupResponse;
import software.amazon.awssdk.services.xray.model.CreateSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.CreateSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.DeleteGroupRequest;
import software.amazon.awssdk.services.xray.model.DeleteGroupResponse;
import software.amazon.awssdk.services.xray.model.DeleteSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.DeleteSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.GetEncryptionConfigRequest;
import software.amazon.awssdk.services.xray.model.GetEncryptionConfigResponse;
import software.amazon.awssdk.services.xray.model.GetGroupRequest;
import software.amazon.awssdk.services.xray.model.GetGroupResponse;
import software.amazon.awssdk.services.xray.model.GetGroupsRequest;
import software.amazon.awssdk.services.xray.model.GetGroupsResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingRulesRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingRulesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingTargetsRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingTargetsResponse;
import software.amazon.awssdk.services.xray.model.GetServiceGraphRequest;
import software.amazon.awssdk.services.xray.model.GetServiceGraphResponse;
import software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsRequest;
import software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsResponse;
import software.amazon.awssdk.services.xray.model.GetTraceGraphRequest;
import software.amazon.awssdk.services.xray.model.GetTraceGraphResponse;
import software.amazon.awssdk.services.xray.model.GetTraceSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetTraceSummariesResponse;
import software.amazon.awssdk.services.xray.model.InvalidRequestException;
import software.amazon.awssdk.services.xray.model.PutEncryptionConfigRequest;
import software.amazon.awssdk.services.xray.model.PutEncryptionConfigResponse;
import software.amazon.awssdk.services.xray.model.PutTelemetryRecordsRequest;
import software.amazon.awssdk.services.xray.model.PutTelemetryRecordsResponse;
import software.amazon.awssdk.services.xray.model.PutTraceSegmentsRequest;
import software.amazon.awssdk.services.xray.model.PutTraceSegmentsResponse;
import software.amazon.awssdk.services.xray.model.RuleLimitExceededException;
import software.amazon.awssdk.services.xray.model.ThrottledException;
import software.amazon.awssdk.services.xray.model.UpdateGroupRequest;
import software.amazon.awssdk.services.xray.model.UpdateGroupResponse;
import software.amazon.awssdk.services.xray.model.UpdateSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.UpdateSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.XRayException;
import software.amazon.awssdk.services.xray.model.XRayRequest;
import software.amazon.awssdk.services.xray.paginators.BatchGetTracesPublisher;
import software.amazon.awssdk.services.xray.paginators.GetGroupsPublisher;
import software.amazon.awssdk.services.xray.paginators.GetSamplingRulesPublisher;
import software.amazon.awssdk.services.xray.paginators.GetSamplingStatisticSummariesPublisher;
import software.amazon.awssdk.services.xray.paginators.GetServiceGraphPublisher;
import software.amazon.awssdk.services.xray.paginators.GetTimeSeriesServiceStatisticsPublisher;
import software.amazon.awssdk.services.xray.paginators.GetTraceGraphPublisher;
import software.amazon.awssdk.services.xray.paginators.GetTraceSummariesPublisher;
import software.amazon.awssdk.services.xray.transform.BatchGetTracesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.CreateSamplingRuleRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteSamplingRuleRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetGroupsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingRulesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingStatisticSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingTargetsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetServiceGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTimeSeriesServiceStatisticsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTraceGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTraceSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutTelemetryRecordsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutTraceSegmentsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UpdateSamplingRuleRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link XRayAsyncClient}.
 *
 * @see XRayAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultXRayAsyncClient implements XRayAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultXRayAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultXRayAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retrieves a list of traces specified by ID. Each trace is a collection of segment documents that originates from
     * a single request. Use <code>GetTraceSummaries</code> to get a list of trace IDs.
     * </p>
     *
     * @param batchGetTracesRequest
     * @return A Java Future containing the result of the BatchGetTraces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.BatchGetTraces
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/BatchGetTraces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetTracesResponse> batchGetTraces(BatchGetTracesRequest batchGetTracesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetTracesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetTracesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetTracesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetTracesRequest, BatchGetTracesResponse>()
                            .withOperationName("BatchGetTraces")
                            .withMarshaller(new BatchGetTracesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchGetTracesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of traces specified by ID. Each trace is a collection of segment documents that originates from
     * a single request. Use <code>GetTraceSummaries</code> to get a list of trace IDs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #batchGetTraces(software.amazon.awssdk.services.xray.model.BatchGetTracesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.BatchGetTracesPublisher publisher = client.batchGetTracesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.BatchGetTracesPublisher publisher = client.batchGetTracesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.xray.model.BatchGetTracesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.xray.model.BatchGetTracesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #batchGetTraces(software.amazon.awssdk.services.xray.model.BatchGetTracesRequest)} operation.</b>
     * </p>
     *
     * @param batchGetTracesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.BatchGetTraces
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/BatchGetTraces" target="_top">AWS API
     *      Documentation</a>
     */
    public BatchGetTracesPublisher batchGetTracesPaginator(BatchGetTracesRequest batchGetTracesRequest) {
        return new BatchGetTracesPublisher(this, applyPaginatorUserAgent(batchGetTracesRequest));
    }

    /**
     * <p>
     * Creates a group resource with a name and a filter expression.
     * </p>
     *
     * @param createGroupRequest
     * @return A Java Future containing the result of the CreateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.CreateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/CreateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGroupResponse> createGroup(CreateGroupRequest createGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                            .withOperationName("CreateGroup").withMarshaller(new CreateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a rule to control sampling behavior for instrumented applications. Services retrieve rules with
     * <a>GetSamplingRules</a>, and evaluate each rule in ascending order of <i>priority</i> for each request. If a rule
     * matches, the service records a trace, borrowing it from the reservoir size. After 10 seconds, the service reports
     * back to X-Ray with <a>GetSamplingTargets</a> to get updated versions of each in-use rule. The updated rule
     * contains a trace quota that the service can use instead of borrowing from the reservoir.
     * </p>
     *
     * @param createSamplingRuleRequest
     * @return A Java Future containing the result of the CreateSamplingRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>RuleLimitExceededException You have reached the maximum number of sampling rules.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.CreateSamplingRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/CreateSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSamplingRuleResponse> createSamplingRule(CreateSamplingRuleRequest createSamplingRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSamplingRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSamplingRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSamplingRuleRequest, CreateSamplingRuleResponse>()
                            .withOperationName("CreateSamplingRule")
                            .withMarshaller(new CreateSamplingRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createSamplingRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a group resource.
     * </p>
     *
     * @param deleteGroupRequest
     * @return A Java Future containing the result of the DeleteGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.DeleteGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGroupResponse> deleteGroup(DeleteGroupRequest deleteGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                            .withOperationName("DeleteGroup").withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a sampling rule.
     * </p>
     *
     * @param deleteSamplingRuleRequest
     * @return A Java Future containing the result of the DeleteSamplingRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.DeleteSamplingRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSamplingRuleResponse> deleteSamplingRule(DeleteSamplingRuleRequest deleteSamplingRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSamplingRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSamplingRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSamplingRuleRequest, DeleteSamplingRuleResponse>()
                            .withOperationName("DeleteSamplingRule")
                            .withMarshaller(new DeleteSamplingRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteSamplingRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current encryption configuration for X-Ray data.
     * </p>
     *
     * @param getEncryptionConfigRequest
     * @return A Java Future containing the result of the GetEncryptionConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetEncryptionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetEncryptionConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetEncryptionConfigResponse> getEncryptionConfig(
            GetEncryptionConfigRequest getEncryptionConfigRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEncryptionConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEncryptionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEncryptionConfigRequest, GetEncryptionConfigResponse>()
                            .withOperationName("GetEncryptionConfig")
                            .withMarshaller(new GetEncryptionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getEncryptionConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves group resource details.
     * </p>
     *
     * @param getGroupRequest
     * @return A Java Future containing the result of the GetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGroupResponse> getGroup(GetGroupRequest getGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGroupRequest, GetGroupResponse>().withOperationName("GetGroup")
                            .withMarshaller(new GetGroupRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all active group details.
     * </p>
     *
     * @param getGroupsRequest
     * @return A Java Future containing the result of the GetGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGroupsResponse> getGroups(GetGroupsRequest getGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGroupsRequest, GetGroupsResponse>().withOperationName("GetGroups")
                            .withMarshaller(new GetGroupsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all active group details.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getGroups(software.amazon.awssdk.services.xray.model.GetGroupsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetGroupsPublisher publisher = client.getGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetGroupsPublisher publisher = client.getGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.xray.model.GetGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.xray.model.GetGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getGroups(software.amazon.awssdk.services.xray.model.GetGroupsRequest)} operation.</b>
     * </p>
     *
     * @param getGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroups" target="_top">AWS API
     *      Documentation</a>
     */
    public GetGroupsPublisher getGroupsPaginator(GetGroupsRequest getGroupsRequest) {
        return new GetGroupsPublisher(this, applyPaginatorUserAgent(getGroupsRequest));
    }

    /**
     * <p>
     * Retrieves all sampling rules.
     * </p>
     *
     * @param getSamplingRulesRequest
     * @return A Java Future containing the result of the GetSamplingRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetSamplingRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSamplingRulesResponse> getSamplingRules(GetSamplingRulesRequest getSamplingRulesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSamplingRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSamplingRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSamplingRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSamplingRulesRequest, GetSamplingRulesResponse>()
                            .withOperationName("GetSamplingRules")
                            .withMarshaller(new GetSamplingRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSamplingRulesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all sampling rules.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getSamplingRules(software.amazon.awssdk.services.xray.model.GetSamplingRulesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetSamplingRulesPublisher publisher = client.getSamplingRulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetSamplingRulesPublisher publisher = client.getSamplingRulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.xray.model.GetSamplingRulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.xray.model.GetSamplingRulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getSamplingRules(software.amazon.awssdk.services.xray.model.GetSamplingRulesRequest)} operation.</b>
     * </p>
     *
     * @param getSamplingRulesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetSamplingRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingRules" target="_top">AWS API
     *      Documentation</a>
     */
    public GetSamplingRulesPublisher getSamplingRulesPaginator(GetSamplingRulesRequest getSamplingRulesRequest) {
        return new GetSamplingRulesPublisher(this, applyPaginatorUserAgent(getSamplingRulesRequest));
    }

    /**
     * <p>
     * Retrieves information about recent sampling results for all sampling rules.
     * </p>
     *
     * @param getSamplingStatisticSummariesRequest
     * @return A Java Future containing the result of the GetSamplingStatisticSummaries operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetSamplingStatisticSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingStatisticSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSamplingStatisticSummariesResponse> getSamplingStatisticSummaries(
            GetSamplingStatisticSummariesRequest getSamplingStatisticSummariesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSamplingStatisticSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSamplingStatisticSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSamplingStatisticSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSamplingStatisticSummariesRequest, GetSamplingStatisticSummariesResponse>()
                            .withOperationName("GetSamplingStatisticSummaries")
                            .withMarshaller(new GetSamplingStatisticSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSamplingStatisticSummariesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about recent sampling results for all sampling rules.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getSamplingStatisticSummaries(software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetSamplingStatisticSummariesPublisher publisher = client.getSamplingStatisticSummariesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetSamplingStatisticSummariesPublisher publisher = client.getSamplingStatisticSummariesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getSamplingStatisticSummaries(software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesRequest)}
     * operation.</b>
     * </p>
     *
     * @param getSamplingStatisticSummariesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetSamplingStatisticSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingStatisticSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    public GetSamplingStatisticSummariesPublisher getSamplingStatisticSummariesPaginator(
            GetSamplingStatisticSummariesRequest getSamplingStatisticSummariesRequest) {
        return new GetSamplingStatisticSummariesPublisher(this, applyPaginatorUserAgent(getSamplingStatisticSummariesRequest));
    }

    /**
     * <p>
     * Requests a sampling quota for rules that the service is using to sample requests.
     * </p>
     *
     * @param getSamplingTargetsRequest
     * @return A Java Future containing the result of the GetSamplingTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetSamplingTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSamplingTargetsResponse> getSamplingTargets(GetSamplingTargetsRequest getSamplingTargetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSamplingTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSamplingTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSamplingTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSamplingTargetsRequest, GetSamplingTargetsResponse>()
                            .withOperationName("GetSamplingTargets")
                            .withMarshaller(new GetSamplingTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSamplingTargetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a document that describes services that process incoming requests, and downstream services that they
     * call as a result. Root services process incoming requests and make calls to downstream services. Root services
     * are applications that use the <a href="https://docs.aws.amazon.com/xray/index.html">AWS X-Ray SDK</a>. Downstream
     * services can be other applications, AWS resources, HTTP web APIs, or SQL databases.
     * </p>
     *
     * @param getServiceGraphRequest
     * @return A Java Future containing the result of the GetServiceGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetServiceGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetServiceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceGraphResponse> getServiceGraph(GetServiceGraphRequest getServiceGraphRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceGraphResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceGraphRequest, GetServiceGraphResponse>()
                            .withOperationName("GetServiceGraph")
                            .withMarshaller(new GetServiceGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getServiceGraphRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a document that describes services that process incoming requests, and downstream services that they
     * call as a result. Root services process incoming requests and make calls to downstream services. Root services
     * are applications that use the <a href="https://docs.aws.amazon.com/xray/index.html">AWS X-Ray SDK</a>. Downstream
     * services can be other applications, AWS resources, HTTP web APIs, or SQL databases.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getServiceGraph(software.amazon.awssdk.services.xray.model.GetServiceGraphRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetServiceGraphPublisher publisher = client.getServiceGraphPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetServiceGraphPublisher publisher = client.getServiceGraphPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.xray.model.GetServiceGraphResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.xray.model.GetServiceGraphResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getServiceGraph(software.amazon.awssdk.services.xray.model.GetServiceGraphRequest)} operation.</b>
     * </p>
     *
     * @param getServiceGraphRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetServiceGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetServiceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    public GetServiceGraphPublisher getServiceGraphPaginator(GetServiceGraphRequest getServiceGraphRequest) {
        return new GetServiceGraphPublisher(this, applyPaginatorUserAgent(getServiceGraphRequest));
    }

    /**
     * <p>
     * Get an aggregation of service statistics defined by a specific time range.
     * </p>
     *
     * @param getTimeSeriesServiceStatisticsRequest
     * @return A Java Future containing the result of the GetTimeSeriesServiceStatistics operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetTimeSeriesServiceStatistics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTimeSeriesServiceStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTimeSeriesServiceStatisticsResponse> getTimeSeriesServiceStatistics(
            GetTimeSeriesServiceStatisticsRequest getTimeSeriesServiceStatisticsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTimeSeriesServiceStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTimeSeriesServiceStatisticsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTimeSeriesServiceStatisticsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTimeSeriesServiceStatisticsRequest, GetTimeSeriesServiceStatisticsResponse>()
                            .withOperationName("GetTimeSeriesServiceStatistics")
                            .withMarshaller(new GetTimeSeriesServiceStatisticsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getTimeSeriesServiceStatisticsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get an aggregation of service statistics defined by a specific time range.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getTimeSeriesServiceStatistics(software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTimeSeriesServiceStatisticsPublisher publisher = client.getTimeSeriesServiceStatisticsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTimeSeriesServiceStatisticsPublisher publisher = client.getTimeSeriesServiceStatisticsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getTimeSeriesServiceStatistics(software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getTimeSeriesServiceStatisticsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetTimeSeriesServiceStatistics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTimeSeriesServiceStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    public GetTimeSeriesServiceStatisticsPublisher getTimeSeriesServiceStatisticsPaginator(
            GetTimeSeriesServiceStatisticsRequest getTimeSeriesServiceStatisticsRequest) {
        return new GetTimeSeriesServiceStatisticsPublisher(this, applyPaginatorUserAgent(getTimeSeriesServiceStatisticsRequest));
    }

    /**
     * <p>
     * Retrieves a service graph for one or more specific trace IDs.
     * </p>
     *
     * @param getTraceGraphRequest
     * @return A Java Future containing the result of the GetTraceGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetTraceGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTraceGraphResponse> getTraceGraph(GetTraceGraphRequest getTraceGraphRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTraceGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTraceGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTraceGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTraceGraphRequest, GetTraceGraphResponse>()
                            .withOperationName("GetTraceGraph")
                            .withMarshaller(new GetTraceGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getTraceGraphRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a service graph for one or more specific trace IDs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getTraceGraph(software.amazon.awssdk.services.xray.model.GetTraceGraphRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTraceGraphPublisher publisher = client.getTraceGraphPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTraceGraphPublisher publisher = client.getTraceGraphPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.xray.model.GetTraceGraphResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.xray.model.GetTraceGraphResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getTraceGraph(software.amazon.awssdk.services.xray.model.GetTraceGraphRequest)} operation.</b>
     * </p>
     *
     * @param getTraceGraphRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetTraceGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    public GetTraceGraphPublisher getTraceGraphPaginator(GetTraceGraphRequest getTraceGraphRequest) {
        return new GetTraceGraphPublisher(this, applyPaginatorUserAgent(getTraceGraphRequest));
    }

    /**
     * <p>
     * Retrieves IDs and annotations for traces available for a specified time frame using an optional filter. To get
     * the full traces, pass the trace IDs to <code>BatchGetTraces</code>.
     * </p>
     * <p>
     * A filter expression can target traced requests that hit specific service nodes or edges, have errors, or come
     * from a known user. For example, the following filter expression targets traces that pass through
     * <code>api.example.com</code>:
     * </p>
     * <p>
     * <code>service("api.example.com")</code>
     * </p>
     * <p>
     * This filter expression finds traces that have an annotation named <code>account</code> with the value
     * <code>12345</code>:
     * </p>
     * <p>
     * <code>annotation.account = "12345"</code>
     * </p>
     * <p>
     * For a full list of indexed fields and keywords that you can use in filter expressions, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-console-filters.html">Using Filter Expressions</a> in
     * the <i>AWS X-Ray Developer Guide</i>.
     * </p>
     *
     * @param getTraceSummariesRequest
     * @return A Java Future containing the result of the GetTraceSummaries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetTraceSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceSummaries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTraceSummariesResponse> getTraceSummaries(GetTraceSummariesRequest getTraceSummariesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTraceSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTraceSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTraceSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTraceSummariesRequest, GetTraceSummariesResponse>()
                            .withOperationName("GetTraceSummaries")
                            .withMarshaller(new GetTraceSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getTraceSummariesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves IDs and annotations for traces available for a specified time frame using an optional filter. To get
     * the full traces, pass the trace IDs to <code>BatchGetTraces</code>.
     * </p>
     * <p>
     * A filter expression can target traced requests that hit specific service nodes or edges, have errors, or come
     * from a known user. For example, the following filter expression targets traces that pass through
     * <code>api.example.com</code>:
     * </p>
     * <p>
     * <code>service("api.example.com")</code>
     * </p>
     * <p>
     * This filter expression finds traces that have an annotation named <code>account</code> with the value
     * <code>12345</code>:
     * </p>
     * <p>
     * <code>annotation.account = "12345"</code>
     * </p>
     * <p>
     * For a full list of indexed fields and keywords that you can use in filter expressions, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-console-filters.html">Using Filter Expressions</a> in
     * the <i>AWS X-Ray Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getTraceSummaries(software.amazon.awssdk.services.xray.model.GetTraceSummariesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTraceSummariesPublisher publisher = client.getTraceSummariesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTraceSummariesPublisher publisher = client.getTraceSummariesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.xray.model.GetTraceSummariesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.xray.model.GetTraceSummariesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getTraceSummaries(software.amazon.awssdk.services.xray.model.GetTraceSummariesRequest)} operation.</b>
     * </p>
     *
     * @param getTraceSummariesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetTraceSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceSummaries" target="_top">AWS API
     *      Documentation</a>
     */
    public GetTraceSummariesPublisher getTraceSummariesPaginator(GetTraceSummariesRequest getTraceSummariesRequest) {
        return new GetTraceSummariesPublisher(this, applyPaginatorUserAgent(getTraceSummariesRequest));
    }

    /**
     * <p>
     * Updates the encryption configuration for X-Ray data.
     * </p>
     *
     * @param putEncryptionConfigRequest
     * @return A Java Future containing the result of the PutEncryptionConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.PutEncryptionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutEncryptionConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutEncryptionConfigResponse> putEncryptionConfig(
            PutEncryptionConfigRequest putEncryptionConfigRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutEncryptionConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEncryptionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEncryptionConfigRequest, PutEncryptionConfigResponse>()
                            .withOperationName("PutEncryptionConfig")
                            .withMarshaller(new PutEncryptionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putEncryptionConfigRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by the AWS X-Ray daemon to upload telemetry.
     * </p>
     *
     * @param putTelemetryRecordsRequest
     * @return A Java Future containing the result of the PutTelemetryRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.PutTelemetryRecords
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutTelemetryRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutTelemetryRecordsResponse> putTelemetryRecords(
            PutTelemetryRecordsRequest putTelemetryRecordsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTelemetryRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutTelemetryRecordsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTelemetryRecordsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTelemetryRecordsRequest, PutTelemetryRecordsResponse>()
                            .withOperationName("PutTelemetryRecords")
                            .withMarshaller(new PutTelemetryRecordsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putTelemetryRecordsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads segment documents to AWS X-Ray. The <a href="https://docs.aws.amazon.com/xray/index.html">X-Ray SDK</a>
     * generates segment documents and sends them to the X-Ray daemon, which uploads them in batches. A segment document
     * can be a completed segment, an in-progress segment, or an array of subsegments.
     * </p>
     * <p>
     * Segments must include the following fields. For the full segment document schema, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-api-segmentdocuments.html">AWS X-Ray Segment
     * Documents</a> in the <i>AWS X-Ray Developer Guide</i>.
     * </p>
     * <p class="title">
     * <b>Required Segment Document Fields</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>name</code> - The name of the service that handled the request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>id</code> - A 64-bit identifier for the segment, unique among segments in the same trace, in 16 hexadecimal
     * digits.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>trace_id</code> - A unique identifier that connects all segments and subsegments originating from a single
     * client request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>start_time</code> - Time the segment or subsegment was created, in floating point seconds in epoch time,
     * accurate to milliseconds. For example, <code>1480615200.010</code> or <code>1.480615200010E9</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>end_time</code> - Time the segment or subsegment was closed. For example, <code>1480615200.090</code> or
     * <code>1.480615200090E9</code>. Specify either an <code>end_time</code> or <code>in_progress</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>in_progress</code> - Set to <code>true</code> instead of specifying an <code>end_time</code> to record that
     * a segment has been started, but is not complete. Send an in progress segment when your application receives a
     * request that will take a long time to serve, to trace the fact that the request was received. When the response
     * is sent, send the complete segment to overwrite the in-progress segment.
     * </p>
     * </li>
     * </ul>
     * <p>
     * A <code>trace_id</code> consists of three numbers separated by hyphens. For example,
     * 1-58406520-a006649127e371903a2de979. This includes:
     * </p>
     * <p class="title">
     * <b>Trace ID Format</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The version number, i.e. <code>1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The time of the original request, in Unix epoch time, in 8 hexadecimal digits. For example, 10:00AM December 2nd,
     * 2016 PST in epoch time is <code>1480615200</code> seconds, or <code>58406520</code> in hexadecimal.
     * </p>
     * </li>
     * <li>
     * <p>
     * A 96-bit identifier for the trace, globally unique, in 24 hexadecimal digits.
     * </p>
     * </li>
     * </ul>
     *
     * @param putTraceSegmentsRequest
     * @return A Java Future containing the result of the PutTraceSegments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.PutTraceSegments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutTraceSegments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutTraceSegmentsResponse> putTraceSegments(PutTraceSegmentsRequest putTraceSegmentsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTraceSegmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutTraceSegmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTraceSegmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTraceSegmentsRequest, PutTraceSegmentsResponse>()
                            .withOperationName("PutTraceSegments")
                            .withMarshaller(new PutTraceSegmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putTraceSegmentsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a group resource.
     * </p>
     *
     * @param updateGroupRequest
     * @return A Java Future containing the result of the UpdateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.UpdateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UpdateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGroupResponse> updateGroup(UpdateGroupRequest updateGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                            .withOperationName("UpdateGroup").withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies a sampling rule's configuration.
     * </p>
     *
     * @param updateSamplingRuleRequest
     * @return A Java Future containing the result of the UpdateSamplingRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.UpdateSamplingRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UpdateSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSamplingRuleResponse> updateSamplingRule(UpdateSamplingRuleRequest updateSamplingRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSamplingRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSamplingRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSamplingRuleRequest, UpdateSamplingRuleResponse>()
                            .withOperationName("UpdateSamplingRule")
                            .withMarshaller(new UpdateSamplingRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateSamplingRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(XRayException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledException")
                                .exceptionBuilderSupplier(ThrottledException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RuleLimitExceededException")
                                .exceptionBuilderSupplier(RuleLimitExceededException::builder).build());
    }

    private <T extends XRayRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
