/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.xray.model.BatchGetTracesRequest;
import software.amazon.awssdk.services.xray.model.BatchGetTracesResponse;
import software.amazon.awssdk.services.xray.model.CreateGroupRequest;
import software.amazon.awssdk.services.xray.model.CreateGroupResponse;
import software.amazon.awssdk.services.xray.model.CreateSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.CreateSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.DeleteGroupRequest;
import software.amazon.awssdk.services.xray.model.DeleteGroupResponse;
import software.amazon.awssdk.services.xray.model.DeleteSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.DeleteSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.GetEncryptionConfigRequest;
import software.amazon.awssdk.services.xray.model.GetEncryptionConfigResponse;
import software.amazon.awssdk.services.xray.model.GetGroupRequest;
import software.amazon.awssdk.services.xray.model.GetGroupResponse;
import software.amazon.awssdk.services.xray.model.GetGroupsRequest;
import software.amazon.awssdk.services.xray.model.GetGroupsResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingRulesRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingRulesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingTargetsRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingTargetsResponse;
import software.amazon.awssdk.services.xray.model.GetServiceGraphRequest;
import software.amazon.awssdk.services.xray.model.GetServiceGraphResponse;
import software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsRequest;
import software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsResponse;
import software.amazon.awssdk.services.xray.model.GetTraceGraphRequest;
import software.amazon.awssdk.services.xray.model.GetTraceGraphResponse;
import software.amazon.awssdk.services.xray.model.GetTraceSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetTraceSummariesResponse;
import software.amazon.awssdk.services.xray.model.InvalidRequestException;
import software.amazon.awssdk.services.xray.model.PutEncryptionConfigRequest;
import software.amazon.awssdk.services.xray.model.PutEncryptionConfigResponse;
import software.amazon.awssdk.services.xray.model.PutTelemetryRecordsRequest;
import software.amazon.awssdk.services.xray.model.PutTelemetryRecordsResponse;
import software.amazon.awssdk.services.xray.model.PutTraceSegmentsRequest;
import software.amazon.awssdk.services.xray.model.PutTraceSegmentsResponse;
import software.amazon.awssdk.services.xray.model.RuleLimitExceededException;
import software.amazon.awssdk.services.xray.model.ThrottledException;
import software.amazon.awssdk.services.xray.model.UpdateGroupRequest;
import software.amazon.awssdk.services.xray.model.UpdateGroupResponse;
import software.amazon.awssdk.services.xray.model.UpdateSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.UpdateSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.XRayException;
import software.amazon.awssdk.services.xray.model.XRayRequest;
import software.amazon.awssdk.services.xray.paginators.BatchGetTracesIterable;
import software.amazon.awssdk.services.xray.paginators.GetGroupsIterable;
import software.amazon.awssdk.services.xray.paginators.GetSamplingRulesIterable;
import software.amazon.awssdk.services.xray.paginators.GetSamplingStatisticSummariesIterable;
import software.amazon.awssdk.services.xray.paginators.GetServiceGraphIterable;
import software.amazon.awssdk.services.xray.paginators.GetTimeSeriesServiceStatisticsIterable;
import software.amazon.awssdk.services.xray.paginators.GetTraceGraphIterable;
import software.amazon.awssdk.services.xray.paginators.GetTraceSummariesIterable;
import software.amazon.awssdk.services.xray.transform.BatchGetTracesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.CreateSamplingRuleRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteSamplingRuleRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetGroupsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingRulesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingStatisticSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingTargetsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetServiceGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTimeSeriesServiceStatisticsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTraceGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTraceSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutTelemetryRecordsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutTraceSegmentsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UpdateSamplingRuleRequestMarshaller;

/**
 * Internal implementation of {@link XRayClient}.
 *
 * @see XRayClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultXRayClient implements XRayClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultXRayClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retrieves a list of traces specified by ID. Each trace is a collection of segment documents that originates from
     * a single request. Use <code>GetTraceSummaries</code> to get a list of trace IDs.
     * </p>
     *
     * @param batchGetTracesRequest
     * @return Result of the BatchGetTraces operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.BatchGetTraces
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/BatchGetTraces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetTracesResponse batchGetTraces(BatchGetTracesRequest batchGetTracesRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetTracesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetTracesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchGetTracesRequest, BatchGetTracesResponse>()
                .withOperationName("BatchGetTraces").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchGetTracesRequest)
                .withMarshaller(new BatchGetTracesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a list of traces specified by ID. Each trace is a collection of segment documents that originates from
     * a single request. Use <code>GetTraceSummaries</code> to get a list of trace IDs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #batchGetTraces(software.amazon.awssdk.services.xray.model.BatchGetTracesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.BatchGetTracesIterable responses = client.batchGetTracesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.xray.paginators.BatchGetTracesIterable responses = client.batchGetTracesPaginator(request);
     *     for (software.amazon.awssdk.services.xray.model.BatchGetTracesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.BatchGetTracesIterable responses = client.batchGetTracesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #batchGetTraces(software.amazon.awssdk.services.xray.model.BatchGetTracesRequest)} operation.</b>
     * </p>
     *
     * @param batchGetTracesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.BatchGetTraces
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/BatchGetTraces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetTracesIterable batchGetTracesPaginator(BatchGetTracesRequest batchGetTracesRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        return new BatchGetTracesIterable(this, applyPaginatorUserAgent(batchGetTracesRequest));
    }

    /**
     * <p>
     * Creates a group resource with a name and a filter expression.
     * </p>
     *
     * @param createGroupRequest
     * @return Result of the CreateGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.CreateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/CreateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateGroupResponse createGroup(CreateGroupRequest createGroupRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                .withOperationName("CreateGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createGroupRequest)
                .withMarshaller(new CreateGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a rule to control sampling behavior for instrumented applications. Services retrieve rules with
     * <a>GetSamplingRules</a>, and evaluate each rule in ascending order of <i>priority</i> for each request. If a rule
     * matches, the service records a trace, borrowing it from the reservoir size. After 10 seconds, the service reports
     * back to X-Ray with <a>GetSamplingTargets</a> to get updated versions of each in-use rule. The updated rule
     * contains a trace quota that the service can use instead of borrowing from the reservoir.
     * </p>
     *
     * @param createSamplingRuleRequest
     * @return Result of the CreateSamplingRule operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws RuleLimitExceededException
     *         You have reached the maximum number of sampling rules.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.CreateSamplingRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/CreateSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSamplingRuleResponse createSamplingRule(CreateSamplingRuleRequest createSamplingRuleRequest)
            throws InvalidRequestException, ThrottledException, RuleLimitExceededException, AwsServiceException,
            SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSamplingRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateSamplingRuleRequest, CreateSamplingRuleResponse>()
                .withOperationName("CreateSamplingRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createSamplingRuleRequest)
                .withMarshaller(new CreateSamplingRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a group resource.
     * </p>
     *
     * @param deleteGroupRequest
     * @return Result of the DeleteGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.DeleteGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteGroupResponse deleteGroup(DeleteGroupRequest deleteGroupRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                .withOperationName("DeleteGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteGroupRequest)
                .withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a sampling rule.
     * </p>
     *
     * @param deleteSamplingRuleRequest
     * @return Result of the DeleteSamplingRule operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.DeleteSamplingRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSamplingRuleResponse deleteSamplingRule(DeleteSamplingRuleRequest deleteSamplingRuleRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSamplingRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteSamplingRuleRequest, DeleteSamplingRuleResponse>()
                .withOperationName("DeleteSamplingRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteSamplingRuleRequest)
                .withMarshaller(new DeleteSamplingRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the current encryption configuration for X-Ray data.
     * </p>
     *
     * @param getEncryptionConfigRequest
     * @return Result of the GetEncryptionConfig operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetEncryptionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetEncryptionConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetEncryptionConfigResponse getEncryptionConfig(GetEncryptionConfigRequest getEncryptionConfigRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEncryptionConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetEncryptionConfigRequest, GetEncryptionConfigResponse>()
                .withOperationName("GetEncryptionConfig").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getEncryptionConfigRequest)
                .withMarshaller(new GetEncryptionConfigRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves group resource details.
     * </p>
     *
     * @param getGroupRequest
     * @return Result of the GetGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGroupResponse getGroup(GetGroupRequest getGroupRequest) throws InvalidRequestException, ThrottledException,
            AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetGroupRequest, GetGroupResponse>().withOperationName("GetGroup")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getGroupRequest)
                .withMarshaller(new GetGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves all active group details.
     * </p>
     *
     * @param getGroupsRequest
     * @return Result of the GetGroups operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGroupsResponse getGroups(GetGroupsRequest getGroupsRequest) throws InvalidRequestException, ThrottledException,
            AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetGroupsRequest, GetGroupsResponse>()
                .withOperationName("GetGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getGroupsRequest)
                .withMarshaller(new GetGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves all active group details.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getGroups(software.amazon.awssdk.services.xray.model.GetGroupsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetGroupsIterable responses = client.getGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.xray.paginators.GetGroupsIterable responses = client.getGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.xray.model.GetGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetGroupsIterable responses = client.getGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getGroups(software.amazon.awssdk.services.xray.model.GetGroupsRequest)} operation.</b>
     * </p>
     *
     * @param getGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGroupsIterable getGroupsPaginator(GetGroupsRequest getGroupsRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        return new GetGroupsIterable(this, applyPaginatorUserAgent(getGroupsRequest));
    }

    /**
     * <p>
     * Retrieves all sampling rules.
     * </p>
     *
     * @param getSamplingRulesRequest
     * @return Result of the GetSamplingRules operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetSamplingRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSamplingRulesResponse getSamplingRules(GetSamplingRulesRequest getSamplingRulesRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSamplingRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSamplingRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetSamplingRulesRequest, GetSamplingRulesResponse>()
                .withOperationName("GetSamplingRules").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getSamplingRulesRequest)
                .withMarshaller(new GetSamplingRulesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves all sampling rules.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getSamplingRules(software.amazon.awssdk.services.xray.model.GetSamplingRulesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetSamplingRulesIterable responses = client.getSamplingRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.xray.paginators.GetSamplingRulesIterable responses = client
     *             .getSamplingRulesPaginator(request);
     *     for (software.amazon.awssdk.services.xray.model.GetSamplingRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetSamplingRulesIterable responses = client.getSamplingRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getSamplingRules(software.amazon.awssdk.services.xray.model.GetSamplingRulesRequest)} operation.</b>
     * </p>
     *
     * @param getSamplingRulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetSamplingRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSamplingRulesIterable getSamplingRulesPaginator(GetSamplingRulesRequest getSamplingRulesRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        return new GetSamplingRulesIterable(this, applyPaginatorUserAgent(getSamplingRulesRequest));
    }

    /**
     * <p>
     * Retrieves information about recent sampling results for all sampling rules.
     * </p>
     *
     * @param getSamplingStatisticSummariesRequest
     * @return Result of the GetSamplingStatisticSummaries operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetSamplingStatisticSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingStatisticSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSamplingStatisticSummariesResponse getSamplingStatisticSummaries(
            GetSamplingStatisticSummariesRequest getSamplingStatisticSummariesRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSamplingStatisticSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSamplingStatisticSummariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetSamplingStatisticSummariesRequest, GetSamplingStatisticSummariesResponse>()
                        .withOperationName("GetSamplingStatisticSummaries").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getSamplingStatisticSummariesRequest)
                        .withMarshaller(new GetSamplingStatisticSummariesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves information about recent sampling results for all sampling rules.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getSamplingStatisticSummaries(software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetSamplingStatisticSummariesIterable responses = client.getSamplingStatisticSummariesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.xray.paginators.GetSamplingStatisticSummariesIterable responses = client
     *             .getSamplingStatisticSummariesPaginator(request);
     *     for (software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetSamplingStatisticSummariesIterable responses = client.getSamplingStatisticSummariesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getSamplingStatisticSummaries(software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesRequest)}
     * operation.</b>
     * </p>
     *
     * @param getSamplingStatisticSummariesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetSamplingStatisticSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingStatisticSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSamplingStatisticSummariesIterable getSamplingStatisticSummariesPaginator(
            GetSamplingStatisticSummariesRequest getSamplingStatisticSummariesRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        return new GetSamplingStatisticSummariesIterable(this, applyPaginatorUserAgent(getSamplingStatisticSummariesRequest));
    }

    /**
     * <p>
     * Requests a sampling quota for rules that the service is using to sample requests.
     * </p>
     *
     * @param getSamplingTargetsRequest
     * @return Result of the GetSamplingTargets operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetSamplingTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSamplingTargetsResponse getSamplingTargets(GetSamplingTargetsRequest getSamplingTargetsRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSamplingTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSamplingTargetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetSamplingTargetsRequest, GetSamplingTargetsResponse>()
                .withOperationName("GetSamplingTargets").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getSamplingTargetsRequest)
                .withMarshaller(new GetSamplingTargetsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a document that describes services that process incoming requests, and downstream services that they
     * call as a result. Root services process incoming requests and make calls to downstream services. Root services
     * are applications that use the <a href="https://docs.aws.amazon.com/xray/index.html">AWS X-Ray SDK</a>. Downstream
     * services can be other applications, AWS resources, HTTP web APIs, or SQL databases.
     * </p>
     *
     * @param getServiceGraphRequest
     * @return Result of the GetServiceGraph operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetServiceGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetServiceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetServiceGraphResponse getServiceGraph(GetServiceGraphRequest getServiceGraphRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetServiceGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetServiceGraphRequest, GetServiceGraphResponse>()
                .withOperationName("GetServiceGraph").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getServiceGraphRequest)
                .withMarshaller(new GetServiceGraphRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a document that describes services that process incoming requests, and downstream services that they
     * call as a result. Root services process incoming requests and make calls to downstream services. Root services
     * are applications that use the <a href="https://docs.aws.amazon.com/xray/index.html">AWS X-Ray SDK</a>. Downstream
     * services can be other applications, AWS resources, HTTP web APIs, or SQL databases.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getServiceGraph(software.amazon.awssdk.services.xray.model.GetServiceGraphRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetServiceGraphIterable responses = client.getServiceGraphPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.xray.paginators.GetServiceGraphIterable responses = client.getServiceGraphPaginator(request);
     *     for (software.amazon.awssdk.services.xray.model.GetServiceGraphResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetServiceGraphIterable responses = client.getServiceGraphPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getServiceGraph(software.amazon.awssdk.services.xray.model.GetServiceGraphRequest)} operation.</b>
     * </p>
     *
     * @param getServiceGraphRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetServiceGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetServiceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetServiceGraphIterable getServiceGraphPaginator(GetServiceGraphRequest getServiceGraphRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        return new GetServiceGraphIterable(this, applyPaginatorUserAgent(getServiceGraphRequest));
    }

    /**
     * <p>
     * Get an aggregation of service statistics defined by a specific time range.
     * </p>
     *
     * @param getTimeSeriesServiceStatisticsRequest
     * @return Result of the GetTimeSeriesServiceStatistics operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetTimeSeriesServiceStatistics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTimeSeriesServiceStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTimeSeriesServiceStatisticsResponse getTimeSeriesServiceStatistics(
            GetTimeSeriesServiceStatisticsRequest getTimeSeriesServiceStatisticsRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTimeSeriesServiceStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetTimeSeriesServiceStatisticsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetTimeSeriesServiceStatisticsRequest, GetTimeSeriesServiceStatisticsResponse>()
                        .withOperationName("GetTimeSeriesServiceStatistics").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getTimeSeriesServiceStatisticsRequest)
                        .withMarshaller(new GetTimeSeriesServiceStatisticsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Get an aggregation of service statistics defined by a specific time range.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getTimeSeriesServiceStatistics(software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTimeSeriesServiceStatisticsIterable responses = client.getTimeSeriesServiceStatisticsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.xray.paginators.GetTimeSeriesServiceStatisticsIterable responses = client
     *             .getTimeSeriesServiceStatisticsPaginator(request);
     *     for (software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTimeSeriesServiceStatisticsIterable responses = client.getTimeSeriesServiceStatisticsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getTimeSeriesServiceStatistics(software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getTimeSeriesServiceStatisticsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetTimeSeriesServiceStatistics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTimeSeriesServiceStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTimeSeriesServiceStatisticsIterable getTimeSeriesServiceStatisticsPaginator(
            GetTimeSeriesServiceStatisticsRequest getTimeSeriesServiceStatisticsRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        return new GetTimeSeriesServiceStatisticsIterable(this, applyPaginatorUserAgent(getTimeSeriesServiceStatisticsRequest));
    }

    /**
     * <p>
     * Retrieves a service graph for one or more specific trace IDs.
     * </p>
     *
     * @param getTraceGraphRequest
     * @return Result of the GetTraceGraph operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetTraceGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTraceGraphResponse getTraceGraph(GetTraceGraphRequest getTraceGraphRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTraceGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTraceGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetTraceGraphRequest, GetTraceGraphResponse>()
                .withOperationName("GetTraceGraph").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getTraceGraphRequest)
                .withMarshaller(new GetTraceGraphRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a service graph for one or more specific trace IDs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getTraceGraph(software.amazon.awssdk.services.xray.model.GetTraceGraphRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTraceGraphIterable responses = client.getTraceGraphPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.xray.paginators.GetTraceGraphIterable responses = client.getTraceGraphPaginator(request);
     *     for (software.amazon.awssdk.services.xray.model.GetTraceGraphResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTraceGraphIterable responses = client.getTraceGraphPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getTraceGraph(software.amazon.awssdk.services.xray.model.GetTraceGraphRequest)} operation.</b>
     * </p>
     *
     * @param getTraceGraphRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetTraceGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTraceGraphIterable getTraceGraphPaginator(GetTraceGraphRequest getTraceGraphRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        return new GetTraceGraphIterable(this, applyPaginatorUserAgent(getTraceGraphRequest));
    }

    /**
     * <p>
     * Retrieves IDs and annotations for traces available for a specified time frame using an optional filter. To get
     * the full traces, pass the trace IDs to <code>BatchGetTraces</code>.
     * </p>
     * <p>
     * A filter expression can target traced requests that hit specific service nodes or edges, have errors, or come
     * from a known user. For example, the following filter expression targets traces that pass through
     * <code>api.example.com</code>:
     * </p>
     * <p>
     * <code>service("api.example.com")</code>
     * </p>
     * <p>
     * This filter expression finds traces that have an annotation named <code>account</code> with the value
     * <code>12345</code>:
     * </p>
     * <p>
     * <code>annotation.account = "12345"</code>
     * </p>
     * <p>
     * For a full list of indexed fields and keywords that you can use in filter expressions, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-console-filters.html">Using Filter Expressions</a> in
     * the <i>AWS X-Ray Developer Guide</i>.
     * </p>
     *
     * @param getTraceSummariesRequest
     * @return Result of the GetTraceSummaries operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetTraceSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceSummaries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTraceSummariesResponse getTraceSummaries(GetTraceSummariesRequest getTraceSummariesRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTraceSummariesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTraceSummariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetTraceSummariesRequest, GetTraceSummariesResponse>()
                .withOperationName("GetTraceSummaries").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getTraceSummariesRequest)
                .withMarshaller(new GetTraceSummariesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves IDs and annotations for traces available for a specified time frame using an optional filter. To get
     * the full traces, pass the trace IDs to <code>BatchGetTraces</code>.
     * </p>
     * <p>
     * A filter expression can target traced requests that hit specific service nodes or edges, have errors, or come
     * from a known user. For example, the following filter expression targets traces that pass through
     * <code>api.example.com</code>:
     * </p>
     * <p>
     * <code>service("api.example.com")</code>
     * </p>
     * <p>
     * This filter expression finds traces that have an annotation named <code>account</code> with the value
     * <code>12345</code>:
     * </p>
     * <p>
     * <code>annotation.account = "12345"</code>
     * </p>
     * <p>
     * For a full list of indexed fields and keywords that you can use in filter expressions, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-console-filters.html">Using Filter Expressions</a> in
     * the <i>AWS X-Ray Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getTraceSummaries(software.amazon.awssdk.services.xray.model.GetTraceSummariesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTraceSummariesIterable responses = client.getTraceSummariesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.xray.paginators.GetTraceSummariesIterable responses = client
     *             .getTraceSummariesPaginator(request);
     *     for (software.amazon.awssdk.services.xray.model.GetTraceSummariesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.xray.paginators.GetTraceSummariesIterable responses = client.getTraceSummariesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getTraceSummaries(software.amazon.awssdk.services.xray.model.GetTraceSummariesRequest)} operation.</b>
     * </p>
     *
     * @param getTraceSummariesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetTraceSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceSummaries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTraceSummariesIterable getTraceSummariesPaginator(GetTraceSummariesRequest getTraceSummariesRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        return new GetTraceSummariesIterable(this, applyPaginatorUserAgent(getTraceSummariesRequest));
    }

    /**
     * <p>
     * Updates the encryption configuration for X-Ray data.
     * </p>
     *
     * @param putEncryptionConfigRequest
     * @return Result of the PutEncryptionConfig operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.PutEncryptionConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutEncryptionConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutEncryptionConfigResponse putEncryptionConfig(PutEncryptionConfigRequest putEncryptionConfigRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutEncryptionConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutEncryptionConfigRequest, PutEncryptionConfigResponse>()
                .withOperationName("PutEncryptionConfig").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putEncryptionConfigRequest)
                .withMarshaller(new PutEncryptionConfigRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Used by the AWS X-Ray daemon to upload telemetry.
     * </p>
     *
     * @param putTelemetryRecordsRequest
     * @return Result of the PutTelemetryRecords operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.PutTelemetryRecords
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutTelemetryRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutTelemetryRecordsResponse putTelemetryRecords(PutTelemetryRecordsRequest putTelemetryRecordsRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutTelemetryRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutTelemetryRecordsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutTelemetryRecordsRequest, PutTelemetryRecordsResponse>()
                .withOperationName("PutTelemetryRecords").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putTelemetryRecordsRequest)
                .withMarshaller(new PutTelemetryRecordsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Uploads segment documents to AWS X-Ray. The <a href="https://docs.aws.amazon.com/xray/index.html">X-Ray SDK</a>
     * generates segment documents and sends them to the X-Ray daemon, which uploads them in batches. A segment document
     * can be a completed segment, an in-progress segment, or an array of subsegments.
     * </p>
     * <p>
     * Segments must include the following fields. For the full segment document schema, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-api-segmentdocuments.html">AWS X-Ray Segment
     * Documents</a> in the <i>AWS X-Ray Developer Guide</i>.
     * </p>
     * <p class="title">
     * <b>Required Segment Document Fields</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>name</code> - The name of the service that handled the request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>id</code> - A 64-bit identifier for the segment, unique among segments in the same trace, in 16 hexadecimal
     * digits.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>trace_id</code> - A unique identifier that connects all segments and subsegments originating from a single
     * client request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>start_time</code> - Time the segment or subsegment was created, in floating point seconds in epoch time,
     * accurate to milliseconds. For example, <code>1480615200.010</code> or <code>1.480615200010E9</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>end_time</code> - Time the segment or subsegment was closed. For example, <code>1480615200.090</code> or
     * <code>1.480615200090E9</code>. Specify either an <code>end_time</code> or <code>in_progress</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>in_progress</code> - Set to <code>true</code> instead of specifying an <code>end_time</code> to record that
     * a segment has been started, but is not complete. Send an in progress segment when your application receives a
     * request that will take a long time to serve, to trace the fact that the request was received. When the response
     * is sent, send the complete segment to overwrite the in-progress segment.
     * </p>
     * </li>
     * </ul>
     * <p>
     * A <code>trace_id</code> consists of three numbers separated by hyphens. For example,
     * 1-58406520-a006649127e371903a2de979. This includes:
     * </p>
     * <p class="title">
     * <b>Trace ID Format</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The version number, i.e. <code>1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The time of the original request, in Unix epoch time, in 8 hexadecimal digits. For example, 10:00AM December 2nd,
     * 2016 PST in epoch time is <code>1480615200</code> seconds, or <code>58406520</code> in hexadecimal.
     * </p>
     * </li>
     * <li>
     * <p>
     * A 96-bit identifier for the trace, globally unique, in 24 hexadecimal digits.
     * </p>
     * </li>
     * </ul>
     *
     * @param putTraceSegmentsRequest
     * @return Result of the PutTraceSegments operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.PutTraceSegments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutTraceSegments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutTraceSegmentsResponse putTraceSegments(PutTraceSegmentsRequest putTraceSegmentsRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutTraceSegmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutTraceSegmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutTraceSegmentsRequest, PutTraceSegmentsResponse>()
                .withOperationName("PutTraceSegments").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putTraceSegmentsRequest)
                .withMarshaller(new PutTraceSegmentsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates a group resource.
     * </p>
     *
     * @param updateGroupRequest
     * @return Result of the UpdateGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.UpdateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UpdateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateGroupResponse updateGroup(UpdateGroupRequest updateGroupRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                .withOperationName("UpdateGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateGroupRequest)
                .withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Modifies a sampling rule's configuration.
     * </p>
     *
     * @param updateSamplingRuleRequest
     * @return Result of the UpdateSamplingRule operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.UpdateSamplingRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UpdateSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateSamplingRuleResponse updateSamplingRule(UpdateSamplingRuleRequest updateSamplingRuleRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSamplingRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateSamplingRuleRequest, UpdateSamplingRuleResponse>()
                .withOperationName("UpdateSamplingRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateSamplingRuleRequest)
                .withMarshaller(new UpdateSamplingRuleRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(XRayException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledException")
                                .exceptionBuilderSupplier(ThrottledException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RuleLimitExceededException")
                                .exceptionBuilderSupplier(RuleLimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends XRayRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
