/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.worklink;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.worklink.model.AssociateDomainRequest;
import software.amazon.awssdk.services.worklink.model.AssociateDomainResponse;
import software.amazon.awssdk.services.worklink.model.AssociateWebsiteAuthorizationProviderRequest;
import software.amazon.awssdk.services.worklink.model.AssociateWebsiteAuthorizationProviderResponse;
import software.amazon.awssdk.services.worklink.model.AssociateWebsiteCertificateAuthorityRequest;
import software.amazon.awssdk.services.worklink.model.AssociateWebsiteCertificateAuthorityResponse;
import software.amazon.awssdk.services.worklink.model.CreateFleetRequest;
import software.amazon.awssdk.services.worklink.model.CreateFleetResponse;
import software.amazon.awssdk.services.worklink.model.DeleteFleetRequest;
import software.amazon.awssdk.services.worklink.model.DeleteFleetResponse;
import software.amazon.awssdk.services.worklink.model.DescribeAuditStreamConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.DescribeAuditStreamConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.DescribeCompanyNetworkConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.DescribeCompanyNetworkConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.DescribeDevicePolicyConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.DescribeDevicePolicyConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.DescribeDeviceRequest;
import software.amazon.awssdk.services.worklink.model.DescribeDeviceResponse;
import software.amazon.awssdk.services.worklink.model.DescribeDomainRequest;
import software.amazon.awssdk.services.worklink.model.DescribeDomainResponse;
import software.amazon.awssdk.services.worklink.model.DescribeFleetMetadataRequest;
import software.amazon.awssdk.services.worklink.model.DescribeFleetMetadataResponse;
import software.amazon.awssdk.services.worklink.model.DescribeIdentityProviderConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.DescribeIdentityProviderConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.DescribeWebsiteCertificateAuthorityRequest;
import software.amazon.awssdk.services.worklink.model.DescribeWebsiteCertificateAuthorityResponse;
import software.amazon.awssdk.services.worklink.model.DisassociateDomainRequest;
import software.amazon.awssdk.services.worklink.model.DisassociateDomainResponse;
import software.amazon.awssdk.services.worklink.model.DisassociateWebsiteAuthorizationProviderRequest;
import software.amazon.awssdk.services.worklink.model.DisassociateWebsiteAuthorizationProviderResponse;
import software.amazon.awssdk.services.worklink.model.DisassociateWebsiteCertificateAuthorityRequest;
import software.amazon.awssdk.services.worklink.model.DisassociateWebsiteCertificateAuthorityResponse;
import software.amazon.awssdk.services.worklink.model.InternalServerErrorException;
import software.amazon.awssdk.services.worklink.model.InvalidRequestException;
import software.amazon.awssdk.services.worklink.model.ListDevicesRequest;
import software.amazon.awssdk.services.worklink.model.ListDevicesResponse;
import software.amazon.awssdk.services.worklink.model.ListDomainsRequest;
import software.amazon.awssdk.services.worklink.model.ListDomainsResponse;
import software.amazon.awssdk.services.worklink.model.ListFleetsRequest;
import software.amazon.awssdk.services.worklink.model.ListFleetsResponse;
import software.amazon.awssdk.services.worklink.model.ListWebsiteAuthorizationProvidersRequest;
import software.amazon.awssdk.services.worklink.model.ListWebsiteAuthorizationProvidersResponse;
import software.amazon.awssdk.services.worklink.model.ListWebsiteCertificateAuthoritiesRequest;
import software.amazon.awssdk.services.worklink.model.ListWebsiteCertificateAuthoritiesResponse;
import software.amazon.awssdk.services.worklink.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.worklink.model.ResourceNotFoundException;
import software.amazon.awssdk.services.worklink.model.RestoreDomainAccessRequest;
import software.amazon.awssdk.services.worklink.model.RestoreDomainAccessResponse;
import software.amazon.awssdk.services.worklink.model.RevokeDomainAccessRequest;
import software.amazon.awssdk.services.worklink.model.RevokeDomainAccessResponse;
import software.amazon.awssdk.services.worklink.model.SignOutUserRequest;
import software.amazon.awssdk.services.worklink.model.SignOutUserResponse;
import software.amazon.awssdk.services.worklink.model.TooManyRequestsException;
import software.amazon.awssdk.services.worklink.model.UnauthorizedException;
import software.amazon.awssdk.services.worklink.model.UpdateAuditStreamConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.UpdateAuditStreamConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.UpdateCompanyNetworkConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.UpdateCompanyNetworkConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.UpdateDevicePolicyConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.UpdateDevicePolicyConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.UpdateDomainMetadataRequest;
import software.amazon.awssdk.services.worklink.model.UpdateDomainMetadataResponse;
import software.amazon.awssdk.services.worklink.model.UpdateFleetMetadataRequest;
import software.amazon.awssdk.services.worklink.model.UpdateFleetMetadataResponse;
import software.amazon.awssdk.services.worklink.model.UpdateIdentityProviderConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.UpdateIdentityProviderConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.WorkLinkException;
import software.amazon.awssdk.services.worklink.model.WorkLinkRequest;
import software.amazon.awssdk.services.worklink.paginators.ListDevicesPublisher;
import software.amazon.awssdk.services.worklink.paginators.ListDomainsPublisher;
import software.amazon.awssdk.services.worklink.paginators.ListFleetsPublisher;
import software.amazon.awssdk.services.worklink.paginators.ListWebsiteAuthorizationProvidersPublisher;
import software.amazon.awssdk.services.worklink.paginators.ListWebsiteCertificateAuthoritiesPublisher;
import software.amazon.awssdk.services.worklink.transform.AssociateDomainRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.AssociateWebsiteAuthorizationProviderRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.AssociateWebsiteCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.CreateFleetRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DeleteFleetRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeAuditStreamConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeCompanyNetworkConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeDevicePolicyConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeDeviceRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeDomainRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeFleetMetadataRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeIdentityProviderConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeWebsiteCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DisassociateDomainRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DisassociateWebsiteAuthorizationProviderRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DisassociateWebsiteCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListDevicesRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListFleetsRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListWebsiteAuthorizationProvidersRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListWebsiteCertificateAuthoritiesRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.RestoreDomainAccessRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.RevokeDomainAccessRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.SignOutUserRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateAuditStreamConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateCompanyNetworkConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateDevicePolicyConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateDomainMetadataRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateFleetMetadataRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateIdentityProviderConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link WorkLinkAsyncClient}.
 *
 * @see WorkLinkAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWorkLinkAsyncClient implements WorkLinkAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultWorkLinkAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWorkLinkAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Specifies a domain to be associated to Amazon WorkLink.
     * </p>
     *
     * @param associateDomainRequest
     * @return A Java Future containing the result of the AssociateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.AssociateDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/AssociateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateDomainResponse> associateDomain(AssociateDomainRequest associateDomainRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateDomainRequest, AssociateDomainResponse>()
                            .withOperationName("AssociateDomain")
                            .withMarshaller(new AssociateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a website authorization provider with a specified fleet. This is used to authorize users against
     * associated websites in the company network.
     * </p>
     *
     * @param associateWebsiteAuthorizationProviderRequest
     * @return A Java Future containing the result of the AssociateWebsiteAuthorizationProvider operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.AssociateWebsiteAuthorizationProvider
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/AssociateWebsiteAuthorizationProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateWebsiteAuthorizationProviderResponse> associateWebsiteAuthorizationProvider(
            AssociateWebsiteAuthorizationProviderRequest associateWebsiteAuthorizationProviderRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateWebsiteAuthorizationProviderResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateWebsiteAuthorizationProviderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateWebsiteAuthorizationProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateWebsiteAuthorizationProviderRequest, AssociateWebsiteAuthorizationProviderResponse>()
                            .withOperationName("AssociateWebsiteAuthorizationProvider")
                            .withMarshaller(new AssociateWebsiteAuthorizationProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateWebsiteAuthorizationProviderRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports the root certificate of a certificate authority (CA) used to obtain TLS certificates used by associated
     * websites within the company network.
     * </p>
     *
     * @param associateWebsiteCertificateAuthorityRequest
     * @return A Java Future containing the result of the AssociateWebsiteCertificateAuthority operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.AssociateWebsiteCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/AssociateWebsiteCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateWebsiteCertificateAuthorityResponse> associateWebsiteCertificateAuthority(
            AssociateWebsiteCertificateAuthorityRequest associateWebsiteCertificateAuthorityRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateWebsiteCertificateAuthorityResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateWebsiteCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateWebsiteCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateWebsiteCertificateAuthorityRequest, AssociateWebsiteCertificateAuthorityResponse>()
                            .withOperationName("AssociateWebsiteCertificateAuthority")
                            .withMarshaller(new AssociateWebsiteCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateWebsiteCertificateAuthorityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a fleet. A fleet consists of resources and the configuration that delivers associated websites to
     * authorized users who download and set up the Amazon WorkLink app.
     * </p>
     *
     * @param createFleetRequest
     * @return A Java Future containing the result of the CreateFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.CreateFleet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/CreateFleet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFleetResponse> createFleet(CreateFleetRequest createFleetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFleetRequest, CreateFleetResponse>()
                            .withOperationName("CreateFleet").withMarshaller(new CreateFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createFleetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a fleet. Prevents users from accessing previously associated websites.
     * </p>
     *
     * @param deleteFleetRequest
     * @return A Java Future containing the result of the DeleteFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DeleteFleet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DeleteFleet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFleetResponse> deleteFleet(DeleteFleetRequest deleteFleetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFleetRequest, DeleteFleetResponse>()
                            .withOperationName("DeleteFleet").withMarshaller(new DeleteFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteFleetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the configuration for delivering audit streams to the customer account.
     * </p>
     *
     * @param describeAuditStreamConfigurationRequest
     * @return A Java Future containing the result of the DescribeAuditStreamConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeAuditStreamConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeAuditStreamConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAuditStreamConfigurationResponse> describeAuditStreamConfiguration(
            DescribeAuditStreamConfigurationRequest describeAuditStreamConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAuditStreamConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeAuditStreamConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAuditStreamConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAuditStreamConfigurationRequest, DescribeAuditStreamConfigurationResponse>()
                            .withOperationName("DescribeAuditStreamConfiguration")
                            .withMarshaller(new DescribeAuditStreamConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAuditStreamConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the networking configuration to access the internal websites associated with the specified fleet.
     * </p>
     *
     * @param describeCompanyNetworkConfigurationRequest
     * @return A Java Future containing the result of the DescribeCompanyNetworkConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeCompanyNetworkConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeCompanyNetworkConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCompanyNetworkConfigurationResponse> describeCompanyNetworkConfiguration(
            DescribeCompanyNetworkConfigurationRequest describeCompanyNetworkConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCompanyNetworkConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeCompanyNetworkConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCompanyNetworkConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCompanyNetworkConfigurationRequest, DescribeCompanyNetworkConfigurationResponse>()
                            .withOperationName("DescribeCompanyNetworkConfiguration")
                            .withMarshaller(new DescribeCompanyNetworkConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeCompanyNetworkConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a user's device.
     * </p>
     *
     * @param describeDeviceRequest
     * @return A Java Future containing the result of the DescribeDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeDevice
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDeviceResponse> describeDevice(DescribeDeviceRequest describeDeviceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDeviceRequest, DescribeDeviceResponse>()
                            .withOperationName("DescribeDevice")
                            .withMarshaller(new DescribeDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDeviceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the device policy configuration for the specified fleet.
     * </p>
     *
     * @param describeDevicePolicyConfigurationRequest
     * @return A Java Future containing the result of the DescribeDevicePolicyConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeDevicePolicyConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeDevicePolicyConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDevicePolicyConfigurationResponse> describeDevicePolicyConfiguration(
            DescribeDevicePolicyConfigurationRequest describeDevicePolicyConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDevicePolicyConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeDevicePolicyConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDevicePolicyConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDevicePolicyConfigurationRequest, DescribeDevicePolicyConfigurationResponse>()
                            .withOperationName("DescribeDevicePolicyConfiguration")
                            .withMarshaller(new DescribeDevicePolicyConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDevicePolicyConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the domain.
     * </p>
     *
     * @param describeDomainRequest
     * @return A Java Future containing the result of the DescribeDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDomainResponse> describeDomain(DescribeDomainRequest describeDomainRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainRequest, DescribeDomainResponse>()
                            .withOperationName("DescribeDomain")
                            .withMarshaller(new DescribeDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides basic information for the specified fleet, excluding identity provider, networking, and device
     * configuration details.
     * </p>
     *
     * @param describeFleetMetadataRequest
     * @return A Java Future containing the result of the DescribeFleetMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeFleetMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeFleetMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFleetMetadataResponse> describeFleetMetadata(
            DescribeFleetMetadataRequest describeFleetMetadataRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFleetMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFleetMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFleetMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFleetMetadataRequest, DescribeFleetMetadataResponse>()
                            .withOperationName("DescribeFleetMetadata")
                            .withMarshaller(new DescribeFleetMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeFleetMetadataRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the identity provider configuration of the specified fleet.
     * </p>
     *
     * @param describeIdentityProviderConfigurationRequest
     * @return A Java Future containing the result of the DescribeIdentityProviderConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeIdentityProviderConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeIdentityProviderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeIdentityProviderConfigurationResponse> describeIdentityProviderConfiguration(
            DescribeIdentityProviderConfigurationRequest describeIdentityProviderConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeIdentityProviderConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeIdentityProviderConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeIdentityProviderConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeIdentityProviderConfigurationRequest, DescribeIdentityProviderConfigurationResponse>()
                            .withOperationName("DescribeIdentityProviderConfiguration")
                            .withMarshaller(new DescribeIdentityProviderConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeIdentityProviderConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the certificate authority.
     * </p>
     *
     * @param describeWebsiteCertificateAuthorityRequest
     * @return A Java Future containing the result of the DescribeWebsiteCertificateAuthority operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeWebsiteCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeWebsiteCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWebsiteCertificateAuthorityResponse> describeWebsiteCertificateAuthority(
            DescribeWebsiteCertificateAuthorityRequest describeWebsiteCertificateAuthorityRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWebsiteCertificateAuthorityResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeWebsiteCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWebsiteCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWebsiteCertificateAuthorityRequest, DescribeWebsiteCertificateAuthorityResponse>()
                            .withOperationName("DescribeWebsiteCertificateAuthority")
                            .withMarshaller(new DescribeWebsiteCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeWebsiteCertificateAuthorityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a domain from Amazon WorkLink. End users lose the ability to access the domain with Amazon
     * WorkLink.
     * </p>
     *
     * @param disassociateDomainRequest
     * @return A Java Future containing the result of the DisassociateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DisassociateDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DisassociateDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateDomainResponse> disassociateDomain(DisassociateDomainRequest disassociateDomainRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateDomainRequest, DisassociateDomainResponse>()
                            .withOperationName("DisassociateDomain")
                            .withMarshaller(new DisassociateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a website authorization provider from a specified fleet. After the disassociation, users can't load
     * any associated websites that require this authorization provider.
     * </p>
     *
     * @param disassociateWebsiteAuthorizationProviderRequest
     * @return A Java Future containing the result of the DisassociateWebsiteAuthorizationProvider operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DisassociateWebsiteAuthorizationProvider
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DisassociateWebsiteAuthorizationProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateWebsiteAuthorizationProviderResponse> disassociateWebsiteAuthorizationProvider(
            DisassociateWebsiteAuthorizationProviderRequest disassociateWebsiteAuthorizationProviderRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateWebsiteAuthorizationProviderResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateWebsiteAuthorizationProviderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateWebsiteAuthorizationProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateWebsiteAuthorizationProviderRequest, DisassociateWebsiteAuthorizationProviderResponse>()
                            .withOperationName("DisassociateWebsiteAuthorizationProvider")
                            .withMarshaller(new DisassociateWebsiteAuthorizationProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateWebsiteAuthorizationProviderRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a certificate authority (CA).
     * </p>
     *
     * @param disassociateWebsiteCertificateAuthorityRequest
     * @return A Java Future containing the result of the DisassociateWebsiteCertificateAuthority operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DisassociateWebsiteCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DisassociateWebsiteCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateWebsiteCertificateAuthorityResponse> disassociateWebsiteCertificateAuthority(
            DisassociateWebsiteCertificateAuthorityRequest disassociateWebsiteCertificateAuthorityRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateWebsiteCertificateAuthorityResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateWebsiteCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateWebsiteCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateWebsiteCertificateAuthorityRequest, DisassociateWebsiteCertificateAuthorityResponse>()
                            .withOperationName("DisassociateWebsiteCertificateAuthority")
                            .withMarshaller(new DisassociateWebsiteCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateWebsiteCertificateAuthorityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of devices registered with the specified fleet.
     * </p>
     *
     * @param listDevicesRequest
     * @return A Java Future containing the result of the ListDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListDevices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListDevices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDevicesResponse> listDevices(ListDevicesRequest listDevicesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDevicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDevicesRequest, ListDevicesResponse>()
                            .withOperationName("ListDevices").withMarshaller(new ListDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDevicesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of devices registered with the specified fleet.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDevices(software.amazon.awssdk.services.worklink.model.ListDevicesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListDevicesPublisher publisher = client.listDevicesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListDevicesPublisher publisher = client.listDevicesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.worklink.model.ListDevicesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.worklink.model.ListDevicesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDevices(software.amazon.awssdk.services.worklink.model.ListDevicesRequest)} operation.</b>
     * </p>
     *
     * @param listDevicesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListDevices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListDevices" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDevicesPublisher listDevicesPaginator(ListDevicesRequest listDevicesRequest) {
        return new ListDevicesPublisher(this, applyPaginatorUserAgent(listDevicesRequest));
    }

    /**
     * <p>
     * Retrieves a list of domains associated to a specified fleet.
     * </p>
     *
     * @param listDomainsRequest
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListDomains
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                            .withOperationName("ListDomains").withMarshaller(new ListDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDomainsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of domains associated to a specified fleet.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDomains(software.amazon.awssdk.services.worklink.model.ListDomainsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.worklink.model.ListDomainsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.worklink.model.ListDomainsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.worklink.model.ListDomainsRequest)} operation.</b>
     * </p>
     *
     * @param listDomainsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListDomains
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDomainsPublisher listDomainsPaginator(ListDomainsRequest listDomainsRequest) {
        return new ListDomainsPublisher(this, applyPaginatorUserAgent(listDomainsRequest));
    }

    /**
     * <p>
     * Retrieves a list of fleets for the current account and Region.
     * </p>
     *
     * @param listFleetsRequest
     * @return A Java Future containing the result of the ListFleets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListFleets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListFleets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFleetsResponse> listFleets(ListFleetsRequest listFleetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFleetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFleetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFleetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFleetsRequest, ListFleetsResponse>().withOperationName("ListFleets")
                            .withMarshaller(new ListFleetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listFleetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of fleets for the current account and Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listFleets(software.amazon.awssdk.services.worklink.model.ListFleetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListFleetsPublisher publisher = client.listFleetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListFleetsPublisher publisher = client.listFleetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.worklink.model.ListFleetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.worklink.model.ListFleetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFleets(software.amazon.awssdk.services.worklink.model.ListFleetsRequest)} operation.</b>
     * </p>
     *
     * @param listFleetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListFleets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListFleets" target="_top">AWS API
     *      Documentation</a>
     */
    public ListFleetsPublisher listFleetsPaginator(ListFleetsRequest listFleetsRequest) {
        return new ListFleetsPublisher(this, applyPaginatorUserAgent(listFleetsRequest));
    }

    /**
     * <p>
     * Retrieves a list of website authorization providers associated with a specified fleet.
     * </p>
     *
     * @param listWebsiteAuthorizationProvidersRequest
     * @return A Java Future containing the result of the ListWebsiteAuthorizationProviders operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListWebsiteAuthorizationProviders
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListWebsiteAuthorizationProviders"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWebsiteAuthorizationProvidersResponse> listWebsiteAuthorizationProviders(
            ListWebsiteAuthorizationProvidersRequest listWebsiteAuthorizationProvidersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWebsiteAuthorizationProvidersResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListWebsiteAuthorizationProvidersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWebsiteAuthorizationProvidersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWebsiteAuthorizationProvidersRequest, ListWebsiteAuthorizationProvidersResponse>()
                            .withOperationName("ListWebsiteAuthorizationProviders")
                            .withMarshaller(new ListWebsiteAuthorizationProvidersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listWebsiteAuthorizationProvidersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of website authorization providers associated with a specified fleet.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWebsiteAuthorizationProviders(software.amazon.awssdk.services.worklink.model.ListWebsiteAuthorizationProvidersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListWebsiteAuthorizationProvidersPublisher publisher = client.listWebsiteAuthorizationProvidersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListWebsiteAuthorizationProvidersPublisher publisher = client.listWebsiteAuthorizationProvidersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.worklink.model.ListWebsiteAuthorizationProvidersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.worklink.model.ListWebsiteAuthorizationProvidersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWebsiteAuthorizationProviders(software.amazon.awssdk.services.worklink.model.ListWebsiteAuthorizationProvidersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listWebsiteAuthorizationProvidersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListWebsiteAuthorizationProviders
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListWebsiteAuthorizationProviders"
     *      target="_top">AWS API Documentation</a>
     */
    public ListWebsiteAuthorizationProvidersPublisher listWebsiteAuthorizationProvidersPaginator(
            ListWebsiteAuthorizationProvidersRequest listWebsiteAuthorizationProvidersRequest) {
        return new ListWebsiteAuthorizationProvidersPublisher(this,
                applyPaginatorUserAgent(listWebsiteAuthorizationProvidersRequest));
    }

    /**
     * <p>
     * Retrieves a list of certificate authorities added for the current account and Region.
     * </p>
     *
     * @param listWebsiteCertificateAuthoritiesRequest
     * @return A Java Future containing the result of the ListWebsiteCertificateAuthorities operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListWebsiteCertificateAuthorities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListWebsiteCertificateAuthorities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWebsiteCertificateAuthoritiesResponse> listWebsiteCertificateAuthorities(
            ListWebsiteCertificateAuthoritiesRequest listWebsiteCertificateAuthoritiesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWebsiteCertificateAuthoritiesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListWebsiteCertificateAuthoritiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWebsiteCertificateAuthoritiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWebsiteCertificateAuthoritiesRequest, ListWebsiteCertificateAuthoritiesResponse>()
                            .withOperationName("ListWebsiteCertificateAuthorities")
                            .withMarshaller(new ListWebsiteCertificateAuthoritiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listWebsiteCertificateAuthoritiesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of certificate authorities added for the current account and Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWebsiteCertificateAuthorities(software.amazon.awssdk.services.worklink.model.ListWebsiteCertificateAuthoritiesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListWebsiteCertificateAuthoritiesPublisher publisher = client.listWebsiteCertificateAuthoritiesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.worklink.paginators.ListWebsiteCertificateAuthoritiesPublisher publisher = client.listWebsiteCertificateAuthoritiesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.worklink.model.ListWebsiteCertificateAuthoritiesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.worklink.model.ListWebsiteCertificateAuthoritiesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWebsiteCertificateAuthorities(software.amazon.awssdk.services.worklink.model.ListWebsiteCertificateAuthoritiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listWebsiteCertificateAuthoritiesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListWebsiteCertificateAuthorities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListWebsiteCertificateAuthorities"
     *      target="_top">AWS API Documentation</a>
     */
    public ListWebsiteCertificateAuthoritiesPublisher listWebsiteCertificateAuthoritiesPaginator(
            ListWebsiteCertificateAuthoritiesRequest listWebsiteCertificateAuthoritiesRequest) {
        return new ListWebsiteCertificateAuthoritiesPublisher(this,
                applyPaginatorUserAgent(listWebsiteCertificateAuthoritiesRequest));
    }

    /**
     * <p>
     * Moves a domain to ACTIVE status if it was in the INACTIVE status.
     * </p>
     *
     * @param restoreDomainAccessRequest
     * @return A Java Future containing the result of the RestoreDomainAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.RestoreDomainAccess
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/RestoreDomainAccess" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreDomainAccessResponse> restoreDomainAccess(
            RestoreDomainAccessRequest restoreDomainAccessRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreDomainAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreDomainAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreDomainAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreDomainAccessRequest, RestoreDomainAccessResponse>()
                            .withOperationName("RestoreDomainAccess")
                            .withMarshaller(new RestoreDomainAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(restoreDomainAccessRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Moves a domain to INACTIVE status if it was in the ACTIVE status.
     * </p>
     *
     * @param revokeDomainAccessRequest
     * @return A Java Future containing the result of the RevokeDomainAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.RevokeDomainAccess
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/RevokeDomainAccess" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RevokeDomainAccessResponse> revokeDomainAccess(RevokeDomainAccessRequest revokeDomainAccessRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RevokeDomainAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RevokeDomainAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RevokeDomainAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RevokeDomainAccessRequest, RevokeDomainAccessResponse>()
                            .withOperationName("RevokeDomainAccess")
                            .withMarshaller(new RevokeDomainAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(revokeDomainAccessRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Signs the user out from all of their devices. The user can sign in again if they have valid credentials.
     * </p>
     *
     * @param signOutUserRequest
     * @return A Java Future containing the result of the SignOutUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.SignOutUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/SignOutUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SignOutUserResponse> signOutUser(SignOutUserRequest signOutUserRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SignOutUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SignOutUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SignOutUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SignOutUserRequest, SignOutUserResponse>()
                            .withOperationName("SignOutUser").withMarshaller(new SignOutUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(signOutUserRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the audit stream configuration for the fleet.
     * </p>
     *
     * @param updateAuditStreamConfigurationRequest
     * @return A Java Future containing the result of the UpdateAuditStreamConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateAuditStreamConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateAuditStreamConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAuditStreamConfigurationResponse> updateAuditStreamConfiguration(
            UpdateAuditStreamConfigurationRequest updateAuditStreamConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAuditStreamConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAuditStreamConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAuditStreamConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAuditStreamConfigurationRequest, UpdateAuditStreamConfigurationResponse>()
                            .withOperationName("UpdateAuditStreamConfiguration")
                            .withMarshaller(new UpdateAuditStreamConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateAuditStreamConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the company network configuration for the fleet.
     * </p>
     *
     * @param updateCompanyNetworkConfigurationRequest
     * @return A Java Future containing the result of the UpdateCompanyNetworkConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateCompanyNetworkConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateCompanyNetworkConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCompanyNetworkConfigurationResponse> updateCompanyNetworkConfiguration(
            UpdateCompanyNetworkConfigurationRequest updateCompanyNetworkConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCompanyNetworkConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateCompanyNetworkConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCompanyNetworkConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCompanyNetworkConfigurationRequest, UpdateCompanyNetworkConfigurationResponse>()
                            .withOperationName("UpdateCompanyNetworkConfiguration")
                            .withMarshaller(new UpdateCompanyNetworkConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateCompanyNetworkConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the device policy configuration for the fleet.
     * </p>
     *
     * @param updateDevicePolicyConfigurationRequest
     * @return A Java Future containing the result of the UpdateDevicePolicyConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateDevicePolicyConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateDevicePolicyConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDevicePolicyConfigurationResponse> updateDevicePolicyConfiguration(
            UpdateDevicePolicyConfigurationRequest updateDevicePolicyConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDevicePolicyConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDevicePolicyConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDevicePolicyConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDevicePolicyConfigurationRequest, UpdateDevicePolicyConfigurationResponse>()
                            .withOperationName("UpdateDevicePolicyConfiguration")
                            .withMarshaller(new UpdateDevicePolicyConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDevicePolicyConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates domain metadata, such as DisplayName.
     * </p>
     *
     * @param updateDomainMetadataRequest
     * @return A Java Future containing the result of the UpdateDomainMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateDomainMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateDomainMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainMetadataResponse> updateDomainMetadata(
            UpdateDomainMetadataRequest updateDomainMetadataRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDomainMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainMetadataRequest, UpdateDomainMetadataResponse>()
                            .withOperationName("UpdateDomainMetadata")
                            .withMarshaller(new UpdateDomainMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDomainMetadataRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates fleet metadata, such as DisplayName.
     * </p>
     *
     * @param updateFleetMetadataRequest
     * @return A Java Future containing the result of the UpdateFleetMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateFleetMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateFleetMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFleetMetadataResponse> updateFleetMetadata(
            UpdateFleetMetadataRequest updateFleetMetadataRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFleetMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFleetMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFleetMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFleetMetadataRequest, UpdateFleetMetadataResponse>()
                            .withOperationName("UpdateFleetMetadata")
                            .withMarshaller(new UpdateFleetMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateFleetMetadataRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the identity provider configuration for the fleet.
     * </p>
     *
     * @param updateIdentityProviderConfigurationRequest
     * @return A Java Future containing the result of the UpdateIdentityProviderConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateIdentityProviderConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateIdentityProviderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateIdentityProviderConfigurationResponse> updateIdentityProviderConfiguration(
            UpdateIdentityProviderConfigurationRequest updateIdentityProviderConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateIdentityProviderConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateIdentityProviderConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateIdentityProviderConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateIdentityProviderConfigurationRequest, UpdateIdentityProviderConfigurationResponse>()
                            .withOperationName("UpdateIdentityProviderConfiguration")
                            .withMarshaller(new UpdateIdentityProviderConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateIdentityProviderConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WorkLinkException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private <T extends WorkLinkRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
