/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workdocs;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.workdocs.model.AbortDocumentVersionUploadRequest;
import software.amazon.awssdk.services.workdocs.model.AbortDocumentVersionUploadResponse;
import software.amazon.awssdk.services.workdocs.model.ActivateUserRequest;
import software.amazon.awssdk.services.workdocs.model.ActivateUserResponse;
import software.amazon.awssdk.services.workdocs.model.AddResourcePermissionsRequest;
import software.amazon.awssdk.services.workdocs.model.AddResourcePermissionsResponse;
import software.amazon.awssdk.services.workdocs.model.ConcurrentModificationException;
import software.amazon.awssdk.services.workdocs.model.ConflictingOperationException;
import software.amazon.awssdk.services.workdocs.model.CreateCommentRequest;
import software.amazon.awssdk.services.workdocs.model.CreateCommentResponse;
import software.amazon.awssdk.services.workdocs.model.CreateCustomMetadataRequest;
import software.amazon.awssdk.services.workdocs.model.CreateCustomMetadataResponse;
import software.amazon.awssdk.services.workdocs.model.CreateFolderRequest;
import software.amazon.awssdk.services.workdocs.model.CreateFolderResponse;
import software.amazon.awssdk.services.workdocs.model.CreateLabelsRequest;
import software.amazon.awssdk.services.workdocs.model.CreateLabelsResponse;
import software.amazon.awssdk.services.workdocs.model.CreateNotificationSubscriptionRequest;
import software.amazon.awssdk.services.workdocs.model.CreateNotificationSubscriptionResponse;
import software.amazon.awssdk.services.workdocs.model.CreateUserRequest;
import software.amazon.awssdk.services.workdocs.model.CreateUserResponse;
import software.amazon.awssdk.services.workdocs.model.CustomMetadataLimitExceededException;
import software.amazon.awssdk.services.workdocs.model.DeactivateUserRequest;
import software.amazon.awssdk.services.workdocs.model.DeactivateUserResponse;
import software.amazon.awssdk.services.workdocs.model.DeactivatingLastSystemUserException;
import software.amazon.awssdk.services.workdocs.model.DeleteCommentRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteCommentResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteCustomMetadataRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteCustomMetadataResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteDocumentRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteDocumentResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteDocumentVersionRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteDocumentVersionResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteFolderContentsRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteFolderContentsResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteFolderRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteFolderResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteLabelsRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteLabelsResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteNotificationSubscriptionRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteNotificationSubscriptionResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteUserRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteUserResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeActivitiesRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeActivitiesResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeCommentsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeCommentsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeDocumentVersionsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeDocumentVersionsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeFolderContentsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeFolderContentsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeGroupsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeGroupsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeNotificationSubscriptionsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeNotificationSubscriptionsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeResourcePermissionsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeResourcePermissionsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeRootFoldersRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeRootFoldersResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeUsersRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeUsersResponse;
import software.amazon.awssdk.services.workdocs.model.DocumentLockedForCommentsException;
import software.amazon.awssdk.services.workdocs.model.DraftUploadOutOfSyncException;
import software.amazon.awssdk.services.workdocs.model.EntityAlreadyExistsException;
import software.amazon.awssdk.services.workdocs.model.EntityNotExistsException;
import software.amazon.awssdk.services.workdocs.model.FailedDependencyException;
import software.amazon.awssdk.services.workdocs.model.GetCurrentUserRequest;
import software.amazon.awssdk.services.workdocs.model.GetCurrentUserResponse;
import software.amazon.awssdk.services.workdocs.model.GetDocumentPathRequest;
import software.amazon.awssdk.services.workdocs.model.GetDocumentPathResponse;
import software.amazon.awssdk.services.workdocs.model.GetDocumentRequest;
import software.amazon.awssdk.services.workdocs.model.GetDocumentResponse;
import software.amazon.awssdk.services.workdocs.model.GetDocumentVersionRequest;
import software.amazon.awssdk.services.workdocs.model.GetDocumentVersionResponse;
import software.amazon.awssdk.services.workdocs.model.GetFolderPathRequest;
import software.amazon.awssdk.services.workdocs.model.GetFolderPathResponse;
import software.amazon.awssdk.services.workdocs.model.GetFolderRequest;
import software.amazon.awssdk.services.workdocs.model.GetFolderResponse;
import software.amazon.awssdk.services.workdocs.model.GetResourcesRequest;
import software.amazon.awssdk.services.workdocs.model.GetResourcesResponse;
import software.amazon.awssdk.services.workdocs.model.IllegalUserStateException;
import software.amazon.awssdk.services.workdocs.model.InitiateDocumentVersionUploadRequest;
import software.amazon.awssdk.services.workdocs.model.InitiateDocumentVersionUploadResponse;
import software.amazon.awssdk.services.workdocs.model.InvalidArgumentException;
import software.amazon.awssdk.services.workdocs.model.InvalidCommentOperationException;
import software.amazon.awssdk.services.workdocs.model.InvalidOperationException;
import software.amazon.awssdk.services.workdocs.model.InvalidPasswordException;
import software.amazon.awssdk.services.workdocs.model.LimitExceededException;
import software.amazon.awssdk.services.workdocs.model.ProhibitedStateException;
import software.amazon.awssdk.services.workdocs.model.RemoveAllResourcePermissionsRequest;
import software.amazon.awssdk.services.workdocs.model.RemoveAllResourcePermissionsResponse;
import software.amazon.awssdk.services.workdocs.model.RemoveResourcePermissionRequest;
import software.amazon.awssdk.services.workdocs.model.RemoveResourcePermissionResponse;
import software.amazon.awssdk.services.workdocs.model.RequestedEntityTooLargeException;
import software.amazon.awssdk.services.workdocs.model.ResourceAlreadyCheckedOutException;
import software.amazon.awssdk.services.workdocs.model.RestoreDocumentVersionsRequest;
import software.amazon.awssdk.services.workdocs.model.RestoreDocumentVersionsResponse;
import software.amazon.awssdk.services.workdocs.model.SearchResourcesRequest;
import software.amazon.awssdk.services.workdocs.model.SearchResourcesResponse;
import software.amazon.awssdk.services.workdocs.model.ServiceUnavailableException;
import software.amazon.awssdk.services.workdocs.model.StorageLimitExceededException;
import software.amazon.awssdk.services.workdocs.model.StorageLimitWillExceedException;
import software.amazon.awssdk.services.workdocs.model.TooManyLabelsException;
import software.amazon.awssdk.services.workdocs.model.TooManySubscriptionsException;
import software.amazon.awssdk.services.workdocs.model.UnauthorizedOperationException;
import software.amazon.awssdk.services.workdocs.model.UnauthorizedResourceAccessException;
import software.amazon.awssdk.services.workdocs.model.UpdateDocumentRequest;
import software.amazon.awssdk.services.workdocs.model.UpdateDocumentResponse;
import software.amazon.awssdk.services.workdocs.model.UpdateDocumentVersionRequest;
import software.amazon.awssdk.services.workdocs.model.UpdateDocumentVersionResponse;
import software.amazon.awssdk.services.workdocs.model.UpdateFolderRequest;
import software.amazon.awssdk.services.workdocs.model.UpdateFolderResponse;
import software.amazon.awssdk.services.workdocs.model.UpdateUserRequest;
import software.amazon.awssdk.services.workdocs.model.UpdateUserResponse;
import software.amazon.awssdk.services.workdocs.model.WorkDocsException;
import software.amazon.awssdk.services.workdocs.transform.AbortDocumentVersionUploadRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.ActivateUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.AddResourcePermissionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateCommentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateCustomMetadataRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateFolderRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateLabelsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateNotificationSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeactivateUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteCommentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteCustomMetadataRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteDocumentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteDocumentVersionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteFolderContentsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteFolderRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteLabelsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteNotificationSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeActivitiesRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeCommentsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeDocumentVersionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeFolderContentsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeGroupsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeNotificationSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeResourcePermissionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeRootFoldersRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeUsersRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetCurrentUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetDocumentPathRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetDocumentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetDocumentVersionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetFolderPathRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetFolderRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetResourcesRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.InitiateDocumentVersionUploadRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.RemoveAllResourcePermissionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.RemoveResourcePermissionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.RestoreDocumentVersionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.SearchResourcesRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.UpdateDocumentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.UpdateDocumentVersionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.UpdateFolderRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link WorkDocsAsyncClient}.
 *
 * @see WorkDocsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWorkDocsAsyncClient implements WorkDocsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultWorkDocsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final WorkDocsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultWorkDocsAsyncClient(WorkDocsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Aborts the upload of the specified document version that was previously initiated by
     * <a>InitiateDocumentVersionUpload</a>. The client should make this call only when it no longer intends to upload
     * the document version, or fails to do so.
     * </p>
     *
     * @param abortDocumentVersionUploadRequest
     * @return A Java Future containing the result of the AbortDocumentVersionUpload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>ConcurrentModificationException The resource hierarchy is changing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.AbortDocumentVersionUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/AbortDocumentVersionUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AbortDocumentVersionUploadResponse> abortDocumentVersionUpload(
            AbortDocumentVersionUploadRequest abortDocumentVersionUploadRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, abortDocumentVersionUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AbortDocumentVersionUpload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AbortDocumentVersionUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AbortDocumentVersionUploadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AbortDocumentVersionUploadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AbortDocumentVersionUploadRequest, AbortDocumentVersionUploadResponse>()
                            .withOperationName("AbortDocumentVersionUpload")
                            .withMarshaller(new AbortDocumentVersionUploadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(abortDocumentVersionUploadRequest));
            CompletableFuture<AbortDocumentVersionUploadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates the specified user. Only active users can access Amazon WorkDocs.
     * </p>
     *
     * @param activateUserRequest
     * @return A Java Future containing the result of the ActivateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.ActivateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/ActivateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateUserResponse> activateUser(ActivateUserRequest activateUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ActivateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateUserRequest, ActivateUserResponse>()
                            .withOperationName("ActivateUser").withMarshaller(new ActivateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(activateUserRequest));
            CompletableFuture<ActivateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a set of permissions for the specified folder or document. The resource permissions are overwritten if
     * the principals already have different permissions.
     * </p>
     *
     * @param addResourcePermissionsRequest
     * @return A Java Future containing the result of the AddResourcePermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.AddResourcePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/AddResourcePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddResourcePermissionsResponse> addResourcePermissions(
            AddResourcePermissionsRequest addResourcePermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addResourcePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddResourcePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddResourcePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddResourcePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddResourcePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddResourcePermissionsRequest, AddResourcePermissionsResponse>()
                            .withOperationName("AddResourcePermissions")
                            .withMarshaller(new AddResourcePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addResourcePermissionsRequest));
            CompletableFuture<AddResourcePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a new comment to the specified document version.
     * </p>
     *
     * @param createCommentRequest
     * @return A Java Future containing the result of the CreateComment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>DocumentLockedForCommentsException This exception is thrown when the document is locked for comments
     *         and user tries to create or delete a comment on that document.</li>
     *         <li>InvalidCommentOperationException The requested operation is not allowed on the specified comment
     *         object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.CreateComment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateComment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCommentResponse> createComment(CreateCommentRequest createCommentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCommentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCommentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateCommentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCommentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCommentRequest, CreateCommentResponse>()
                            .withOperationName("CreateComment")
                            .withMarshaller(new CreateCommentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCommentRequest));
            CompletableFuture<CreateCommentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more custom properties to the specified resource (a folder, document, or version).
     * </p>
     *
     * @param createCustomMetadataRequest
     * @return A Java Future containing the result of the CreateCustomMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>CustomMetadataLimitExceededException The limit has been reached on the number of custom properties
     *         for the specified resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.CreateCustomMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateCustomMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCustomMetadataResponse> createCustomMetadata(
            CreateCustomMetadataRequest createCustomMetadataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCustomMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCustomMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCustomMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCustomMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCustomMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCustomMetadataRequest, CreateCustomMetadataResponse>()
                            .withOperationName("CreateCustomMetadata")
                            .withMarshaller(new CreateCustomMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCustomMetadataRequest));
            CompletableFuture<CreateCustomMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a folder with the specified name and parent folder.
     * </p>
     *
     * @param createFolderRequest
     * @return A Java Future containing the result of the CreateFolder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>EntityAlreadyExistsException The resource already exists.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>ConflictingOperationException Another operation is in progress on the resource that conflicts with
     *         the current operation.</li>
     *         <li>ConcurrentModificationException The resource hierarchy is changing.</li>
     *         <li>LimitExceededException The maximum of 100,000 files and folders under the parent folder has been
     *         exceeded.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.CreateFolder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateFolder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFolderResponse> createFolder(CreateFolderRequest createFolderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFolderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFolder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFolderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFolderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFolderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFolderRequest, CreateFolderResponse>()
                            .withOperationName("CreateFolder").withMarshaller(new CreateFolderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFolderRequest));
            CompletableFuture<CreateFolderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified list of labels to the given resource (a document or folder)
     * </p>
     *
     * @param createLabelsRequest
     * @return A Java Future containing the result of the CreateLabels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>TooManyLabelsException The limit has been reached on the number of labels for the specified resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.CreateLabels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateLabels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLabelsResponse> createLabels(CreateLabelsRequest createLabelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLabelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLabels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLabelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLabelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLabelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLabelsRequest, CreateLabelsResponse>()
                            .withOperationName("CreateLabels").withMarshaller(new CreateLabelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLabelsRequest));
            CompletableFuture<CreateLabelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configure Amazon WorkDocs to use Amazon SNS notifications. The endpoint receives a confirmation message, and must
     * confirm the subscription.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/workdocs/latest/developerguide/manage-notifications.html">Setting up
     * notifications for an IAM user or role</a> in the <i>Amazon WorkDocs Developer Guide</i>.
     * </p>
     *
     * @param createNotificationSubscriptionRequest
     * @return A Java Future containing the result of the CreateNotificationSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>TooManySubscriptionsException You've reached the limit on the number of subscriptions for the
     *         WorkDocs instance.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.CreateNotificationSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateNotificationSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNotificationSubscriptionResponse> createNotificationSubscription(
            CreateNotificationSubscriptionRequest createNotificationSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createNotificationSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNotificationSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNotificationSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateNotificationSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNotificationSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNotificationSubscriptionRequest, CreateNotificationSubscriptionResponse>()
                            .withOperationName("CreateNotificationSubscription")
                            .withMarshaller(new CreateNotificationSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createNotificationSubscriptionRequest));
            CompletableFuture<CreateNotificationSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a user in a Simple AD or Microsoft AD directory. The status of a newly created user is "ACTIVE". New
     * users can access Amazon WorkDocs.
     * </p>
     *
     * @param createUserRequest
     * @return A Java Future containing the result of the CreateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityAlreadyExistsException The resource already exists.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateUserResponse> createUser(CreateUserRequest createUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createUserRequest));
            CompletableFuture<CreateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deactivates the specified user, which revokes the user's access to Amazon WorkDocs.
     * </p>
     *
     * @param deactivateUserRequest
     * @return A Java Future containing the result of the DeactivateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeactivateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeactivateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeactivateUserResponse> deactivateUser(DeactivateUserRequest deactivateUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeactivateUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeactivateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeactivateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeactivateUserRequest, DeactivateUserResponse>()
                            .withOperationName("DeactivateUser")
                            .withMarshaller(new DeactivateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deactivateUserRequest));
            CompletableFuture<DeactivateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified comment from the document version.
     * </p>
     *
     * @param deleteCommentRequest
     * @return A Java Future containing the result of the DeleteComment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>DocumentLockedForCommentsException This exception is thrown when the document is locked for comments
     *         and user tries to create or delete a comment on that document.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeleteComment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteComment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCommentResponse> deleteComment(DeleteCommentRequest deleteCommentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCommentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteComment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCommentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteCommentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCommentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCommentRequest, DeleteCommentResponse>()
                            .withOperationName("DeleteComment")
                            .withMarshaller(new DeleteCommentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteCommentRequest));
            CompletableFuture<DeleteCommentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes custom metadata from the specified resource.
     * </p>
     *
     * @param deleteCustomMetadataRequest
     * @return A Java Future containing the result of the DeleteCustomMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeleteCustomMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteCustomMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCustomMetadataResponse> deleteCustomMetadata(
            DeleteCustomMetadataRequest deleteCustomMetadataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCustomMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCustomMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCustomMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCustomMetadataRequest, DeleteCustomMetadataResponse>()
                            .withOperationName("DeleteCustomMetadata")
                            .withMarshaller(new DeleteCustomMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteCustomMetadataRequest));
            CompletableFuture<DeleteCustomMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes the specified document and its associated metadata.
     * </p>
     *
     * @param deleteDocumentRequest
     * @return A Java Future containing the result of the DeleteDocument operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>LimitExceededException The maximum of 100,000 files and folders under the parent folder has been
     *         exceeded.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>ConflictingOperationException Another operation is in progress on the resource that conflicts with
     *         the current operation.</li>
     *         <li>ConcurrentModificationException The resource hierarchy is changing.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeleteDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDocumentResponse> deleteDocument(DeleteDocumentRequest deleteDocumentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDocument");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDocumentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDocumentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDocumentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDocumentRequest, DeleteDocumentResponse>()
                            .withOperationName("DeleteDocument")
                            .withMarshaller(new DeleteDocumentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDocumentRequest));
            CompletableFuture<DeleteDocumentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specific version of a document.
     * </p>
     *
     * @param deleteDocumentVersionRequest
     * @return A Java Future containing the result of the DeleteDocumentVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>ConflictingOperationException Another operation is in progress on the resource that conflicts with
     *         the current operation.</li>
     *         <li>ConcurrentModificationException The resource hierarchy is changing.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>InvalidOperationException The operation is invalid.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeleteDocumentVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteDocumentVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDocumentVersionResponse> deleteDocumentVersion(
            DeleteDocumentVersionRequest deleteDocumentVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDocumentVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDocumentVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDocumentVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDocumentVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDocumentVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDocumentVersionRequest, DeleteDocumentVersionResponse>()
                            .withOperationName("DeleteDocumentVersion")
                            .withMarshaller(new DeleteDocumentVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDocumentVersionRequest));
            CompletableFuture<DeleteDocumentVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes the specified folder and its contents.
     * </p>
     *
     * @param deleteFolderRequest
     * @return A Java Future containing the result of the DeleteFolder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>LimitExceededException The maximum of 100,000 files and folders under the parent folder has been
     *         exceeded.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>ConflictingOperationException Another operation is in progress on the resource that conflicts with
     *         the current operation.</li>
     *         <li>ConcurrentModificationException The resource hierarchy is changing.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeleteFolder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteFolder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFolderResponse> deleteFolder(DeleteFolderRequest deleteFolderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFolderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFolder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFolderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFolderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFolderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFolderRequest, DeleteFolderResponse>()
                            .withOperationName("DeleteFolder").withMarshaller(new DeleteFolderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFolderRequest));
            CompletableFuture<DeleteFolderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the contents of the specified folder.
     * </p>
     *
     * @param deleteFolderContentsRequest
     * @return A Java Future containing the result of the DeleteFolderContents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>ConflictingOperationException Another operation is in progress on the resource that conflicts with
     *         the current operation.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeleteFolderContents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteFolderContents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFolderContentsResponse> deleteFolderContents(
            DeleteFolderContentsRequest deleteFolderContentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFolderContentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFolderContents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFolderContentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFolderContentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFolderContentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFolderContentsRequest, DeleteFolderContentsResponse>()
                            .withOperationName("DeleteFolderContents")
                            .withMarshaller(new DeleteFolderContentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFolderContentsRequest));
            CompletableFuture<DeleteFolderContentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified list of labels from a resource.
     * </p>
     *
     * @param deleteLabelsRequest
     * @return A Java Future containing the result of the DeleteLabels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeleteLabels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteLabels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLabelsResponse> deleteLabels(DeleteLabelsRequest deleteLabelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLabelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLabels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLabelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLabelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLabelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLabelsRequest, DeleteLabelsResponse>()
                            .withOperationName("DeleteLabels").withMarshaller(new DeleteLabelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLabelsRequest));
            CompletableFuture<DeleteLabelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified subscription from the specified organization.
     * </p>
     *
     * @param deleteNotificationSubscriptionRequest
     * @return A Java Future containing the result of the DeleteNotificationSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeleteNotificationSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteNotificationSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNotificationSubscriptionResponse> deleteNotificationSubscription(
            DeleteNotificationSubscriptionRequest deleteNotificationSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteNotificationSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotificationSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNotificationSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNotificationSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNotificationSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationSubscriptionRequest, DeleteNotificationSubscriptionResponse>()
                            .withOperationName("DeleteNotificationSubscription")
                            .withMarshaller(new DeleteNotificationSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteNotificationSubscriptionRequest));
            CompletableFuture<DeleteNotificationSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified user from a Simple AD or Microsoft AD directory.
     * </p>
     * <important>
     * <p>
     * Deleting a user immediately and permanently deletes all content in that user's folder structure. Site retention
     * policies do NOT apply to this type of deletion.
     * </p>
     * </important>
     *
     * @param deleteUserRequest
     * @return A Java Future containing the result of the DeleteUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DeleteUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteUserResponse> deleteUser(DeleteUserRequest deleteUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>().withOperationName("DeleteUser")
                            .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteUserRequest));
            CompletableFuture<DeleteUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the user activities in a specified time period.
     * </p>
     *
     * @param describeActivitiesRequest
     * @return A Java Future containing the result of the DescribeActivities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DescribeActivities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeActivities" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeActivitiesResponse> describeActivities(DescribeActivitiesRequest describeActivitiesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeActivitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeActivities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeActivitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeActivitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeActivitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeActivitiesRequest, DescribeActivitiesResponse>()
                            .withOperationName("DescribeActivities")
                            .withMarshaller(new DescribeActivitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeActivitiesRequest));
            CompletableFuture<DescribeActivitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all the comments for the specified document version.
     * </p>
     *
     * @param describeCommentsRequest
     * @return A Java Future containing the result of the DescribeComments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DescribeComments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeComments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCommentsResponse> describeComments(DescribeCommentsRequest describeCommentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCommentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeComments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCommentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeCommentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCommentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCommentsRequest, DescribeCommentsResponse>()
                            .withOperationName("DescribeComments")
                            .withMarshaller(new DescribeCommentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeCommentsRequest));
            CompletableFuture<DescribeCommentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the document versions for the specified document.
     * </p>
     * <p>
     * By default, only active versions are returned.
     * </p>
     *
     * @param describeDocumentVersionsRequest
     * @return A Java Future containing the result of the DescribeDocumentVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>InvalidPasswordException The password is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DescribeDocumentVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeDocumentVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDocumentVersionsResponse> describeDocumentVersions(
            DescribeDocumentVersionsRequest describeDocumentVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDocumentVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDocumentVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDocumentVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDocumentVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDocumentVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDocumentVersionsRequest, DescribeDocumentVersionsResponse>()
                            .withOperationName("DescribeDocumentVersions")
                            .withMarshaller(new DescribeDocumentVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDocumentVersionsRequest));
            CompletableFuture<DescribeDocumentVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the contents of the specified folder, including its documents and subfolders.
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns the first 100 active document and folder metadata items. If there are more
     * results, the response includes a marker that you can use to request the next set of results. You can also request
     * initialized documents.
     * </p>
     *
     * @param describeFolderContentsRequest
     * @return A Java Future containing the result of the DescribeFolderContents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DescribeFolderContents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeFolderContents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFolderContentsResponse> describeFolderContents(
            DescribeFolderContentsRequest describeFolderContentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFolderContentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFolderContents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFolderContentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFolderContentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFolderContentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFolderContentsRequest, DescribeFolderContentsResponse>()
                            .withOperationName("DescribeFolderContents")
                            .withMarshaller(new DescribeFolderContentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFolderContentsRequest));
            CompletableFuture<DescribeFolderContentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the groups specified by the query. Groups are defined by the underlying Active Directory.
     * </p>
     *
     * @param describeGroupsRequest
     * @return A Java Future containing the result of the DescribeGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DescribeGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGroupsResponse> describeGroups(DescribeGroupsRequest describeGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGroupsRequest, DescribeGroupsResponse>()
                            .withOperationName("DescribeGroups")
                            .withMarshaller(new DescribeGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGroupsRequest));
            CompletableFuture<DescribeGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the specified notification subscriptions.
     * </p>
     *
     * @param describeNotificationSubscriptionsRequest
     * @return A Java Future containing the result of the DescribeNotificationSubscriptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DescribeNotificationSubscriptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeNotificationSubscriptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeNotificationSubscriptionsResponse> describeNotificationSubscriptions(
            DescribeNotificationSubscriptionsRequest describeNotificationSubscriptionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeNotificationSubscriptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotificationSubscriptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeNotificationSubscriptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeNotificationSubscriptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeNotificationSubscriptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationSubscriptionsRequest, DescribeNotificationSubscriptionsResponse>()
                            .withOperationName("DescribeNotificationSubscriptions")
                            .withMarshaller(new DescribeNotificationSubscriptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeNotificationSubscriptionsRequest));
            CompletableFuture<DescribeNotificationSubscriptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the permissions of a specified resource.
     * </p>
     *
     * @param describeResourcePermissionsRequest
     * @return A Java Future containing the result of the DescribeResourcePermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DescribeResourcePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeResourcePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeResourcePermissionsResponse> describeResourcePermissions(
            DescribeResourcePermissionsRequest describeResourcePermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeResourcePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeResourcePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeResourcePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeResourcePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeResourcePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeResourcePermissionsRequest, DescribeResourcePermissionsResponse>()
                            .withOperationName("DescribeResourcePermissions")
                            .withMarshaller(new DescribeResourcePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeResourcePermissionsRequest));
            CompletableFuture<DescribeResourcePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the current user's special folders; the <code>RootFolder</code> and the <code>RecycleBin</code>.
     * <code>RootFolder</code> is the root of user's files and folders and <code>RecycleBin</code> is the root of
     * recycled items. This is not a valid action for SigV4 (administrative API) clients.
     * </p>
     * <p>
     * This action requires an authentication token. To get an authentication token, register an application with Amazon
     * WorkDocs. For more information, see <a
     * href="https://docs.aws.amazon.com/workdocs/latest/developerguide/wd-auth-user.html">Authentication and Access
     * Control for User Applications</a> in the <i>Amazon WorkDocs Developer Guide</i>.
     * </p>
     *
     * @param describeRootFoldersRequest
     * @return A Java Future containing the result of the DescribeRootFolders operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DescribeRootFolders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeRootFolders" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRootFoldersResponse> describeRootFolders(
            DescribeRootFoldersRequest describeRootFoldersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRootFoldersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRootFolders");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRootFoldersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRootFoldersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRootFoldersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRootFoldersRequest, DescribeRootFoldersResponse>()
                            .withOperationName("DescribeRootFolders")
                            .withMarshaller(new DescribeRootFoldersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRootFoldersRequest));
            CompletableFuture<DescribeRootFoldersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified users. You can describe all users or filter the results (for example, by status or
     * organization).
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns the first 24 active or pending users. If there are more results, the response
     * includes a marker that you can use to request the next set of results.
     * </p>
     *
     * @param describeUsersRequest
     * @return A Java Future containing the result of the DescribeUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>RequestedEntityTooLargeException The response is too large to return. The request must include a
     *         filter to reduce the size of the response.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.DescribeUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUsersResponse> describeUsers(DescribeUsersRequest describeUsersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeUsersRequest, DescribeUsersResponse>()
                            .withOperationName("DescribeUsers")
                            .withMarshaller(new DescribeUsersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeUsersRequest));
            CompletableFuture<DescribeUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details of the current user for whom the authentication token was generated. This is not a valid action
     * for SigV4 (administrative API) clients.
     * </p>
     * <p>
     * This action requires an authentication token. To get an authentication token, register an application with Amazon
     * WorkDocs. For more information, see <a
     * href="https://docs.aws.amazon.com/workdocs/latest/developerguide/wd-auth-user.html">Authentication and Access
     * Control for User Applications</a> in the <i>Amazon WorkDocs Developer Guide</i>.
     * </p>
     *
     * @param getCurrentUserRequest
     * @return A Java Future containing the result of the GetCurrentUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.GetCurrentUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetCurrentUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCurrentUserResponse> getCurrentUser(GetCurrentUserRequest getCurrentUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCurrentUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCurrentUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCurrentUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCurrentUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCurrentUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCurrentUserRequest, GetCurrentUserResponse>()
                            .withOperationName("GetCurrentUser")
                            .withMarshaller(new GetCurrentUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCurrentUserRequest));
            CompletableFuture<GetCurrentUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details of a document.
     * </p>
     *
     * @param getDocumentRequest
     * @return A Java Future containing the result of the GetDocument operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>InvalidPasswordException The password is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.GetDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDocumentResponse> getDocument(GetDocumentRequest getDocumentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocument");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDocumentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDocumentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDocumentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDocumentRequest, GetDocumentResponse>()
                            .withOperationName("GetDocument").withMarshaller(new GetDocumentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDocumentRequest));
            CompletableFuture<GetDocumentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the path information (the hierarchy from the root folder) for the requested document.
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns a maximum of 100 levels upwards from the requested document and only includes
     * the IDs of the parent folders in the path. You can limit the maximum number of levels. You can also request the
     * names of the parent folders.
     * </p>
     *
     * @param getDocumentPathRequest
     * @return A Java Future containing the result of the GetDocumentPath operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.GetDocumentPath
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetDocumentPath" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDocumentPathResponse> getDocumentPath(GetDocumentPathRequest getDocumentPathRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentPathRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocumentPath");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDocumentPathResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDocumentPathResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDocumentPathResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDocumentPathRequest, GetDocumentPathResponse>()
                            .withOperationName("GetDocumentPath")
                            .withMarshaller(new GetDocumentPathRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDocumentPathRequest));
            CompletableFuture<GetDocumentPathResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves version metadata for the specified document.
     * </p>
     *
     * @param getDocumentVersionRequest
     * @return A Java Future containing the result of the GetDocumentVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>InvalidPasswordException The password is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.GetDocumentVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetDocumentVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDocumentVersionResponse> getDocumentVersion(GetDocumentVersionRequest getDocumentVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocumentVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDocumentVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDocumentVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDocumentVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDocumentVersionRequest, GetDocumentVersionResponse>()
                            .withOperationName("GetDocumentVersion")
                            .withMarshaller(new GetDocumentVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDocumentVersionRequest));
            CompletableFuture<GetDocumentVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the metadata of the specified folder.
     * </p>
     *
     * @param getFolderRequest
     * @return A Java Future containing the result of the GetFolder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.GetFolder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetFolder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFolderResponse> getFolder(GetFolderRequest getFolderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFolderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFolder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFolderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFolderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFolderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFolderRequest, GetFolderResponse>().withOperationName("GetFolder")
                            .withMarshaller(new GetFolderRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getFolderRequest));
            CompletableFuture<GetFolderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the path information (the hierarchy from the root folder) for the specified folder.
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns a maximum of 100 levels upwards from the requested folder and only includes
     * the IDs of the parent folders in the path. You can limit the maximum number of levels. You can also request the
     * parent folder names.
     * </p>
     *
     * @param getFolderPathRequest
     * @return A Java Future containing the result of the GetFolderPath operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.GetFolderPath
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetFolderPath" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFolderPathResponse> getFolderPath(GetFolderPathRequest getFolderPathRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFolderPathRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFolderPath");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFolderPathResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFolderPathResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFolderPathResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFolderPathRequest, GetFolderPathResponse>()
                            .withOperationName("GetFolderPath")
                            .withMarshaller(new GetFolderPathRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFolderPathRequest));
            CompletableFuture<GetFolderPathResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a collection of resources, including folders and documents. The only <code>CollectionType</code>
     * supported is <code>SHARED_WITH_ME</code>.
     * </p>
     *
     * @param getResourcesRequest
     * @return A Java Future containing the result of the GetResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.GetResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcesResponse> getResources(GetResourcesRequest getResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcesRequest, GetResourcesResponse>()
                            .withOperationName("GetResources").withMarshaller(new GetResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourcesRequest));
            CompletableFuture<GetResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new document object and version object.
     * </p>
     * <p>
     * The client specifies the parent folder ID and name of the document to upload. The ID is optionally specified when
     * creating a new version of an existing document. This is the first step to upload a document. Next, upload the
     * document to the URL returned from the call, and then call <a>UpdateDocumentVersion</a>.
     * </p>
     * <p>
     * To cancel the document upload, call <a>AbortDocumentVersionUpload</a>.
     * </p>
     *
     * @param initiateDocumentVersionUploadRequest
     * @return A Java Future containing the result of the InitiateDocumentVersionUpload operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>EntityAlreadyExistsException The resource already exists.</li>
     *         <li>StorageLimitExceededException The storage limit has been exceeded.</li>
     *         <li>StorageLimitWillExceedException The storage limit will be exceeded.</li>
     *         <li>LimitExceededException The maximum of 100,000 files and folders under the parent folder has been
     *         exceeded.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>DraftUploadOutOfSyncException This exception is thrown when a valid checkout ID is not presented on
     *         document version upload calls for a document that has been checked out from Web client.</li>
     *         <li>ResourceAlreadyCheckedOutException The resource is already checked out.</li>
     *         <li>InvalidPasswordException The password is invalid.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.InitiateDocumentVersionUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/InitiateDocumentVersionUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<InitiateDocumentVersionUploadResponse> initiateDocumentVersionUpload(
            InitiateDocumentVersionUploadRequest initiateDocumentVersionUploadRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                initiateDocumentVersionUploadRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateDocumentVersionUpload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InitiateDocumentVersionUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InitiateDocumentVersionUploadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InitiateDocumentVersionUploadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InitiateDocumentVersionUploadRequest, InitiateDocumentVersionUploadResponse>()
                            .withOperationName("InitiateDocumentVersionUpload")
                            .withMarshaller(new InitiateDocumentVersionUploadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(initiateDocumentVersionUploadRequest));
            CompletableFuture<InitiateDocumentVersionUploadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes all the permissions from the specified resource.
     * </p>
     *
     * @param removeAllResourcePermissionsRequest
     * @return A Java Future containing the result of the RemoveAllResourcePermissions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.RemoveAllResourcePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/RemoveAllResourcePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveAllResourcePermissionsResponse> removeAllResourcePermissions(
            RemoveAllResourcePermissionsRequest removeAllResourcePermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeAllResourcePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveAllResourcePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveAllResourcePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveAllResourcePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveAllResourcePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveAllResourcePermissionsRequest, RemoveAllResourcePermissionsResponse>()
                            .withOperationName("RemoveAllResourcePermissions")
                            .withMarshaller(new RemoveAllResourcePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeAllResourcePermissionsRequest));
            CompletableFuture<RemoveAllResourcePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the permission for the specified principal from the specified resource.
     * </p>
     *
     * @param removeResourcePermissionRequest
     * @return A Java Future containing the result of the RemoveResourcePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.RemoveResourcePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/RemoveResourcePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveResourcePermissionResponse> removeResourcePermission(
            RemoveResourcePermissionRequest removeResourcePermissionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeResourcePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveResourcePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveResourcePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveResourcePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveResourcePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveResourcePermissionRequest, RemoveResourcePermissionResponse>()
                            .withOperationName("RemoveResourcePermission")
                            .withMarshaller(new RemoveResourcePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeResourcePermissionRequest));
            CompletableFuture<RemoveResourcePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Recovers a deleted version of an Amazon WorkDocs document.
     * </p>
     *
     * @param restoreDocumentVersionsRequest
     * @return A Java Future containing the result of the RestoreDocumentVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>ConflictingOperationException Another operation is in progress on the resource that conflicts with
     *         the current operation.</li>
     *         <li>ConcurrentModificationException The resource hierarchy is changing.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>InvalidOperationException The operation is invalid.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.RestoreDocumentVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/RestoreDocumentVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreDocumentVersionsResponse> restoreDocumentVersions(
            RestoreDocumentVersionsRequest restoreDocumentVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreDocumentVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreDocumentVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreDocumentVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreDocumentVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreDocumentVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreDocumentVersionsRequest, RestoreDocumentVersionsResponse>()
                            .withOperationName("RestoreDocumentVersions")
                            .withMarshaller(new RestoreDocumentVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(restoreDocumentVersionsRequest));
            CompletableFuture<RestoreDocumentVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches metadata and the content of folders, documents, document versions, and comments.
     * </p>
     *
     * @param searchResourcesRequest
     * @return A Java Future containing the result of the SearchResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.SearchResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/SearchResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SearchResourcesResponse> searchResources(SearchResourcesRequest searchResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchResourcesRequest, SearchResourcesResponse>()
                            .withOperationName("SearchResources")
                            .withMarshaller(new SearchResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchResourcesRequest));
            CompletableFuture<SearchResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified attributes of a document. The user must have access to both the document and its parent
     * folder, if applicable.
     * </p>
     *
     * @param updateDocumentRequest
     * @return A Java Future containing the result of the UpdateDocument operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>EntityAlreadyExistsException The resource already exists.</li>
     *         <li>LimitExceededException The maximum of 100,000 files and folders under the parent folder has been
     *         exceeded.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>ConflictingOperationException Another operation is in progress on the resource that conflicts with
     *         the current operation.</li>
     *         <li>ConcurrentModificationException The resource hierarchy is changing.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.UpdateDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/UpdateDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDocumentResponse> updateDocument(UpdateDocumentRequest updateDocumentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDocument");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDocumentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDocumentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDocumentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDocumentRequest, UpdateDocumentResponse>()
                            .withOperationName("UpdateDocument")
                            .withMarshaller(new UpdateDocumentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDocumentRequest));
            CompletableFuture<UpdateDocumentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the status of the document version to ACTIVE.
     * </p>
     * <p>
     * Amazon WorkDocs also sets its document container to ACTIVE. This is the last step in a document upload, after the
     * client uploads the document to an S3-presigned URL returned by <a>InitiateDocumentVersionUpload</a>.
     * </p>
     *
     * @param updateDocumentVersionRequest
     * @return A Java Future containing the result of the UpdateDocumentVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>ConcurrentModificationException The resource hierarchy is changing.</li>
     *         <li>InvalidOperationException The operation is invalid.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.UpdateDocumentVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/UpdateDocumentVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDocumentVersionResponse> updateDocumentVersion(
            UpdateDocumentVersionRequest updateDocumentVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDocumentVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDocumentVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDocumentVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDocumentVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDocumentVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDocumentVersionRequest, UpdateDocumentVersionResponse>()
                            .withOperationName("UpdateDocumentVersion")
                            .withMarshaller(new UpdateDocumentVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDocumentVersionRequest));
            CompletableFuture<UpdateDocumentVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified attributes of the specified folder. The user must have access to both the folder and its
     * parent folder, if applicable.
     * </p>
     *
     * @param updateFolderRequest
     * @return A Java Future containing the result of the UpdateFolder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>EntityAlreadyExistsException The resource already exists.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>ConflictingOperationException Another operation is in progress on the resource that conflicts with
     *         the current operation.</li>
     *         <li>ConcurrentModificationException The resource hierarchy is changing.</li>
     *         <li>LimitExceededException The maximum of 100,000 files and folders under the parent folder has been
     *         exceeded.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.UpdateFolder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/UpdateFolder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFolderResponse> updateFolder(UpdateFolderRequest updateFolderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFolderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFolder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFolderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateFolderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFolderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFolderRequest, UpdateFolderResponse>()
                            .withOperationName("UpdateFolder").withMarshaller(new UpdateFolderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFolderRequest));
            CompletableFuture<UpdateFolderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified attributes of the specified user, and grants or revokes administrative privileges to the
     * Amazon WorkDocs site.
     * </p>
     *
     * @param updateUserRequest
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotExistsException The resource does not exist.</li>
     *         <li>UnauthorizedOperationException The operation is not permitted.</li>
     *         <li>UnauthorizedResourceAccessException The caller does not have access to perform the action on the
     *         resource.</li>
     *         <li>IllegalUserStateException The user is undergoing transfer of ownership.</li>
     *         <li>ProhibitedStateException The specified document version is not in the INITIALIZED state.</li>
     *         <li>FailedDependencyException The Directory Service cannot reach an on-premises instance. Or a dependency
     *         under the control of the organization is failing, such as a connected Active Directory.</li>
     *         <li>ServiceUnavailableException One or more of the dependencies is unavailable.</li>
     *         <li>DeactivatingLastSystemUserException The last user in the organization is being deactivated.</li>
     *         <li>InvalidArgumentException The pagination marker or limit fields are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkDocsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkDocsAsyncClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserResponse> updateUser(UpdateUserRequest updateUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>().withOperationName("UpdateUser")
                            .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateUserRequest));
            CompletableFuture<UpdateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final WorkDocsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WorkDocsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FailedDependencyException")
                                .exceptionBuilderSupplier(FailedDependencyException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).httpStatusCode(405).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomMetadataLimitExceededException")
                                .exceptionBuilderSupplier(CustomMetadataLimitExceededException::builder).httpStatusCode(429)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedResourceAccessException")
                                .exceptionBuilderSupplier(UnauthorizedResourceAccessException::builder).httpStatusCode(404)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManySubscriptionsException")
                                .exceptionBuilderSupplier(TooManySubscriptionsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntityAlreadyExistsException")
                                .exceptionBuilderSupplier(EntityAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntityNotExistsException")
                                .exceptionBuilderSupplier(EntityNotExistsException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DocumentLockedForCommentsException")
                                .exceptionBuilderSupplier(DocumentLockedForCommentsException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StorageLimitExceededException")
                                .exceptionBuilderSupplier(StorageLimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StorageLimitWillExceedException")
                                .exceptionBuilderSupplier(StorageLimitWillExceedException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProhibitedStateException")
                                .exceptionBuilderSupplier(ProhibitedStateException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalUserStateException")
                                .exceptionBuilderSupplier(IllegalUserStateException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictingOperationException")
                                .exceptionBuilderSupplier(ConflictingOperationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCommentOperationException")
                                .exceptionBuilderSupplier(InvalidCommentOperationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPasswordException")
                                .exceptionBuilderSupplier(InvalidPasswordException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedOperationException")
                                .exceptionBuilderSupplier(UnauthorizedOperationException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyLabelsException")
                                .exceptionBuilderSupplier(TooManyLabelsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DraftUploadOutOfSyncException")
                                .exceptionBuilderSupplier(DraftUploadOutOfSyncException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyCheckedOutException")
                                .exceptionBuilderSupplier(ResourceAlreadyCheckedOutException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestedEntityTooLargeException")
                                .exceptionBuilderSupplier(RequestedEntityTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeactivatingLastSystemUserException")
                                .exceptionBuilderSupplier(DeactivatingLastSystemUserException::builder).httpStatusCode(409)
                                .build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
