/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains one or more IP addresses or blocks of IP addresses specified in Classless Inter-Domain Routing (CIDR)
 * notation. AWS WAF supports any CIDR range. For information about CIDR notation, see the Wikipedia entry <a
 * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing">Classless Inter-Domain Routing</a>.
 * </p>
 * <p>
 * AWS WAF assigns an ARN to each <code>IPSet</code> that you create. To use an IP set in a rule, you provide the ARN to
 * the <a>Rule</a> statement <a>IPSetReferenceStatement</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IPSet implements SdkPojo, Serializable, ToCopyableBuilder<IPSet.Builder, IPSet> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(IPSet::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(IPSet::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ARN")
            .getter(getter(IPSet::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(IPSet::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> IP_ADDRESS_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IPAddressVersion").getter(getter(IPSet::ipAddressVersionAsString))
            .setter(setter(Builder::ipAddressVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IPAddressVersion").build()).build();

    private static final SdkField<List<String>> ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Addresses")
            .getter(getter(IPSet::addresses))
            .setter(setter(Builder::addresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Addresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ID_FIELD,
            ARN_FIELD, DESCRIPTION_FIELD, IP_ADDRESS_VERSION_FIELD, ADDRESSES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String id;

    private final String arn;

    private final String description;

    private final String ipAddressVersion;

    private final List<String> addresses;

    private IPSet(BuilderImpl builder) {
        this.name = builder.name;
        this.id = builder.id;
        this.arn = builder.arn;
        this.description = builder.description;
        this.ipAddressVersion = builder.ipAddressVersion;
        this.addresses = builder.addresses;
    }

    /**
     * <p>
     * The name of the IP set. You cannot change the name of an <code>IPSet</code> after you create it.
     * </p>
     * 
     * @return The name of the IP set. You cannot change the name of an <code>IPSet</code> after you create it.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A unique identifier for the set. This ID is returned in the responses to create and list commands. You provide it
     * to operations like update and delete.
     * </p>
     * 
     * @return A unique identifier for the set. This ID is returned in the responses to create and list commands. You
     *         provide it to operations like update and delete.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the entity.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the entity.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * A description of the IP set that helps with identification.
     * </p>
     * 
     * @return A description of the IP set that helps with identification.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Specify IPV4 or IPV6.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressVersion}
     * will return {@link IPAddressVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #ipAddressVersionAsString}.
     * </p>
     * 
     * @return Specify IPV4 or IPV6.
     * @see IPAddressVersion
     */
    public final IPAddressVersion ipAddressVersion() {
        return IPAddressVersion.fromValue(ipAddressVersion);
    }

    /**
     * <p>
     * Specify IPV4 or IPV6.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressVersion}
     * will return {@link IPAddressVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #ipAddressVersionAsString}.
     * </p>
     * 
     * @return Specify IPV4 or IPV6.
     * @see IPAddressVersion
     */
    public final String ipAddressVersionAsString() {
        return ipAddressVersion;
    }

    /**
     * Returns true if the Addresses property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasAddresses() {
        return addresses != null && !(addresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains an array of strings that specify one or more IP addresses or blocks of IP addresses in Classless
     * Inter-Domain Routing (CIDR) notation. AWS WAF supports all address ranges for IP versions IPv4 and IPv6.
     * </p>
     * <p>
     * Examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To configure AWS WAF to allow, block, or count requests that originated from the IP address 192.0.2.44, specify
     * <code>192.0.2.44/32</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To configure AWS WAF to allow, block, or count requests that originated from IP addresses from 192.0.2.0 to
     * 192.0.2.255, specify <code>192.0.2.0/24</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To configure AWS WAF to allow, block, or count requests that originated from the IP address
     * 1111:0000:0000:0000:0000:0000:0000:0111, specify <code>1111:0000:0000:0000:0000:0000:0000:0111/128</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To configure AWS WAF to allow, block, or count requests that originated from IP addresses
     * 1111:0000:0000:0000:0000:0000:0000:0000 to 1111:0000:0000:0000:ffff:ffff:ffff:ffff, specify
     * <code>1111:0000:0000:0000:0000:0000:0000:0000/64</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about CIDR notation, see the Wikipedia entry <a
     * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing">Classless Inter-Domain Routing</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAddresses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Contains an array of strings that specify one or more IP addresses or blocks of IP addresses in Classless
     *         Inter-Domain Routing (CIDR) notation. AWS WAF supports all address ranges for IP versions IPv4 and IPv6.
     *         </p>
     *         <p>
     *         Examples:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To configure AWS WAF to allow, block, or count requests that originated from the IP address 192.0.2.44,
     *         specify <code>192.0.2.44/32</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To configure AWS WAF to allow, block, or count requests that originated from IP addresses from 192.0.2.0
     *         to 192.0.2.255, specify <code>192.0.2.0/24</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To configure AWS WAF to allow, block, or count requests that originated from the IP address
     *         1111:0000:0000:0000:0000:0000:0000:0111, specify <code>1111:0000:0000:0000:0000:0000:0000:0111/128</code>
     *         .
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To configure AWS WAF to allow, block, or count requests that originated from IP addresses
     *         1111:0000:0000:0000:0000:0000:0000:0000 to 1111:0000:0000:0000:ffff:ffff:ffff:ffff, specify
     *         <code>1111:0000:0000:0000:0000:0000:0000:0000/64</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about CIDR notation, see the Wikipedia entry <a
     *         href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing">Classless Inter-Domain Routing</a>.
     */
    public final List<String> addresses() {
        return addresses;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressVersionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAddresses() ? addresses() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IPSet)) {
            return false;
        }
        IPSet other = (IPSet) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(description(), other.description())
                && Objects.equals(ipAddressVersionAsString(), other.ipAddressVersionAsString())
                && hasAddresses() == other.hasAddresses() && Objects.equals(addresses(), other.addresses());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IPSet").add("Name", name()).add("Id", id()).add("ARN", arn()).add("Description", description())
                .add("IPAddressVersion", ipAddressVersionAsString()).add("Addresses", hasAddresses() ? addresses() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "IPAddressVersion":
            return Optional.ofNullable(clazz.cast(ipAddressVersionAsString()));
        case "Addresses":
            return Optional.ofNullable(clazz.cast(addresses()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IPSet, T> g) {
        return obj -> g.apply((IPSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IPSet> {
        /**
         * <p>
         * The name of the IP set. You cannot change the name of an <code>IPSet</code> after you create it.
         * </p>
         * 
         * @param name
         *        The name of the IP set. You cannot change the name of an <code>IPSet</code> after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A unique identifier for the set. This ID is returned in the responses to create and list commands. You
         * provide it to operations like update and delete.
         * </p>
         * 
         * @param id
         *        A unique identifier for the set. This ID is returned in the responses to create and list commands. You
         *        provide it to operations like update and delete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the entity.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * A description of the IP set that helps with identification.
         * </p>
         * 
         * @param description
         *        A description of the IP set that helps with identification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Specify IPV4 or IPV6.
         * </p>
         * 
         * @param ipAddressVersion
         *        Specify IPV4 or IPV6.
         * @see IPAddressVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IPAddressVersion
         */
        Builder ipAddressVersion(String ipAddressVersion);

        /**
         * <p>
         * Specify IPV4 or IPV6.
         * </p>
         * 
         * @param ipAddressVersion
         *        Specify IPV4 or IPV6.
         * @see IPAddressVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IPAddressVersion
         */
        Builder ipAddressVersion(IPAddressVersion ipAddressVersion);

        /**
         * <p>
         * Contains an array of strings that specify one or more IP addresses or blocks of IP addresses in Classless
         * Inter-Domain Routing (CIDR) notation. AWS WAF supports all address ranges for IP versions IPv4 and IPv6.
         * </p>
         * <p>
         * Examples:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To configure AWS WAF to allow, block, or count requests that originated from the IP address 192.0.2.44,
         * specify <code>192.0.2.44/32</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To configure AWS WAF to allow, block, or count requests that originated from IP addresses from 192.0.2.0 to
         * 192.0.2.255, specify <code>192.0.2.0/24</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To configure AWS WAF to allow, block, or count requests that originated from the IP address
         * 1111:0000:0000:0000:0000:0000:0000:0111, specify <code>1111:0000:0000:0000:0000:0000:0000:0111/128</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To configure AWS WAF to allow, block, or count requests that originated from IP addresses
         * 1111:0000:0000:0000:0000:0000:0000:0000 to 1111:0000:0000:0000:ffff:ffff:ffff:ffff, specify
         * <code>1111:0000:0000:0000:0000:0000:0000:0000/64</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about CIDR notation, see the Wikipedia entry <a
         * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing">Classless Inter-Domain Routing</a>.
         * </p>
         * 
         * @param addresses
         *        Contains an array of strings that specify one or more IP addresses or blocks of IP addresses in
         *        Classless Inter-Domain Routing (CIDR) notation. AWS WAF supports all address ranges for IP versions
         *        IPv4 and IPv6. </p>
         *        <p>
         *        Examples:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To configure AWS WAF to allow, block, or count requests that originated from the IP address
         *        192.0.2.44, specify <code>192.0.2.44/32</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To configure AWS WAF to allow, block, or count requests that originated from IP addresses from
         *        192.0.2.0 to 192.0.2.255, specify <code>192.0.2.0/24</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To configure AWS WAF to allow, block, or count requests that originated from the IP address
         *        1111:0000:0000:0000:0000:0000:0000:0111, specify
         *        <code>1111:0000:0000:0000:0000:0000:0000:0111/128</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To configure AWS WAF to allow, block, or count requests that originated from IP addresses
         *        1111:0000:0000:0000:0000:0000:0000:0000 to 1111:0000:0000:0000:ffff:ffff:ffff:ffff, specify
         *        <code>1111:0000:0000:0000:0000:0000:0000:0000/64</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about CIDR notation, see the Wikipedia entry <a
         *        href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing">Classless Inter-Domain
         *        Routing</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addresses(Collection<String> addresses);

        /**
         * <p>
         * Contains an array of strings that specify one or more IP addresses or blocks of IP addresses in Classless
         * Inter-Domain Routing (CIDR) notation. AWS WAF supports all address ranges for IP versions IPv4 and IPv6.
         * </p>
         * <p>
         * Examples:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To configure AWS WAF to allow, block, or count requests that originated from the IP address 192.0.2.44,
         * specify <code>192.0.2.44/32</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To configure AWS WAF to allow, block, or count requests that originated from IP addresses from 192.0.2.0 to
         * 192.0.2.255, specify <code>192.0.2.0/24</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To configure AWS WAF to allow, block, or count requests that originated from the IP address
         * 1111:0000:0000:0000:0000:0000:0000:0111, specify <code>1111:0000:0000:0000:0000:0000:0000:0111/128</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To configure AWS WAF to allow, block, or count requests that originated from IP addresses
         * 1111:0000:0000:0000:0000:0000:0000:0000 to 1111:0000:0000:0000:ffff:ffff:ffff:ffff, specify
         * <code>1111:0000:0000:0000:0000:0000:0000:0000/64</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about CIDR notation, see the Wikipedia entry <a
         * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing">Classless Inter-Domain Routing</a>.
         * </p>
         * 
         * @param addresses
         *        Contains an array of strings that specify one or more IP addresses or blocks of IP addresses in
         *        Classless Inter-Domain Routing (CIDR) notation. AWS WAF supports all address ranges for IP versions
         *        IPv4 and IPv6. </p>
         *        <p>
         *        Examples:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To configure AWS WAF to allow, block, or count requests that originated from the IP address
         *        192.0.2.44, specify <code>192.0.2.44/32</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To configure AWS WAF to allow, block, or count requests that originated from IP addresses from
         *        192.0.2.0 to 192.0.2.255, specify <code>192.0.2.0/24</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To configure AWS WAF to allow, block, or count requests that originated from the IP address
         *        1111:0000:0000:0000:0000:0000:0000:0111, specify
         *        <code>1111:0000:0000:0000:0000:0000:0000:0111/128</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To configure AWS WAF to allow, block, or count requests that originated from IP addresses
         *        1111:0000:0000:0000:0000:0000:0000:0000 to 1111:0000:0000:0000:ffff:ffff:ffff:ffff, specify
         *        <code>1111:0000:0000:0000:0000:0000:0000:0000/64</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about CIDR notation, see the Wikipedia entry <a
         *        href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing">Classless Inter-Domain
         *        Routing</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addresses(String... addresses);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String id;

        private String arn;

        private String description;

        private String ipAddressVersion;

        private List<String> addresses = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(IPSet model) {
            name(model.name);
            id(model.id);
            arn(model.arn);
            description(model.description);
            ipAddressVersion(model.ipAddressVersion);
            addresses(model.addresses);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getIpAddressVersion() {
            return ipAddressVersion;
        }

        @Override
        public final Builder ipAddressVersion(String ipAddressVersion) {
            this.ipAddressVersion = ipAddressVersion;
            return this;
        }

        @Override
        public final Builder ipAddressVersion(IPAddressVersion ipAddressVersion) {
            this.ipAddressVersion(ipAddressVersion == null ? null : ipAddressVersion.toString());
            return this;
        }

        public final void setIpAddressVersion(String ipAddressVersion) {
            this.ipAddressVersion = ipAddressVersion;
        }

        public final Collection<String> getAddresses() {
            if (addresses instanceof SdkAutoConstructList) {
                return null;
            }
            return addresses;
        }

        @Override
        public final Builder addresses(Collection<String> addresses) {
            this.addresses = IPAddressesCopier.copy(addresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder addresses(String... addresses) {
            addresses(Arrays.asList(addresses));
            return this;
        }

        public final void setAddresses(Collection<String> addresses) {
            this.addresses = IPAddressesCopier.copy(addresses);
        }

        @Override
        public IPSet build() {
            return new IPSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
