/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is the latest version of <b>AWS WAF</b>, named AWS WAFV2, released in November, 2019. For information, including
 * how to migrate your AWS WAF resources from the prior release, see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>.
 * </p>
 * </note>
 * <p>
 * A rule group defines a collection of rules to inspect and control web requests that you can use in a <a>WebACL</a>.
 * When you create a rule group, you define an immutable capacity limit. If you update a rule group, you must stay
 * within the capacity. This allows others to reuse the rule group with confidence in its capacity requirements.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RuleGroup implements SdkPojo, Serializable, ToCopyableBuilder<RuleGroup.Builder, RuleGroup> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RuleGroup::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RuleGroup::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<Long> CAPACITY_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(RuleGroup::capacity)).setter(setter(Builder::capacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Capacity").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RuleGroup::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RuleGroup::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<Rule>> RULES_FIELD = SdkField
            .<List<Rule>> builder(MarshallingType.LIST)
            .getter(getter(RuleGroup::rules))
            .setter(setter(Builder::rules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Rules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Rule> builder(MarshallingType.SDK_POJO)
                                            .constructor(Rule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<VisibilityConfig> VISIBILITY_CONFIG_FIELD = SdkField
            .<VisibilityConfig> builder(MarshallingType.SDK_POJO).getter(getter(RuleGroup::visibilityConfig))
            .setter(setter(Builder::visibilityConfig)).constructor(VisibilityConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VisibilityConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ID_FIELD,
            CAPACITY_FIELD, ARN_FIELD, DESCRIPTION_FIELD, RULES_FIELD, VISIBILITY_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String id;

    private final Long capacity;

    private final String arn;

    private final String description;

    private final List<Rule> rules;

    private final VisibilityConfig visibilityConfig;

    private RuleGroup(BuilderImpl builder) {
        this.name = builder.name;
        this.id = builder.id;
        this.capacity = builder.capacity;
        this.arn = builder.arn;
        this.description = builder.description;
        this.rules = builder.rules;
        this.visibilityConfig = builder.visibilityConfig;
    }

    /**
     * <p>
     * A friendly name of the rule group. You cannot change the name of a rule group after you create it.
     * </p>
     * 
     * @return A friendly name of the rule group. You cannot change the name of a rule group after you create it.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A unique identifier for the rule group. This ID is returned in the responses to create and list commands. You
     * provide it to operations like update and delete.
     * </p>
     * 
     * @return A unique identifier for the rule group. This ID is returned in the responses to create and list commands.
     *         You provide it to operations like update and delete.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The web ACL capacity units (WCUs) required for this rule group.
     * </p>
     * <p>
     * When you create your own rule group, you define this, and you cannot change it after creation. When you add or
     * modify the rules in a rule group, AWS WAF enforces this limit. You can check the capacity for a set of rules
     * using <a>CheckCapacity</a>.
     * </p>
     * <p>
     * AWS WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule groups,
     * and web ACLs. AWS WAF calculates capacity differently for each rule type, to reflect the relative cost of each
     * rule. Simple rules that cost little to run use fewer WCUs than more complex rules that use more processing power.
     * Rule group capacity is fixed at creation, which helps users plan their web ACL WCU usage when they use a rule
     * group. The WCU limit for web ACLs is 1,500.
     * </p>
     * 
     * @return The web ACL capacity units (WCUs) required for this rule group.</p>
     *         <p>
     *         When you create your own rule group, you define this, and you cannot change it after creation. When you
     *         add or modify the rules in a rule group, AWS WAF enforces this limit. You can check the capacity for a
     *         set of rules using <a>CheckCapacity</a>.
     *         </p>
     *         <p>
     *         AWS WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule
     *         groups, and web ACLs. AWS WAF calculates capacity differently for each rule type, to reflect the relative
     *         cost of each rule. Simple rules that cost little to run use fewer WCUs than more complex rules that use
     *         more processing power. Rule group capacity is fixed at creation, which helps users plan their web ACL WCU
     *         usage when they use a rule group. The WCU limit for web ACLs is 1,500.
     */
    public Long capacity() {
        return capacity;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the entity.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the entity.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * A friendly description of the rule group. You cannot change the description of a rule group after you create it.
     * </p>
     * 
     * @return A friendly description of the rule group. You cannot change the description of a rule group after you
     *         create it.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the Rules property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRules() {
        return rules != null && !(rules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count. Each rule
     * includes one top-level statement that AWS WAF uses to identify matching web requests, and parameters that govern
     * how AWS WAF handles them.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count.
     *         Each rule includes one top-level statement that AWS WAF uses to identify matching web requests, and
     *         parameters that govern how AWS WAF handles them.
     */
    public List<Rule> rules() {
        return rules;
    }

    /**
     * <p>
     * Defines and enables Amazon CloudWatch metrics and web request sample collection.
     * </p>
     * 
     * @return Defines and enables Amazon CloudWatch metrics and web request sample collection.
     */
    public VisibilityConfig visibilityConfig() {
        return visibilityConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(capacity());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(rules());
        hashCode = 31 * hashCode + Objects.hashCode(visibilityConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RuleGroup)) {
            return false;
        }
        RuleGroup other = (RuleGroup) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(id(), other.id())
                && Objects.equals(capacity(), other.capacity()) && Objects.equals(arn(), other.arn())
                && Objects.equals(description(), other.description()) && Objects.equals(rules(), other.rules())
                && Objects.equals(visibilityConfig(), other.visibilityConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RuleGroup").add("Name", name()).add("Id", id()).add("Capacity", capacity()).add("ARN", arn())
                .add("Description", description()).add("Rules", rules()).add("VisibilityConfig", visibilityConfig()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Capacity":
            return Optional.ofNullable(clazz.cast(capacity()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Rules":
            return Optional.ofNullable(clazz.cast(rules()));
        case "VisibilityConfig":
            return Optional.ofNullable(clazz.cast(visibilityConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RuleGroup, T> g) {
        return obj -> g.apply((RuleGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RuleGroup> {
        /**
         * <p>
         * A friendly name of the rule group. You cannot change the name of a rule group after you create it.
         * </p>
         * 
         * @param name
         *        A friendly name of the rule group. You cannot change the name of a rule group after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A unique identifier for the rule group. This ID is returned in the responses to create and list commands. You
         * provide it to operations like update and delete.
         * </p>
         * 
         * @param id
         *        A unique identifier for the rule group. This ID is returned in the responses to create and list
         *        commands. You provide it to operations like update and delete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The web ACL capacity units (WCUs) required for this rule group.
         * </p>
         * <p>
         * When you create your own rule group, you define this, and you cannot change it after creation. When you add
         * or modify the rules in a rule group, AWS WAF enforces this limit. You can check the capacity for a set of
         * rules using <a>CheckCapacity</a>.
         * </p>
         * <p>
         * AWS WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule
         * groups, and web ACLs. AWS WAF calculates capacity differently for each rule type, to reflect the relative
         * cost of each rule. Simple rules that cost little to run use fewer WCUs than more complex rules that use more
         * processing power. Rule group capacity is fixed at creation, which helps users plan their web ACL WCU usage
         * when they use a rule group. The WCU limit for web ACLs is 1,500.
         * </p>
         * 
         * @param capacity
         *        The web ACL capacity units (WCUs) required for this rule group.</p>
         *        <p>
         *        When you create your own rule group, you define this, and you cannot change it after creation. When
         *        you add or modify the rules in a rule group, AWS WAF enforces this limit. You can check the capacity
         *        for a set of rules using <a>CheckCapacity</a>.
         *        </p>
         *        <p>
         *        AWS WAF uses WCUs to calculate and control the operating resources that are used to run your rules,
         *        rule groups, and web ACLs. AWS WAF calculates capacity differently for each rule type, to reflect the
         *        relative cost of each rule. Simple rules that cost little to run use fewer WCUs than more complex
         *        rules that use more processing power. Rule group capacity is fixed at creation, which helps users plan
         *        their web ACL WCU usage when they use a rule group. The WCU limit for web ACLs is 1,500.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacity(Long capacity);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the entity.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * A friendly description of the rule group. You cannot change the description of a rule group after you create
         * it.
         * </p>
         * 
         * @param description
         *        A friendly description of the rule group. You cannot change the description of a rule group after you
         *        create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count. Each
         * rule includes one top-level statement that AWS WAF uses to identify matching web requests, and parameters
         * that govern how AWS WAF handles them.
         * </p>
         * 
         * @param rules
         *        The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count.
         *        Each rule includes one top-level statement that AWS WAF uses to identify matching web requests, and
         *        parameters that govern how AWS WAF handles them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Collection<Rule> rules);

        /**
         * <p>
         * The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count. Each
         * rule includes one top-level statement that AWS WAF uses to identify matching web requests, and parameters
         * that govern how AWS WAF handles them.
         * </p>
         * 
         * @param rules
         *        The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count.
         *        Each rule includes one top-level statement that AWS WAF uses to identify matching web requests, and
         *        parameters that govern how AWS WAF handles them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Rule... rules);

        /**
         * <p>
         * The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count. Each
         * rule includes one top-level statement that AWS WAF uses to identify matching web requests, and parameters
         * that govern how AWS WAF handles them.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Rule>.Builder} avoiding the need to create
         * one manually via {@link List<Rule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Rule>.Builder#build()} is called immediately and its result
         * is passed to {@link #rules(List<Rule>)}.
         * 
         * @param rules
         *        a consumer that will call methods on {@link List<Rule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rules(List<Rule>)
         */
        Builder rules(Consumer<Rule.Builder>... rules);

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * 
         * @param visibilityConfig
         *        Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder visibilityConfig(VisibilityConfig visibilityConfig);

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * This is a convenience that creates an instance of the {@link VisibilityConfig.Builder} avoiding the need to
         * create one manually via {@link VisibilityConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link VisibilityConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #visibilityConfig(VisibilityConfig)}.
         * 
         * @param visibilityConfig
         *        a consumer that will call methods on {@link VisibilityConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #visibilityConfig(VisibilityConfig)
         */
        default Builder visibilityConfig(Consumer<VisibilityConfig.Builder> visibilityConfig) {
            return visibilityConfig(VisibilityConfig.builder().applyMutation(visibilityConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String id;

        private Long capacity;

        private String arn;

        private String description;

        private List<Rule> rules = DefaultSdkAutoConstructList.getInstance();

        private VisibilityConfig visibilityConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(RuleGroup model) {
            name(model.name);
            id(model.id);
            capacity(model.capacity);
            arn(model.arn);
            description(model.description);
            rules(model.rules);
            visibilityConfig(model.visibilityConfig);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final Long getCapacity() {
            return capacity;
        }

        @Override
        public final Builder capacity(Long capacity) {
            this.capacity = capacity;
            return this;
        }

        public final void setCapacity(Long capacity) {
            this.capacity = capacity;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<Rule.Builder> getRules() {
            return rules != null ? rules.stream().map(Rule::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder rules(Collection<Rule> rules) {
            this.rules = RulesCopier.copy(rules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(Rule... rules) {
            rules(Arrays.asList(rules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(Consumer<Rule.Builder>... rules) {
            rules(Stream.of(rules).map(c -> Rule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRules(Collection<Rule.BuilderImpl> rules) {
            this.rules = RulesCopier.copyFromBuilder(rules);
        }

        public final VisibilityConfig.Builder getVisibilityConfig() {
            return visibilityConfig != null ? visibilityConfig.toBuilder() : null;
        }

        @Override
        public final Builder visibilityConfig(VisibilityConfig visibilityConfig) {
            this.visibilityConfig = visibilityConfig;
            return this;
        }

        public final void setVisibilityConfig(VisibilityConfig.BuilderImpl visibilityConfig) {
            this.visibilityConfig = visibilityConfig != null ? visibilityConfig.build() : null;
        }

        @Override
        public RuleGroup build() {
            return new RuleGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
