/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is the latest version of <b>AWS WAF</b>, named AWS WAFV2, released in November, 2019. For information, including
 * how to migrate your AWS WAF resources from the prior release, see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>.
 * </p>
 * </note>
 * <p>
 * A single rule, which you can use in a <a>WebACL</a> or <a>RuleGroup</a> to identify web requests that you want to
 * allow, block, or count. Each rule includes one top-level <a>Statement</a> that AWS WAF uses to identify matching web
 * requests, and parameters that govern how AWS WAF handles them.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Rule implements SdkPojo, Serializable, ToCopyableBuilder<Rule.Builder, Rule> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Rule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Rule::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<Statement> STATEMENT_FIELD = SdkField.<Statement> builder(MarshallingType.SDK_POJO)
            .getter(getter(Rule::statement)).setter(setter(Builder::statement)).constructor(Statement::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statement").build()).build();

    private static final SdkField<RuleAction> ACTION_FIELD = SdkField.<RuleAction> builder(MarshallingType.SDK_POJO)
            .getter(getter(Rule::action)).setter(setter(Builder::action)).constructor(RuleAction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<OverrideAction> OVERRIDE_ACTION_FIELD = SdkField
            .<OverrideAction> builder(MarshallingType.SDK_POJO).getter(getter(Rule::overrideAction))
            .setter(setter(Builder::overrideAction)).constructor(OverrideAction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OverrideAction").build()).build();

    private static final SdkField<VisibilityConfig> VISIBILITY_CONFIG_FIELD = SdkField
            .<VisibilityConfig> builder(MarshallingType.SDK_POJO).getter(getter(Rule::visibilityConfig))
            .setter(setter(Builder::visibilityConfig)).constructor(VisibilityConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VisibilityConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, PRIORITY_FIELD,
            STATEMENT_FIELD, ACTION_FIELD, OVERRIDE_ACTION_FIELD, VISIBILITY_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Integer priority;

    private final Statement statement;

    private final RuleAction action;

    private final OverrideAction overrideAction;

    private final VisibilityConfig visibilityConfig;

    private Rule(BuilderImpl builder) {
        this.name = builder.name;
        this.priority = builder.priority;
        this.statement = builder.statement;
        this.action = builder.action;
        this.overrideAction = builder.overrideAction;
        this.visibilityConfig = builder.visibilityConfig;
    }

    /**
     * <p>
     * A friendly name of the rule. You can't change the name of a <code>Rule</code> after you create it.
     * </p>
     * 
     * @return A friendly name of the rule. You can't change the name of a <code>Rule</code> after you create it.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * If you define more than one <code>Rule</code> in a <code>WebACL</code>, AWS WAF evaluates each request against
     * the <code>Rules</code> in order based on the value of <code>Priority</code>. AWS WAF processes rules with lower
     * priority first. The priorities don't need to be consecutive, but they must all be different.
     * </p>
     * 
     * @return If you define more than one <code>Rule</code> in a <code>WebACL</code>, AWS WAF evaluates each request
     *         against the <code>Rules</code> in order based on the value of <code>Priority</code>. AWS WAF processes
     *         rules with lower priority first. The priorities don't need to be consecutive, but they must all be
     *         different.
     */
    public Integer priority() {
        return priority;
    }

    /**
     * <p>
     * The AWS WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
     * <a>SizeConstraintStatement</a>.
     * </p>
     * 
     * @return The AWS WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
     *         <a>SizeConstraintStatement</a>.
     */
    public Statement statement() {
        return statement;
    }

    /**
     * <p>
     * The action that AWS WAF should take on a web request when it matches the rule's statement. Settings at the web
     * ACL level can override the rule action setting.
     * </p>
     * 
     * @return The action that AWS WAF should take on a web request when it matches the rule's statement. Settings at
     *         the web ACL level can override the rule action setting.
     */
    public RuleAction action() {
        return action;
    }

    /**
     * <p>
     * The action to use to override the rule's <code>Action</code> setting. You can use no override action, in which
     * case the rule action is in effect, or count action, in which case, if the rule matches a web request, it only
     * counts the match.
     * </p>
     * 
     * @return The action to use to override the rule's <code>Action</code> setting. You can use no override action, in
     *         which case the rule action is in effect, or count action, in which case, if the rule matches a web
     *         request, it only counts the match.
     */
    public OverrideAction overrideAction() {
        return overrideAction;
    }

    /**
     * <p>
     * Defines and enables Amazon CloudWatch metrics and web request sample collection.
     * </p>
     * 
     * @return Defines and enables Amazon CloudWatch metrics and web request sample collection.
     */
    public VisibilityConfig visibilityConfig() {
        return visibilityConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(statement());
        hashCode = 31 * hashCode + Objects.hashCode(action());
        hashCode = 31 * hashCode + Objects.hashCode(overrideAction());
        hashCode = 31 * hashCode + Objects.hashCode(visibilityConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Rule)) {
            return false;
        }
        Rule other = (Rule) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(priority(), other.priority())
                && Objects.equals(statement(), other.statement()) && Objects.equals(action(), other.action())
                && Objects.equals(overrideAction(), other.overrideAction())
                && Objects.equals(visibilityConfig(), other.visibilityConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Rule").add("Name", name()).add("Priority", priority()).add("Statement", statement())
                .add("Action", action()).add("OverrideAction", overrideAction()).add("VisibilityConfig", visibilityConfig())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "Statement":
            return Optional.ofNullable(clazz.cast(statement()));
        case "Action":
            return Optional.ofNullable(clazz.cast(action()));
        case "OverrideAction":
            return Optional.ofNullable(clazz.cast(overrideAction()));
        case "VisibilityConfig":
            return Optional.ofNullable(clazz.cast(visibilityConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Rule, T> g) {
        return obj -> g.apply((Rule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Rule> {
        /**
         * <p>
         * A friendly name of the rule. You can't change the name of a <code>Rule</code> after you create it.
         * </p>
         * 
         * @param name
         *        A friendly name of the rule. You can't change the name of a <code>Rule</code> after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If you define more than one <code>Rule</code> in a <code>WebACL</code>, AWS WAF evaluates each request
         * against the <code>Rules</code> in order based on the value of <code>Priority</code>. AWS WAF processes rules
         * with lower priority first. The priorities don't need to be consecutive, but they must all be different.
         * </p>
         * 
         * @param priority
         *        If you define more than one <code>Rule</code> in a <code>WebACL</code>, AWS WAF evaluates each request
         *        against the <code>Rules</code> in order based on the value of <code>Priority</code>. AWS WAF processes
         *        rules with lower priority first. The priorities don't need to be consecutive, but they must all be
         *        different.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * The AWS WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
         * <a>SizeConstraintStatement</a>.
         * </p>
         * 
         * @param statement
         *        The AWS WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
         *        <a>SizeConstraintStatement</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statement(Statement statement);

        /**
         * <p>
         * The AWS WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
         * <a>SizeConstraintStatement</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link Statement.Builder} avoiding the need to create
         * one manually via {@link Statement#builder()}.
         *
         * When the {@link Consumer} completes, {@link Statement.Builder#build()} is called immediately and its result
         * is passed to {@link #statement(Statement)}.
         * 
         * @param statement
         *        a consumer that will call methods on {@link Statement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statement(Statement)
         */
        default Builder statement(Consumer<Statement.Builder> statement) {
            return statement(Statement.builder().applyMutation(statement).build());
        }

        /**
         * <p>
         * The action that AWS WAF should take on a web request when it matches the rule's statement. Settings at the
         * web ACL level can override the rule action setting.
         * </p>
         * 
         * @param action
         *        The action that AWS WAF should take on a web request when it matches the rule's statement. Settings at
         *        the web ACL level can override the rule action setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder action(RuleAction action);

        /**
         * <p>
         * The action that AWS WAF should take on a web request when it matches the rule's statement. Settings at the
         * web ACL level can override the rule action setting.
         * </p>
         * This is a convenience that creates an instance of the {@link RuleAction.Builder} avoiding the need to create
         * one manually via {@link RuleAction#builder()}.
         *
         * When the {@link Consumer} completes, {@link RuleAction.Builder#build()} is called immediately and its result
         * is passed to {@link #action(RuleAction)}.
         * 
         * @param action
         *        a consumer that will call methods on {@link RuleAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #action(RuleAction)
         */
        default Builder action(Consumer<RuleAction.Builder> action) {
            return action(RuleAction.builder().applyMutation(action).build());
        }

        /**
         * <p>
         * The action to use to override the rule's <code>Action</code> setting. You can use no override action, in
         * which case the rule action is in effect, or count action, in which case, if the rule matches a web request,
         * it only counts the match.
         * </p>
         * 
         * @param overrideAction
         *        The action to use to override the rule's <code>Action</code> setting. You can use no override action,
         *        in which case the rule action is in effect, or count action, in which case, if the rule matches a web
         *        request, it only counts the match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrideAction(OverrideAction overrideAction);

        /**
         * <p>
         * The action to use to override the rule's <code>Action</code> setting. You can use no override action, in
         * which case the rule action is in effect, or count action, in which case, if the rule matches a web request,
         * it only counts the match.
         * </p>
         * This is a convenience that creates an instance of the {@link OverrideAction.Builder} avoiding the need to
         * create one manually via {@link OverrideAction#builder()}.
         *
         * When the {@link Consumer} completes, {@link OverrideAction.Builder#build()} is called immediately and its
         * result is passed to {@link #overrideAction(OverrideAction)}.
         * 
         * @param overrideAction
         *        a consumer that will call methods on {@link OverrideAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overrideAction(OverrideAction)
         */
        default Builder overrideAction(Consumer<OverrideAction.Builder> overrideAction) {
            return overrideAction(OverrideAction.builder().applyMutation(overrideAction).build());
        }

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * 
         * @param visibilityConfig
         *        Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder visibilityConfig(VisibilityConfig visibilityConfig);

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * This is a convenience that creates an instance of the {@link VisibilityConfig.Builder} avoiding the need to
         * create one manually via {@link VisibilityConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link VisibilityConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #visibilityConfig(VisibilityConfig)}.
         * 
         * @param visibilityConfig
         *        a consumer that will call methods on {@link VisibilityConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #visibilityConfig(VisibilityConfig)
         */
        default Builder visibilityConfig(Consumer<VisibilityConfig.Builder> visibilityConfig) {
            return visibilityConfig(VisibilityConfig.builder().applyMutation(visibilityConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Integer priority;

        private Statement statement;

        private RuleAction action;

        private OverrideAction overrideAction;

        private VisibilityConfig visibilityConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Rule model) {
            name(model.name);
            priority(model.priority);
            statement(model.statement);
            action(model.action);
            overrideAction(model.overrideAction);
            visibilityConfig(model.visibilityConfig);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Integer getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        public final Statement.Builder getStatement() {
            return statement != null ? statement.toBuilder() : null;
        }

        @Override
        public final Builder statement(Statement statement) {
            this.statement = statement;
            return this;
        }

        public final void setStatement(Statement.BuilderImpl statement) {
            this.statement = statement != null ? statement.build() : null;
        }

        public final RuleAction.Builder getAction() {
            return action != null ? action.toBuilder() : null;
        }

        @Override
        public final Builder action(RuleAction action) {
            this.action = action;
            return this;
        }

        public final void setAction(RuleAction.BuilderImpl action) {
            this.action = action != null ? action.build() : null;
        }

        public final OverrideAction.Builder getOverrideAction() {
            return overrideAction != null ? overrideAction.toBuilder() : null;
        }

        @Override
        public final Builder overrideAction(OverrideAction overrideAction) {
            this.overrideAction = overrideAction;
            return this;
        }

        public final void setOverrideAction(OverrideAction.BuilderImpl overrideAction) {
            this.overrideAction = overrideAction != null ? overrideAction.build() : null;
        }

        public final VisibilityConfig.Builder getVisibilityConfig() {
            return visibilityConfig != null ? visibilityConfig.toBuilder() : null;
        }

        @Override
        public final Builder visibilityConfig(VisibilityConfig visibilityConfig) {
            this.visibilityConfig = visibilityConfig;
            return this;
        }

        public final void setVisibilityConfig(VisibilityConfig.BuilderImpl visibilityConfig) {
            this.visibilityConfig = visibilityConfig != null ? visibilityConfig.build() : null;
        }

        @Override
        public Rule build() {
            return new Rule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
