/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.waf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is <b>AWS WAF Classic</b> documentation. For more information, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/classic-waf-chapter.html">AWS WAF Classic</a> in the
 * developer guide.
 * </p>
 * <p>
 * <b>For the latest version of AWS WAF</b>, use the AWS WAFV2 API and see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>. With the
 * latest version, AWS WAF has a single set of endpoints for regional and global use.
 * </p>
 * </note>
 * <p>
 * Specifies a constraint on the size of a part of the web request. AWS WAF uses the <code>Size</code>,
 * <code>ComparisonOperator</code>, and <code>FieldToMatch</code> to build an expression in the form of "
 * <code>Size</code> <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is
 * true, the <code>SizeConstraint</code> is considered to match.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SizeConstraint implements SdkPojo, Serializable, ToCopyableBuilder<SizeConstraint.Builder, SizeConstraint> {
    private static final SdkField<FieldToMatch> FIELD_TO_MATCH_FIELD = SdkField.<FieldToMatch> builder(MarshallingType.SDK_POJO)
            .memberName("FieldToMatch").getter(getter(SizeConstraint::fieldToMatch)).setter(setter(Builder::fieldToMatch))
            .constructor(FieldToMatch::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FieldToMatch").build()).build();

    private static final SdkField<String> TEXT_TRANSFORMATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TextTransformation").getter(getter(SizeConstraint::textTransformationAsString))
            .setter(setter(Builder::textTransformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TextTransformation").build())
            .build();

    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComparisonOperator").getter(getter(SizeConstraint::comparisonOperatorAsString))
            .setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComparisonOperator").build())
            .build();

    private static final SdkField<Long> SIZE_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Size")
            .getter(getter(SizeConstraint::size)).setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Size").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FIELD_TO_MATCH_FIELD,
            TEXT_TRANSFORMATION_FIELD, COMPARISON_OPERATOR_FIELD, SIZE_FIELD));

    private static final long serialVersionUID = 1L;

    private final FieldToMatch fieldToMatch;

    private final String textTransformation;

    private final String comparisonOperator;

    private final Long size;

    private SizeConstraint(BuilderImpl builder) {
        this.fieldToMatch = builder.fieldToMatch;
        this.textTransformation = builder.textTransformation;
        this.comparisonOperator = builder.comparisonOperator;
        this.size = builder.size;
    }

    /**
     * <p>
     * Specifies where in a web request to look for the size constraint.
     * </p>
     * 
     * @return Specifies where in a web request to look for the size constraint.
     */
    public final FieldToMatch fieldToMatch() {
        return fieldToMatch;
    }

    /**
     * <p>
     * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort to
     * bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on <code>FieldToMatch</code>
     * before inspecting it for a match.
     * </p>
     * <p>
     * You can only specify a single type of TextTransformation.
     * </p>
     * <p>
     * Note that if you choose <code>BODY</code> for the value of <code>Type</code>, you must choose <code>NONE</code>
     * for <code>TextTransformation</code> because CloudFront forwards only the first 8192 bytes for inspection.
     * </p>
     * <p>
     * <b>NONE</b>
     * </p>
     * <p>
     * Specify <code>NONE</code> if you don't want to perform any text transformations.
     * </p>
     * <p>
     * <b>CMD_LINE</b>
     * </p>
     * <p>
     * When you're concerned that attackers are injecting an operating system command line command and using unusual
     * formatting to disguise some or all of the command, use this option to perform the following transformations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Delete the following characters: \ " ' ^
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete spaces before the following characters: / (
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace the following characters with a space: , ;
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace multiple spaces with one space
     * </p>
     * </li>
     * <li>
     * <p>
     * Convert uppercase letters (A-Z) to lowercase (a-z)
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>COMPRESS_WHITE_SPACE</b>
     * </p>
     * <p>
     * Use this option to replace the following characters with a space character (decimal 32):
     * </p>
     * <ul>
     * <li>
     * <p>
     * \f, formfeed, decimal 12
     * </p>
     * </li>
     * <li>
     * <p>
     * \t, tab, decimal 9
     * </p>
     * </li>
     * <li>
     * <p>
     * \n, newline, decimal 10
     * </p>
     * </li>
     * <li>
     * <p>
     * \r, carriage return, decimal 13
     * </p>
     * </li>
     * <li>
     * <p>
     * \v, vertical tab, decimal 11
     * </p>
     * </li>
     * <li>
     * <p>
     * non-breaking space, decimal 160
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     * </p>
     * <p>
     * <b>HTML_ENTITY_DECODE</b>
     * </p>
     * <p>
     * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
     * performs the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Replaces <code>(ampersand)quot;</code> with <code>"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>LOWERCASE</b>
     * </p>
     * <p>
     * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     * </p>
     * <p>
     * <b>URL_DECODE</b>
     * </p>
     * <p>
     * Use this option to decode a URL-encoded value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #textTransformation} will return {@link TextTransformation#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #textTransformationAsString}.
     * </p>
     * 
     * @return Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
     *         effort to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
     *         <code>FieldToMatch</code> before inspecting it for a match.</p>
     *         <p>
     *         You can only specify a single type of TextTransformation.
     *         </p>
     *         <p>
     *         Note that if you choose <code>BODY</code> for the value of <code>Type</code>, you must choose
     *         <code>NONE</code> for <code>TextTransformation</code> because CloudFront forwards only the first 8192
     *         bytes for inspection.
     *         </p>
     *         <p>
     *         <b>NONE</b>
     *         </p>
     *         <p>
     *         Specify <code>NONE</code> if you don't want to perform any text transformations.
     *         </p>
     *         <p>
     *         <b>CMD_LINE</b>
     *         </p>
     *         <p>
     *         When you're concerned that attackers are injecting an operating system command line command and using
     *         unusual formatting to disguise some or all of the command, use this option to perform the following
     *         transformations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Delete the following characters: \ " ' ^
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete spaces before the following characters: / (
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace the following characters with a space: , ;
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace multiple spaces with one space
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Convert uppercase letters (A-Z) to lowercase (a-z)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>COMPRESS_WHITE_SPACE</b>
     *         </p>
     *         <p>
     *         Use this option to replace the following characters with a space character (decimal 32):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         \f, formfeed, decimal 12
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \t, tab, decimal 9
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \n, newline, decimal 10
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \r, carriage return, decimal 13
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \v, vertical tab, decimal 11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     *         </p>
     *         <p>
     *         <b>HTML_ENTITY_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to replace HTML-encoded characters with unencoded characters.
     *         <code>HTML_ENTITY_DECODE</code> performs the following operations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)quot;</code> with <code>"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>LOWERCASE</b>
     *         </p>
     *         <p>
     *         Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     *         </p>
     *         <p>
     *         <b>URL_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to decode a URL-encoded value.
     * @see TextTransformation
     */
    public final TextTransformation textTransformation() {
        return TextTransformation.fromValue(textTransformation);
    }

    /**
     * <p>
     * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort to
     * bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on <code>FieldToMatch</code>
     * before inspecting it for a match.
     * </p>
     * <p>
     * You can only specify a single type of TextTransformation.
     * </p>
     * <p>
     * Note that if you choose <code>BODY</code> for the value of <code>Type</code>, you must choose <code>NONE</code>
     * for <code>TextTransformation</code> because CloudFront forwards only the first 8192 bytes for inspection.
     * </p>
     * <p>
     * <b>NONE</b>
     * </p>
     * <p>
     * Specify <code>NONE</code> if you don't want to perform any text transformations.
     * </p>
     * <p>
     * <b>CMD_LINE</b>
     * </p>
     * <p>
     * When you're concerned that attackers are injecting an operating system command line command and using unusual
     * formatting to disguise some or all of the command, use this option to perform the following transformations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Delete the following characters: \ " ' ^
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete spaces before the following characters: / (
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace the following characters with a space: , ;
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace multiple spaces with one space
     * </p>
     * </li>
     * <li>
     * <p>
     * Convert uppercase letters (A-Z) to lowercase (a-z)
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>COMPRESS_WHITE_SPACE</b>
     * </p>
     * <p>
     * Use this option to replace the following characters with a space character (decimal 32):
     * </p>
     * <ul>
     * <li>
     * <p>
     * \f, formfeed, decimal 12
     * </p>
     * </li>
     * <li>
     * <p>
     * \t, tab, decimal 9
     * </p>
     * </li>
     * <li>
     * <p>
     * \n, newline, decimal 10
     * </p>
     * </li>
     * <li>
     * <p>
     * \r, carriage return, decimal 13
     * </p>
     * </li>
     * <li>
     * <p>
     * \v, vertical tab, decimal 11
     * </p>
     * </li>
     * <li>
     * <p>
     * non-breaking space, decimal 160
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     * </p>
     * <p>
     * <b>HTML_ENTITY_DECODE</b>
     * </p>
     * <p>
     * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
     * performs the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Replaces <code>(ampersand)quot;</code> with <code>"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>LOWERCASE</b>
     * </p>
     * <p>
     * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     * </p>
     * <p>
     * <b>URL_DECODE</b>
     * </p>
     * <p>
     * Use this option to decode a URL-encoded value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #textTransformation} will return {@link TextTransformation#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #textTransformationAsString}.
     * </p>
     * 
     * @return Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
     *         effort to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
     *         <code>FieldToMatch</code> before inspecting it for a match.</p>
     *         <p>
     *         You can only specify a single type of TextTransformation.
     *         </p>
     *         <p>
     *         Note that if you choose <code>BODY</code> for the value of <code>Type</code>, you must choose
     *         <code>NONE</code> for <code>TextTransformation</code> because CloudFront forwards only the first 8192
     *         bytes for inspection.
     *         </p>
     *         <p>
     *         <b>NONE</b>
     *         </p>
     *         <p>
     *         Specify <code>NONE</code> if you don't want to perform any text transformations.
     *         </p>
     *         <p>
     *         <b>CMD_LINE</b>
     *         </p>
     *         <p>
     *         When you're concerned that attackers are injecting an operating system command line command and using
     *         unusual formatting to disguise some or all of the command, use this option to perform the following
     *         transformations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Delete the following characters: \ " ' ^
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete spaces before the following characters: / (
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace the following characters with a space: , ;
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace multiple spaces with one space
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Convert uppercase letters (A-Z) to lowercase (a-z)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>COMPRESS_WHITE_SPACE</b>
     *         </p>
     *         <p>
     *         Use this option to replace the following characters with a space character (decimal 32):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         \f, formfeed, decimal 12
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \t, tab, decimal 9
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \n, newline, decimal 10
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \r, carriage return, decimal 13
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \v, vertical tab, decimal 11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     *         </p>
     *         <p>
     *         <b>HTML_ENTITY_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to replace HTML-encoded characters with unencoded characters.
     *         <code>HTML_ENTITY_DECODE</code> performs the following operations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)quot;</code> with <code>"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>LOWERCASE</b>
     *         </p>
     *         <p>
     *         Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     *         </p>
     *         <p>
     *         <b>URL_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to decode a URL-encoded value.
     * @see TextTransformation
     */
    public final String textTransformationAsString() {
        return textTransformation;
    }

    /**
     * <p>
     * The type of comparison you want AWS WAF to perform. AWS WAF uses this in combination with the provided
     * <code>Size</code> and <code>FieldToMatch</code> to build an expression in the form of "<code>Size</code>
     * <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is true, the
     * <code>SizeConstraint</code> is considered to match.
     * </p>
     * <p>
     * <b>EQ</b>: Used to test if the <code>Size</code> is equal to the size of the <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>NE</b>: Used to test if the <code>Size</code> is not equal to the size of the <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>LE</b>: Used to test if the <code>Size</code> is less than or equal to the size of the
     * <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>LT</b>: Used to test if the <code>Size</code> is strictly less than the size of the <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>GE</b>: Used to test if the <code>Size</code> is greater than or equal to the size of the
     * <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>GT</b>: Used to test if the <code>Size</code> is strictly greater than the size of the
     * <code>FieldToMatch</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The type of comparison you want AWS WAF to perform. AWS WAF uses this in combination with the provided
     *         <code>Size</code> and <code>FieldToMatch</code> to build an expression in the form of "<code>Size</code>
     *         <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is true,
     *         the <code>SizeConstraint</code> is considered to match.</p>
     *         <p>
     *         <b>EQ</b>: Used to test if the <code>Size</code> is equal to the size of the <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>NE</b>: Used to test if the <code>Size</code> is not equal to the size of the
     *         <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>LE</b>: Used to test if the <code>Size</code> is less than or equal to the size of the
     *         <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>LT</b>: Used to test if the <code>Size</code> is strictly less than the size of the
     *         <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>GE</b>: Used to test if the <code>Size</code> is greater than or equal to the size of the
     *         <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>GT</b>: Used to test if the <code>Size</code> is strictly greater than the size of the
     *         <code>FieldToMatch</code>
     * @see ComparisonOperator
     */
    public final ComparisonOperator comparisonOperator() {
        return ComparisonOperator.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * The type of comparison you want AWS WAF to perform. AWS WAF uses this in combination with the provided
     * <code>Size</code> and <code>FieldToMatch</code> to build an expression in the form of "<code>Size</code>
     * <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is true, the
     * <code>SizeConstraint</code> is considered to match.
     * </p>
     * <p>
     * <b>EQ</b>: Used to test if the <code>Size</code> is equal to the size of the <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>NE</b>: Used to test if the <code>Size</code> is not equal to the size of the <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>LE</b>: Used to test if the <code>Size</code> is less than or equal to the size of the
     * <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>LT</b>: Used to test if the <code>Size</code> is strictly less than the size of the <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>GE</b>: Used to test if the <code>Size</code> is greater than or equal to the size of the
     * <code>FieldToMatch</code>
     * </p>
     * <p>
     * <b>GT</b>: Used to test if the <code>Size</code> is strictly greater than the size of the
     * <code>FieldToMatch</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The type of comparison you want AWS WAF to perform. AWS WAF uses this in combination with the provided
     *         <code>Size</code> and <code>FieldToMatch</code> to build an expression in the form of "<code>Size</code>
     *         <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is true,
     *         the <code>SizeConstraint</code> is considered to match.</p>
     *         <p>
     *         <b>EQ</b>: Used to test if the <code>Size</code> is equal to the size of the <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>NE</b>: Used to test if the <code>Size</code> is not equal to the size of the
     *         <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>LE</b>: Used to test if the <code>Size</code> is less than or equal to the size of the
     *         <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>LT</b>: Used to test if the <code>Size</code> is strictly less than the size of the
     *         <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>GE</b>: Used to test if the <code>Size</code> is greater than or equal to the size of the
     *         <code>FieldToMatch</code>
     *         </p>
     *         <p>
     *         <b>GT</b>: Used to test if the <code>Size</code> is strictly greater than the size of the
     *         <code>FieldToMatch</code>
     * @see ComparisonOperator
     */
    public final String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * <p>
     * The size in bytes that you want AWS WAF to compare against the size of the specified <code>FieldToMatch</code>.
     * AWS WAF uses this in combination with <code>ComparisonOperator</code> and <code>FieldToMatch</code> to build an
     * expression in the form of "<code>Size</code> <code>ComparisonOperator</code> size in bytes of
     * <code>FieldToMatch</code>". If that expression is true, the <code>SizeConstraint</code> is considered to match.
     * </p>
     * <p>
     * Valid values for size are 0 - 21474836480 bytes (0 - 20 GB).
     * </p>
     * <p>
     * If you specify <code>URI</code> for the value of <code>Type</code>, the / in the URI counts as one character. For
     * example, the URI <code>/logo.jpg</code> is nine characters long.
     * </p>
     * 
     * @return The size in bytes that you want AWS WAF to compare against the size of the specified
     *         <code>FieldToMatch</code>. AWS WAF uses this in combination with <code>ComparisonOperator</code> and
     *         <code>FieldToMatch</code> to build an expression in the form of "<code>Size</code>
     *         <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is true,
     *         the <code>SizeConstraint</code> is considered to match.</p>
     *         <p>
     *         Valid values for size are 0 - 21474836480 bytes (0 - 20 GB).
     *         </p>
     *         <p>
     *         If you specify <code>URI</code> for the value of <code>Type</code>, the / in the URI counts as one
     *         character. For example, the URI <code>/logo.jpg</code> is nine characters long.
     */
    public final Long size() {
        return size;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fieldToMatch());
        hashCode = 31 * hashCode + Objects.hashCode(textTransformationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(size());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SizeConstraint)) {
            return false;
        }
        SizeConstraint other = (SizeConstraint) obj;
        return Objects.equals(fieldToMatch(), other.fieldToMatch())
                && Objects.equals(textTransformationAsString(), other.textTransformationAsString())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(size(), other.size());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SizeConstraint").add("FieldToMatch", fieldToMatch())
                .add("TextTransformation", textTransformationAsString()).add("ComparisonOperator", comparisonOperatorAsString())
                .add("Size", size()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FieldToMatch":
            return Optional.ofNullable(clazz.cast(fieldToMatch()));
        case "TextTransformation":
            return Optional.ofNullable(clazz.cast(textTransformationAsString()));
        case "ComparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        case "Size":
            return Optional.ofNullable(clazz.cast(size()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SizeConstraint, T> g) {
        return obj -> g.apply((SizeConstraint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SizeConstraint> {
        /**
         * <p>
         * Specifies where in a web request to look for the size constraint.
         * </p>
         * 
         * @param fieldToMatch
         *        Specifies where in a web request to look for the size constraint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldToMatch(FieldToMatch fieldToMatch);

        /**
         * <p>
         * Specifies where in a web request to look for the size constraint.
         * </p>
         * This is a convenience method that creates an instance of the {@link FieldToMatch.Builder} avoiding the need
         * to create one manually via {@link FieldToMatch#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FieldToMatch.Builder#build()} is called immediately and its
         * result is passed to {@link #fieldToMatch(FieldToMatch)}.
         * 
         * @param fieldToMatch
         *        a consumer that will call methods on {@link FieldToMatch.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fieldToMatch(FieldToMatch)
         */
        default Builder fieldToMatch(Consumer<FieldToMatch.Builder> fieldToMatch) {
            return fieldToMatch(FieldToMatch.builder().applyMutation(fieldToMatch).build());
        }

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
         * <code>FieldToMatch</code> before inspecting it for a match.
         * </p>
         * <p>
         * You can only specify a single type of TextTransformation.
         * </p>
         * <p>
         * Note that if you choose <code>BODY</code> for the value of <code>Type</code>, you must choose
         * <code>NONE</code> for <code>TextTransformation</code> because CloudFront forwards only the first 8192 bytes
         * for inspection.
         * </p>
         * <p>
         * <b>NONE</b>
         * </p>
         * <p>
         * Specify <code>NONE</code> if you don't want to perform any text transformations.
         * </p>
         * <p>
         * <b>CMD_LINE</b>
         * </p>
         * <p>
         * When you're concerned that attackers are injecting an operating system command line command and using unusual
         * formatting to disguise some or all of the command, use this option to perform the following transformations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Delete the following characters: \ " ' ^
         * </p>
         * </li>
         * <li>
         * <p>
         * Delete spaces before the following characters: / (
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace the following characters with a space: , ;
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace multiple spaces with one space
         * </p>
         * </li>
         * <li>
         * <p>
         * Convert uppercase letters (A-Z) to lowercase (a-z)
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>COMPRESS_WHITE_SPACE</b>
         * </p>
         * <p>
         * Use this option to replace the following characters with a space character (decimal 32):
         * </p>
         * <ul>
         * <li>
         * <p>
         * \f, formfeed, decimal 12
         * </p>
         * </li>
         * <li>
         * <p>
         * \t, tab, decimal 9
         * </p>
         * </li>
         * <li>
         * <p>
         * \n, newline, decimal 10
         * </p>
         * </li>
         * <li>
         * <p>
         * \r, carriage return, decimal 13
         * </p>
         * </li>
         * <li>
         * <p>
         * \v, vertical tab, decimal 11
         * </p>
         * </li>
         * <li>
         * <p>
         * non-breaking space, decimal 160
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         * </p>
         * <p>
         * <b>HTML_ENTITY_DECODE</b>
         * </p>
         * <p>
         * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
         * performs the following operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Replaces <code>(ampersand)quot;</code> with <code>"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>LOWERCASE</b>
         * </p>
         * <p>
         * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         * </p>
         * <p>
         * <b>URL_DECODE</b>
         * </p>
         * <p>
         * Use this option to decode a URL-encoded value.
         * </p>
         * 
         * @param textTransformation
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
         *        <code>FieldToMatch</code> before inspecting it for a match.</p>
         *        <p>
         *        You can only specify a single type of TextTransformation.
         *        </p>
         *        <p>
         *        Note that if you choose <code>BODY</code> for the value of <code>Type</code>, you must choose
         *        <code>NONE</code> for <code>TextTransformation</code> because CloudFront forwards only the first 8192
         *        bytes for inspection.
         *        </p>
         *        <p>
         *        <b>NONE</b>
         *        </p>
         *        <p>
         *        Specify <code>NONE</code> if you don't want to perform any text transformations.
         *        </p>
         *        <p>
         *        <b>CMD_LINE</b>
         *        </p>
         *        <p>
         *        When you're concerned that attackers are injecting an operating system command line command and using
         *        unusual formatting to disguise some or all of the command, use this option to perform the following
         *        transformations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Delete the following characters: \ " ' ^
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delete spaces before the following characters: / (
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace the following characters with a space: , ;
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace multiple spaces with one space
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Convert uppercase letters (A-Z) to lowercase (a-z)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>COMPRESS_WHITE_SPACE</b>
         *        </p>
         *        <p>
         *        Use this option to replace the following characters with a space character (decimal 32):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        \f, formfeed, decimal 12
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \t, tab, decimal 9
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \n, newline, decimal 10
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \r, carriage return, decimal 13
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \v, vertical tab, decimal 11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         *        </p>
         *        <p>
         *        <b>HTML_ENTITY_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to replace HTML-encoded characters with unencoded characters.
         *        <code>HTML_ENTITY_DECODE</code> performs the following operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)quot;</code> with <code>"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with
         *        the corresponding characters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         *        corresponding characters
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>LOWERCASE</b>
         *        </p>
         *        <p>
         *        Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         *        </p>
         *        <p>
         *        <b>URL_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to decode a URL-encoded value.
         * @see TextTransformation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTransformation
         */
        Builder textTransformation(String textTransformation);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
         * <code>FieldToMatch</code> before inspecting it for a match.
         * </p>
         * <p>
         * You can only specify a single type of TextTransformation.
         * </p>
         * <p>
         * Note that if you choose <code>BODY</code> for the value of <code>Type</code>, you must choose
         * <code>NONE</code> for <code>TextTransformation</code> because CloudFront forwards only the first 8192 bytes
         * for inspection.
         * </p>
         * <p>
         * <b>NONE</b>
         * </p>
         * <p>
         * Specify <code>NONE</code> if you don't want to perform any text transformations.
         * </p>
         * <p>
         * <b>CMD_LINE</b>
         * </p>
         * <p>
         * When you're concerned that attackers are injecting an operating system command line command and using unusual
         * formatting to disguise some or all of the command, use this option to perform the following transformations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Delete the following characters: \ " ' ^
         * </p>
         * </li>
         * <li>
         * <p>
         * Delete spaces before the following characters: / (
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace the following characters with a space: , ;
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace multiple spaces with one space
         * </p>
         * </li>
         * <li>
         * <p>
         * Convert uppercase letters (A-Z) to lowercase (a-z)
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>COMPRESS_WHITE_SPACE</b>
         * </p>
         * <p>
         * Use this option to replace the following characters with a space character (decimal 32):
         * </p>
         * <ul>
         * <li>
         * <p>
         * \f, formfeed, decimal 12
         * </p>
         * </li>
         * <li>
         * <p>
         * \t, tab, decimal 9
         * </p>
         * </li>
         * <li>
         * <p>
         * \n, newline, decimal 10
         * </p>
         * </li>
         * <li>
         * <p>
         * \r, carriage return, decimal 13
         * </p>
         * </li>
         * <li>
         * <p>
         * \v, vertical tab, decimal 11
         * </p>
         * </li>
         * <li>
         * <p>
         * non-breaking space, decimal 160
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         * </p>
         * <p>
         * <b>HTML_ENTITY_DECODE</b>
         * </p>
         * <p>
         * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
         * performs the following operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Replaces <code>(ampersand)quot;</code> with <code>"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>LOWERCASE</b>
         * </p>
         * <p>
         * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         * </p>
         * <p>
         * <b>URL_DECODE</b>
         * </p>
         * <p>
         * Use this option to decode a URL-encoded value.
         * </p>
         * 
         * @param textTransformation
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
         *        <code>FieldToMatch</code> before inspecting it for a match.</p>
         *        <p>
         *        You can only specify a single type of TextTransformation.
         *        </p>
         *        <p>
         *        Note that if you choose <code>BODY</code> for the value of <code>Type</code>, you must choose
         *        <code>NONE</code> for <code>TextTransformation</code> because CloudFront forwards only the first 8192
         *        bytes for inspection.
         *        </p>
         *        <p>
         *        <b>NONE</b>
         *        </p>
         *        <p>
         *        Specify <code>NONE</code> if you don't want to perform any text transformations.
         *        </p>
         *        <p>
         *        <b>CMD_LINE</b>
         *        </p>
         *        <p>
         *        When you're concerned that attackers are injecting an operating system command line command and using
         *        unusual formatting to disguise some or all of the command, use this option to perform the following
         *        transformations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Delete the following characters: \ " ' ^
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delete spaces before the following characters: / (
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace the following characters with a space: , ;
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace multiple spaces with one space
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Convert uppercase letters (A-Z) to lowercase (a-z)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>COMPRESS_WHITE_SPACE</b>
         *        </p>
         *        <p>
         *        Use this option to replace the following characters with a space character (decimal 32):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        \f, formfeed, decimal 12
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \t, tab, decimal 9
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \n, newline, decimal 10
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \r, carriage return, decimal 13
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \v, vertical tab, decimal 11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         *        </p>
         *        <p>
         *        <b>HTML_ENTITY_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to replace HTML-encoded characters with unencoded characters.
         *        <code>HTML_ENTITY_DECODE</code> performs the following operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)quot;</code> with <code>"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with
         *        the corresponding characters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         *        corresponding characters
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>LOWERCASE</b>
         *        </p>
         *        <p>
         *        Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         *        </p>
         *        <p>
         *        <b>URL_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to decode a URL-encoded value.
         * @see TextTransformation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTransformation
         */
        Builder textTransformation(TextTransformation textTransformation);

        /**
         * <p>
         * The type of comparison you want AWS WAF to perform. AWS WAF uses this in combination with the provided
         * <code>Size</code> and <code>FieldToMatch</code> to build an expression in the form of "<code>Size</code>
         * <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is true, the
         * <code>SizeConstraint</code> is considered to match.
         * </p>
         * <p>
         * <b>EQ</b>: Used to test if the <code>Size</code> is equal to the size of the <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>NE</b>: Used to test if the <code>Size</code> is not equal to the size of the <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>LE</b>: Used to test if the <code>Size</code> is less than or equal to the size of the
         * <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>LT</b>: Used to test if the <code>Size</code> is strictly less than the size of the
         * <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>GE</b>: Used to test if the <code>Size</code> is greater than or equal to the size of the
         * <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>GT</b>: Used to test if the <code>Size</code> is strictly greater than the size of the
         * <code>FieldToMatch</code>
         * </p>
         * 
         * @param comparisonOperator
         *        The type of comparison you want AWS WAF to perform. AWS WAF uses this in combination with the provided
         *        <code>Size</code> and <code>FieldToMatch</code> to build an expression in the form of "
         *        <code>Size</code> <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that
         *        expression is true, the <code>SizeConstraint</code> is considered to match.</p>
         *        <p>
         *        <b>EQ</b>: Used to test if the <code>Size</code> is equal to the size of the <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>NE</b>: Used to test if the <code>Size</code> is not equal to the size of the
         *        <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>LE</b>: Used to test if the <code>Size</code> is less than or equal to the size of the
         *        <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>LT</b>: Used to test if the <code>Size</code> is strictly less than the size of the
         *        <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>GE</b>: Used to test if the <code>Size</code> is greater than or equal to the size of the
         *        <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>GT</b>: Used to test if the <code>Size</code> is strictly greater than the size of the
         *        <code>FieldToMatch</code>
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * The type of comparison you want AWS WAF to perform. AWS WAF uses this in combination with the provided
         * <code>Size</code> and <code>FieldToMatch</code> to build an expression in the form of "<code>Size</code>
         * <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is true, the
         * <code>SizeConstraint</code> is considered to match.
         * </p>
         * <p>
         * <b>EQ</b>: Used to test if the <code>Size</code> is equal to the size of the <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>NE</b>: Used to test if the <code>Size</code> is not equal to the size of the <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>LE</b>: Used to test if the <code>Size</code> is less than or equal to the size of the
         * <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>LT</b>: Used to test if the <code>Size</code> is strictly less than the size of the
         * <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>GE</b>: Used to test if the <code>Size</code> is greater than or equal to the size of the
         * <code>FieldToMatch</code>
         * </p>
         * <p>
         * <b>GT</b>: Used to test if the <code>Size</code> is strictly greater than the size of the
         * <code>FieldToMatch</code>
         * </p>
         * 
         * @param comparisonOperator
         *        The type of comparison you want AWS WAF to perform. AWS WAF uses this in combination with the provided
         *        <code>Size</code> and <code>FieldToMatch</code> to build an expression in the form of "
         *        <code>Size</code> <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that
         *        expression is true, the <code>SizeConstraint</code> is considered to match.</p>
         *        <p>
         *        <b>EQ</b>: Used to test if the <code>Size</code> is equal to the size of the <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>NE</b>: Used to test if the <code>Size</code> is not equal to the size of the
         *        <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>LE</b>: Used to test if the <code>Size</code> is less than or equal to the size of the
         *        <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>LT</b>: Used to test if the <code>Size</code> is strictly less than the size of the
         *        <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>GE</b>: Used to test if the <code>Size</code> is greater than or equal to the size of the
         *        <code>FieldToMatch</code>
         *        </p>
         *        <p>
         *        <b>GT</b>: Used to test if the <code>Size</code> is strictly greater than the size of the
         *        <code>FieldToMatch</code>
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);

        /**
         * <p>
         * The size in bytes that you want AWS WAF to compare against the size of the specified
         * <code>FieldToMatch</code>. AWS WAF uses this in combination with <code>ComparisonOperator</code> and
         * <code>FieldToMatch</code> to build an expression in the form of "<code>Size</code>
         * <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is true, the
         * <code>SizeConstraint</code> is considered to match.
         * </p>
         * <p>
         * Valid values for size are 0 - 21474836480 bytes (0 - 20 GB).
         * </p>
         * <p>
         * If you specify <code>URI</code> for the value of <code>Type</code>, the / in the URI counts as one character.
         * For example, the URI <code>/logo.jpg</code> is nine characters long.
         * </p>
         * 
         * @param size
         *        The size in bytes that you want AWS WAF to compare against the size of the specified
         *        <code>FieldToMatch</code>. AWS WAF uses this in combination with <code>ComparisonOperator</code> and
         *        <code>FieldToMatch</code> to build an expression in the form of "<code>Size</code>
         *        <code>ComparisonOperator</code> size in bytes of <code>FieldToMatch</code>". If that expression is
         *        true, the <code>SizeConstraint</code> is considered to match.</p>
         *        <p>
         *        Valid values for size are 0 - 21474836480 bytes (0 - 20 GB).
         *        </p>
         *        <p>
         *        If you specify <code>URI</code> for the value of <code>Type</code>, the / in the URI counts as one
         *        character. For example, the URI <code>/logo.jpg</code> is nine characters long.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder size(Long size);
    }

    static final class BuilderImpl implements Builder {
        private FieldToMatch fieldToMatch;

        private String textTransformation;

        private String comparisonOperator;

        private Long size;

        private BuilderImpl() {
        }

        private BuilderImpl(SizeConstraint model) {
            fieldToMatch(model.fieldToMatch);
            textTransformation(model.textTransformation);
            comparisonOperator(model.comparisonOperator);
            size(model.size);
        }

        public final FieldToMatch.Builder getFieldToMatch() {
            return fieldToMatch != null ? fieldToMatch.toBuilder() : null;
        }

        public final void setFieldToMatch(FieldToMatch.BuilderImpl fieldToMatch) {
            this.fieldToMatch = fieldToMatch != null ? fieldToMatch.build() : null;
        }

        @Override
        public final Builder fieldToMatch(FieldToMatch fieldToMatch) {
            this.fieldToMatch = fieldToMatch;
            return this;
        }

        public final String getTextTransformation() {
            return textTransformation;
        }

        public final void setTextTransformation(String textTransformation) {
            this.textTransformation = textTransformation;
        }

        @Override
        public final Builder textTransformation(String textTransformation) {
            this.textTransformation = textTransformation;
            return this;
        }

        @Override
        public final Builder textTransformation(TextTransformation textTransformation) {
            this.textTransformation(textTransformation == null ? null : textTransformation.toString());
            return this;
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        public final Long getSize() {
            return size;
        }

        public final void setSize(Long size) {
            this.size = size;
        }

        @Override
        public final Builder size(Long size) {
            this.size = size;
            return this;
        }

        @Override
        public SizeConstraint build() {
            return new SizeConstraint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
