/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.waf.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is <b>AWS WAF Classic</b> documentation. For more information, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/classic-waf-chapter.html">AWS WAF Classic</a> in the
 * developer guide.
 * </p>
 * <p>
 * <b>For the latest version of AWS WAF</b>, use the AWS WAFV2 API and see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>. With the
 * latest version, AWS WAF has a single set of endpoints for regional and global use.
 * </p>
 * </note>
 * <p>
 * The bytes (typically a string that corresponds with ASCII characters) that you want AWS WAF to search for in web
 * requests, the location in requests that you want AWS WAF to search, and other settings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ByteMatchTuple implements SdkPojo, Serializable, ToCopyableBuilder<ByteMatchTuple.Builder, ByteMatchTuple> {
    private static final SdkField<FieldToMatch> FIELD_TO_MATCH_FIELD = SdkField.<FieldToMatch> builder(MarshallingType.SDK_POJO)
            .memberName("FieldToMatch").getter(getter(ByteMatchTuple::fieldToMatch)).setter(setter(Builder::fieldToMatch))
            .constructor(FieldToMatch::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FieldToMatch").build()).build();

    private static final SdkField<SdkBytes> TARGET_STRING_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("TargetString").getter(getter(ByteMatchTuple::targetString)).setter(setter(Builder::targetString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetString").build()).build();

    private static final SdkField<String> TEXT_TRANSFORMATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TextTransformation").getter(getter(ByteMatchTuple::textTransformationAsString))
            .setter(setter(Builder::textTransformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TextTransformation").build())
            .build();

    private static final SdkField<String> POSITIONAL_CONSTRAINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PositionalConstraint").getter(getter(ByteMatchTuple::positionalConstraintAsString))
            .setter(setter(Builder::positionalConstraint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PositionalConstraint").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FIELD_TO_MATCH_FIELD,
            TARGET_STRING_FIELD, TEXT_TRANSFORMATION_FIELD, POSITIONAL_CONSTRAINT_FIELD));

    private static final long serialVersionUID = 1L;

    private final FieldToMatch fieldToMatch;

    private final SdkBytes targetString;

    private final String textTransformation;

    private final String positionalConstraint;

    private ByteMatchTuple(BuilderImpl builder) {
        this.fieldToMatch = builder.fieldToMatch;
        this.targetString = builder.targetString;
        this.textTransformation = builder.textTransformation;
        this.positionalConstraint = builder.positionalConstraint;
    }

    /**
     * <p>
     * The part of a web request that you want AWS WAF to search, such as a specified header or a query string. For more
     * information, see <a>FieldToMatch</a>.
     * </p>
     * 
     * @return The part of a web request that you want AWS WAF to search, such as a specified header or a query string.
     *         For more information, see <a>FieldToMatch</a>.
     */
    public final FieldToMatch fieldToMatch() {
        return fieldToMatch;
    }

    /**
     * <p>
     * The value that you want AWS WAF to search for. AWS WAF searches for the specified string in the part of web
     * requests that you specified in <code>FieldToMatch</code>. The maximum length of the value is 50 bytes.
     * </p>
     * <p>
     * Valid values depend on the values that you specified for <code>FieldToMatch</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HEADER</code>: The value that you want AWS WAF to search for in the request header that you specified in
     * <a>FieldToMatch</a>, for example, the value of the <code>User-Agent</code> or <code>Referer</code> header.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>METHOD</code>: The HTTP method, which indicates the type of operation specified in the request. CloudFront
     * supports the following methods: <code>DELETE</code>, <code>GET</code>, <code>HEAD</code>, <code>OPTIONS</code>,
     * <code>PATCH</code>, <code>POST</code>, and <code>PUT</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>QUERY_STRING</code>: The value that you want AWS WAF to search for in the query string, which is the part
     * of a URL that appears after a <code>?</code> character.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>URI</code>: The value that you want AWS WAF to search for in the part of a URL that identifies a resource,
     * for example, <code>/images/daily-ad.jpg</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BODY</code>: The part of a request that contains any additional data that you want to send to your web
     * server as the HTTP request body, such as data from a form. The request body immediately follows the request
     * headers. Note that only the first <code>8192</code> bytes of the request body are forwarded to AWS WAF for
     * inspection. To allow or block requests based on the length of the body, you can create a size constraint set. For
     * more information, see <a>CreateSizeConstraintSet</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_QUERY_ARG</code>: The parameter in the query string that you will inspect, such as <i>UserName</i>
     * or <i>SalesRegion</i>. The maximum length for <code>SINGLE_QUERY_ARG</code> is 30 characters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL_QUERY_ARGS</code>: Similar to <code>SINGLE_QUERY_ARG</code>, but instead of inspecting a single
     * parameter, AWS WAF inspects all parameters within the query string for the value or regex pattern that you
     * specify in <code>TargetString</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If <code>TargetString</code> includes alphabetic characters A-Z and a-z, note that the value is case sensitive.
     * </p>
     * <p>
     * <b>If you're using the AWS WAF API</b>
     * </p>
     * <p>
     * Specify a base64-encoded version of the value. The maximum length of the value before you base64-encode it is 50
     * bytes.
     * </p>
     * <p>
     * For example, suppose the value of <code>Type</code> is <code>HEADER</code> and the value of <code>Data</code> is
     * <code>User-Agent</code>. If you want to search the <code>User-Agent</code> header for the value
     * <code>BadBot</code>, you base64-encode <code>BadBot</code> using MIME base64-encoding and include the resulting
     * value, <code>QmFkQm90</code>, in the value of <code>TargetString</code>.
     * </p>
     * <p>
     * <b>If you're using the AWS CLI or one of the AWS SDKs</b>
     * </p>
     * <p>
     * The value that you want AWS WAF to search for. The SDK automatically base64 encodes the value.
     * </p>
     * 
     * @return The value that you want AWS WAF to search for. AWS WAF searches for the specified string in the part of
     *         web requests that you specified in <code>FieldToMatch</code>. The maximum length of the value is 50
     *         bytes.</p>
     *         <p>
     *         Valid values depend on the values that you specified for <code>FieldToMatch</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HEADER</code>: The value that you want AWS WAF to search for in the request header that you
     *         specified in <a>FieldToMatch</a>, for example, the value of the <code>User-Agent</code> or
     *         <code>Referer</code> header.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>METHOD</code>: The HTTP method, which indicates the type of operation specified in the request.
     *         CloudFront supports the following methods: <code>DELETE</code>, <code>GET</code>, <code>HEAD</code>,
     *         <code>OPTIONS</code>, <code>PATCH</code>, <code>POST</code>, and <code>PUT</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>QUERY_STRING</code>: The value that you want AWS WAF to search for in the query string, which is
     *         the part of a URL that appears after a <code>?</code> character.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>URI</code>: The value that you want AWS WAF to search for in the part of a URL that identifies a
     *         resource, for example, <code>/images/daily-ad.jpg</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BODY</code>: The part of a request that contains any additional data that you want to send to your
     *         web server as the HTTP request body, such as data from a form. The request body immediately follows the
     *         request headers. Note that only the first <code>8192</code> bytes of the request body are forwarded to
     *         AWS WAF for inspection. To allow or block requests based on the length of the body, you can create a size
     *         constraint set. For more information, see <a>CreateSizeConstraintSet</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_QUERY_ARG</code>: The parameter in the query string that you will inspect, such as
     *         <i>UserName</i> or <i>SalesRegion</i>. The maximum length for <code>SINGLE_QUERY_ARG</code> is 30
     *         characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL_QUERY_ARGS</code>: Similar to <code>SINGLE_QUERY_ARG</code>, but instead of inspecting a single
     *         parameter, AWS WAF inspects all parameters within the query string for the value or regex pattern that
     *         you specify in <code>TargetString</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If <code>TargetString</code> includes alphabetic characters A-Z and a-z, note that the value is case
     *         sensitive.
     *         </p>
     *         <p>
     *         <b>If you're using the AWS WAF API</b>
     *         </p>
     *         <p>
     *         Specify a base64-encoded version of the value. The maximum length of the value before you base64-encode
     *         it is 50 bytes.
     *         </p>
     *         <p>
     *         For example, suppose the value of <code>Type</code> is <code>HEADER</code> and the value of
     *         <code>Data</code> is <code>User-Agent</code>. If you want to search the <code>User-Agent</code> header
     *         for the value <code>BadBot</code>, you base64-encode <code>BadBot</code> using MIME base64-encoding and
     *         include the resulting value, <code>QmFkQm90</code>, in the value of <code>TargetString</code>.
     *         </p>
     *         <p>
     *         <b>If you're using the AWS CLI or one of the AWS SDKs</b>
     *         </p>
     *         <p>
     *         The value that you want AWS WAF to search for. The SDK automatically base64 encodes the value.
     */
    public final SdkBytes targetString() {
        return targetString;
    }

    /**
     * <p>
     * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort to
     * bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on <code>FieldToMatch</code>
     * before inspecting it for a match.
     * </p>
     * <p>
     * You can only specify a single type of TextTransformation.
     * </p>
     * <p>
     * <b>CMD_LINE</b>
     * </p>
     * <p>
     * When you're concerned that attackers are injecting an operating system command line command and using unusual
     * formatting to disguise some or all of the command, use this option to perform the following transformations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Delete the following characters: \ " ' ^
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete spaces before the following characters: / (
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace the following characters with a space: , ;
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace multiple spaces with one space
     * </p>
     * </li>
     * <li>
     * <p>
     * Convert uppercase letters (A-Z) to lowercase (a-z)
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>COMPRESS_WHITE_SPACE</b>
     * </p>
     * <p>
     * Use this option to replace the following characters with a space character (decimal 32):
     * </p>
     * <ul>
     * <li>
     * <p>
     * \f, formfeed, decimal 12
     * </p>
     * </li>
     * <li>
     * <p>
     * \t, tab, decimal 9
     * </p>
     * </li>
     * <li>
     * <p>
     * \n, newline, decimal 10
     * </p>
     * </li>
     * <li>
     * <p>
     * \r, carriage return, decimal 13
     * </p>
     * </li>
     * <li>
     * <p>
     * \v, vertical tab, decimal 11
     * </p>
     * </li>
     * <li>
     * <p>
     * non-breaking space, decimal 160
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     * </p>
     * <p>
     * <b>HTML_ENTITY_DECODE</b>
     * </p>
     * <p>
     * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
     * performs the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Replaces <code>(ampersand)quot;</code> with <code>"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>LOWERCASE</b>
     * </p>
     * <p>
     * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     * </p>
     * <p>
     * <b>URL_DECODE</b>
     * </p>
     * <p>
     * Use this option to decode a URL-encoded value.
     * </p>
     * <p>
     * <b>NONE</b>
     * </p>
     * <p>
     * Specify <code>NONE</code> if you don't want to perform any text transformations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #textTransformation} will return {@link TextTransformation#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #textTransformationAsString}.
     * </p>
     * 
     * @return Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
     *         effort to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
     *         <code>FieldToMatch</code> before inspecting it for a match.</p>
     *         <p>
     *         You can only specify a single type of TextTransformation.
     *         </p>
     *         <p>
     *         <b>CMD_LINE</b>
     *         </p>
     *         <p>
     *         When you're concerned that attackers are injecting an operating system command line command and using
     *         unusual formatting to disguise some or all of the command, use this option to perform the following
     *         transformations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Delete the following characters: \ " ' ^
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete spaces before the following characters: / (
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace the following characters with a space: , ;
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace multiple spaces with one space
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Convert uppercase letters (A-Z) to lowercase (a-z)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>COMPRESS_WHITE_SPACE</b>
     *         </p>
     *         <p>
     *         Use this option to replace the following characters with a space character (decimal 32):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         \f, formfeed, decimal 12
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \t, tab, decimal 9
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \n, newline, decimal 10
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \r, carriage return, decimal 13
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \v, vertical tab, decimal 11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     *         </p>
     *         <p>
     *         <b>HTML_ENTITY_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to replace HTML-encoded characters with unencoded characters.
     *         <code>HTML_ENTITY_DECODE</code> performs the following operations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)quot;</code> with <code>"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>LOWERCASE</b>
     *         </p>
     *         <p>
     *         Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     *         </p>
     *         <p>
     *         <b>URL_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to decode a URL-encoded value.
     *         </p>
     *         <p>
     *         <b>NONE</b>
     *         </p>
     *         <p>
     *         Specify <code>NONE</code> if you don't want to perform any text transformations.
     * @see TextTransformation
     */
    public final TextTransformation textTransformation() {
        return TextTransformation.fromValue(textTransformation);
    }

    /**
     * <p>
     * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort to
     * bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on <code>FieldToMatch</code>
     * before inspecting it for a match.
     * </p>
     * <p>
     * You can only specify a single type of TextTransformation.
     * </p>
     * <p>
     * <b>CMD_LINE</b>
     * </p>
     * <p>
     * When you're concerned that attackers are injecting an operating system command line command and using unusual
     * formatting to disguise some or all of the command, use this option to perform the following transformations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Delete the following characters: \ " ' ^
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete spaces before the following characters: / (
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace the following characters with a space: , ;
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace multiple spaces with one space
     * </p>
     * </li>
     * <li>
     * <p>
     * Convert uppercase letters (A-Z) to lowercase (a-z)
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>COMPRESS_WHITE_SPACE</b>
     * </p>
     * <p>
     * Use this option to replace the following characters with a space character (decimal 32):
     * </p>
     * <ul>
     * <li>
     * <p>
     * \f, formfeed, decimal 12
     * </p>
     * </li>
     * <li>
     * <p>
     * \t, tab, decimal 9
     * </p>
     * </li>
     * <li>
     * <p>
     * \n, newline, decimal 10
     * </p>
     * </li>
     * <li>
     * <p>
     * \r, carriage return, decimal 13
     * </p>
     * </li>
     * <li>
     * <p>
     * \v, vertical tab, decimal 11
     * </p>
     * </li>
     * <li>
     * <p>
     * non-breaking space, decimal 160
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     * </p>
     * <p>
     * <b>HTML_ENTITY_DECODE</b>
     * </p>
     * <p>
     * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
     * performs the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Replaces <code>(ampersand)quot;</code> with <code>"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>LOWERCASE</b>
     * </p>
     * <p>
     * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     * </p>
     * <p>
     * <b>URL_DECODE</b>
     * </p>
     * <p>
     * Use this option to decode a URL-encoded value.
     * </p>
     * <p>
     * <b>NONE</b>
     * </p>
     * <p>
     * Specify <code>NONE</code> if you don't want to perform any text transformations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #textTransformation} will return {@link TextTransformation#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #textTransformationAsString}.
     * </p>
     * 
     * @return Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
     *         effort to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
     *         <code>FieldToMatch</code> before inspecting it for a match.</p>
     *         <p>
     *         You can only specify a single type of TextTransformation.
     *         </p>
     *         <p>
     *         <b>CMD_LINE</b>
     *         </p>
     *         <p>
     *         When you're concerned that attackers are injecting an operating system command line command and using
     *         unusual formatting to disguise some or all of the command, use this option to perform the following
     *         transformations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Delete the following characters: \ " ' ^
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete spaces before the following characters: / (
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace the following characters with a space: , ;
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace multiple spaces with one space
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Convert uppercase letters (A-Z) to lowercase (a-z)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>COMPRESS_WHITE_SPACE</b>
     *         </p>
     *         <p>
     *         Use this option to replace the following characters with a space character (decimal 32):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         \f, formfeed, decimal 12
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \t, tab, decimal 9
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \n, newline, decimal 10
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \r, carriage return, decimal 13
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \v, vertical tab, decimal 11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     *         </p>
     *         <p>
     *         <b>HTML_ENTITY_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to replace HTML-encoded characters with unencoded characters.
     *         <code>HTML_ENTITY_DECODE</code> performs the following operations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)quot;</code> with <code>"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>LOWERCASE</b>
     *         </p>
     *         <p>
     *         Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     *         </p>
     *         <p>
     *         <b>URL_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to decode a URL-encoded value.
     *         </p>
     *         <p>
     *         <b>NONE</b>
     *         </p>
     *         <p>
     *         Specify <code>NONE</code> if you don't want to perform any text transformations.
     * @see TextTransformation
     */
    public final String textTransformationAsString() {
        return textTransformation;
    }

    /**
     * <p>
     * Within the portion of a web request that you want to search (for example, in the query string, if any), specify
     * where you want AWS WAF to search. Valid values include the following:
     * </p>
     * <p>
     * <b>CONTAINS</b>
     * </p>
     * <p>
     * The specified part of the web request must include the value of <code>TargetString</code>, but the location
     * doesn't matter.
     * </p>
     * <p>
     * <b>CONTAINS_WORD</b>
     * </p>
     * <p>
     * The specified part of the web request must include the value of <code>TargetString</code>, and
     * <code>TargetString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _). In
     * addition, <code>TargetString</code> must be a word, which means one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TargetString</code> exactly matches the value of the specified part of the web request, such as the value
     * of a header.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TargetString</code> is at the beginning of the specified part of the web request and is followed by a
     * character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TargetString</code> is at the end of the specified part of the web request and is preceded by a character
     * other than an alphanumeric character or underscore (_), for example, <code>;BadBot</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TargetString</code> is in the middle of the specified part of the web request and is preceded and followed
     * by characters other than alphanumeric characters or underscore (_), for example, <code>-BadBot;</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>EXACTLY</b>
     * </p>
     * <p>
     * The value of the specified part of the web request must exactly match the value of <code>TargetString</code>.
     * </p>
     * <p>
     * <b>STARTS_WITH</b>
     * </p>
     * <p>
     * The value of <code>TargetString</code> must appear at the beginning of the specified part of the web request.
     * </p>
     * <p>
     * <b>ENDS_WITH</b>
     * </p>
     * <p>
     * The value of <code>TargetString</code> must appear at the end of the specified part of the web request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #positionalConstraint} will return {@link PositionalConstraint#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #positionalConstraintAsString}.
     * </p>
     * 
     * @return Within the portion of a web request that you want to search (for example, in the query string, if any),
     *         specify where you want AWS WAF to search. Valid values include the following:</p>
     *         <p>
     *         <b>CONTAINS</b>
     *         </p>
     *         <p>
     *         The specified part of the web request must include the value of <code>TargetString</code>, but the
     *         location doesn't matter.
     *         </p>
     *         <p>
     *         <b>CONTAINS_WORD</b>
     *         </p>
     *         <p>
     *         The specified part of the web request must include the value of <code>TargetString</code>, and
     *         <code>TargetString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _).
     *         In addition, <code>TargetString</code> must be a word, which means one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TargetString</code> exactly matches the value of the specified part of the web request, such as the
     *         value of a header.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TargetString</code> is at the beginning of the specified part of the web request and is followed by
     *         a character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TargetString</code> is at the end of the specified part of the web request and is preceded by a
     *         character other than an alphanumeric character or underscore (_), for example, <code>;BadBot</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TargetString</code> is in the middle of the specified part of the web request and is preceded and
     *         followed by characters other than alphanumeric characters or underscore (_), for example,
     *         <code>-BadBot;</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>EXACTLY</b>
     *         </p>
     *         <p>
     *         The value of the specified part of the web request must exactly match the value of
     *         <code>TargetString</code>.
     *         </p>
     *         <p>
     *         <b>STARTS_WITH</b>
     *         </p>
     *         <p>
     *         The value of <code>TargetString</code> must appear at the beginning of the specified part of the web
     *         request.
     *         </p>
     *         <p>
     *         <b>ENDS_WITH</b>
     *         </p>
     *         <p>
     *         The value of <code>TargetString</code> must appear at the end of the specified part of the web request.
     * @see PositionalConstraint
     */
    public final PositionalConstraint positionalConstraint() {
        return PositionalConstraint.fromValue(positionalConstraint);
    }

    /**
     * <p>
     * Within the portion of a web request that you want to search (for example, in the query string, if any), specify
     * where you want AWS WAF to search. Valid values include the following:
     * </p>
     * <p>
     * <b>CONTAINS</b>
     * </p>
     * <p>
     * The specified part of the web request must include the value of <code>TargetString</code>, but the location
     * doesn't matter.
     * </p>
     * <p>
     * <b>CONTAINS_WORD</b>
     * </p>
     * <p>
     * The specified part of the web request must include the value of <code>TargetString</code>, and
     * <code>TargetString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _). In
     * addition, <code>TargetString</code> must be a word, which means one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TargetString</code> exactly matches the value of the specified part of the web request, such as the value
     * of a header.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TargetString</code> is at the beginning of the specified part of the web request and is followed by a
     * character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TargetString</code> is at the end of the specified part of the web request and is preceded by a character
     * other than an alphanumeric character or underscore (_), for example, <code>;BadBot</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TargetString</code> is in the middle of the specified part of the web request and is preceded and followed
     * by characters other than alphanumeric characters or underscore (_), for example, <code>-BadBot;</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>EXACTLY</b>
     * </p>
     * <p>
     * The value of the specified part of the web request must exactly match the value of <code>TargetString</code>.
     * </p>
     * <p>
     * <b>STARTS_WITH</b>
     * </p>
     * <p>
     * The value of <code>TargetString</code> must appear at the beginning of the specified part of the web request.
     * </p>
     * <p>
     * <b>ENDS_WITH</b>
     * </p>
     * <p>
     * The value of <code>TargetString</code> must appear at the end of the specified part of the web request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #positionalConstraint} will return {@link PositionalConstraint#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #positionalConstraintAsString}.
     * </p>
     * 
     * @return Within the portion of a web request that you want to search (for example, in the query string, if any),
     *         specify where you want AWS WAF to search. Valid values include the following:</p>
     *         <p>
     *         <b>CONTAINS</b>
     *         </p>
     *         <p>
     *         The specified part of the web request must include the value of <code>TargetString</code>, but the
     *         location doesn't matter.
     *         </p>
     *         <p>
     *         <b>CONTAINS_WORD</b>
     *         </p>
     *         <p>
     *         The specified part of the web request must include the value of <code>TargetString</code>, and
     *         <code>TargetString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _).
     *         In addition, <code>TargetString</code> must be a word, which means one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TargetString</code> exactly matches the value of the specified part of the web request, such as the
     *         value of a header.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TargetString</code> is at the beginning of the specified part of the web request and is followed by
     *         a character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TargetString</code> is at the end of the specified part of the web request and is preceded by a
     *         character other than an alphanumeric character or underscore (_), for example, <code>;BadBot</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TargetString</code> is in the middle of the specified part of the web request and is preceded and
     *         followed by characters other than alphanumeric characters or underscore (_), for example,
     *         <code>-BadBot;</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>EXACTLY</b>
     *         </p>
     *         <p>
     *         The value of the specified part of the web request must exactly match the value of
     *         <code>TargetString</code>.
     *         </p>
     *         <p>
     *         <b>STARTS_WITH</b>
     *         </p>
     *         <p>
     *         The value of <code>TargetString</code> must appear at the beginning of the specified part of the web
     *         request.
     *         </p>
     *         <p>
     *         <b>ENDS_WITH</b>
     *         </p>
     *         <p>
     *         The value of <code>TargetString</code> must appear at the end of the specified part of the web request.
     * @see PositionalConstraint
     */
    public final String positionalConstraintAsString() {
        return positionalConstraint;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fieldToMatch());
        hashCode = 31 * hashCode + Objects.hashCode(targetString());
        hashCode = 31 * hashCode + Objects.hashCode(textTransformationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(positionalConstraintAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ByteMatchTuple)) {
            return false;
        }
        ByteMatchTuple other = (ByteMatchTuple) obj;
        return Objects.equals(fieldToMatch(), other.fieldToMatch()) && Objects.equals(targetString(), other.targetString())
                && Objects.equals(textTransformationAsString(), other.textTransformationAsString())
                && Objects.equals(positionalConstraintAsString(), other.positionalConstraintAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ByteMatchTuple").add("FieldToMatch", fieldToMatch()).add("TargetString", targetString())
                .add("TextTransformation", textTransformationAsString())
                .add("PositionalConstraint", positionalConstraintAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FieldToMatch":
            return Optional.ofNullable(clazz.cast(fieldToMatch()));
        case "TargetString":
            return Optional.ofNullable(clazz.cast(targetString()));
        case "TextTransformation":
            return Optional.ofNullable(clazz.cast(textTransformationAsString()));
        case "PositionalConstraint":
            return Optional.ofNullable(clazz.cast(positionalConstraintAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ByteMatchTuple, T> g) {
        return obj -> g.apply((ByteMatchTuple) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ByteMatchTuple> {
        /**
         * <p>
         * The part of a web request that you want AWS WAF to search, such as a specified header or a query string. For
         * more information, see <a>FieldToMatch</a>.
         * </p>
         * 
         * @param fieldToMatch
         *        The part of a web request that you want AWS WAF to search, such as a specified header or a query
         *        string. For more information, see <a>FieldToMatch</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldToMatch(FieldToMatch fieldToMatch);

        /**
         * <p>
         * The part of a web request that you want AWS WAF to search, such as a specified header or a query string. For
         * more information, see <a>FieldToMatch</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link FieldToMatch.Builder} avoiding the need to
         * create one manually via {@link FieldToMatch#builder()}.
         *
         * When the {@link Consumer} completes, {@link FieldToMatch.Builder#build()} is called immediately and its
         * result is passed to {@link #fieldToMatch(FieldToMatch)}.
         * 
         * @param fieldToMatch
         *        a consumer that will call methods on {@link FieldToMatch.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fieldToMatch(FieldToMatch)
         */
        default Builder fieldToMatch(Consumer<FieldToMatch.Builder> fieldToMatch) {
            return fieldToMatch(FieldToMatch.builder().applyMutation(fieldToMatch).build());
        }

        /**
         * <p>
         * The value that you want AWS WAF to search for. AWS WAF searches for the specified string in the part of web
         * requests that you specified in <code>FieldToMatch</code>. The maximum length of the value is 50 bytes.
         * </p>
         * <p>
         * Valid values depend on the values that you specified for <code>FieldToMatch</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HEADER</code>: The value that you want AWS WAF to search for in the request header that you specified
         * in <a>FieldToMatch</a>, for example, the value of the <code>User-Agent</code> or <code>Referer</code> header.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>METHOD</code>: The HTTP method, which indicates the type of operation specified in the request.
         * CloudFront supports the following methods: <code>DELETE</code>, <code>GET</code>, <code>HEAD</code>,
         * <code>OPTIONS</code>, <code>PATCH</code>, <code>POST</code>, and <code>PUT</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>QUERY_STRING</code>: The value that you want AWS WAF to search for in the query string, which is the
         * part of a URL that appears after a <code>?</code> character.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>URI</code>: The value that you want AWS WAF to search for in the part of a URL that identifies a
         * resource, for example, <code>/images/daily-ad.jpg</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BODY</code>: The part of a request that contains any additional data that you want to send to your web
         * server as the HTTP request body, such as data from a form. The request body immediately follows the request
         * headers. Note that only the first <code>8192</code> bytes of the request body are forwarded to AWS WAF for
         * inspection. To allow or block requests based on the length of the body, you can create a size constraint set.
         * For more information, see <a>CreateSizeConstraintSet</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_QUERY_ARG</code>: The parameter in the query string that you will inspect, such as
         * <i>UserName</i> or <i>SalesRegion</i>. The maximum length for <code>SINGLE_QUERY_ARG</code> is 30 characters.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL_QUERY_ARGS</code>: Similar to <code>SINGLE_QUERY_ARG</code>, but instead of inspecting a single
         * parameter, AWS WAF inspects all parameters within the query string for the value or regex pattern that you
         * specify in <code>TargetString</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If <code>TargetString</code> includes alphabetic characters A-Z and a-z, note that the value is case
         * sensitive.
         * </p>
         * <p>
         * <b>If you're using the AWS WAF API</b>
         * </p>
         * <p>
         * Specify a base64-encoded version of the value. The maximum length of the value before you base64-encode it is
         * 50 bytes.
         * </p>
         * <p>
         * For example, suppose the value of <code>Type</code> is <code>HEADER</code> and the value of <code>Data</code>
         * is <code>User-Agent</code>. If you want to search the <code>User-Agent</code> header for the value
         * <code>BadBot</code>, you base64-encode <code>BadBot</code> using MIME base64-encoding and include the
         * resulting value, <code>QmFkQm90</code>, in the value of <code>TargetString</code>.
         * </p>
         * <p>
         * <b>If you're using the AWS CLI or one of the AWS SDKs</b>
         * </p>
         * <p>
         * The value that you want AWS WAF to search for. The SDK automatically base64 encodes the value.
         * </p>
         * 
         * @param targetString
         *        The value that you want AWS WAF to search for. AWS WAF searches for the specified string in the part
         *        of web requests that you specified in <code>FieldToMatch</code>. The maximum length of the value is 50
         *        bytes.</p>
         *        <p>
         *        Valid values depend on the values that you specified for <code>FieldToMatch</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HEADER</code>: The value that you want AWS WAF to search for in the request header that you
         *        specified in <a>FieldToMatch</a>, for example, the value of the <code>User-Agent</code> or
         *        <code>Referer</code> header.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>METHOD</code>: The HTTP method, which indicates the type of operation specified in the request.
         *        CloudFront supports the following methods: <code>DELETE</code>, <code>GET</code>, <code>HEAD</code>,
         *        <code>OPTIONS</code>, <code>PATCH</code>, <code>POST</code>, and <code>PUT</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>QUERY_STRING</code>: The value that you want AWS WAF to search for in the query string, which is
         *        the part of a URL that appears after a <code>?</code> character.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>URI</code>: The value that you want AWS WAF to search for in the part of a URL that identifies a
         *        resource, for example, <code>/images/daily-ad.jpg</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BODY</code>: The part of a request that contains any additional data that you want to send to
         *        your web server as the HTTP request body, such as data from a form. The request body immediately
         *        follows the request headers. Note that only the first <code>8192</code> bytes of the request body are
         *        forwarded to AWS WAF for inspection. To allow or block requests based on the length of the body, you
         *        can create a size constraint set. For more information, see <a>CreateSizeConstraintSet</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_QUERY_ARG</code>: The parameter in the query string that you will inspect, such as
         *        <i>UserName</i> or <i>SalesRegion</i>. The maximum length for <code>SINGLE_QUERY_ARG</code> is 30
         *        characters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL_QUERY_ARGS</code>: Similar to <code>SINGLE_QUERY_ARG</code>, but instead of inspecting a
         *        single parameter, AWS WAF inspects all parameters within the query string for the value or regex
         *        pattern that you specify in <code>TargetString</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If <code>TargetString</code> includes alphabetic characters A-Z and a-z, note that the value is case
         *        sensitive.
         *        </p>
         *        <p>
         *        <b>If you're using the AWS WAF API</b>
         *        </p>
         *        <p>
         *        Specify a base64-encoded version of the value. The maximum length of the value before you
         *        base64-encode it is 50 bytes.
         *        </p>
         *        <p>
         *        For example, suppose the value of <code>Type</code> is <code>HEADER</code> and the value of
         *        <code>Data</code> is <code>User-Agent</code>. If you want to search the <code>User-Agent</code> header
         *        for the value <code>BadBot</code>, you base64-encode <code>BadBot</code> using MIME base64-encoding
         *        and include the resulting value, <code>QmFkQm90</code>, in the value of <code>TargetString</code>.
         *        </p>
         *        <p>
         *        <b>If you're using the AWS CLI or one of the AWS SDKs</b>
         *        </p>
         *        <p>
         *        The value that you want AWS WAF to search for. The SDK automatically base64 encodes the value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetString(SdkBytes targetString);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
         * <code>FieldToMatch</code> before inspecting it for a match.
         * </p>
         * <p>
         * You can only specify a single type of TextTransformation.
         * </p>
         * <p>
         * <b>CMD_LINE</b>
         * </p>
         * <p>
         * When you're concerned that attackers are injecting an operating system command line command and using unusual
         * formatting to disguise some or all of the command, use this option to perform the following transformations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Delete the following characters: \ " ' ^
         * </p>
         * </li>
         * <li>
         * <p>
         * Delete spaces before the following characters: / (
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace the following characters with a space: , ;
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace multiple spaces with one space
         * </p>
         * </li>
         * <li>
         * <p>
         * Convert uppercase letters (A-Z) to lowercase (a-z)
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>COMPRESS_WHITE_SPACE</b>
         * </p>
         * <p>
         * Use this option to replace the following characters with a space character (decimal 32):
         * </p>
         * <ul>
         * <li>
         * <p>
         * \f, formfeed, decimal 12
         * </p>
         * </li>
         * <li>
         * <p>
         * \t, tab, decimal 9
         * </p>
         * </li>
         * <li>
         * <p>
         * \n, newline, decimal 10
         * </p>
         * </li>
         * <li>
         * <p>
         * \r, carriage return, decimal 13
         * </p>
         * </li>
         * <li>
         * <p>
         * \v, vertical tab, decimal 11
         * </p>
         * </li>
         * <li>
         * <p>
         * non-breaking space, decimal 160
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         * </p>
         * <p>
         * <b>HTML_ENTITY_DECODE</b>
         * </p>
         * <p>
         * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
         * performs the following operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Replaces <code>(ampersand)quot;</code> with <code>"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>LOWERCASE</b>
         * </p>
         * <p>
         * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         * </p>
         * <p>
         * <b>URL_DECODE</b>
         * </p>
         * <p>
         * Use this option to decode a URL-encoded value.
         * </p>
         * <p>
         * <b>NONE</b>
         * </p>
         * <p>
         * Specify <code>NONE</code> if you don't want to perform any text transformations.
         * </p>
         * 
         * @param textTransformation
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
         *        <code>FieldToMatch</code> before inspecting it for a match.</p>
         *        <p>
         *        You can only specify a single type of TextTransformation.
         *        </p>
         *        <p>
         *        <b>CMD_LINE</b>
         *        </p>
         *        <p>
         *        When you're concerned that attackers are injecting an operating system command line command and using
         *        unusual formatting to disguise some or all of the command, use this option to perform the following
         *        transformations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Delete the following characters: \ " ' ^
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delete spaces before the following characters: / (
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace the following characters with a space: , ;
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace multiple spaces with one space
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Convert uppercase letters (A-Z) to lowercase (a-z)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>COMPRESS_WHITE_SPACE</b>
         *        </p>
         *        <p>
         *        Use this option to replace the following characters with a space character (decimal 32):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        \f, formfeed, decimal 12
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \t, tab, decimal 9
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \n, newline, decimal 10
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \r, carriage return, decimal 13
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \v, vertical tab, decimal 11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         *        </p>
         *        <p>
         *        <b>HTML_ENTITY_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to replace HTML-encoded characters with unencoded characters.
         *        <code>HTML_ENTITY_DECODE</code> performs the following operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)quot;</code> with <code>"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with
         *        the corresponding characters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         *        corresponding characters
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>LOWERCASE</b>
         *        </p>
         *        <p>
         *        Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         *        </p>
         *        <p>
         *        <b>URL_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to decode a URL-encoded value.
         *        </p>
         *        <p>
         *        <b>NONE</b>
         *        </p>
         *        <p>
         *        Specify <code>NONE</code> if you don't want to perform any text transformations.
         * @see TextTransformation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTransformation
         */
        Builder textTransformation(String textTransformation);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
         * <code>FieldToMatch</code> before inspecting it for a match.
         * </p>
         * <p>
         * You can only specify a single type of TextTransformation.
         * </p>
         * <p>
         * <b>CMD_LINE</b>
         * </p>
         * <p>
         * When you're concerned that attackers are injecting an operating system command line command and using unusual
         * formatting to disguise some or all of the command, use this option to perform the following transformations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Delete the following characters: \ " ' ^
         * </p>
         * </li>
         * <li>
         * <p>
         * Delete spaces before the following characters: / (
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace the following characters with a space: , ;
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace multiple spaces with one space
         * </p>
         * </li>
         * <li>
         * <p>
         * Convert uppercase letters (A-Z) to lowercase (a-z)
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>COMPRESS_WHITE_SPACE</b>
         * </p>
         * <p>
         * Use this option to replace the following characters with a space character (decimal 32):
         * </p>
         * <ul>
         * <li>
         * <p>
         * \f, formfeed, decimal 12
         * </p>
         * </li>
         * <li>
         * <p>
         * \t, tab, decimal 9
         * </p>
         * </li>
         * <li>
         * <p>
         * \n, newline, decimal 10
         * </p>
         * </li>
         * <li>
         * <p>
         * \r, carriage return, decimal 13
         * </p>
         * </li>
         * <li>
         * <p>
         * \v, vertical tab, decimal 11
         * </p>
         * </li>
         * <li>
         * <p>
         * non-breaking space, decimal 160
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         * </p>
         * <p>
         * <b>HTML_ENTITY_DECODE</b>
         * </p>
         * <p>
         * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
         * performs the following operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Replaces <code>(ampersand)quot;</code> with <code>"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>LOWERCASE</b>
         * </p>
         * <p>
         * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         * </p>
         * <p>
         * <b>URL_DECODE</b>
         * </p>
         * <p>
         * Use this option to decode a URL-encoded value.
         * </p>
         * <p>
         * <b>NONE</b>
         * </p>
         * <p>
         * Specify <code>NONE</code> if you don't want to perform any text transformations.
         * </p>
         * 
         * @param textTransformation
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass AWS WAF. If you specify a transformation, AWS WAF performs the transformation on
         *        <code>FieldToMatch</code> before inspecting it for a match.</p>
         *        <p>
         *        You can only specify a single type of TextTransformation.
         *        </p>
         *        <p>
         *        <b>CMD_LINE</b>
         *        </p>
         *        <p>
         *        When you're concerned that attackers are injecting an operating system command line command and using
         *        unusual formatting to disguise some or all of the command, use this option to perform the following
         *        transformations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Delete the following characters: \ " ' ^
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delete spaces before the following characters: / (
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace the following characters with a space: , ;
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace multiple spaces with one space
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Convert uppercase letters (A-Z) to lowercase (a-z)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>COMPRESS_WHITE_SPACE</b>
         *        </p>
         *        <p>
         *        Use this option to replace the following characters with a space character (decimal 32):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        \f, formfeed, decimal 12
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \t, tab, decimal 9
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \n, newline, decimal 10
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \r, carriage return, decimal 13
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \v, vertical tab, decimal 11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         *        </p>
         *        <p>
         *        <b>HTML_ENTITY_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to replace HTML-encoded characters with unencoded characters.
         *        <code>HTML_ENTITY_DECODE</code> performs the following operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)quot;</code> with <code>"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with
         *        the corresponding characters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         *        corresponding characters
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>LOWERCASE</b>
         *        </p>
         *        <p>
         *        Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         *        </p>
         *        <p>
         *        <b>URL_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to decode a URL-encoded value.
         *        </p>
         *        <p>
         *        <b>NONE</b>
         *        </p>
         *        <p>
         *        Specify <code>NONE</code> if you don't want to perform any text transformations.
         * @see TextTransformation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTransformation
         */
        Builder textTransformation(TextTransformation textTransformation);

        /**
         * <p>
         * Within the portion of a web request that you want to search (for example, in the query string, if any),
         * specify where you want AWS WAF to search. Valid values include the following:
         * </p>
         * <p>
         * <b>CONTAINS</b>
         * </p>
         * <p>
         * The specified part of the web request must include the value of <code>TargetString</code>, but the location
         * doesn't matter.
         * </p>
         * <p>
         * <b>CONTAINS_WORD</b>
         * </p>
         * <p>
         * The specified part of the web request must include the value of <code>TargetString</code>, and
         * <code>TargetString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _). In
         * addition, <code>TargetString</code> must be a word, which means one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TargetString</code> exactly matches the value of the specified part of the web request, such as the
         * value of a header.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TargetString</code> is at the beginning of the specified part of the web request and is followed by a
         * character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TargetString</code> is at the end of the specified part of the web request and is preceded by a
         * character other than an alphanumeric character or underscore (_), for example, <code>;BadBot</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TargetString</code> is in the middle of the specified part of the web request and is preceded and
         * followed by characters other than alphanumeric characters or underscore (_), for example,
         * <code>-BadBot;</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>EXACTLY</b>
         * </p>
         * <p>
         * The value of the specified part of the web request must exactly match the value of <code>TargetString</code>.
         * </p>
         * <p>
         * <b>STARTS_WITH</b>
         * </p>
         * <p>
         * The value of <code>TargetString</code> must appear at the beginning of the specified part of the web request.
         * </p>
         * <p>
         * <b>ENDS_WITH</b>
         * </p>
         * <p>
         * The value of <code>TargetString</code> must appear at the end of the specified part of the web request.
         * </p>
         * 
         * @param positionalConstraint
         *        Within the portion of a web request that you want to search (for example, in the query string, if
         *        any), specify where you want AWS WAF to search. Valid values include the following:</p>
         *        <p>
         *        <b>CONTAINS</b>
         *        </p>
         *        <p>
         *        The specified part of the web request must include the value of <code>TargetString</code>, but the
         *        location doesn't matter.
         *        </p>
         *        <p>
         *        <b>CONTAINS_WORD</b>
         *        </p>
         *        <p>
         *        The specified part of the web request must include the value of <code>TargetString</code>, and
         *        <code>TargetString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or
         *        _). In addition, <code>TargetString</code> must be a word, which means one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TargetString</code> exactly matches the value of the specified part of the web request, such as
         *        the value of a header.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TargetString</code> is at the beginning of the specified part of the web request and is followed
         *        by a character other than an alphanumeric character or underscore (_), for example,
         *        <code>BadBot;</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TargetString</code> is at the end of the specified part of the web request and is preceded by a
         *        character other than an alphanumeric character or underscore (_), for example, <code>;BadBot</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TargetString</code> is in the middle of the specified part of the web request and is preceded
         *        and followed by characters other than alphanumeric characters or underscore (_), for example,
         *        <code>-BadBot;</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>EXACTLY</b>
         *        </p>
         *        <p>
         *        The value of the specified part of the web request must exactly match the value of
         *        <code>TargetString</code>.
         *        </p>
         *        <p>
         *        <b>STARTS_WITH</b>
         *        </p>
         *        <p>
         *        The value of <code>TargetString</code> must appear at the beginning of the specified part of the web
         *        request.
         *        </p>
         *        <p>
         *        <b>ENDS_WITH</b>
         *        </p>
         *        <p>
         *        The value of <code>TargetString</code> must appear at the end of the specified part of the web
         *        request.
         * @see PositionalConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PositionalConstraint
         */
        Builder positionalConstraint(String positionalConstraint);

        /**
         * <p>
         * Within the portion of a web request that you want to search (for example, in the query string, if any),
         * specify where you want AWS WAF to search. Valid values include the following:
         * </p>
         * <p>
         * <b>CONTAINS</b>
         * </p>
         * <p>
         * The specified part of the web request must include the value of <code>TargetString</code>, but the location
         * doesn't matter.
         * </p>
         * <p>
         * <b>CONTAINS_WORD</b>
         * </p>
         * <p>
         * The specified part of the web request must include the value of <code>TargetString</code>, and
         * <code>TargetString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _). In
         * addition, <code>TargetString</code> must be a word, which means one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TargetString</code> exactly matches the value of the specified part of the web request, such as the
         * value of a header.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TargetString</code> is at the beginning of the specified part of the web request and is followed by a
         * character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TargetString</code> is at the end of the specified part of the web request and is preceded by a
         * character other than an alphanumeric character or underscore (_), for example, <code>;BadBot</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TargetString</code> is in the middle of the specified part of the web request and is preceded and
         * followed by characters other than alphanumeric characters or underscore (_), for example,
         * <code>-BadBot;</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>EXACTLY</b>
         * </p>
         * <p>
         * The value of the specified part of the web request must exactly match the value of <code>TargetString</code>.
         * </p>
         * <p>
         * <b>STARTS_WITH</b>
         * </p>
         * <p>
         * The value of <code>TargetString</code> must appear at the beginning of the specified part of the web request.
         * </p>
         * <p>
         * <b>ENDS_WITH</b>
         * </p>
         * <p>
         * The value of <code>TargetString</code> must appear at the end of the specified part of the web request.
         * </p>
         * 
         * @param positionalConstraint
         *        Within the portion of a web request that you want to search (for example, in the query string, if
         *        any), specify where you want AWS WAF to search. Valid values include the following:</p>
         *        <p>
         *        <b>CONTAINS</b>
         *        </p>
         *        <p>
         *        The specified part of the web request must include the value of <code>TargetString</code>, but the
         *        location doesn't matter.
         *        </p>
         *        <p>
         *        <b>CONTAINS_WORD</b>
         *        </p>
         *        <p>
         *        The specified part of the web request must include the value of <code>TargetString</code>, and
         *        <code>TargetString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or
         *        _). In addition, <code>TargetString</code> must be a word, which means one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TargetString</code> exactly matches the value of the specified part of the web request, such as
         *        the value of a header.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TargetString</code> is at the beginning of the specified part of the web request and is followed
         *        by a character other than an alphanumeric character or underscore (_), for example,
         *        <code>BadBot;</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TargetString</code> is at the end of the specified part of the web request and is preceded by a
         *        character other than an alphanumeric character or underscore (_), for example, <code>;BadBot</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TargetString</code> is in the middle of the specified part of the web request and is preceded
         *        and followed by characters other than alphanumeric characters or underscore (_), for example,
         *        <code>-BadBot;</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>EXACTLY</b>
         *        </p>
         *        <p>
         *        The value of the specified part of the web request must exactly match the value of
         *        <code>TargetString</code>.
         *        </p>
         *        <p>
         *        <b>STARTS_WITH</b>
         *        </p>
         *        <p>
         *        The value of <code>TargetString</code> must appear at the beginning of the specified part of the web
         *        request.
         *        </p>
         *        <p>
         *        <b>ENDS_WITH</b>
         *        </p>
         *        <p>
         *        The value of <code>TargetString</code> must appear at the end of the specified part of the web
         *        request.
         * @see PositionalConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PositionalConstraint
         */
        Builder positionalConstraint(PositionalConstraint positionalConstraint);
    }

    static final class BuilderImpl implements Builder {
        private FieldToMatch fieldToMatch;

        private SdkBytes targetString;

        private String textTransformation;

        private String positionalConstraint;

        private BuilderImpl() {
        }

        private BuilderImpl(ByteMatchTuple model) {
            fieldToMatch(model.fieldToMatch);
            targetString(model.targetString);
            textTransformation(model.textTransformation);
            positionalConstraint(model.positionalConstraint);
        }

        public final FieldToMatch.Builder getFieldToMatch() {
            return fieldToMatch != null ? fieldToMatch.toBuilder() : null;
        }

        @Override
        public final Builder fieldToMatch(FieldToMatch fieldToMatch) {
            this.fieldToMatch = fieldToMatch;
            return this;
        }

        public final void setFieldToMatch(FieldToMatch.BuilderImpl fieldToMatch) {
            this.fieldToMatch = fieldToMatch != null ? fieldToMatch.build() : null;
        }

        public final ByteBuffer getTargetString() {
            return targetString == null ? null : targetString.asByteBuffer();
        }

        @Override
        public final Builder targetString(SdkBytes targetString) {
            this.targetString = StandardMemberCopier.copy(targetString);
            return this;
        }

        public final void setTargetString(ByteBuffer targetString) {
            targetString(targetString == null ? null : SdkBytes.fromByteBuffer(targetString));
        }

        public final String getTextTransformation() {
            return textTransformation;
        }

        @Override
        public final Builder textTransformation(String textTransformation) {
            this.textTransformation = textTransformation;
            return this;
        }

        @Override
        public final Builder textTransformation(TextTransformation textTransformation) {
            this.textTransformation(textTransformation == null ? null : textTransformation.toString());
            return this;
        }

        public final void setTextTransformation(String textTransformation) {
            this.textTransformation = textTransformation;
        }

        public final String getPositionalConstraint() {
            return positionalConstraint;
        }

        @Override
        public final Builder positionalConstraint(String positionalConstraint) {
            this.positionalConstraint = positionalConstraint;
            return this;
        }

        @Override
        public final Builder positionalConstraint(PositionalConstraint positionalConstraint) {
            this.positionalConstraint(positionalConstraint == null ? null : positionalConstraint.toString());
            return this;
        }

        public final void setPositionalConstraint(String positionalConstraint) {
            this.positionalConstraint = positionalConstraint;
        }

        @Override
        public ByteMatchTuple build() {
            return new ByteMatchTuple(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
