/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.waf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is <b>AWS WAF Classic</b> documentation. For more information, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/classic-waf-chapter.html">AWS WAF Classic</a> in the
 * developer guide.
 * </p>
 * <p>
 * <b>For the latest version of AWS WAF</b>, use the AWS WAFV2 API and see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>. With the
 * latest version, AWS WAF has a single set of endpoints for regional and global use.
 * </p>
 * </note>
 * <p>
 * Contains the <code>Rules</code> that identify the requests that you want to allow, block, or count. In a
 * <code>WebACL</code>, you also specify a default action (<code>ALLOW</code> or <code>BLOCK</code>), and the action for
 * each <code>Rule</code> that you add to a <code>WebACL</code>, for example, block requests from specified IP addresses
 * or block requests from specified referrers. You also associate the <code>WebACL</code> with a CloudFront distribution
 * to identify the requests that you want AWS WAF to filter. If you add more than one <code>Rule</code> to a
 * <code>WebACL</code>, a request needs to match only one of the specifications to be allowed, blocked, or counted. For
 * more information, see <a>UpdateWebACL</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WebACL implements SdkPojo, Serializable, ToCopyableBuilder<WebACL.Builder, WebACL> {
    private static final SdkField<String> WEB_ACL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WebACL::webACLId)).setter(setter(Builder::webACLId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WebACLId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WebACL::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WebACL::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<WafAction> DEFAULT_ACTION_FIELD = SdkField.<WafAction> builder(MarshallingType.SDK_POJO)
            .getter(getter(WebACL::defaultAction)).setter(setter(Builder::defaultAction)).constructor(WafAction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultAction").build()).build();

    private static final SdkField<List<ActivatedRule>> RULES_FIELD = SdkField
            .<List<ActivatedRule>> builder(MarshallingType.LIST)
            .getter(getter(WebACL::rules))
            .setter(setter(Builder::rules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Rules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ActivatedRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(ActivatedRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> WEB_ACL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WebACL::webACLArn)).setter(setter(Builder::webACLArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WebACLArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WEB_ACL_ID_FIELD, NAME_FIELD,
            METRIC_NAME_FIELD, DEFAULT_ACTION_FIELD, RULES_FIELD, WEB_ACL_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String webACLId;

    private final String name;

    private final String metricName;

    private final WafAction defaultAction;

    private final List<ActivatedRule> rules;

    private final String webACLArn;

    private WebACL(BuilderImpl builder) {
        this.webACLId = builder.webACLId;
        this.name = builder.name;
        this.metricName = builder.metricName;
        this.defaultAction = builder.defaultAction;
        this.rules = builder.rules;
        this.webACLArn = builder.webACLArn;
    }

    /**
     * <p>
     * A unique identifier for a <code>WebACL</code>. You use <code>WebACLId</code> to get information about a
     * <code>WebACL</code> (see <a>GetWebACL</a>), update a <code>WebACL</code> (see <a>UpdateWebACL</a>), and delete a
     * <code>WebACL</code> from AWS WAF (see <a>DeleteWebACL</a>).
     * </p>
     * <p>
     * <code>WebACLId</code> is returned by <a>CreateWebACL</a> and by <a>ListWebACLs</a>.
     * </p>
     * 
     * @return A unique identifier for a <code>WebACL</code>. You use <code>WebACLId</code> to get information about a
     *         <code>WebACL</code> (see <a>GetWebACL</a>), update a <code>WebACL</code> (see <a>UpdateWebACL</a>), and
     *         delete a <code>WebACL</code> from AWS WAF (see <a>DeleteWebACL</a>).</p>
     *         <p>
     *         <code>WebACLId</code> is returned by <a>CreateWebACL</a> and by <a>ListWebACLs</a>.
     */
    public String webACLId() {
        return webACLId;
    }

    /**
     * <p>
     * A friendly name or description of the <code>WebACL</code>. You can't change the name of a <code>WebACL</code>
     * after you create it.
     * </p>
     * 
     * @return A friendly name or description of the <code>WebACL</code>. You can't change the name of a
     *         <code>WebACL</code> after you create it.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A friendly name or description for the metrics for this <code>WebACL</code>. The name can contain only
     * alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't contain
     * whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You can't change
     * <code>MetricName</code> after you create the <code>WebACL</code>.
     * </p>
     * 
     * @return A friendly name or description for the metrics for this <code>WebACL</code>. The name can contain only
     *         alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't contain
     *         whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You can't change
     *         <code>MetricName</code> after you create the <code>WebACL</code>.
     */
    public String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match. The action is
     * specified by the <a>WafAction</a> object.
     * </p>
     * 
     * @return The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match. The
     *         action is specified by the <a>WafAction</a> object.
     */
    public WafAction defaultAction() {
        return defaultAction;
    }

    /**
     * Returns true if the Rules property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRules() {
        return rules != null && !(rules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array that contains the action for each <code>Rule</code> in a <code>WebACL</code>, the priority of the
     * <code>Rule</code>, and the ID of the <code>Rule</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array that contains the action for each <code>Rule</code> in a <code>WebACL</code>, the priority of
     *         the <code>Rule</code>, and the ID of the <code>Rule</code>.
     */
    public List<ActivatedRule> rules() {
        return rules;
    }

    /**
     * <p>
     * Tha Amazon Resource Name (ARN) of the web ACL.
     * </p>
     * 
     * @return Tha Amazon Resource Name (ARN) of the web ACL.
     */
    public String webACLArn() {
        return webACLArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(webACLId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(defaultAction());
        hashCode = 31 * hashCode + Objects.hashCode(rules());
        hashCode = 31 * hashCode + Objects.hashCode(webACLArn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WebACL)) {
            return false;
        }
        WebACL other = (WebACL) obj;
        return Objects.equals(webACLId(), other.webACLId()) && Objects.equals(name(), other.name())
                && Objects.equals(metricName(), other.metricName()) && Objects.equals(defaultAction(), other.defaultAction())
                && Objects.equals(rules(), other.rules()) && Objects.equals(webACLArn(), other.webACLArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("WebACL").add("WebACLId", webACLId()).add("Name", name()).add("MetricName", metricName())
                .add("DefaultAction", defaultAction()).add("Rules", rules()).add("WebACLArn", webACLArn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WebACLId":
            return Optional.ofNullable(clazz.cast(webACLId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "DefaultAction":
            return Optional.ofNullable(clazz.cast(defaultAction()));
        case "Rules":
            return Optional.ofNullable(clazz.cast(rules()));
        case "WebACLArn":
            return Optional.ofNullable(clazz.cast(webACLArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WebACL, T> g) {
        return obj -> g.apply((WebACL) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WebACL> {
        /**
         * <p>
         * A unique identifier for a <code>WebACL</code>. You use <code>WebACLId</code> to get information about a
         * <code>WebACL</code> (see <a>GetWebACL</a>), update a <code>WebACL</code> (see <a>UpdateWebACL</a>), and
         * delete a <code>WebACL</code> from AWS WAF (see <a>DeleteWebACL</a>).
         * </p>
         * <p>
         * <code>WebACLId</code> is returned by <a>CreateWebACL</a> and by <a>ListWebACLs</a>.
         * </p>
         * 
         * @param webACLId
         *        A unique identifier for a <code>WebACL</code>. You use <code>WebACLId</code> to get information about
         *        a <code>WebACL</code> (see <a>GetWebACL</a>), update a <code>WebACL</code> (see <a>UpdateWebACL</a>),
         *        and delete a <code>WebACL</code> from AWS WAF (see <a>DeleteWebACL</a>).</p>
         *        <p>
         *        <code>WebACLId</code> is returned by <a>CreateWebACL</a> and by <a>ListWebACLs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder webACLId(String webACLId);

        /**
         * <p>
         * A friendly name or description of the <code>WebACL</code>. You can't change the name of a <code>WebACL</code>
         * after you create it.
         * </p>
         * 
         * @param name
         *        A friendly name or description of the <code>WebACL</code>. You can't change the name of a
         *        <code>WebACL</code> after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A friendly name or description for the metrics for this <code>WebACL</code>. The name can contain only
         * alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't contain
         * whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You can't change
         * <code>MetricName</code> after you create the <code>WebACL</code>.
         * </p>
         * 
         * @param metricName
         *        A friendly name or description for the metrics for this <code>WebACL</code>. The name can contain only
         *        alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't
         *        contain whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You
         *        can't change <code>MetricName</code> after you create the <code>WebACL</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match. The
         * action is specified by the <a>WafAction</a> object.
         * </p>
         * 
         * @param defaultAction
         *        The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match.
         *        The action is specified by the <a>WafAction</a> object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultAction(WafAction defaultAction);

        /**
         * <p>
         * The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match. The
         * action is specified by the <a>WafAction</a> object.
         * </p>
         * This is a convenience that creates an instance of the {@link WafAction.Builder} avoiding the need to create
         * one manually via {@link WafAction#builder()}.
         *
         * When the {@link Consumer} completes, {@link WafAction.Builder#build()} is called immediately and its result
         * is passed to {@link #defaultAction(WafAction)}.
         * 
         * @param defaultAction
         *        a consumer that will call methods on {@link WafAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultAction(WafAction)
         */
        default Builder defaultAction(Consumer<WafAction.Builder> defaultAction) {
            return defaultAction(WafAction.builder().applyMutation(defaultAction).build());
        }

        /**
         * <p>
         * An array that contains the action for each <code>Rule</code> in a <code>WebACL</code>, the priority of the
         * <code>Rule</code>, and the ID of the <code>Rule</code>.
         * </p>
         * 
         * @param rules
         *        An array that contains the action for each <code>Rule</code> in a <code>WebACL</code>, the priority of
         *        the <code>Rule</code>, and the ID of the <code>Rule</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Collection<ActivatedRule> rules);

        /**
         * <p>
         * An array that contains the action for each <code>Rule</code> in a <code>WebACL</code>, the priority of the
         * <code>Rule</code>, and the ID of the <code>Rule</code>.
         * </p>
         * 
         * @param rules
         *        An array that contains the action for each <code>Rule</code> in a <code>WebACL</code>, the priority of
         *        the <code>Rule</code>, and the ID of the <code>Rule</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(ActivatedRule... rules);

        /**
         * <p>
         * An array that contains the action for each <code>Rule</code> in a <code>WebACL</code>, the priority of the
         * <code>Rule</code>, and the ID of the <code>Rule</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ActivatedRule>.Builder} avoiding the need
         * to create one manually via {@link List<ActivatedRule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ActivatedRule>.Builder#build()} is called immediately and
         * its result is passed to {@link #rules(List<ActivatedRule>)}.
         * 
         * @param rules
         *        a consumer that will call methods on {@link List<ActivatedRule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rules(List<ActivatedRule>)
         */
        Builder rules(Consumer<ActivatedRule.Builder>... rules);

        /**
         * <p>
         * Tha Amazon Resource Name (ARN) of the web ACL.
         * </p>
         * 
         * @param webACLArn
         *        Tha Amazon Resource Name (ARN) of the web ACL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder webACLArn(String webACLArn);
    }

    static final class BuilderImpl implements Builder {
        private String webACLId;

        private String name;

        private String metricName;

        private WafAction defaultAction;

        private List<ActivatedRule> rules = DefaultSdkAutoConstructList.getInstance();

        private String webACLArn;

        private BuilderImpl() {
        }

        private BuilderImpl(WebACL model) {
            webACLId(model.webACLId);
            name(model.name);
            metricName(model.metricName);
            defaultAction(model.defaultAction);
            rules(model.rules);
            webACLArn(model.webACLArn);
        }

        public final String getWebACLId() {
            return webACLId;
        }

        @Override
        public final Builder webACLId(String webACLId) {
            this.webACLId = webACLId;
            return this;
        }

        public final void setWebACLId(String webACLId) {
            this.webACLId = webACLId;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final WafAction.Builder getDefaultAction() {
            return defaultAction != null ? defaultAction.toBuilder() : null;
        }

        @Override
        public final Builder defaultAction(WafAction defaultAction) {
            this.defaultAction = defaultAction;
            return this;
        }

        public final void setDefaultAction(WafAction.BuilderImpl defaultAction) {
            this.defaultAction = defaultAction != null ? defaultAction.build() : null;
        }

        public final Collection<ActivatedRule.Builder> getRules() {
            return rules != null ? rules.stream().map(ActivatedRule::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder rules(Collection<ActivatedRule> rules) {
            this.rules = ActivatedRulesCopier.copy(rules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(ActivatedRule... rules) {
            rules(Arrays.asList(rules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(Consumer<ActivatedRule.Builder>... rules) {
            rules(Stream.of(rules).map(c -> ActivatedRule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRules(Collection<ActivatedRule.BuilderImpl> rules) {
            this.rules = ActivatedRulesCopier.copyFromBuilder(rules);
        }

        public final String getWebACLArn() {
            return webACLArn;
        }

        @Override
        public final Builder webACLArn(String webACLArn) {
            this.webACLArn = webACLArn;
            return this;
        }

        public final void setWebACLArn(String webACLArn) {
            this.webACLArn = webACLArn;
        }

        @Override
        public WebACL build() {
            return new WebACL(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
