/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.waf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is <b>AWS WAF Classic</b> documentation. For more information, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/classic-waf-chapter.html">AWS WAF Classic</a> in the
 * developer guide.
 * </p>
 * <p>
 * <b>For the latest version of AWS WAF</b>, use the AWS WAFV2 API and see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>. With the
 * latest version, AWS WAF has a single set of endpoints for regional and global use.
 * </p>
 * </note>
 * <p>
 * A <code>RateBasedRule</code> is identical to a regular <a>Rule</a>, with one addition: a <code>RateBasedRule</code>
 * counts the number of requests that arrive from a specified IP address every five minutes. For example, based on
 * recent requests that you've seen from an attacker, you might create a <code>RateBasedRule</code> that includes the
 * following conditions:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The requests come from 192.0.2.44.
 * </p>
 * </li>
 * <li>
 * <p>
 * They contain the value <code>BadBot</code> in the <code>User-Agent</code> header.
 * </p>
 * </li>
 * </ul>
 * <p>
 * In the rule, you also define the rate limit as 1,000.
 * </p>
 * <p>
 * Requests that meet both of these conditions and exceed 1,000 requests every five minutes trigger the rule's action
 * (block or count), which is defined in the web ACL.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RateBasedRule implements SdkPojo, Serializable, ToCopyableBuilder<RateBasedRule.Builder, RateBasedRule> {
    private static final SdkField<String> RULE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RateBasedRule::ruleId)).setter(setter(Builder::ruleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RateBasedRule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RateBasedRule::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<List<Predicate>> MATCH_PREDICATES_FIELD = SdkField
            .<List<Predicate>> builder(MarshallingType.LIST)
            .getter(getter(RateBasedRule::matchPredicates))
            .setter(setter(Builder::matchPredicates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchPredicates").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Predicate> builder(MarshallingType.SDK_POJO)
                                            .constructor(Predicate::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RATE_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RateBasedRule::rateKeyAsString)).setter(setter(Builder::rateKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RateKey").build()).build();

    private static final SdkField<Long> RATE_LIMIT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(RateBasedRule::rateLimit)).setter(setter(Builder::rateLimit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RateLimit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_ID_FIELD, NAME_FIELD,
            METRIC_NAME_FIELD, MATCH_PREDICATES_FIELD, RATE_KEY_FIELD, RATE_LIMIT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ruleId;

    private final String name;

    private final String metricName;

    private final List<Predicate> matchPredicates;

    private final String rateKey;

    private final Long rateLimit;

    private RateBasedRule(BuilderImpl builder) {
        this.ruleId = builder.ruleId;
        this.name = builder.name;
        this.metricName = builder.metricName;
        this.matchPredicates = builder.matchPredicates;
        this.rateKey = builder.rateKey;
        this.rateLimit = builder.rateLimit;
    }

    /**
     * <p>
     * A unique identifier for a <code>RateBasedRule</code>. You use <code>RuleId</code> to get more information about a
     * <code>RateBasedRule</code> (see <a>GetRateBasedRule</a>), update a <code>RateBasedRule</code> (see
     * <a>UpdateRateBasedRule</a>), insert a <code>RateBasedRule</code> into a <code>WebACL</code> or delete one from a
     * <code>WebACL</code> (see <a>UpdateWebACL</a>), or delete a <code>RateBasedRule</code> from AWS WAF (see
     * <a>DeleteRateBasedRule</a>).
     * </p>
     * 
     * @return A unique identifier for a <code>RateBasedRule</code>. You use <code>RuleId</code> to get more information
     *         about a <code>RateBasedRule</code> (see <a>GetRateBasedRule</a>), update a <code>RateBasedRule</code>
     *         (see <a>UpdateRateBasedRule</a>), insert a <code>RateBasedRule</code> into a <code>WebACL</code> or
     *         delete one from a <code>WebACL</code> (see <a>UpdateWebACL</a>), or delete a <code>RateBasedRule</code>
     *         from AWS WAF (see <a>DeleteRateBasedRule</a>).
     */
    public String ruleId() {
        return ruleId;
    }

    /**
     * <p>
     * A friendly name or description for a <code>RateBasedRule</code>. You can't change the name of a
     * <code>RateBasedRule</code> after you create it.
     * </p>
     * 
     * @return A friendly name or description for a <code>RateBasedRule</code>. You can't change the name of a
     *         <code>RateBasedRule</code> after you create it.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A friendly name or description for the metrics for a <code>RateBasedRule</code>. The name can contain only
     * alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't contain
     * whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You can't change the name
     * of the metric after you create the <code>RateBasedRule</code>.
     * </p>
     * 
     * @return A friendly name or description for the metrics for a <code>RateBasedRule</code>. The name can contain
     *         only alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't
     *         contain whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You can't
     *         change the name of the metric after you create the <code>RateBasedRule</code>.
     */
    public String metricName() {
        return metricName;
    }

    /**
     * Returns true if the MatchPredicates property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasMatchPredicates() {
        return matchPredicates != null && !(matchPredicates instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <code>Predicates</code> object contains one <code>Predicate</code> element for each <a>ByteMatchSet</a>,
     * <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a <code>RateBasedRule</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMatchPredicates()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The <code>Predicates</code> object contains one <code>Predicate</code> element for each
     *         <a>ByteMatchSet</a>, <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a
     *         <code>RateBasedRule</code>.
     */
    public List<Predicate> matchPredicates() {
        return matchPredicates;
    }

    /**
     * <p>
     * The field that AWS WAF uses to determine if requests are likely arriving from single source and thus subject to
     * rate monitoring. The only valid value for <code>RateKey</code> is <code>IP</code>. <code>IP</code> indicates that
     * requests arriving from the same IP address are subject to the <code>RateLimit</code> that is specified in the
     * <code>RateBasedRule</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rateKey} will
     * return {@link RateKey#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #rateKeyAsString}.
     * </p>
     * 
     * @return The field that AWS WAF uses to determine if requests are likely arriving from single source and thus
     *         subject to rate monitoring. The only valid value for <code>RateKey</code> is <code>IP</code>.
     *         <code>IP</code> indicates that requests arriving from the same IP address are subject to the
     *         <code>RateLimit</code> that is specified in the <code>RateBasedRule</code>.
     * @see RateKey
     */
    public RateKey rateKey() {
        return RateKey.fromValue(rateKey);
    }

    /**
     * <p>
     * The field that AWS WAF uses to determine if requests are likely arriving from single source and thus subject to
     * rate monitoring. The only valid value for <code>RateKey</code> is <code>IP</code>. <code>IP</code> indicates that
     * requests arriving from the same IP address are subject to the <code>RateLimit</code> that is specified in the
     * <code>RateBasedRule</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rateKey} will
     * return {@link RateKey#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #rateKeyAsString}.
     * </p>
     * 
     * @return The field that AWS WAF uses to determine if requests are likely arriving from single source and thus
     *         subject to rate monitoring. The only valid value for <code>RateKey</code> is <code>IP</code>.
     *         <code>IP</code> indicates that requests arriving from the same IP address are subject to the
     *         <code>RateLimit</code> that is specified in the <code>RateBasedRule</code>.
     * @see RateKey
     */
    public String rateKeyAsString() {
        return rateKey;
    }

    /**
     * <p>
     * The maximum number of requests, which have an identical value in the field specified by the <code>RateKey</code>,
     * allowed in a five-minute period. If the number of requests exceeds the <code>RateLimit</code> and the other
     * predicates specified in the rule are also met, AWS WAF triggers the action that is specified for this rule.
     * </p>
     * 
     * @return The maximum number of requests, which have an identical value in the field specified by the
     *         <code>RateKey</code>, allowed in a five-minute period. If the number of requests exceeds the
     *         <code>RateLimit</code> and the other predicates specified in the rule are also met, AWS WAF triggers the
     *         action that is specified for this rule.
     */
    public Long rateLimit() {
        return rateLimit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(matchPredicates());
        hashCode = 31 * hashCode + Objects.hashCode(rateKeyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(rateLimit());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RateBasedRule)) {
            return false;
        }
        RateBasedRule other = (RateBasedRule) obj;
        return Objects.equals(ruleId(), other.ruleId()) && Objects.equals(name(), other.name())
                && Objects.equals(metricName(), other.metricName()) && Objects.equals(matchPredicates(), other.matchPredicates())
                && Objects.equals(rateKeyAsString(), other.rateKeyAsString()) && Objects.equals(rateLimit(), other.rateLimit());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RateBasedRule").add("RuleId", ruleId()).add("Name", name()).add("MetricName", metricName())
                .add("MatchPredicates", matchPredicates()).add("RateKey", rateKeyAsString()).add("RateLimit", rateLimit())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleId":
            return Optional.ofNullable(clazz.cast(ruleId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "MatchPredicates":
            return Optional.ofNullable(clazz.cast(matchPredicates()));
        case "RateKey":
            return Optional.ofNullable(clazz.cast(rateKeyAsString()));
        case "RateLimit":
            return Optional.ofNullable(clazz.cast(rateLimit()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RateBasedRule, T> g) {
        return obj -> g.apply((RateBasedRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RateBasedRule> {
        /**
         * <p>
         * A unique identifier for a <code>RateBasedRule</code>. You use <code>RuleId</code> to get more information
         * about a <code>RateBasedRule</code> (see <a>GetRateBasedRule</a>), update a <code>RateBasedRule</code> (see
         * <a>UpdateRateBasedRule</a>), insert a <code>RateBasedRule</code> into a <code>WebACL</code> or delete one
         * from a <code>WebACL</code> (see <a>UpdateWebACL</a>), or delete a <code>RateBasedRule</code> from AWS WAF
         * (see <a>DeleteRateBasedRule</a>).
         * </p>
         * 
         * @param ruleId
         *        A unique identifier for a <code>RateBasedRule</code>. You use <code>RuleId</code> to get more
         *        information about a <code>RateBasedRule</code> (see <a>GetRateBasedRule</a>), update a
         *        <code>RateBasedRule</code> (see <a>UpdateRateBasedRule</a>), insert a <code>RateBasedRule</code> into
         *        a <code>WebACL</code> or delete one from a <code>WebACL</code> (see <a>UpdateWebACL</a>), or delete a
         *        <code>RateBasedRule</code> from AWS WAF (see <a>DeleteRateBasedRule</a>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleId(String ruleId);

        /**
         * <p>
         * A friendly name or description for a <code>RateBasedRule</code>. You can't change the name of a
         * <code>RateBasedRule</code> after you create it.
         * </p>
         * 
         * @param name
         *        A friendly name or description for a <code>RateBasedRule</code>. You can't change the name of a
         *        <code>RateBasedRule</code> after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A friendly name or description for the metrics for a <code>RateBasedRule</code>. The name can contain only
         * alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't contain
         * whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You can't change the
         * name of the metric after you create the <code>RateBasedRule</code>.
         * </p>
         * 
         * @param metricName
         *        A friendly name or description for the metrics for a <code>RateBasedRule</code>. The name can contain
         *        only alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't
         *        contain whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You
         *        can't change the name of the metric after you create the <code>RateBasedRule</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The <code>Predicates</code> object contains one <code>Predicate</code> element for each <a>ByteMatchSet</a>,
         * <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a <code>RateBasedRule</code>.
         * </p>
         * 
         * @param matchPredicates
         *        The <code>Predicates</code> object contains one <code>Predicate</code> element for each
         *        <a>ByteMatchSet</a>, <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a
         *        <code>RateBasedRule</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchPredicates(Collection<Predicate> matchPredicates);

        /**
         * <p>
         * The <code>Predicates</code> object contains one <code>Predicate</code> element for each <a>ByteMatchSet</a>,
         * <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a <code>RateBasedRule</code>.
         * </p>
         * 
         * @param matchPredicates
         *        The <code>Predicates</code> object contains one <code>Predicate</code> element for each
         *        <a>ByteMatchSet</a>, <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a
         *        <code>RateBasedRule</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchPredicates(Predicate... matchPredicates);

        /**
         * <p>
         * The <code>Predicates</code> object contains one <code>Predicate</code> element for each <a>ByteMatchSet</a>,
         * <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a <code>RateBasedRule</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Predicate>.Builder} avoiding the need to
         * create one manually via {@link List<Predicate>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Predicate>.Builder#build()} is called immediately and its
         * result is passed to {@link #matchPredicates(List<Predicate>)}.
         * 
         * @param matchPredicates
         *        a consumer that will call methods on {@link List<Predicate>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #matchPredicates(List<Predicate>)
         */
        Builder matchPredicates(Consumer<Predicate.Builder>... matchPredicates);

        /**
         * <p>
         * The field that AWS WAF uses to determine if requests are likely arriving from single source and thus subject
         * to rate monitoring. The only valid value for <code>RateKey</code> is <code>IP</code>. <code>IP</code>
         * indicates that requests arriving from the same IP address are subject to the <code>RateLimit</code> that is
         * specified in the <code>RateBasedRule</code>.
         * </p>
         * 
         * @param rateKey
         *        The field that AWS WAF uses to determine if requests are likely arriving from single source and thus
         *        subject to rate monitoring. The only valid value for <code>RateKey</code> is <code>IP</code>.
         *        <code>IP</code> indicates that requests arriving from the same IP address are subject to the
         *        <code>RateLimit</code> that is specified in the <code>RateBasedRule</code>.
         * @see RateKey
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RateKey
         */
        Builder rateKey(String rateKey);

        /**
         * <p>
         * The field that AWS WAF uses to determine if requests are likely arriving from single source and thus subject
         * to rate monitoring. The only valid value for <code>RateKey</code> is <code>IP</code>. <code>IP</code>
         * indicates that requests arriving from the same IP address are subject to the <code>RateLimit</code> that is
         * specified in the <code>RateBasedRule</code>.
         * </p>
         * 
         * @param rateKey
         *        The field that AWS WAF uses to determine if requests are likely arriving from single source and thus
         *        subject to rate monitoring. The only valid value for <code>RateKey</code> is <code>IP</code>.
         *        <code>IP</code> indicates that requests arriving from the same IP address are subject to the
         *        <code>RateLimit</code> that is specified in the <code>RateBasedRule</code>.
         * @see RateKey
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RateKey
         */
        Builder rateKey(RateKey rateKey);

        /**
         * <p>
         * The maximum number of requests, which have an identical value in the field specified by the
         * <code>RateKey</code>, allowed in a five-minute period. If the number of requests exceeds the
         * <code>RateLimit</code> and the other predicates specified in the rule are also met, AWS WAF triggers the
         * action that is specified for this rule.
         * </p>
         * 
         * @param rateLimit
         *        The maximum number of requests, which have an identical value in the field specified by the
         *        <code>RateKey</code>, allowed in a five-minute period. If the number of requests exceeds the
         *        <code>RateLimit</code> and the other predicates specified in the rule are also met, AWS WAF triggers
         *        the action that is specified for this rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rateLimit(Long rateLimit);
    }

    static final class BuilderImpl implements Builder {
        private String ruleId;

        private String name;

        private String metricName;

        private List<Predicate> matchPredicates = DefaultSdkAutoConstructList.getInstance();

        private String rateKey;

        private Long rateLimit;

        private BuilderImpl() {
        }

        private BuilderImpl(RateBasedRule model) {
            ruleId(model.ruleId);
            name(model.name);
            metricName(model.metricName);
            matchPredicates(model.matchPredicates);
            rateKey(model.rateKey);
            rateLimit(model.rateLimit);
        }

        public final String getRuleId() {
            return ruleId;
        }

        @Override
        public final Builder ruleId(String ruleId) {
            this.ruleId = ruleId;
            return this;
        }

        public final void setRuleId(String ruleId) {
            this.ruleId = ruleId;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final Collection<Predicate.Builder> getMatchPredicates() {
            return matchPredicates != null ? matchPredicates.stream().map(Predicate::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder matchPredicates(Collection<Predicate> matchPredicates) {
            this.matchPredicates = PredicatesCopier.copy(matchPredicates);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchPredicates(Predicate... matchPredicates) {
            matchPredicates(Arrays.asList(matchPredicates));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchPredicates(Consumer<Predicate.Builder>... matchPredicates) {
            matchPredicates(Stream.of(matchPredicates).map(c -> Predicate.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMatchPredicates(Collection<Predicate.BuilderImpl> matchPredicates) {
            this.matchPredicates = PredicatesCopier.copyFromBuilder(matchPredicates);
        }

        public final String getRateKey() {
            return rateKey;
        }

        @Override
        public final Builder rateKey(String rateKey) {
            this.rateKey = rateKey;
            return this;
        }

        @Override
        public final Builder rateKey(RateKey rateKey) {
            this.rateKey(rateKey == null ? null : rateKey.toString());
            return this;
        }

        public final void setRateKey(String rateKey) {
            this.rateKey = rateKey;
        }

        public final Long getRateLimit() {
            return rateLimit;
        }

        @Override
        public final Builder rateLimit(Long rateLimit) {
            this.rateLimit = rateLimit;
            return this;
        }

        public final void setRateLimit(Long rateLimit) {
            this.rateLimit = rateLimit;
        }

        @Override
        public RateBasedRule build() {
            return new RateBasedRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
