/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.waf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The <code>ActivatedRule</code> object in an <a>UpdateWebACL</a> request specifies a <code>Rule</code> that you want
 * to insert or delete, the priority of the <code>Rule</code> in the <code>WebACL</code>, and the action that you want
 * AWS WAF to take when a web request matches the <code>Rule</code> (<code>ALLOW</code>, <code>BLOCK</code>, or
 * <code>COUNT</code>).
 * </p>
 * <p>
 * To specify whether to insert or delete a <code>Rule</code>, use the <code>Action</code> parameter in the
 * <a>WebACLUpdate</a> data type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ActivatedRule implements SdkPojo, Serializable, ToCopyableBuilder<ActivatedRule.Builder, ActivatedRule> {
    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ActivatedRule::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<String> RULE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ActivatedRule::ruleId)).setter(setter(Builder::ruleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleId").build()).build();

    private static final SdkField<WafAction> ACTION_FIELD = SdkField.<WafAction> builder(MarshallingType.SDK_POJO)
            .getter(getter(ActivatedRule::action)).setter(setter(Builder::action)).constructor(WafAction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<WafOverrideAction> OVERRIDE_ACTION_FIELD = SdkField
            .<WafOverrideAction> builder(MarshallingType.SDK_POJO).getter(getter(ActivatedRule::overrideAction))
            .setter(setter(Builder::overrideAction)).constructor(WafOverrideAction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OverrideAction").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ActivatedRule::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<ExcludedRule>> EXCLUDED_RULES_FIELD = SdkField
            .<List<ExcludedRule>> builder(MarshallingType.LIST)
            .getter(getter(ActivatedRule::excludedRules))
            .setter(setter(Builder::excludedRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludedRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ExcludedRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(ExcludedRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PRIORITY_FIELD, RULE_ID_FIELD,
            ACTION_FIELD, OVERRIDE_ACTION_FIELD, TYPE_FIELD, EXCLUDED_RULES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer priority;

    private final String ruleId;

    private final WafAction action;

    private final WafOverrideAction overrideAction;

    private final String type;

    private final List<ExcludedRule> excludedRules;

    private ActivatedRule(BuilderImpl builder) {
        this.priority = builder.priority;
        this.ruleId = builder.ruleId;
        this.action = builder.action;
        this.overrideAction = builder.overrideAction;
        this.type = builder.type;
        this.excludedRules = builder.excludedRules;
    }

    /**
     * <p>
     * Specifies the order in which the <code>Rules</code> in a <code>WebACL</code> are evaluated. Rules with a lower
     * value for <code>Priority</code> are evaluated before <code>Rules</code> with a higher value. The value must be a
     * unique integer. If you add multiple <code>Rules</code> to a <code>WebACL</code>, the values don't need to be
     * consecutive.
     * </p>
     * 
     * @return Specifies the order in which the <code>Rules</code> in a <code>WebACL</code> are evaluated. Rules with a
     *         lower value for <code>Priority</code> are evaluated before <code>Rules</code> with a higher value. The
     *         value must be a unique integer. If you add multiple <code>Rules</code> to a <code>WebACL</code>, the
     *         values don't need to be consecutive.
     */
    public Integer priority() {
        return priority;
    }

    /**
     * <p>
     * The <code>RuleId</code> for a <code>Rule</code>. You use <code>RuleId</code> to get more information about a
     * <code>Rule</code> (see <a>GetRule</a>), update a <code>Rule</code> (see <a>UpdateRule</a>), insert a
     * <code>Rule</code> into a <code>WebACL</code> or delete a one from a <code>WebACL</code> (see
     * <a>UpdateWebACL</a>), or delete a <code>Rule</code> from AWS WAF (see <a>DeleteRule</a>).
     * </p>
     * <p>
     * <code>RuleId</code> is returned by <a>CreateRule</a> and by <a>ListRules</a>.
     * </p>
     * 
     * @return The <code>RuleId</code> for a <code>Rule</code>. You use <code>RuleId</code> to get more information
     *         about a <code>Rule</code> (see <a>GetRule</a>), update a <code>Rule</code> (see <a>UpdateRule</a>),
     *         insert a <code>Rule</code> into a <code>WebACL</code> or delete a one from a <code>WebACL</code> (see
     *         <a>UpdateWebACL</a>), or delete a <code>Rule</code> from AWS WAF (see <a>DeleteRule</a>).</p>
     *         <p>
     *         <code>RuleId</code> is returned by <a>CreateRule</a> and by <a>ListRules</a>.
     */
    public String ruleId() {
        return ruleId;
    }

    /**
     * <p>
     * Specifies the action that CloudFront or AWS WAF takes when a web request matches the conditions in the
     * <code>Rule</code>. Valid values for <code>Action</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALLOW</code>: CloudFront responds with the requested object.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BLOCK</code>: CloudFront responds with an HTTP 403 (Forbidden) status code.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COUNT</code>: AWS WAF increments a counter of requests that match the conditions in the rule and then
     * continues to inspect the web request based on the remaining rules in the web ACL.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a <code>RuleGroup</code> to a
     * <code>WebACL</code>. In this case, you do not use <code>ActivatedRule|Action</code>. For all other update
     * requests, <code>ActivatedRule|Action</code> is used instead of <code>ActivatedRule|OverrideAction</code>.
     * </p>
     * 
     * @return Specifies the action that CloudFront or AWS WAF takes when a web request matches the conditions in the
     *         <code>Rule</code>. Valid values for <code>Action</code> include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALLOW</code>: CloudFront responds with the requested object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BLOCK</code>: CloudFront responds with an HTTP 403 (Forbidden) status code.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COUNT</code>: AWS WAF increments a counter of requests that match the conditions in the rule and
     *         then continues to inspect the web request based on the remaining rules in the web ACL.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a <code>RuleGroup</code>
     *         to a <code>WebACL</code>. In this case, you do not use <code>ActivatedRule|Action</code>. For all other
     *         update requests, <code>ActivatedRule|Action</code> is used instead of
     *         <code>ActivatedRule|OverrideAction</code>.
     */
    public WafAction action() {
        return action;
    }

    /**
     * <p>
     * Use the <code>OverrideAction</code> to test your <code>RuleGroup</code>.
     * </p>
     * <p>
     * Any rule in a <code>RuleGroup</code> can potentially block a request. If you set the <code>OverrideAction</code>
     * to <code>None</code>, the <code>RuleGroup</code> will block a request if any individual rule in the
     * <code>RuleGroup</code> matches the request and is configured to block that request. However if you first want to
     * test the <code>RuleGroup</code>, set the <code>OverrideAction</code> to <code>Count</code>. The
     * <code>RuleGroup</code> will then override any block action specified by individual rules contained within the
     * group. Instead of blocking matching requests, those requests will be counted. You can view a record of counted
     * requests using <a>GetSampledRequests</a>.
     * </p>
     * <p>
     * <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a <code>RuleGroup</code> to a
     * <code>WebACL</code>. In this case you do not use <code>ActivatedRule|Action</code>. For all other update
     * requests, <code>ActivatedRule|Action</code> is used instead of <code>ActivatedRule|OverrideAction</code>.
     * </p>
     * 
     * @return Use the <code>OverrideAction</code> to test your <code>RuleGroup</code>.</p>
     *         <p>
     *         Any rule in a <code>RuleGroup</code> can potentially block a request. If you set the
     *         <code>OverrideAction</code> to <code>None</code>, the <code>RuleGroup</code> will block a request if any
     *         individual rule in the <code>RuleGroup</code> matches the request and is configured to block that
     *         request. However if you first want to test the <code>RuleGroup</code>, set the
     *         <code>OverrideAction</code> to <code>Count</code>. The <code>RuleGroup</code> will then override any
     *         block action specified by individual rules contained within the group. Instead of blocking matching
     *         requests, those requests will be counted. You can view a record of counted requests using
     *         <a>GetSampledRequests</a>.
     *         </p>
     *         <p>
     *         <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a <code>RuleGroup</code>
     *         to a <code>WebACL</code>. In this case you do not use <code>ActivatedRule|Action</code>. For all other
     *         update requests, <code>ActivatedRule|Action</code> is used instead of
     *         <code>ActivatedRule|OverrideAction</code>.
     */
    public WafOverrideAction overrideAction() {
        return overrideAction;
    }

    /**
     * <p>
     * The rule type, either <code>REGULAR</code>, as defined by <a>Rule</a>, <code>RATE_BASED</code>, as defined by
     * <a>RateBasedRule</a>, or <code>GROUP</code>, as defined by <a>RuleGroup</a>. The default is REGULAR. Although
     * this field is optional, be aware that if you try to add a RATE_BASED rule to a web ACL without setting the type,
     * the <a>UpdateWebACL</a> request will fail because the request tries to add a REGULAR rule with the specified ID,
     * which does not exist.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link WafRuleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The rule type, either <code>REGULAR</code>, as defined by <a>Rule</a>, <code>RATE_BASED</code>, as
     *         defined by <a>RateBasedRule</a>, or <code>GROUP</code>, as defined by <a>RuleGroup</a>. The default is
     *         REGULAR. Although this field is optional, be aware that if you try to add a RATE_BASED rule to a web ACL
     *         without setting the type, the <a>UpdateWebACL</a> request will fail because the request tries to add a
     *         REGULAR rule with the specified ID, which does not exist.
     * @see WafRuleType
     */
    public WafRuleType type() {
        return WafRuleType.fromValue(type);
    }

    /**
     * <p>
     * The rule type, either <code>REGULAR</code>, as defined by <a>Rule</a>, <code>RATE_BASED</code>, as defined by
     * <a>RateBasedRule</a>, or <code>GROUP</code>, as defined by <a>RuleGroup</a>. The default is REGULAR. Although
     * this field is optional, be aware that if you try to add a RATE_BASED rule to a web ACL without setting the type,
     * the <a>UpdateWebACL</a> request will fail because the request tries to add a REGULAR rule with the specified ID,
     * which does not exist.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link WafRuleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The rule type, either <code>REGULAR</code>, as defined by <a>Rule</a>, <code>RATE_BASED</code>, as
     *         defined by <a>RateBasedRule</a>, or <code>GROUP</code>, as defined by <a>RuleGroup</a>. The default is
     *         REGULAR. Although this field is optional, be aware that if you try to add a RATE_BASED rule to a web ACL
     *         without setting the type, the <a>UpdateWebACL</a> request will fail because the request tries to add a
     *         REGULAR rule with the specified ID, which does not exist.
     * @see WafRuleType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * Returns true if the ExcludedRules property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasExcludedRules() {
        return excludedRules != null && !(excludedRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of rules to exclude from a rule group. This is applicable only when the <code>ActivatedRule</code>
     * refers to a <code>RuleGroup</code>.
     * </p>
     * <p>
     * Sometimes it is necessary to troubleshoot rule groups that are blocking traffic unexpectedly (false positives).
     * One troubleshooting technique is to identify the specific rule within the rule group that is blocking the
     * legitimate traffic and then disable (exclude) that particular rule. You can exclude rules from both your own rule
     * groups and AWS Marketplace rule groups that have been associated with a web ACL.
     * </p>
     * <p>
     * Specifying <code>ExcludedRules</code> does not remove those rules from the rule group. Rather, it changes the
     * action for the rules to <code>COUNT</code>. Therefore, requests that match an <code>ExcludedRule</code> are
     * counted but not blocked. The <code>RuleGroup</code> owner will receive COUNT metrics for each
     * <code>ExcludedRule</code>.
     * </p>
     * <p>
     * If you want to exclude rules from a rule group that is already associated with a web ACL, perform the following
     * steps:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Use the AWS WAF logs to identify the IDs of the rules that you want to exclude. For more information about the
     * logs, see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging Web ACL Traffic
     * Information</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Submit an <a>UpdateWebACL</a> request that has two actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The first action deletes the existing rule group from the web ACL. That is, in the <a>UpdateWebACL</a> request,
     * the first <code>Updates:Action</code> should be <code>DELETE</code> and <code>Updates:ActivatedRule:RuleId</code>
     * should be the rule group that contains the rules that you want to exclude.
     * </p>
     * </li>
     * <li>
     * <p>
     * The second action inserts the same rule group back in, but specifying the rules to exclude. That is, the second
     * <code>Updates:Action</code> should be <code>INSERT</code>, <code>Updates:ActivatedRule:RuleId</code> should be
     * the rule group that you just removed, and <code>ExcludedRules</code> should contain the rules that you want to
     * exclude.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ol>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasExcludedRules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of rules to exclude from a rule group. This is applicable only when the
     *         <code>ActivatedRule</code> refers to a <code>RuleGroup</code>.</p>
     *         <p>
     *         Sometimes it is necessary to troubleshoot rule groups that are blocking traffic unexpectedly (false
     *         positives). One troubleshooting technique is to identify the specific rule within the rule group that is
     *         blocking the legitimate traffic and then disable (exclude) that particular rule. You can exclude rules
     *         from both your own rule groups and AWS Marketplace rule groups that have been associated with a web ACL.
     *         </p>
     *         <p>
     *         Specifying <code>ExcludedRules</code> does not remove those rules from the rule group. Rather, it changes
     *         the action for the rules to <code>COUNT</code>. Therefore, requests that match an
     *         <code>ExcludedRule</code> are counted but not blocked. The <code>RuleGroup</code> owner will receive
     *         COUNT metrics for each <code>ExcludedRule</code>.
     *         </p>
     *         <p>
     *         If you want to exclude rules from a rule group that is already associated with a web ACL, perform the
     *         following steps:
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         Use the AWS WAF logs to identify the IDs of the rules that you want to exclude. For more information
     *         about the logs, see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging
     *         Web ACL Traffic Information</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Submit an <a>UpdateWebACL</a> request that has two actions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The first action deletes the existing rule group from the web ACL. That is, in the <a>UpdateWebACL</a>
     *         request, the first <code>Updates:Action</code> should be <code>DELETE</code> and
     *         <code>Updates:ActivatedRule:RuleId</code> should be the rule group that contains the rules that you want
     *         to exclude.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The second action inserts the same rule group back in, but specifying the rules to exclude. That is, the
     *         second <code>Updates:Action</code> should be <code>INSERT</code>,
     *         <code>Updates:ActivatedRule:RuleId</code> should be the rule group that you just removed, and
     *         <code>ExcludedRules</code> should contain the rules that you want to exclude.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public List<ExcludedRule> excludedRules() {
        return excludedRules;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(ruleId());
        hashCode = 31 * hashCode + Objects.hashCode(action());
        hashCode = 31 * hashCode + Objects.hashCode(overrideAction());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(excludedRules());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ActivatedRule)) {
            return false;
        }
        ActivatedRule other = (ActivatedRule) obj;
        return Objects.equals(priority(), other.priority()) && Objects.equals(ruleId(), other.ruleId())
                && Objects.equals(action(), other.action()) && Objects.equals(overrideAction(), other.overrideAction())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(excludedRules(), other.excludedRules());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ActivatedRule").add("Priority", priority()).add("RuleId", ruleId()).add("Action", action())
                .add("OverrideAction", overrideAction()).add("Type", typeAsString()).add("ExcludedRules", excludedRules())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "RuleId":
            return Optional.ofNullable(clazz.cast(ruleId()));
        case "Action":
            return Optional.ofNullable(clazz.cast(action()));
        case "OverrideAction":
            return Optional.ofNullable(clazz.cast(overrideAction()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "ExcludedRules":
            return Optional.ofNullable(clazz.cast(excludedRules()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ActivatedRule, T> g) {
        return obj -> g.apply((ActivatedRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ActivatedRule> {
        /**
         * <p>
         * Specifies the order in which the <code>Rules</code> in a <code>WebACL</code> are evaluated. Rules with a
         * lower value for <code>Priority</code> are evaluated before <code>Rules</code> with a higher value. The value
         * must be a unique integer. If you add multiple <code>Rules</code> to a <code>WebACL</code>, the values don't
         * need to be consecutive.
         * </p>
         * 
         * @param priority
         *        Specifies the order in which the <code>Rules</code> in a <code>WebACL</code> are evaluated. Rules with
         *        a lower value for <code>Priority</code> are evaluated before <code>Rules</code> with a higher value.
         *        The value must be a unique integer. If you add multiple <code>Rules</code> to a <code>WebACL</code>,
         *        the values don't need to be consecutive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * The <code>RuleId</code> for a <code>Rule</code>. You use <code>RuleId</code> to get more information about a
         * <code>Rule</code> (see <a>GetRule</a>), update a <code>Rule</code> (see <a>UpdateRule</a>), insert a
         * <code>Rule</code> into a <code>WebACL</code> or delete a one from a <code>WebACL</code> (see
         * <a>UpdateWebACL</a>), or delete a <code>Rule</code> from AWS WAF (see <a>DeleteRule</a>).
         * </p>
         * <p>
         * <code>RuleId</code> is returned by <a>CreateRule</a> and by <a>ListRules</a>.
         * </p>
         * 
         * @param ruleId
         *        The <code>RuleId</code> for a <code>Rule</code>. You use <code>RuleId</code> to get more information
         *        about a <code>Rule</code> (see <a>GetRule</a>), update a <code>Rule</code> (see <a>UpdateRule</a>),
         *        insert a <code>Rule</code> into a <code>WebACL</code> or delete a one from a <code>WebACL</code> (see
         *        <a>UpdateWebACL</a>), or delete a <code>Rule</code> from AWS WAF (see <a>DeleteRule</a>).</p>
         *        <p>
         *        <code>RuleId</code> is returned by <a>CreateRule</a> and by <a>ListRules</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleId(String ruleId);

        /**
         * <p>
         * Specifies the action that CloudFront or AWS WAF takes when a web request matches the conditions in the
         * <code>Rule</code>. Valid values for <code>Action</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALLOW</code>: CloudFront responds with the requested object.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BLOCK</code>: CloudFront responds with an HTTP 403 (Forbidden) status code.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COUNT</code>: AWS WAF increments a counter of requests that match the conditions in the rule and then
         * continues to inspect the web request based on the remaining rules in the web ACL.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a <code>RuleGroup</code> to a
         * <code>WebACL</code>. In this case, you do not use <code>ActivatedRule|Action</code>. For all other update
         * requests, <code>ActivatedRule|Action</code> is used instead of <code>ActivatedRule|OverrideAction</code>.
         * </p>
         * 
         * @param action
         *        Specifies the action that CloudFront or AWS WAF takes when a web request matches the conditions in the
         *        <code>Rule</code>. Valid values for <code>Action</code> include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALLOW</code>: CloudFront responds with the requested object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BLOCK</code>: CloudFront responds with an HTTP 403 (Forbidden) status code.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COUNT</code>: AWS WAF increments a counter of requests that match the conditions in the rule and
         *        then continues to inspect the web request based on the remaining rules in the web ACL.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a
         *        <code>RuleGroup</code> to a <code>WebACL</code>. In this case, you do not use
         *        <code>ActivatedRule|Action</code>. For all other update requests, <code>ActivatedRule|Action</code> is
         *        used instead of <code>ActivatedRule|OverrideAction</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder action(WafAction action);

        /**
         * <p>
         * Specifies the action that CloudFront or AWS WAF takes when a web request matches the conditions in the
         * <code>Rule</code>. Valid values for <code>Action</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALLOW</code>: CloudFront responds with the requested object.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BLOCK</code>: CloudFront responds with an HTTP 403 (Forbidden) status code.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COUNT</code>: AWS WAF increments a counter of requests that match the conditions in the rule and then
         * continues to inspect the web request based on the remaining rules in the web ACL.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a <code>RuleGroup</code> to a
         * <code>WebACL</code>. In this case, you do not use <code>ActivatedRule|Action</code>. For all other update
         * requests, <code>ActivatedRule|Action</code> is used instead of <code>ActivatedRule|OverrideAction</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link WafAction.Builder} avoiding the need to create
         * one manually via {@link WafAction#builder()}.
         *
         * When the {@link Consumer} completes, {@link WafAction.Builder#build()} is called immediately and its result
         * is passed to {@link #action(WafAction)}.
         * 
         * @param action
         *        a consumer that will call methods on {@link WafAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #action(WafAction)
         */
        default Builder action(Consumer<WafAction.Builder> action) {
            return action(WafAction.builder().applyMutation(action).build());
        }

        /**
         * <p>
         * Use the <code>OverrideAction</code> to test your <code>RuleGroup</code>.
         * </p>
         * <p>
         * Any rule in a <code>RuleGroup</code> can potentially block a request. If you set the
         * <code>OverrideAction</code> to <code>None</code>, the <code>RuleGroup</code> will block a request if any
         * individual rule in the <code>RuleGroup</code> matches the request and is configured to block that request.
         * However if you first want to test the <code>RuleGroup</code>, set the <code>OverrideAction</code> to
         * <code>Count</code>. The <code>RuleGroup</code> will then override any block action specified by individual
         * rules contained within the group. Instead of blocking matching requests, those requests will be counted. You
         * can view a record of counted requests using <a>GetSampledRequests</a>.
         * </p>
         * <p>
         * <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a <code>RuleGroup</code> to a
         * <code>WebACL</code>. In this case you do not use <code>ActivatedRule|Action</code>. For all other update
         * requests, <code>ActivatedRule|Action</code> is used instead of <code>ActivatedRule|OverrideAction</code>.
         * </p>
         * 
         * @param overrideAction
         *        Use the <code>OverrideAction</code> to test your <code>RuleGroup</code>.</p>
         *        <p>
         *        Any rule in a <code>RuleGroup</code> can potentially block a request. If you set the
         *        <code>OverrideAction</code> to <code>None</code>, the <code>RuleGroup</code> will block a request if
         *        any individual rule in the <code>RuleGroup</code> matches the request and is configured to block that
         *        request. However if you first want to test the <code>RuleGroup</code>, set the
         *        <code>OverrideAction</code> to <code>Count</code>. The <code>RuleGroup</code> will then override any
         *        block action specified by individual rules contained within the group. Instead of blocking matching
         *        requests, those requests will be counted. You can view a record of counted requests using
         *        <a>GetSampledRequests</a>.
         *        </p>
         *        <p>
         *        <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a
         *        <code>RuleGroup</code> to a <code>WebACL</code>. In this case you do not use
         *        <code>ActivatedRule|Action</code>. For all other update requests, <code>ActivatedRule|Action</code> is
         *        used instead of <code>ActivatedRule|OverrideAction</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrideAction(WafOverrideAction overrideAction);

        /**
         * <p>
         * Use the <code>OverrideAction</code> to test your <code>RuleGroup</code>.
         * </p>
         * <p>
         * Any rule in a <code>RuleGroup</code> can potentially block a request. If you set the
         * <code>OverrideAction</code> to <code>None</code>, the <code>RuleGroup</code> will block a request if any
         * individual rule in the <code>RuleGroup</code> matches the request and is configured to block that request.
         * However if you first want to test the <code>RuleGroup</code>, set the <code>OverrideAction</code> to
         * <code>Count</code>. The <code>RuleGroup</code> will then override any block action specified by individual
         * rules contained within the group. Instead of blocking matching requests, those requests will be counted. You
         * can view a record of counted requests using <a>GetSampledRequests</a>.
         * </p>
         * <p>
         * <code>ActivatedRule|OverrideAction</code> applies only when updating or adding a <code>RuleGroup</code> to a
         * <code>WebACL</code>. In this case you do not use <code>ActivatedRule|Action</code>. For all other update
         * requests, <code>ActivatedRule|Action</code> is used instead of <code>ActivatedRule|OverrideAction</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link WafOverrideAction.Builder} avoiding the need to
         * create one manually via {@link WafOverrideAction#builder()}.
         *
         * When the {@link Consumer} completes, {@link WafOverrideAction.Builder#build()} is called immediately and its
         * result is passed to {@link #overrideAction(WafOverrideAction)}.
         * 
         * @param overrideAction
         *        a consumer that will call methods on {@link WafOverrideAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overrideAction(WafOverrideAction)
         */
        default Builder overrideAction(Consumer<WafOverrideAction.Builder> overrideAction) {
            return overrideAction(WafOverrideAction.builder().applyMutation(overrideAction).build());
        }

        /**
         * <p>
         * The rule type, either <code>REGULAR</code>, as defined by <a>Rule</a>, <code>RATE_BASED</code>, as defined by
         * <a>RateBasedRule</a>, or <code>GROUP</code>, as defined by <a>RuleGroup</a>. The default is REGULAR. Although
         * this field is optional, be aware that if you try to add a RATE_BASED rule to a web ACL without setting the
         * type, the <a>UpdateWebACL</a> request will fail because the request tries to add a REGULAR rule with the
         * specified ID, which does not exist.
         * </p>
         * 
         * @param type
         *        The rule type, either <code>REGULAR</code>, as defined by <a>Rule</a>, <code>RATE_BASED</code>, as
         *        defined by <a>RateBasedRule</a>, or <code>GROUP</code>, as defined by <a>RuleGroup</a>. The default is
         *        REGULAR. Although this field is optional, be aware that if you try to add a RATE_BASED rule to a web
         *        ACL without setting the type, the <a>UpdateWebACL</a> request will fail because the request tries to
         *        add a REGULAR rule with the specified ID, which does not exist.
         * @see WafRuleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WafRuleType
         */
        Builder type(String type);

        /**
         * <p>
         * The rule type, either <code>REGULAR</code>, as defined by <a>Rule</a>, <code>RATE_BASED</code>, as defined by
         * <a>RateBasedRule</a>, or <code>GROUP</code>, as defined by <a>RuleGroup</a>. The default is REGULAR. Although
         * this field is optional, be aware that if you try to add a RATE_BASED rule to a web ACL without setting the
         * type, the <a>UpdateWebACL</a> request will fail because the request tries to add a REGULAR rule with the
         * specified ID, which does not exist.
         * </p>
         * 
         * @param type
         *        The rule type, either <code>REGULAR</code>, as defined by <a>Rule</a>, <code>RATE_BASED</code>, as
         *        defined by <a>RateBasedRule</a>, or <code>GROUP</code>, as defined by <a>RuleGroup</a>. The default is
         *        REGULAR. Although this field is optional, be aware that if you try to add a RATE_BASED rule to a web
         *        ACL without setting the type, the <a>UpdateWebACL</a> request will fail because the request tries to
         *        add a REGULAR rule with the specified ID, which does not exist.
         * @see WafRuleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WafRuleType
         */
        Builder type(WafRuleType type);

        /**
         * <p>
         * An array of rules to exclude from a rule group. This is applicable only when the <code>ActivatedRule</code>
         * refers to a <code>RuleGroup</code>.
         * </p>
         * <p>
         * Sometimes it is necessary to troubleshoot rule groups that are blocking traffic unexpectedly (false
         * positives). One troubleshooting technique is to identify the specific rule within the rule group that is
         * blocking the legitimate traffic and then disable (exclude) that particular rule. You can exclude rules from
         * both your own rule groups and AWS Marketplace rule groups that have been associated with a web ACL.
         * </p>
         * <p>
         * Specifying <code>ExcludedRules</code> does not remove those rules from the rule group. Rather, it changes the
         * action for the rules to <code>COUNT</code>. Therefore, requests that match an <code>ExcludedRule</code> are
         * counted but not blocked. The <code>RuleGroup</code> owner will receive COUNT metrics for each
         * <code>ExcludedRule</code>.
         * </p>
         * <p>
         * If you want to exclude rules from a rule group that is already associated with a web ACL, perform the
         * following steps:
         * </p>
         * <ol>
         * <li>
         * <p>
         * Use the AWS WAF logs to identify the IDs of the rules that you want to exclude. For more information about
         * the logs, see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging Web ACL
         * Traffic Information</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Submit an <a>UpdateWebACL</a> request that has two actions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The first action deletes the existing rule group from the web ACL. That is, in the <a>UpdateWebACL</a>
         * request, the first <code>Updates:Action</code> should be <code>DELETE</code> and
         * <code>Updates:ActivatedRule:RuleId</code> should be the rule group that contains the rules that you want to
         * exclude.
         * </p>
         * </li>
         * <li>
         * <p>
         * The second action inserts the same rule group back in, but specifying the rules to exclude. That is, the
         * second <code>Updates:Action</code> should be <code>INSERT</code>, <code>Updates:ActivatedRule:RuleId</code>
         * should be the rule group that you just removed, and <code>ExcludedRules</code> should contain the rules that
         * you want to exclude.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ol>
         * 
         * @param excludedRules
         *        An array of rules to exclude from a rule group. This is applicable only when the
         *        <code>ActivatedRule</code> refers to a <code>RuleGroup</code>.</p>
         *        <p>
         *        Sometimes it is necessary to troubleshoot rule groups that are blocking traffic unexpectedly (false
         *        positives). One troubleshooting technique is to identify the specific rule within the rule group that
         *        is blocking the legitimate traffic and then disable (exclude) that particular rule. You can exclude
         *        rules from both your own rule groups and AWS Marketplace rule groups that have been associated with a
         *        web ACL.
         *        </p>
         *        <p>
         *        Specifying <code>ExcludedRules</code> does not remove those rules from the rule group. Rather, it
         *        changes the action for the rules to <code>COUNT</code>. Therefore, requests that match an
         *        <code>ExcludedRule</code> are counted but not blocked. The <code>RuleGroup</code> owner will receive
         *        COUNT metrics for each <code>ExcludedRule</code>.
         *        </p>
         *        <p>
         *        If you want to exclude rules from a rule group that is already associated with a web ACL, perform the
         *        following steps:
         *        </p>
         *        <ol>
         *        <li>
         *        <p>
         *        Use the AWS WAF logs to identify the IDs of the rules that you want to exclude. For more information
         *        about the logs, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging Web ACL Traffic
         *        Information</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Submit an <a>UpdateWebACL</a> request that has two actions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The first action deletes the existing rule group from the web ACL. That is, in the <a>UpdateWebACL</a>
         *        request, the first <code>Updates:Action</code> should be <code>DELETE</code> and
         *        <code>Updates:ActivatedRule:RuleId</code> should be the rule group that contains the rules that you
         *        want to exclude.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The second action inserts the same rule group back in, but specifying the rules to exclude. That is,
         *        the second <code>Updates:Action</code> should be <code>INSERT</code>,
         *        <code>Updates:ActivatedRule:RuleId</code> should be the rule group that you just removed, and
         *        <code>ExcludedRules</code> should contain the rules that you want to exclude.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedRules(Collection<ExcludedRule> excludedRules);

        /**
         * <p>
         * An array of rules to exclude from a rule group. This is applicable only when the <code>ActivatedRule</code>
         * refers to a <code>RuleGroup</code>.
         * </p>
         * <p>
         * Sometimes it is necessary to troubleshoot rule groups that are blocking traffic unexpectedly (false
         * positives). One troubleshooting technique is to identify the specific rule within the rule group that is
         * blocking the legitimate traffic and then disable (exclude) that particular rule. You can exclude rules from
         * both your own rule groups and AWS Marketplace rule groups that have been associated with a web ACL.
         * </p>
         * <p>
         * Specifying <code>ExcludedRules</code> does not remove those rules from the rule group. Rather, it changes the
         * action for the rules to <code>COUNT</code>. Therefore, requests that match an <code>ExcludedRule</code> are
         * counted but not blocked. The <code>RuleGroup</code> owner will receive COUNT metrics for each
         * <code>ExcludedRule</code>.
         * </p>
         * <p>
         * If you want to exclude rules from a rule group that is already associated with a web ACL, perform the
         * following steps:
         * </p>
         * <ol>
         * <li>
         * <p>
         * Use the AWS WAF logs to identify the IDs of the rules that you want to exclude. For more information about
         * the logs, see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging Web ACL
         * Traffic Information</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Submit an <a>UpdateWebACL</a> request that has two actions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The first action deletes the existing rule group from the web ACL. That is, in the <a>UpdateWebACL</a>
         * request, the first <code>Updates:Action</code> should be <code>DELETE</code> and
         * <code>Updates:ActivatedRule:RuleId</code> should be the rule group that contains the rules that you want to
         * exclude.
         * </p>
         * </li>
         * <li>
         * <p>
         * The second action inserts the same rule group back in, but specifying the rules to exclude. That is, the
         * second <code>Updates:Action</code> should be <code>INSERT</code>, <code>Updates:ActivatedRule:RuleId</code>
         * should be the rule group that you just removed, and <code>ExcludedRules</code> should contain the rules that
         * you want to exclude.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ol>
         * 
         * @param excludedRules
         *        An array of rules to exclude from a rule group. This is applicable only when the
         *        <code>ActivatedRule</code> refers to a <code>RuleGroup</code>.</p>
         *        <p>
         *        Sometimes it is necessary to troubleshoot rule groups that are blocking traffic unexpectedly (false
         *        positives). One troubleshooting technique is to identify the specific rule within the rule group that
         *        is blocking the legitimate traffic and then disable (exclude) that particular rule. You can exclude
         *        rules from both your own rule groups and AWS Marketplace rule groups that have been associated with a
         *        web ACL.
         *        </p>
         *        <p>
         *        Specifying <code>ExcludedRules</code> does not remove those rules from the rule group. Rather, it
         *        changes the action for the rules to <code>COUNT</code>. Therefore, requests that match an
         *        <code>ExcludedRule</code> are counted but not blocked. The <code>RuleGroup</code> owner will receive
         *        COUNT metrics for each <code>ExcludedRule</code>.
         *        </p>
         *        <p>
         *        If you want to exclude rules from a rule group that is already associated with a web ACL, perform the
         *        following steps:
         *        </p>
         *        <ol>
         *        <li>
         *        <p>
         *        Use the AWS WAF logs to identify the IDs of the rules that you want to exclude. For more information
         *        about the logs, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging Web ACL Traffic
         *        Information</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Submit an <a>UpdateWebACL</a> request that has two actions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The first action deletes the existing rule group from the web ACL. That is, in the <a>UpdateWebACL</a>
         *        request, the first <code>Updates:Action</code> should be <code>DELETE</code> and
         *        <code>Updates:ActivatedRule:RuleId</code> should be the rule group that contains the rules that you
         *        want to exclude.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The second action inserts the same rule group back in, but specifying the rules to exclude. That is,
         *        the second <code>Updates:Action</code> should be <code>INSERT</code>,
         *        <code>Updates:ActivatedRule:RuleId</code> should be the rule group that you just removed, and
         *        <code>ExcludedRules</code> should contain the rules that you want to exclude.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedRules(ExcludedRule... excludedRules);

        /**
         * <p>
         * An array of rules to exclude from a rule group. This is applicable only when the <code>ActivatedRule</code>
         * refers to a <code>RuleGroup</code>.
         * </p>
         * <p>
         * Sometimes it is necessary to troubleshoot rule groups that are blocking traffic unexpectedly (false
         * positives). One troubleshooting technique is to identify the specific rule within the rule group that is
         * blocking the legitimate traffic and then disable (exclude) that particular rule. You can exclude rules from
         * both your own rule groups and AWS Marketplace rule groups that have been associated with a web ACL.
         * </p>
         * <p>
         * Specifying <code>ExcludedRules</code> does not remove those rules from the rule group. Rather, it changes the
         * action for the rules to <code>COUNT</code>. Therefore, requests that match an <code>ExcludedRule</code> are
         * counted but not blocked. The <code>RuleGroup</code> owner will receive COUNT metrics for each
         * <code>ExcludedRule</code>.
         * </p>
         * <p>
         * If you want to exclude rules from a rule group that is already associated with a web ACL, perform the
         * following steps:
         * </p>
         * <ol>
         * <li>
         * <p>
         * Use the AWS WAF logs to identify the IDs of the rules that you want to exclude. For more information about
         * the logs, see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging Web ACL
         * Traffic Information</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Submit an <a>UpdateWebACL</a> request that has two actions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The first action deletes the existing rule group from the web ACL. That is, in the <a>UpdateWebACL</a>
         * request, the first <code>Updates:Action</code> should be <code>DELETE</code> and
         * <code>Updates:ActivatedRule:RuleId</code> should be the rule group that contains the rules that you want to
         * exclude.
         * </p>
         * </li>
         * <li>
         * <p>
         * The second action inserts the same rule group back in, but specifying the rules to exclude. That is, the
         * second <code>Updates:Action</code> should be <code>INSERT</code>, <code>Updates:ActivatedRule:RuleId</code>
         * should be the rule group that you just removed, and <code>ExcludedRules</code> should contain the rules that
         * you want to exclude.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ol>
         * This is a convenience that creates an instance of the {@link List<ExcludedRule>.Builder} avoiding the need to
         * create one manually via {@link List<ExcludedRule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ExcludedRule>.Builder#build()} is called immediately and its
         * result is passed to {@link #excludedRules(List<ExcludedRule>)}.
         * 
         * @param excludedRules
         *        a consumer that will call methods on {@link List<ExcludedRule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #excludedRules(List<ExcludedRule>)
         */
        Builder excludedRules(Consumer<ExcludedRule.Builder>... excludedRules);
    }

    static final class BuilderImpl implements Builder {
        private Integer priority;

        private String ruleId;

        private WafAction action;

        private WafOverrideAction overrideAction;

        private String type;

        private List<ExcludedRule> excludedRules = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ActivatedRule model) {
            priority(model.priority);
            ruleId(model.ruleId);
            action(model.action);
            overrideAction(model.overrideAction);
            type(model.type);
            excludedRules(model.excludedRules);
        }

        public final Integer getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        public final String getRuleId() {
            return ruleId;
        }

        @Override
        public final Builder ruleId(String ruleId) {
            this.ruleId = ruleId;
            return this;
        }

        public final void setRuleId(String ruleId) {
            this.ruleId = ruleId;
        }

        public final WafAction.Builder getAction() {
            return action != null ? action.toBuilder() : null;
        }

        @Override
        public final Builder action(WafAction action) {
            this.action = action;
            return this;
        }

        public final void setAction(WafAction.BuilderImpl action) {
            this.action = action != null ? action.build() : null;
        }

        public final WafOverrideAction.Builder getOverrideAction() {
            return overrideAction != null ? overrideAction.toBuilder() : null;
        }

        @Override
        public final Builder overrideAction(WafOverrideAction overrideAction) {
            this.overrideAction = overrideAction;
            return this;
        }

        public final void setOverrideAction(WafOverrideAction.BuilderImpl overrideAction) {
            this.overrideAction = overrideAction != null ? overrideAction.build() : null;
        }

        public final String getTypeAsString() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(WafRuleType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Collection<ExcludedRule.Builder> getExcludedRules() {
            return excludedRules != null ? excludedRules.stream().map(ExcludedRule::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder excludedRules(Collection<ExcludedRule> excludedRules) {
            this.excludedRules = ExcludedRulesCopier.copy(excludedRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedRules(ExcludedRule... excludedRules) {
            excludedRules(Arrays.asList(excludedRules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedRules(Consumer<ExcludedRule.Builder>... excludedRules) {
            excludedRules(Stream.of(excludedRules).map(c -> ExcludedRule.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setExcludedRules(Collection<ExcludedRule.BuilderImpl> excludedRules) {
            this.excludedRules = ExcludedRulesCopier.copyFromBuilder(excludedRules);
        }

        @Override
        public ActivatedRule build() {
            return new ActivatedRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
