/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.transfer.model.ConflictException;
import software.amazon.awssdk.services.transfer.model.CreateServerRequest;
import software.amazon.awssdk.services.transfer.model.CreateServerResponse;
import software.amazon.awssdk.services.transfer.model.CreateUserRequest;
import software.amazon.awssdk.services.transfer.model.CreateUserResponse;
import software.amazon.awssdk.services.transfer.model.DeleteServerRequest;
import software.amazon.awssdk.services.transfer.model.DeleteServerResponse;
import software.amazon.awssdk.services.transfer.model.DeleteSshPublicKeyRequest;
import software.amazon.awssdk.services.transfer.model.DeleteSshPublicKeyResponse;
import software.amazon.awssdk.services.transfer.model.DeleteUserRequest;
import software.amazon.awssdk.services.transfer.model.DeleteUserResponse;
import software.amazon.awssdk.services.transfer.model.DescribeServerRequest;
import software.amazon.awssdk.services.transfer.model.DescribeServerResponse;
import software.amazon.awssdk.services.transfer.model.DescribeUserRequest;
import software.amazon.awssdk.services.transfer.model.DescribeUserResponse;
import software.amazon.awssdk.services.transfer.model.ImportSshPublicKeyRequest;
import software.amazon.awssdk.services.transfer.model.ImportSshPublicKeyResponse;
import software.amazon.awssdk.services.transfer.model.InternalServiceErrorException;
import software.amazon.awssdk.services.transfer.model.InvalidNextTokenException;
import software.amazon.awssdk.services.transfer.model.InvalidRequestException;
import software.amazon.awssdk.services.transfer.model.ListServersRequest;
import software.amazon.awssdk.services.transfer.model.ListServersResponse;
import software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.transfer.model.ListUsersRequest;
import software.amazon.awssdk.services.transfer.model.ListUsersResponse;
import software.amazon.awssdk.services.transfer.model.ResourceExistsException;
import software.amazon.awssdk.services.transfer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.transfer.model.ServiceUnavailableException;
import software.amazon.awssdk.services.transfer.model.StartServerRequest;
import software.amazon.awssdk.services.transfer.model.StartServerResponse;
import software.amazon.awssdk.services.transfer.model.StopServerRequest;
import software.amazon.awssdk.services.transfer.model.StopServerResponse;
import software.amazon.awssdk.services.transfer.model.TagResourceRequest;
import software.amazon.awssdk.services.transfer.model.TagResourceResponse;
import software.amazon.awssdk.services.transfer.model.TestIdentityProviderRequest;
import software.amazon.awssdk.services.transfer.model.TestIdentityProviderResponse;
import software.amazon.awssdk.services.transfer.model.ThrottlingException;
import software.amazon.awssdk.services.transfer.model.TransferException;
import software.amazon.awssdk.services.transfer.model.TransferRequest;
import software.amazon.awssdk.services.transfer.model.UntagResourceRequest;
import software.amazon.awssdk.services.transfer.model.UntagResourceResponse;
import software.amazon.awssdk.services.transfer.model.UpdateServerRequest;
import software.amazon.awssdk.services.transfer.model.UpdateServerResponse;
import software.amazon.awssdk.services.transfer.model.UpdateUserRequest;
import software.amazon.awssdk.services.transfer.model.UpdateUserResponse;
import software.amazon.awssdk.services.transfer.paginators.ListServersIterable;
import software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable;
import software.amazon.awssdk.services.transfer.paginators.ListUsersIterable;
import software.amazon.awssdk.services.transfer.transform.CreateServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DeleteServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DeleteSshPublicKeyRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DescribeServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ImportSshPublicKeyRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListServersRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.StartServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.StopServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.TestIdentityProviderRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.UpdateServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.UpdateUserRequestMarshaller;

/**
 * Internal implementation of {@link TransferClient}.
 *
 * @see TransferClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTransferClient implements TransferClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTransferClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Instantiates an autoscaling virtual server based on Secure File Transfer Protocol (SFTP) in AWS. When you make
     * updates to your server or when you work with users, use the service-generated <code>ServerId</code> property that
     * is assigned to the newly created server.
     * </p>
     *
     * @param createServerRequest
     * @return Result of the CreateServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.CreateServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateServerResponse createServer(CreateServerRequest createServerRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceExistsException, AwsServiceException,
            SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateServerRequest, CreateServerResponse>()
                .withOperationName("CreateServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createServerRequest)
                .withMarshaller(new CreateServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a user and associates them with an existing Secure File Transfer Protocol (SFTP) server. You can only
     * create and associate users with SFTP servers that have the <code>IdentityProviderType</code> set to
     * <code>SERVICE_MANAGED</code>. Using parameters for <code>CreateUser</code>, you can specify the user name, set
     * the home directory, store the user's public key, and assign the user's AWS Identity and Access Management (IAM)
     * role. You can also optionally add a scope-down policy, and assign metadata with tags that can be used to group
     * and search for users.
     * </p>
     *
     * @param createUserRequest
     * @return Result of the CreateUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.CreateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateUserResponse createUser(CreateUserRequest createUserRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceExistsException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>()
                .withOperationName("CreateUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createUserRequest)
                .withMarshaller(new CreateUserRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the Secure File Transfer Protocol (SFTP) server that you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     *
     * @param deleteServerRequest
     * @return Result of the DeleteServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DeleteServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteServerResponse deleteServer(DeleteServerRequest deleteServerRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteServerRequest, DeleteServerResponse>()
                .withOperationName("DeleteServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteServerRequest)
                .withMarshaller(new DeleteServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a user's Secure Shell (SSH) public key.
     * </p>
     * <p>
     * No response is returned from this operation.
     * </p>
     *
     * @param deleteSshPublicKeyRequest
     * @return Result of the DeleteSshPublicKey operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DeleteSshPublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteSshPublicKeyResponse deleteSshPublicKey(DeleteSshPublicKeyRequest deleteSshPublicKeyRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSshPublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSshPublicKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteSshPublicKeyRequest, DeleteSshPublicKeyResponse>()
                .withOperationName("DeleteSshPublicKey").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteSshPublicKeyRequest)
                .withMarshaller(new DeleteSshPublicKeyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the user belonging to the server you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     * <note>
     * <p>
     * When you delete a user from a server, the user's information is lost.
     * </p>
     * </note>
     *
     * @param deleteUserRequest
     * @return Result of the DeleteUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DeleteUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteUserResponse deleteUser(DeleteUserRequest deleteUserRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>()
                .withOperationName("DeleteUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteUserRequest)
                .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes the server that you specify by passing the <code>ServerId</code> parameter.
     * </p>
     * <p>
     * The response contains a description of the server's properties. When you set <code>EndpointType</code> to VPC,
     * the response will contain the <code>EndpointDetails</code>.
     * </p>
     *
     * @param describeServerRequest
     * @return Result of the DescribeServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DescribeServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeServerResponse describeServer(DescribeServerRequest describeServerRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeServerRequest, DescribeServerResponse>()
                .withOperationName("DescribeServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeServerRequest)
                .withMarshaller(new DescribeServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes the user assigned to a specific server, as identified by its <code>ServerId</code> property.
     * </p>
     * <p>
     * The response from this call returns the properties of the user associated with the <code>ServerId</code> value
     * that was specified.
     * </p>
     *
     * @param describeUserRequest
     * @return Result of the DescribeUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DescribeUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeUserResponse describeUser(DescribeUserRequest describeUserRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                .withOperationName("DescribeUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeUserRequest)
                .withMarshaller(new DescribeUserRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds a Secure Shell (SSH) public key to a user account identified by a <code>UserName</code> value assigned to a
     * specific server, identified by <code>ServerId</code>.
     * </p>
     * <p>
     * The response returns the <code>UserName</code> value, the <code>ServerId</code> value, and the name of the
     * <code>SshPublicKeyId</code>.
     * </p>
     *
     * @param importSshPublicKeyRequest
     * @return Result of the ImportSshPublicKey operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ImportSshPublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ImportSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ImportSshPublicKeyResponse importSshPublicKey(ImportSshPublicKeyRequest importSshPublicKeyRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException, ResourceExistsException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportSshPublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ImportSshPublicKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ImportSshPublicKeyRequest, ImportSshPublicKeyResponse>()
                .withOperationName("ImportSshPublicKey").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(importSshPublicKeyRequest)
                .withMarshaller(new ImportSshPublicKeyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the Secure File Transfer Protocol (SFTP) servers that are associated with your AWS account.
     * </p>
     *
     * @param listServersRequest
     * @return Result of the ListServers operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListServersResponse listServers(ListServersRequest listServersRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, AwsServiceException,
            SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListServersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListServersRequest, ListServersResponse>()
                .withOperationName("ListServers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listServersRequest)
                .withMarshaller(new ListServersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the Secure File Transfer Protocol (SFTP) servers that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListServersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)} operation.</b>
     * </p>
     *
     * @param listServersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListServersIterable listServersPaginator(ListServersRequest listServersRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, AwsServiceException,
            SdkClientException, TransferException {
        return new ListServersIterable(this, applyPaginatorUserAgent(listServersRequest));
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidNextTokenException,
            InvalidRequestException, AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceIterable listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidNextTokenException,
            InvalidRequestException, AwsServiceException, SdkClientException, TransferException {
        return new ListTagsForResourceIterable(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Lists the users for the server that you specify by passing the <code>ServerId</code> parameter.
     * </p>
     *
     * @param listUsersRequest
     * @return Result of the ListUsers operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersResponse listUsers(ListUsersRequest listUsersRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>()
                .withOperationName("ListUsers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listUsersRequest)
                .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the users for the server that you specify by passing the <code>ServerId</code> parameter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListUsersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)} operation.</b>
     * </p>
     *
     * @param listUsersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersIterable listUsersPaginator(ListUsersRequest listUsersRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TransferException {
        return new ListUsersIterable(this, applyPaginatorUserAgent(listUsersRequest));
    }

    /**
     * <p>
     * Changes the state of a Secure File Transfer Protocol (SFTP) server from <code>OFFLINE</code> to
     * <code>ONLINE</code>. It has no impact on an SFTP server that is already <code>ONLINE</code>. An
     * <code>ONLINE</code> server can accept and process file transfer jobs.
     * </p>
     * <p>
     * The state of <code>STARTING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully online. The values of <code>START_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param startServerRequest
     * @return Result of the StartServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.StartServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StartServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartServerResponse startServer(StartServerRequest startServerRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartServerRequest, StartServerResponse>()
                .withOperationName("StartServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(startServerRequest)
                .withMarshaller(new StartServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Changes the state of an SFTP server from <code>ONLINE</code> to <code>OFFLINE</code>. An <code>OFFLINE</code>
     * server cannot accept and process file transfer jobs. Information tied to your server such as server and user
     * properties are not affected by stopping your server. Stopping a server will not reduce or impact your Secure File
     * Transfer Protocol (SFTP) endpoint billing.
     * </p>
     * <p>
     * The state of <code>STOPPING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully offline. The values of <code>STOP_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param stopServerRequest
     * @return Result of the StopServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.StopServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StopServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopServerResponse stopServer(StopServerRequest stopServerRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StopServerRequest, StopServerResponse>()
                .withOperationName("StopServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(stopServerRequest)
                .withMarshaller(new StopServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Attaches a key-value pair to a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * There is no response returned from this call.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * If the <code>IdentityProviderType</code> of the server is <code>API_Gateway</code>, tests whether your API
     * Gateway is set up successfully. We highly recommend that you call this operation to test your authentication
     * method as soon as you create your server. By doing so, you can troubleshoot issues with the API Gateway
     * integration to ensure that your users can successfully use the service.
     * </p>
     *
     * @param testIdentityProviderRequest
     * @return Result of the TestIdentityProvider operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.TestIdentityProvider
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TestIdentityProvider" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TestIdentityProviderResponse testIdentityProvider(TestIdentityProviderRequest testIdentityProviderRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TestIdentityProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, TestIdentityProviderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TestIdentityProviderRequest, TestIdentityProviderResponse>()
                .withOperationName("TestIdentityProvider").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(testIdentityProviderRequest)
                .withMarshaller(new TestIdentityProviderRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Detaches a key-value pair from a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the server properties after that server has been created.
     * </p>
     * <p>
     * The <code>UpdateServer</code> call returns the <code>ServerId</code> of the Secure File Transfer Protocol (SFTP)
     * server you updated.
     * </p>
     *
     * @param updateServerRequest
     * @return Result of the UpdateServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws ConflictException
     *         This exception is thrown when the <code>UpdatServer</code> is called for a server that has VPC as the
     *         endpoint type and the server's <code>VpcEndpointID</code> is not in the available state.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.UpdateServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateServerResponse updateServer(UpdateServerRequest updateServerRequest) throws ServiceUnavailableException,
            ConflictException, InternalServiceErrorException, InvalidRequestException, ResourceExistsException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateServerRequest, UpdateServerResponse>()
                .withOperationName("UpdateServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateServerRequest)
                .withMarshaller(new UpdateServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Assigns new properties to a user. Parameters you pass modify any or all of the following: the home directory,
     * role, and policy for the <code>UserName</code> and <code>ServerId</code> you specify.
     * </p>
     * <p>
     * The response returns the <code>ServerId</code> and the <code>UserName</code> for the updated user.
     * </p>
     *
     * @param updateUserRequest
     * @return Result of the UpdateUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.UpdateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateUserResponse updateUser(UpdateUserRequest updateUserRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, TransferException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>()
                .withOperationName("UpdateUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateUserRequest)
                .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TransferException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends TransferRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
