/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The input document, either as bytes or as an S3 object.
 * </p>
 * <p>
 * You pass image bytes to an Amazon Textract API operation by using the <code>Bytes</code> property. For example, you
 * would use the <code>Bytes</code> property to pass a document loaded from a local file system. Image bytes passed by
 * using the <code>Bytes</code> property must be base64 encoded. Your code might not need to encode document file bytes
 * if you're using an AWS SDK to call Amazon Textract API operations.
 * </p>
 * <p>
 * You pass images stored in an S3 bucket to an Amazon Textract API operation by using the <code>S3Object</code>
 * property. Documents stored in an S3 bucket don't need to be base64 encoded.
 * </p>
 * <p>
 * The AWS Region for the S3 bucket that contains the S3 object must match the AWS Region that you use for Amazon
 * Textract operations.
 * </p>
 * <p>
 * If you use the AWS CLI to call Amazon Textract operations, passing image bytes using the Bytes property isn't
 * supported. You must first upload the document to an Amazon S3 bucket, and then call the operation using the S3Object
 * property.
 * </p>
 * <p>
 * For Amazon Textract to process an S3 object, the user must have permission to access the S3 object.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Document implements SdkPojo, Serializable, ToCopyableBuilder<Document.Builder, Document> {
    private static final SdkField<SdkBytes> BYTES_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("Bytes").getter(getter(Document::bytes)).setter(setter(Builder::bytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bytes").build()).build();

    private static final SdkField<S3Object> S3_OBJECT_FIELD = SdkField.<S3Object> builder(MarshallingType.SDK_POJO)
            .memberName("S3Object").getter(getter(Document::s3Object)).setter(setter(Builder::s3Object))
            .constructor(S3Object::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Object").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BYTES_FIELD, S3_OBJECT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final SdkBytes bytes;

    private final S3Object s3Object;

    private Document(BuilderImpl builder) {
        this.bytes = builder.bytes;
        this.s3Object = builder.s3Object;
    }

    /**
     * <p>
     * A blob of base64-encoded document bytes. The maximum size of a document that's provided in a blob of bytes is 5
     * MB. The document bytes must be in PNG or JPEG format.
     * </p>
     * <p>
     * If you're using an AWS SDK to call Amazon Textract, you might not need to base64-encode image bytes passed using
     * the <code>Bytes</code> field.
     * </p>
     * 
     * @return A blob of base64-encoded document bytes. The maximum size of a document that's provided in a blob of
     *         bytes is 5 MB. The document bytes must be in PNG or JPEG format.</p>
     *         <p>
     *         If you're using an AWS SDK to call Amazon Textract, you might not need to base64-encode image bytes
     *         passed using the <code>Bytes</code> field.
     */
    public final SdkBytes bytes() {
        return bytes;
    }

    /**
     * <p>
     * Identifies an S3 object as the document source. The maximum size of a document that's stored in an S3 bucket is 5
     * MB.
     * </p>
     * 
     * @return Identifies an S3 object as the document source. The maximum size of a document that's stored in an S3
     *         bucket is 5 MB.
     */
    public final S3Object s3Object() {
        return s3Object;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bytes());
        hashCode = 31 * hashCode + Objects.hashCode(s3Object());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Document)) {
            return false;
        }
        Document other = (Document) obj;
        return Objects.equals(bytes(), other.bytes()) && Objects.equals(s3Object(), other.s3Object());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Document").add("Bytes", bytes()).add("S3Object", s3Object()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bytes":
            return Optional.ofNullable(clazz.cast(bytes()));
        case "S3Object":
            return Optional.ofNullable(clazz.cast(s3Object()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Bytes", BYTES_FIELD);
        map.put("S3Object", S3_OBJECT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Document, T> g) {
        return obj -> g.apply((Document) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Document> {
        /**
         * <p>
         * A blob of base64-encoded document bytes. The maximum size of a document that's provided in a blob of bytes is
         * 5 MB. The document bytes must be in PNG or JPEG format.
         * </p>
         * <p>
         * If you're using an AWS SDK to call Amazon Textract, you might not need to base64-encode image bytes passed
         * using the <code>Bytes</code> field.
         * </p>
         * 
         * @param bytes
         *        A blob of base64-encoded document bytes. The maximum size of a document that's provided in a blob of
         *        bytes is 5 MB. The document bytes must be in PNG or JPEG format.</p>
         *        <p>
         *        If you're using an AWS SDK to call Amazon Textract, you might not need to base64-encode image bytes
         *        passed using the <code>Bytes</code> field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytes(SdkBytes bytes);

        /**
         * <p>
         * Identifies an S3 object as the document source. The maximum size of a document that's stored in an S3 bucket
         * is 5 MB.
         * </p>
         * 
         * @param s3Object
         *        Identifies an S3 object as the document source. The maximum size of a document that's stored in an S3
         *        bucket is 5 MB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Object(S3Object s3Object);

        /**
         * <p>
         * Identifies an S3 object as the document source. The maximum size of a document that's stored in an S3 bucket
         * is 5 MB.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3Object.Builder} avoiding the need to
         * create one manually via {@link S3Object#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3Object.Builder#build()} is called immediately and its result is
         * passed to {@link #s3Object(S3Object)}.
         * 
         * @param s3Object
         *        a consumer that will call methods on {@link S3Object.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Object(S3Object)
         */
        default Builder s3Object(Consumer<S3Object.Builder> s3Object) {
            return s3Object(S3Object.builder().applyMutation(s3Object).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private SdkBytes bytes;

        private S3Object s3Object;

        private BuilderImpl() {
        }

        private BuilderImpl(Document model) {
            bytes(model.bytes);
            s3Object(model.s3Object);
        }

        public final ByteBuffer getBytes() {
            return bytes == null ? null : bytes.asByteBuffer();
        }

        public final void setBytes(ByteBuffer bytes) {
            bytes(bytes == null ? null : SdkBytes.fromByteBuffer(bytes));
        }

        @Override
        public final Builder bytes(SdkBytes bytes) {
            this.bytes = bytes;
            return this;
        }

        public final S3Object.Builder getS3Object() {
            return s3Object != null ? s3Object.toBuilder() : null;
        }

        public final void setS3Object(S3Object.BuilderImpl s3Object) {
            this.s3Object = s3Object != null ? s3Object.build() : null;
        }

        @Override
        public final Builder s3Object(S3Object s3Object) {
            this.s3Object = s3Object;
            return this;
        }

        @Override
        public Document build() {
            return new Document(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
