/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sso;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sso.model.GetRoleCredentialsRequest;
import software.amazon.awssdk.services.sso.model.GetRoleCredentialsResponse;
import software.amazon.awssdk.services.sso.model.InvalidRequestException;
import software.amazon.awssdk.services.sso.model.ListAccountRolesRequest;
import software.amazon.awssdk.services.sso.model.ListAccountRolesResponse;
import software.amazon.awssdk.services.sso.model.ListAccountsRequest;
import software.amazon.awssdk.services.sso.model.ListAccountsResponse;
import software.amazon.awssdk.services.sso.model.LogoutRequest;
import software.amazon.awssdk.services.sso.model.LogoutResponse;
import software.amazon.awssdk.services.sso.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sso.model.SsoException;
import software.amazon.awssdk.services.sso.model.SsoRequest;
import software.amazon.awssdk.services.sso.model.TooManyRequestsException;
import software.amazon.awssdk.services.sso.model.UnauthorizedException;
import software.amazon.awssdk.services.sso.paginators.ListAccountRolesPublisher;
import software.amazon.awssdk.services.sso.paginators.ListAccountsPublisher;
import software.amazon.awssdk.services.sso.transform.GetRoleCredentialsRequestMarshaller;
import software.amazon.awssdk.services.sso.transform.ListAccountRolesRequestMarshaller;
import software.amazon.awssdk.services.sso.transform.ListAccountsRequestMarshaller;
import software.amazon.awssdk.services.sso.transform.LogoutRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SsoAsyncClient}.
 *
 * @see SsoAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsoAsyncClient implements SsoAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSsoAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSsoAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Returns the STS short-term credentials for a given role name that is assigned to the user.
     * </p>
     *
     * @param getRoleCredentialsRequest
     * @return A Java Future containing the result of the GetRoleCredentials operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that a problem occurred with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>UnauthorizedException Indicates that the request is not authorized. This can happen due to an invalid
     *         access token in the request.</li>
     *         <li>TooManyRequestsException Indicates that the request is being made too frequently and is more than
     *         what the server can handle.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SsoAsyncClient.GetRoleCredentials
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/GetRoleCredentials" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRoleCredentialsResponse> getRoleCredentials(GetRoleCredentialsRequest getRoleCredentialsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRoleCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRoleCredentialsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRoleCredentialsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRoleCredentialsRequest, GetRoleCredentialsResponse>()
                            .withOperationName("GetRoleCredentials")
                            .withMarshaller(new GetRoleCredentialsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getRoleCredentialsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all roles that are assigned to the user for a given AWS account.
     * </p>
     *
     * @param listAccountRolesRequest
     * @return A Java Future containing the result of the ListAccountRoles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that a problem occurred with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>UnauthorizedException Indicates that the request is not authorized. This can happen due to an invalid
     *         access token in the request.</li>
     *         <li>TooManyRequestsException Indicates that the request is being made too frequently and is more than
     *         what the server can handle.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SsoAsyncClient.ListAccountRoles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/ListAccountRoles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccountRolesResponse> listAccountRoles(ListAccountRolesRequest listAccountRolesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccountRolesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAccountRolesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccountRolesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccountRolesRequest, ListAccountRolesResponse>()
                            .withOperationName("ListAccountRoles")
                            .withMarshaller(new ListAccountRolesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listAccountRolesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all roles that are assigned to the user for a given AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAccountRoles(software.amazon.awssdk.services.sso.model.ListAccountRolesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sso.paginators.ListAccountRolesPublisher publisher = client.listAccountRolesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sso.paginators.ListAccountRolesPublisher publisher = client.listAccountRolesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sso.model.ListAccountRolesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sso.model.ListAccountRolesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccountRoles(software.amazon.awssdk.services.sso.model.ListAccountRolesRequest)} operation.</b>
     * </p>
     *
     * @param listAccountRolesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that a problem occurred with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>UnauthorizedException Indicates that the request is not authorized. This can happen due to an invalid
     *         access token in the request.</li>
     *         <li>TooManyRequestsException Indicates that the request is being made too frequently and is more than
     *         what the server can handle.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SsoAsyncClient.ListAccountRoles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/ListAccountRoles" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAccountRolesPublisher listAccountRolesPaginator(ListAccountRolesRequest listAccountRolesRequest) {
        return new ListAccountRolesPublisher(this, applyPaginatorUserAgent(listAccountRolesRequest));
    }

    /**
     * <p>
     * Lists all AWS accounts assigned to the user. These AWS accounts are assigned by the administrator of the account.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/singlesignon/latest/userguide/useraccess.html#assignusers">Assign User
     * Access</a> in the <i>AWS SSO User Guide</i>. This operation returns a paginated response.
     * </p>
     *
     * @param listAccountsRequest
     * @return A Java Future containing the result of the ListAccounts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that a problem occurred with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>UnauthorizedException Indicates that the request is not authorized. This can happen due to an invalid
     *         access token in the request.</li>
     *         <li>TooManyRequestsException Indicates that the request is being made too frequently and is more than
     *         what the server can handle.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SsoAsyncClient.ListAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/ListAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccountsResponse> listAccounts(ListAccountsRequest listAccountsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccountsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccountsRequest, ListAccountsResponse>()
                            .withOperationName("ListAccounts").withMarshaller(new ListAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listAccountsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all AWS accounts assigned to the user. These AWS accounts are assigned by the administrator of the account.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/singlesignon/latest/userguide/useraccess.html#assignusers">Assign User
     * Access</a> in the <i>AWS SSO User Guide</i>. This operation returns a paginated response.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAccounts(software.amazon.awssdk.services.sso.model.ListAccountsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sso.paginators.ListAccountsPublisher publisher = client.listAccountsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sso.paginators.ListAccountsPublisher publisher = client.listAccountsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sso.model.ListAccountsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sso.model.ListAccountsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccounts(software.amazon.awssdk.services.sso.model.ListAccountsRequest)} operation.</b>
     * </p>
     *
     * @param listAccountsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that a problem occurred with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>UnauthorizedException Indicates that the request is not authorized. This can happen due to an invalid
     *         access token in the request.</li>
     *         <li>TooManyRequestsException Indicates that the request is being made too frequently and is more than
     *         what the server can handle.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SsoAsyncClient.ListAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/ListAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAccountsPublisher listAccountsPaginator(ListAccountsRequest listAccountsRequest) {
        return new ListAccountsPublisher(this, applyPaginatorUserAgent(listAccountsRequest));
    }

    /**
     * <p>
     * Removes the client- and server-side session that is associated with the user.
     * </p>
     *
     * @param logoutRequest
     * @return A Java Future containing the result of the Logout operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that a problem occurred with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>UnauthorizedException Indicates that the request is not authorized. This can happen due to an invalid
     *         access token in the request.</li>
     *         <li>TooManyRequestsException Indicates that the request is being made too frequently and is more than
     *         what the server can handle.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SsoAsyncClient.Logout
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/Logout" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<LogoutResponse> logout(LogoutRequest logoutRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<LogoutResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    LogoutResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<LogoutResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<LogoutRequest, LogoutResponse>().withOperationName("Logout")
                            .withMarshaller(new LogoutRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(logoutRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsoException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    private <T extends SsoRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
