/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmquicksetup;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.ssmquicksetup.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.ssmquicksetup.internal.SsmQuickSetupServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ssmquicksetup.model.AccessDeniedException;
import software.amazon.awssdk.services.ssmquicksetup.model.ConflictException;
import software.amazon.awssdk.services.ssmquicksetup.model.CreateConfigurationManagerRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.CreateConfigurationManagerResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.DeleteConfigurationManagerRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.DeleteConfigurationManagerResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.GetConfigurationManagerRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.GetConfigurationManagerResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.GetConfigurationRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.GetConfigurationResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.GetServiceSettingsRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.GetServiceSettingsResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.InternalServerException;
import software.amazon.awssdk.services.ssmquicksetup.model.ListConfigurationManagersRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.ListConfigurationManagersResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ListConfigurationsRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.ListConfigurationsResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ListQuickSetupTypesRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.ListQuickSetupTypesResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ssmquicksetup.model.SsmQuickSetupException;
import software.amazon.awssdk.services.ssmquicksetup.model.TagResourceRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.TagResourceResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ThrottlingException;
import software.amazon.awssdk.services.ssmquicksetup.model.UntagResourceRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.UntagResourceResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateConfigurationDefinitionRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateConfigurationDefinitionResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateConfigurationManagerRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateConfigurationManagerResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateServiceSettingsRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateServiceSettingsResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ValidationException;
import software.amazon.awssdk.services.ssmquicksetup.transform.CreateConfigurationManagerRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.DeleteConfigurationManagerRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.GetConfigurationManagerRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.GetConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.GetServiceSettingsRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.ListConfigurationManagersRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.ListConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.ListQuickSetupTypesRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.UpdateConfigurationDefinitionRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.UpdateConfigurationManagerRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.UpdateServiceSettingsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SsmQuickSetupClient}.
 *
 * @see SsmQuickSetupClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsmQuickSetupClient implements SsmQuickSetupClient {
    private static final Logger log = Logger.loggerFor(DefaultSsmQuickSetupClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSsmQuickSetupClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "SSM_QuickSetup" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a Quick Setup configuration manager resource. This object is a collection of desired state configurations
     * for multiple configuration definitions and summaries describing the deployments of those definitions.
     * </p>
     *
     * @param createConfigurationManagerRequest
     * @return Result of the CreateConfigurationManager operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.CreateConfigurationManager
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/CreateConfigurationManager"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateConfigurationManagerResponse createConfigurationManager(
            CreateConfigurationManagerRequest createConfigurationManagerRequest) throws ValidationException, ConflictException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConfigurationManagerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateConfigurationManagerResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigurationManagerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigurationManagerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfigurationManager");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationManagerRequest, CreateConfigurationManagerResponse>()
                            .withOperationName("CreateConfigurationManager").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createConfigurationManagerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateConfigurationManagerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a configuration manager.
     * </p>
     *
     * @param deleteConfigurationManagerRequest
     * @return Result of the DeleteConfigurationManager operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found. Check the ID or name and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.DeleteConfigurationManager
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/DeleteConfigurationManager"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteConfigurationManagerResponse deleteConfigurationManager(
            DeleteConfigurationManagerRequest deleteConfigurationManagerRequest) throws ValidationException, ConflictException,
            InternalServerException, AccessDeniedException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConfigurationManagerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteConfigurationManagerResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConfigurationManagerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfigurationManagerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfigurationManager");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationManagerRequest, DeleteConfigurationManagerResponse>()
                            .withOperationName("DeleteConfigurationManager").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteConfigurationManagerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteConfigurationManagerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about the specified configuration.
     * </p>
     *
     * @param getConfigurationRequest
     * @return Result of the GetConfiguration operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found. Check the ID or name and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.GetConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/GetConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetConfigurationResponse getConfiguration(GetConfigurationRequest getConfigurationRequest) throws ValidationException,
            ConflictException, InternalServerException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfiguration");

            return clientHandler.execute(new ClientExecutionParams<GetConfigurationRequest, GetConfigurationResponse>()
                    .withOperationName("GetConfiguration").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a configuration manager.
     * </p>
     *
     * @param getConfigurationManagerRequest
     * @return Result of the GetConfigurationManager operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found. Check the ID or name and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.GetConfigurationManager
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/GetConfigurationManager"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetConfigurationManagerResponse getConfigurationManager(GetConfigurationManagerRequest getConfigurationManagerRequest)
            throws ValidationException, ConflictException, InternalServerException, AccessDeniedException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConfigurationManagerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetConfigurationManagerResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigurationManagerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationManagerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfigurationManager");

            return clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationManagerRequest, GetConfigurationManagerResponse>()
                            .withOperationName("GetConfigurationManager").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getConfigurationManagerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetConfigurationManagerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns settings configured for Quick Setup in the requesting Amazon Web Services account and Amazon Web Services
     * Region.
     * </p>
     *
     * @param getServiceSettingsRequest
     * @return Result of the GetServiceSettings operation returned by the service.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.GetServiceSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/GetServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetServiceSettingsResponse getServiceSettings(GetServiceSettingsRequest getServiceSettingsRequest)
            throws ConflictException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetServiceSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceSettings");

            return clientHandler.execute(new ClientExecutionParams<GetServiceSettingsRequest, GetServiceSettingsResponse>()
                    .withOperationName("GetServiceSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getServiceSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetServiceSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns Quick Setup configuration managers.
     * </p>
     *
     * @param listConfigurationManagersRequest
     * @return Result of the ListConfigurationManagers operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.ListConfigurationManagers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/ListConfigurationManagers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListConfigurationManagersResponse listConfigurationManagers(
            ListConfigurationManagersRequest listConfigurationManagersRequest) throws ValidationException, ConflictException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConfigurationManagersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListConfigurationManagersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigurationManagersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationManagersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurationManagers");

            return clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationManagersRequest, ListConfigurationManagersResponse>()
                            .withOperationName("ListConfigurationManagers").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listConfigurationManagersRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListConfigurationManagersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns configurations deployed by Quick Setup in the requesting Amazon Web Services account and Amazon Web
     * Services Region.
     * </p>
     *
     * @param listConfigurationsRequest
     * @return Result of the ListConfigurations operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found. Check the ID or name and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.ListConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/ListConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListConfigurationsResponse listConfigurations(ListConfigurationsRequest listConfigurationsRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListConfigurationsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurations");

            return clientHandler.execute(new ClientExecutionParams<ListConfigurationsRequest, ListConfigurationsResponse>()
                    .withOperationName("ListConfigurations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listConfigurationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the available Quick Setup types.
     * </p>
     *
     * @param listQuickSetupTypesRequest
     * @return Result of the ListQuickSetupTypes operation returned by the service.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.ListQuickSetupTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/ListQuickSetupTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListQuickSetupTypesResponse listQuickSetupTypes(ListQuickSetupTypesRequest listQuickSetupTypesRequest)
            throws ConflictException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListQuickSetupTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListQuickSetupTypesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQuickSetupTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQuickSetupTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQuickSetupTypes");

            return clientHandler.execute(new ClientExecutionParams<ListQuickSetupTypesRequest, ListQuickSetupTypesResponse>()
                    .withOperationName("ListQuickSetupTypes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listQuickSetupTypesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListQuickSetupTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns tags assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found. Check the ID or name and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ConflictException, InternalServerException, AccessDeniedException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns key-value pairs of metadata to Amazon Web Services resources.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found. Check the ID or name and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException, ConflictException,
            InternalServerException, AccessDeniedException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found. Check the ID or name and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ConflictException, InternalServerException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a Quick Setup configuration definition.
     * </p>
     *
     * @param updateConfigurationDefinitionRequest
     * @return Result of the UpdateConfigurationDefinition operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found. Check the ID or name and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.UpdateConfigurationDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/UpdateConfigurationDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateConfigurationDefinitionResponse updateConfigurationDefinition(
            UpdateConfigurationDefinitionRequest updateConfigurationDefinitionRequest) throws ValidationException,
            ConflictException, InternalServerException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConfigurationDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateConfigurationDefinitionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigurationDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateConfigurationDefinitionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfigurationDefinition");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationDefinitionRequest, UpdateConfigurationDefinitionResponse>()
                            .withOperationName("UpdateConfigurationDefinition").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateConfigurationDefinitionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateConfigurationDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a Quick Setup configuration manager.
     * </p>
     *
     * @param updateConfigurationManagerRequest
     * @return Result of the UpdateConfigurationManager operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found. Check the ID or name and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.UpdateConfigurationManager
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/UpdateConfigurationManager"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateConfigurationManagerResponse updateConfigurationManager(
            UpdateConfigurationManagerRequest updateConfigurationManagerRequest) throws ValidationException, ConflictException,
            InternalServerException, AccessDeniedException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConfigurationManagerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateConfigurationManagerResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigurationManagerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfigurationManagerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfigurationManager");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationManagerRequest, UpdateConfigurationManagerResponse>()
                            .withOperationName("UpdateConfigurationManager").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateConfigurationManagerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateConfigurationManagerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates settings configured for Quick Setup.
     * </p>
     *
     * @param updateServiceSettingsRequest
     * @return Result of the UpdateServiceSettings operation returned by the service.
     * @throws ValidationException
     *         The request is invalid. Verify the values provided for the request parameters are accurate.
     * @throws ConflictException
     *         Another request is being processed. Wait a few minutes and try again.
     * @throws InternalServerException
     *         An error occurred on the server side.
     * @throws AccessDeniedException
     *         The requester has insufficient permissions to perform the operation.
     * @throws ThrottlingException
     *         The request or operation exceeds the maximum allowed request rate per Amazon Web Services account and
     *         Amazon Web Services Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmQuickSetupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmQuickSetupClient.UpdateServiceSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/UpdateServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateServiceSettingsResponse updateServiceSettings(UpdateServiceSettingsRequest updateServiceSettingsRequest)
            throws ValidationException, ConflictException, InternalServerException, AccessDeniedException, ThrottlingException,
            AwsServiceException, SdkClientException, SsmQuickSetupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateServiceSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceSettings");

            return clientHandler.execute(new ClientExecutionParams<UpdateServiceSettingsRequest, UpdateServiceSettingsResponse>()
                    .withOperationName("UpdateServiceSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateServiceSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateServiceSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        SsmQuickSetupServiceClientConfigurationBuilder serviceConfigBuilder = new SsmQuickSetupServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(SsmQuickSetupException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final SsmQuickSetupServiceClientConfiguration serviceClientConfiguration() {
        return new SsmQuickSetupServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
