/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes plugin details.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CommandPlugin implements SdkPojo, Serializable, ToCopyableBuilder<CommandPlugin.Builder, CommandPlugin> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandPlugin::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandPlugin::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandPlugin::statusDetails)).setter(setter(Builder::statusDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusDetails").build()).build();

    private static final SdkField<Integer> RESPONSE_CODE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CommandPlugin::responseCode)).setter(setter(Builder::responseCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResponseCode").build()).build();

    private static final SdkField<Instant> RESPONSE_START_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(CommandPlugin::responseStartDateTime)).setter(setter(Builder::responseStartDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResponseStartDateTime").build())
            .build();

    private static final SdkField<Instant> RESPONSE_FINISH_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(CommandPlugin::responseFinishDateTime)).setter(setter(Builder::responseFinishDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResponseFinishDateTime").build())
            .build();

    private static final SdkField<String> OUTPUT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandPlugin::output)).setter(setter(Builder::output))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Output").build()).build();

    private static final SdkField<String> STANDARD_OUTPUT_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandPlugin::standardOutputUrl)).setter(setter(Builder::standardOutputUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardOutputUrl").build()).build();

    private static final SdkField<String> STANDARD_ERROR_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandPlugin::standardErrorUrl)).setter(setter(Builder::standardErrorUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardErrorUrl").build()).build();

    private static final SdkField<String> OUTPUT_S3_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandPlugin::outputS3Region)).setter(setter(Builder::outputS3Region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputS3Region").build()).build();

    private static final SdkField<String> OUTPUT_S3_BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandPlugin::outputS3BucketName)).setter(setter(Builder::outputS3BucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputS3BucketName").build())
            .build();

    private static final SdkField<String> OUTPUT_S3_KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandPlugin::outputS3KeyPrefix)).setter(setter(Builder::outputS3KeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputS3KeyPrefix").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, STATUS_FIELD,
            STATUS_DETAILS_FIELD, RESPONSE_CODE_FIELD, RESPONSE_START_DATE_TIME_FIELD, RESPONSE_FINISH_DATE_TIME_FIELD,
            OUTPUT_FIELD, STANDARD_OUTPUT_URL_FIELD, STANDARD_ERROR_URL_FIELD, OUTPUT_S3_REGION_FIELD,
            OUTPUT_S3_BUCKET_NAME_FIELD, OUTPUT_S3_KEY_PREFIX_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String status;

    private final String statusDetails;

    private final Integer responseCode;

    private final Instant responseStartDateTime;

    private final Instant responseFinishDateTime;

    private final String output;

    private final String standardOutputUrl;

    private final String standardErrorUrl;

    private final String outputS3Region;

    private final String outputS3BucketName;

    private final String outputS3KeyPrefix;

    private CommandPlugin(BuilderImpl builder) {
        this.name = builder.name;
        this.status = builder.status;
        this.statusDetails = builder.statusDetails;
        this.responseCode = builder.responseCode;
        this.responseStartDateTime = builder.responseStartDateTime;
        this.responseFinishDateTime = builder.responseFinishDateTime;
        this.output = builder.output;
        this.standardOutputUrl = builder.standardOutputUrl;
        this.standardErrorUrl = builder.standardErrorUrl;
        this.outputS3Region = builder.outputS3Region;
        this.outputS3BucketName = builder.outputS3BucketName;
        this.outputS3KeyPrefix = builder.outputS3KeyPrefix;
    }

    /**
     * <p>
     * The name of the plugin. Must be one of the following: aws:updateAgent, aws:domainjoin, aws:applications,
     * aws:runPowerShellScript, aws:psmodule, aws:cloudWatch, aws:runShellScript, or aws:updateSSMAgent.
     * </p>
     * 
     * @return The name of the plugin. Must be one of the following: aws:updateAgent, aws:domainjoin, aws:applications,
     *         aws:runPowerShellScript, aws:psmodule, aws:cloudWatch, aws:runShellScript, or aws:updateSSMAgent.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The status of this plugin. You can run a document with multiple plugins.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CommandPluginStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of this plugin. You can run a document with multiple plugins.
     * @see CommandPluginStatus
     */
    public CommandPluginStatus status() {
        return CommandPluginStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of this plugin. You can run a document with multiple plugins.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CommandPluginStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of this plugin. You can run a document with multiple plugins.
     * @see CommandPluginStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A detailed status of the plugin execution. StatusDetails includes more information than Status because it
     * includes states resulting from error and concurrency control parameters. StatusDetails can show different results
     * than Status. For more information about these statuses, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding command
     * statuses</a> in the <i>AWS Systems Manager User Guide</i>. StatusDetails can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Pending: The command has not been sent to the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * In Progress: The command has been sent to the instance but has not reached a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Success: The execution of the command or plugin was successfully completed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delivery Timed Out: The command was not delivered to the instance before the delivery timeout expired. Delivery
     * timeouts do not count against the parent command's MaxErrors limit, but they do contribute to whether the parent
     * command status is Success or Incomplete. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Execution Timed Out: Command execution started on the instance, but the execution was not complete before the
     * execution timeout expired. Execution timeouts count against the MaxErrors limit of the parent command. This is a
     * terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The command was not successful on the instance. For a plugin, this indicates that the result code was not
     * zero. For a command invocation, this indicates that the result code for one or more plugins was not zero.
     * Invocation failures count against the MaxErrors limit of the parent command. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Canceled: The command was terminated before it was completed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Undeliverable: The command can't be delivered to the instance. The instance might not exist, or it might not be
     * responding. Undeliverable invocations don't count against the parent command's MaxErrors limit, and they don't
     * contribute to whether the parent command status is Success or Incomplete. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were canceled by
     * the system. This is a terminal state.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A detailed status of the plugin execution. StatusDetails includes more information than Status because it
     *         includes states resulting from error and concurrency control parameters. StatusDetails can show different
     *         results than Status. For more information about these statuses, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding
     *         command statuses</a> in the <i>AWS Systems Manager User Guide</i>. StatusDetails can be one of the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Pending: The command has not been sent to the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In Progress: The command has been sent to the instance but has not reached a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Success: The execution of the command or plugin was successfully completed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delivery Timed Out: The command was not delivered to the instance before the delivery timeout expired.
     *         Delivery timeouts do not count against the parent command's MaxErrors limit, but they do contribute to
     *         whether the parent command status is Success or Incomplete. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Execution Timed Out: Command execution started on the instance, but the execution was not complete before
     *         the execution timeout expired. Execution timeouts count against the MaxErrors limit of the parent
     *         command. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The command was not successful on the instance. For a plugin, this indicates that the result code
     *         was not zero. For a command invocation, this indicates that the result code for one or more plugins was
     *         not zero. Invocation failures count against the MaxErrors limit of the parent command. This is a terminal
     *         state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Canceled: The command was terminated before it was completed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Undeliverable: The command can't be delivered to the instance. The instance might not exist, or it might
     *         not be responding. Undeliverable invocations don't count against the parent command's MaxErrors limit,
     *         and they don't contribute to whether the parent command status is Success or Incomplete. This is a
     *         terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were
     *         canceled by the system. This is a terminal state.
     *         </p>
     *         </li>
     */
    public String statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * A numeric response code generated after running the plugin.
     * </p>
     * 
     * @return A numeric response code generated after running the plugin.
     */
    public Integer responseCode() {
        return responseCode;
    }

    /**
     * <p>
     * The time the plugin started running.
     * </p>
     * 
     * @return The time the plugin started running.
     */
    public Instant responseStartDateTime() {
        return responseStartDateTime;
    }

    /**
     * <p>
     * The time the plugin stopped running. Could stop prematurely if, for example, a cancel command was sent.
     * </p>
     * 
     * @return The time the plugin stopped running. Could stop prematurely if, for example, a cancel command was sent.
     */
    public Instant responseFinishDateTime() {
        return responseFinishDateTime;
    }

    /**
     * <p>
     * Output of the plugin execution.
     * </p>
     * 
     * @return Output of the plugin execution.
     */
    public String output() {
        return output;
    }

    /**
     * <p>
     * The URL for the complete text written by the plugin to stdout in Amazon S3. If the S3 bucket for the command was
     * not specified, then this string is empty.
     * </p>
     * 
     * @return The URL for the complete text written by the plugin to stdout in Amazon S3. If the S3 bucket for the
     *         command was not specified, then this string is empty.
     */
    public String standardOutputUrl() {
        return standardOutputUrl;
    }

    /**
     * <p>
     * The URL for the complete text written by the plugin to stderr. If execution is not yet complete, then this string
     * is empty.
     * </p>
     * 
     * @return The URL for the complete text written by the plugin to stderr. If execution is not yet complete, then
     *         this string is empty.
     */
    public String standardErrorUrl() {
        return standardErrorUrl;
    }

    /**
     * <p>
     * (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
     * automatically determines the S3 bucket region.
     * </p>
     * 
     * @return (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
     *         automatically determines the S3 bucket region.
     */
    public String outputS3Region() {
        return outputS3Region;
    }

    /**
     * <p>
     * The S3 bucket where the responses to the command executions should be stored. This was requested when issuing the
     * command. For example, in the following response:
     * </p>
     * <p>
     * test_folder/ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix/i-1234567876543/awsrunShellScript
     * </p>
     * <p>
     * test_folder is the name of the S3 bucket;
     * </p>
     * <p>
     * ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix is the name of the S3 prefix;
     * </p>
     * <p>
     * i-1234567876543 is the instance ID;
     * </p>
     * <p>
     * awsrunShellScript is the name of the plugin.
     * </p>
     * 
     * @return The S3 bucket where the responses to the command executions should be stored. This was requested when
     *         issuing the command. For example, in the following response:</p>
     *         <p>
     *         test_folder/ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix/i-1234567876543/awsrunShellScript
     *         </p>
     *         <p>
     *         test_folder is the name of the S3 bucket;
     *         </p>
     *         <p>
     *         ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix is the name of the S3 prefix;
     *         </p>
     *         <p>
     *         i-1234567876543 is the instance ID;
     *         </p>
     *         <p>
     *         awsrunShellScript is the name of the plugin.
     */
    public String outputS3BucketName() {
        return outputS3BucketName;
    }

    /**
     * <p>
     * The S3 directory path inside the bucket where the responses to the command executions should be stored. This was
     * requested when issuing the command. For example, in the following response:
     * </p>
     * <p>
     * test_folder/ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix/i-1234567876543/awsrunShellScript
     * </p>
     * <p>
     * test_folder is the name of the S3 bucket;
     * </p>
     * <p>
     * ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix is the name of the S3 prefix;
     * </p>
     * <p>
     * i-1234567876543 is the instance ID;
     * </p>
     * <p>
     * awsrunShellScript is the name of the plugin.
     * </p>
     * 
     * @return The S3 directory path inside the bucket where the responses to the command executions should be stored.
     *         This was requested when issuing the command. For example, in the following response:</p>
     *         <p>
     *         test_folder/ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix/i-1234567876543/awsrunShellScript
     *         </p>
     *         <p>
     *         test_folder is the name of the S3 bucket;
     *         </p>
     *         <p>
     *         ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix is the name of the S3 prefix;
     *         </p>
     *         <p>
     *         i-1234567876543 is the instance ID;
     *         </p>
     *         <p>
     *         awsrunShellScript is the name of the plugin.
     */
    public String outputS3KeyPrefix() {
        return outputS3KeyPrefix;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetails());
        hashCode = 31 * hashCode + Objects.hashCode(responseCode());
        hashCode = 31 * hashCode + Objects.hashCode(responseStartDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(responseFinishDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(output());
        hashCode = 31 * hashCode + Objects.hashCode(standardOutputUrl());
        hashCode = 31 * hashCode + Objects.hashCode(standardErrorUrl());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3Region());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3BucketName());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3KeyPrefix());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CommandPlugin)) {
            return false;
        }
        CommandPlugin other = (CommandPlugin) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusDetails(), other.statusDetails()) && Objects.equals(responseCode(), other.responseCode())
                && Objects.equals(responseStartDateTime(), other.responseStartDateTime())
                && Objects.equals(responseFinishDateTime(), other.responseFinishDateTime())
                && Objects.equals(output(), other.output()) && Objects.equals(standardOutputUrl(), other.standardOutputUrl())
                && Objects.equals(standardErrorUrl(), other.standardErrorUrl())
                && Objects.equals(outputS3Region(), other.outputS3Region())
                && Objects.equals(outputS3BucketName(), other.outputS3BucketName())
                && Objects.equals(outputS3KeyPrefix(), other.outputS3KeyPrefix());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CommandPlugin").add("Name", name()).add("Status", statusAsString())
                .add("StatusDetails", statusDetails()).add("ResponseCode", responseCode())
                .add("ResponseStartDateTime", responseStartDateTime()).add("ResponseFinishDateTime", responseFinishDateTime())
                .add("Output", output()).add("StandardOutputUrl", standardOutputUrl())
                .add("StandardErrorUrl", standardErrorUrl()).add("OutputS3Region", outputS3Region())
                .add("OutputS3BucketName", outputS3BucketName()).add("OutputS3KeyPrefix", outputS3KeyPrefix()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusDetails":
            return Optional.ofNullable(clazz.cast(statusDetails()));
        case "ResponseCode":
            return Optional.ofNullable(clazz.cast(responseCode()));
        case "ResponseStartDateTime":
            return Optional.ofNullable(clazz.cast(responseStartDateTime()));
        case "ResponseFinishDateTime":
            return Optional.ofNullable(clazz.cast(responseFinishDateTime()));
        case "Output":
            return Optional.ofNullable(clazz.cast(output()));
        case "StandardOutputUrl":
            return Optional.ofNullable(clazz.cast(standardOutputUrl()));
        case "StandardErrorUrl":
            return Optional.ofNullable(clazz.cast(standardErrorUrl()));
        case "OutputS3Region":
            return Optional.ofNullable(clazz.cast(outputS3Region()));
        case "OutputS3BucketName":
            return Optional.ofNullable(clazz.cast(outputS3BucketName()));
        case "OutputS3KeyPrefix":
            return Optional.ofNullable(clazz.cast(outputS3KeyPrefix()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CommandPlugin, T> g) {
        return obj -> g.apply((CommandPlugin) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CommandPlugin> {
        /**
         * <p>
         * The name of the plugin. Must be one of the following: aws:updateAgent, aws:domainjoin, aws:applications,
         * aws:runPowerShellScript, aws:psmodule, aws:cloudWatch, aws:runShellScript, or aws:updateSSMAgent.
         * </p>
         * 
         * @param name
         *        The name of the plugin. Must be one of the following: aws:updateAgent, aws:domainjoin,
         *        aws:applications, aws:runPowerShellScript, aws:psmodule, aws:cloudWatch, aws:runShellScript, or
         *        aws:updateSSMAgent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The status of this plugin. You can run a document with multiple plugins.
         * </p>
         * 
         * @param status
         *        The status of this plugin. You can run a document with multiple plugins.
         * @see CommandPluginStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandPluginStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of this plugin. You can run a document with multiple plugins.
         * </p>
         * 
         * @param status
         *        The status of this plugin. You can run a document with multiple plugins.
         * @see CommandPluginStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandPluginStatus
         */
        Builder status(CommandPluginStatus status);

        /**
         * <p>
         * A detailed status of the plugin execution. StatusDetails includes more information than Status because it
         * includes states resulting from error and concurrency control parameters. StatusDetails can show different
         * results than Status. For more information about these statuses, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding
         * command statuses</a> in the <i>AWS Systems Manager User Guide</i>. StatusDetails can be one of the following
         * values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Pending: The command has not been sent to the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * In Progress: The command has been sent to the instance but has not reached a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Success: The execution of the command or plugin was successfully completed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Delivery Timed Out: The command was not delivered to the instance before the delivery timeout expired.
         * Delivery timeouts do not count against the parent command's MaxErrors limit, but they do contribute to
         * whether the parent command status is Success or Incomplete. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Execution Timed Out: Command execution started on the instance, but the execution was not complete before the
         * execution timeout expired. Execution timeouts count against the MaxErrors limit of the parent command. This
         * is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The command was not successful on the instance. For a plugin, this indicates that the result code was
         * not zero. For a command invocation, this indicates that the result code for one or more plugins was not zero.
         * Invocation failures count against the MaxErrors limit of the parent command. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Canceled: The command was terminated before it was completed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Undeliverable: The command can't be delivered to the instance. The instance might not exist, or it might not
         * be responding. Undeliverable invocations don't count against the parent command's MaxErrors limit, and they
         * don't contribute to whether the parent command status is Success or Incomplete. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were canceled
         * by the system. This is a terminal state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statusDetails
         *        A detailed status of the plugin execution. StatusDetails includes more information than Status because
         *        it includes states resulting from error and concurrency control parameters. StatusDetails can show
         *        different results than Status. For more information about these statuses, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html"
         *        >Understanding command statuses</a> in the <i>AWS Systems Manager User Guide</i>. StatusDetails can be
         *        one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Pending: The command has not been sent to the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In Progress: The command has been sent to the instance but has not reached a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Success: The execution of the command or plugin was successfully completed. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delivery Timed Out: The command was not delivered to the instance before the delivery timeout expired.
         *        Delivery timeouts do not count against the parent command's MaxErrors limit, but they do contribute to
         *        whether the parent command status is Success or Incomplete. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Execution Timed Out: Command execution started on the instance, but the execution was not complete
         *        before the execution timeout expired. Execution timeouts count against the MaxErrors limit of the
         *        parent command. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The command was not successful on the instance. For a plugin, this indicates that the result
         *        code was not zero. For a command invocation, this indicates that the result code for one or more
         *        plugins was not zero. Invocation failures count against the MaxErrors limit of the parent command.
         *        This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Canceled: The command was terminated before it was completed. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Undeliverable: The command can't be delivered to the instance. The instance might not exist, or it
         *        might not be responding. Undeliverable invocations don't count against the parent command's MaxErrors
         *        limit, and they don't contribute to whether the parent command status is Success or Incomplete. This
         *        is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were
         *        canceled by the system. This is a terminal state.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(String statusDetails);

        /**
         * <p>
         * A numeric response code generated after running the plugin.
         * </p>
         * 
         * @param responseCode
         *        A numeric response code generated after running the plugin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseCode(Integer responseCode);

        /**
         * <p>
         * The time the plugin started running.
         * </p>
         * 
         * @param responseStartDateTime
         *        The time the plugin started running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseStartDateTime(Instant responseStartDateTime);

        /**
         * <p>
         * The time the plugin stopped running. Could stop prematurely if, for example, a cancel command was sent.
         * </p>
         * 
         * @param responseFinishDateTime
         *        The time the plugin stopped running. Could stop prematurely if, for example, a cancel command was
         *        sent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseFinishDateTime(Instant responseFinishDateTime);

        /**
         * <p>
         * Output of the plugin execution.
         * </p>
         * 
         * @param output
         *        Output of the plugin execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder output(String output);

        /**
         * <p>
         * The URL for the complete text written by the plugin to stdout in Amazon S3. If the S3 bucket for the command
         * was not specified, then this string is empty.
         * </p>
         * 
         * @param standardOutputUrl
         *        The URL for the complete text written by the plugin to stdout in Amazon S3. If the S3 bucket for the
         *        command was not specified, then this string is empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardOutputUrl(String standardOutputUrl);

        /**
         * <p>
         * The URL for the complete text written by the plugin to stderr. If execution is not yet complete, then this
         * string is empty.
         * </p>
         * 
         * @param standardErrorUrl
         *        The URL for the complete text written by the plugin to stderr. If execution is not yet complete, then
         *        this string is empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardErrorUrl(String standardErrorUrl);

        /**
         * <p>
         * (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
         * automatically determines the S3 bucket region.
         * </p>
         * 
         * @param outputS3Region
         *        (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
         *        automatically determines the S3 bucket region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3Region(String outputS3Region);

        /**
         * <p>
         * The S3 bucket where the responses to the command executions should be stored. This was requested when issuing
         * the command. For example, in the following response:
         * </p>
         * <p>
         * test_folder/ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix/i-1234567876543/awsrunShellScript
         * </p>
         * <p>
         * test_folder is the name of the S3 bucket;
         * </p>
         * <p>
         * ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix is the name of the S3 prefix;
         * </p>
         * <p>
         * i-1234567876543 is the instance ID;
         * </p>
         * <p>
         * awsrunShellScript is the name of the plugin.
         * </p>
         * 
         * @param outputS3BucketName
         *        The S3 bucket where the responses to the command executions should be stored. This was requested when
         *        issuing the command. For example, in the following response:</p>
         *        <p>
         *        test_folder/ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix/i-1234567876543/awsrunShellScript
         *        </p>
         *        <p>
         *        test_folder is the name of the S3 bucket;
         *        </p>
         *        <p>
         *        ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix is the name of the S3 prefix;
         *        </p>
         *        <p>
         *        i-1234567876543 is the instance ID;
         *        </p>
         *        <p>
         *        awsrunShellScript is the name of the plugin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3BucketName(String outputS3BucketName);

        /**
         * <p>
         * The S3 directory path inside the bucket where the responses to the command executions should be stored. This
         * was requested when issuing the command. For example, in the following response:
         * </p>
         * <p>
         * test_folder/ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix/i-1234567876543/awsrunShellScript
         * </p>
         * <p>
         * test_folder is the name of the S3 bucket;
         * </p>
         * <p>
         * ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix is the name of the S3 prefix;
         * </p>
         * <p>
         * i-1234567876543 is the instance ID;
         * </p>
         * <p>
         * awsrunShellScript is the name of the plugin.
         * </p>
         * 
         * @param outputS3KeyPrefix
         *        The S3 directory path inside the bucket where the responses to the command executions should be
         *        stored. This was requested when issuing the command. For example, in the following response:</p>
         *        <p>
         *        test_folder/ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix/i-1234567876543/awsrunShellScript
         *        </p>
         *        <p>
         *        test_folder is the name of the S3 bucket;
         *        </p>
         *        <p>
         *        ab19cb99-a030-46dd-9dfc-8eSAMPLEPre-Fix is the name of the S3 prefix;
         *        </p>
         *        <p>
         *        i-1234567876543 is the instance ID;
         *        </p>
         *        <p>
         *        awsrunShellScript is the name of the plugin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3KeyPrefix(String outputS3KeyPrefix);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String status;

        private String statusDetails;

        private Integer responseCode;

        private Instant responseStartDateTime;

        private Instant responseFinishDateTime;

        private String output;

        private String standardOutputUrl;

        private String standardErrorUrl;

        private String outputS3Region;

        private String outputS3BucketName;

        private String outputS3KeyPrefix;

        private BuilderImpl() {
        }

        private BuilderImpl(CommandPlugin model) {
            name(model.name);
            status(model.status);
            statusDetails(model.statusDetails);
            responseCode(model.responseCode);
            responseStartDateTime(model.responseStartDateTime);
            responseFinishDateTime(model.responseFinishDateTime);
            output(model.output);
            standardOutputUrl(model.standardOutputUrl);
            standardErrorUrl(model.standardErrorUrl);
            outputS3Region(model.outputS3Region);
            outputS3BucketName(model.outputS3BucketName);
            outputS3KeyPrefix(model.outputS3KeyPrefix);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CommandPluginStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusDetails() {
            return statusDetails;
        }

        @Override
        public final Builder statusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        public final void setStatusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
        }

        public final Integer getResponseCode() {
            return responseCode;
        }

        @Override
        public final Builder responseCode(Integer responseCode) {
            this.responseCode = responseCode;
            return this;
        }

        public final void setResponseCode(Integer responseCode) {
            this.responseCode = responseCode;
        }

        public final Instant getResponseStartDateTime() {
            return responseStartDateTime;
        }

        @Override
        public final Builder responseStartDateTime(Instant responseStartDateTime) {
            this.responseStartDateTime = responseStartDateTime;
            return this;
        }

        public final void setResponseStartDateTime(Instant responseStartDateTime) {
            this.responseStartDateTime = responseStartDateTime;
        }

        public final Instant getResponseFinishDateTime() {
            return responseFinishDateTime;
        }

        @Override
        public final Builder responseFinishDateTime(Instant responseFinishDateTime) {
            this.responseFinishDateTime = responseFinishDateTime;
            return this;
        }

        public final void setResponseFinishDateTime(Instant responseFinishDateTime) {
            this.responseFinishDateTime = responseFinishDateTime;
        }

        public final String getOutput() {
            return output;
        }

        @Override
        public final Builder output(String output) {
            this.output = output;
            return this;
        }

        public final void setOutput(String output) {
            this.output = output;
        }

        public final String getStandardOutputUrl() {
            return standardOutputUrl;
        }

        @Override
        public final Builder standardOutputUrl(String standardOutputUrl) {
            this.standardOutputUrl = standardOutputUrl;
            return this;
        }

        public final void setStandardOutputUrl(String standardOutputUrl) {
            this.standardOutputUrl = standardOutputUrl;
        }

        public final String getStandardErrorUrl() {
            return standardErrorUrl;
        }

        @Override
        public final Builder standardErrorUrl(String standardErrorUrl) {
            this.standardErrorUrl = standardErrorUrl;
            return this;
        }

        public final void setStandardErrorUrl(String standardErrorUrl) {
            this.standardErrorUrl = standardErrorUrl;
        }

        public final String getOutputS3Region() {
            return outputS3Region;
        }

        @Override
        public final Builder outputS3Region(String outputS3Region) {
            this.outputS3Region = outputS3Region;
            return this;
        }

        public final void setOutputS3Region(String outputS3Region) {
            this.outputS3Region = outputS3Region;
        }

        public final String getOutputS3BucketName() {
            return outputS3BucketName;
        }

        @Override
        public final Builder outputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
            return this;
        }

        public final void setOutputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
        }

        public final String getOutputS3KeyPrefix() {
            return outputS3KeyPrefix;
        }

        @Override
        public final Builder outputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
            return this;
        }

        public final void setOutputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
        }

        @Override
        public CommandPlugin build() {
            return new CommandPlugin(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
