/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a Systems Manager document.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentDescription.Builder, DocumentDescription> {
    private static final SdkField<String> SHA1_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::sha1)).setter(setter(Builder::sha1))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sha1").build()).build();

    private static final SdkField<String> HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::hash)).setter(setter(Builder::hash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Hash").build()).build();

    private static final SdkField<String> HASH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::hashTypeAsString)).setter(setter(Builder::hashType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HashType").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VERSION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::versionName)).setter(setter(Builder::versionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionName").build()).build();

    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::owner)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Owner").build()).build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(DocumentDescription::createdDate)).setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedDate").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_INFORMATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::statusInformation)).setter(setter(Builder::statusInformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusInformation").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::documentVersion)).setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<DocumentParameter>> PARAMETERS_FIELD = SdkField
            .<List<DocumentParameter>> builder(MarshallingType.LIST)
            .getter(getter(DocumentDescription::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentParameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentParameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> PLATFORM_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DocumentDescription::platformTypesAsStrings))
            .setter(setter(Builder::platformTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DOCUMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::documentTypeAsString)).setter(setter(Builder::documentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentType").build()).build();

    private static final SdkField<String> SCHEMA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::schemaVersion)).setter(setter(Builder::schemaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaVersion").build()).build();

    private static final SdkField<String> LATEST_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::latestVersion)).setter(setter(Builder::latestVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LatestVersion").build()).build();

    private static final SdkField<String> DEFAULT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::defaultVersion)).setter(setter(Builder::defaultVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultVersion").build()).build();

    private static final SdkField<String> DOCUMENT_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::documentFormatAsString)).setter(setter(Builder::documentFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentFormat").build()).build();

    private static final SdkField<String> TARGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentDescription::targetType)).setter(setter(Builder::targetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(DocumentDescription::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AttachmentInformation>> ATTACHMENTS_INFORMATION_FIELD = SdkField
            .<List<AttachmentInformation>> builder(MarshallingType.LIST)
            .getter(getter(DocumentDescription::attachmentsInformation))
            .setter(setter(Builder::attachmentsInformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentsInformation").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttachmentInformation> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttachmentInformation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DocumentRequires>> REQUIRES_FIELD = SdkField
            .<List<DocumentRequires>> builder(MarshallingType.LIST)
            .getter(getter(DocumentDescription::requires))
            .setter(setter(Builder::requires))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Requires").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentRequires> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentRequires::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SHA1_FIELD, HASH_FIELD,
            HASH_TYPE_FIELD, NAME_FIELD, VERSION_NAME_FIELD, OWNER_FIELD, CREATED_DATE_FIELD, STATUS_FIELD,
            STATUS_INFORMATION_FIELD, DOCUMENT_VERSION_FIELD, DESCRIPTION_FIELD, PARAMETERS_FIELD, PLATFORM_TYPES_FIELD,
            DOCUMENT_TYPE_FIELD, SCHEMA_VERSION_FIELD, LATEST_VERSION_FIELD, DEFAULT_VERSION_FIELD, DOCUMENT_FORMAT_FIELD,
            TARGET_TYPE_FIELD, TAGS_FIELD, ATTACHMENTS_INFORMATION_FIELD, REQUIRES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String sha1;

    private final String hash;

    private final String hashType;

    private final String name;

    private final String versionName;

    private final String owner;

    private final Instant createdDate;

    private final String status;

    private final String statusInformation;

    private final String documentVersion;

    private final String description;

    private final List<DocumentParameter> parameters;

    private final List<String> platformTypes;

    private final String documentType;

    private final String schemaVersion;

    private final String latestVersion;

    private final String defaultVersion;

    private final String documentFormat;

    private final String targetType;

    private final List<Tag> tags;

    private final List<AttachmentInformation> attachmentsInformation;

    private final List<DocumentRequires> requires;

    private DocumentDescription(BuilderImpl builder) {
        this.sha1 = builder.sha1;
        this.hash = builder.hash;
        this.hashType = builder.hashType;
        this.name = builder.name;
        this.versionName = builder.versionName;
        this.owner = builder.owner;
        this.createdDate = builder.createdDate;
        this.status = builder.status;
        this.statusInformation = builder.statusInformation;
        this.documentVersion = builder.documentVersion;
        this.description = builder.description;
        this.parameters = builder.parameters;
        this.platformTypes = builder.platformTypes;
        this.documentType = builder.documentType;
        this.schemaVersion = builder.schemaVersion;
        this.latestVersion = builder.latestVersion;
        this.defaultVersion = builder.defaultVersion;
        this.documentFormat = builder.documentFormat;
        this.targetType = builder.targetType;
        this.tags = builder.tags;
        this.attachmentsInformation = builder.attachmentsInformation;
        this.requires = builder.requires;
    }

    /**
     * <p>
     * The SHA1 hash of the document, which you can use for verification.
     * </p>
     * 
     * @return The SHA1 hash of the document, which you can use for verification.
     */
    public String sha1() {
        return sha1;
    }

    /**
     * <p>
     * The Sha256 or Sha1 hash created by the system when the document was created.
     * </p>
     * <note>
     * <p>
     * Sha1 hashes have been deprecated.
     * </p>
     * </note>
     * 
     * @return The Sha256 or Sha1 hash created by the system when the document was created. </p> <note>
     *         <p>
     *         Sha1 hashes have been deprecated.
     *         </p>
     */
    public String hash() {
        return hash;
    }

    /**
     * <p>
     * The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.
     * </p>
     * <note>
     * <p>
     * Sha1 hashes have been deprecated.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hashType} will
     * return {@link DocumentHashType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hashTypeAsString}.
     * </p>
     * 
     * @return The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.</p> <note>
     *         <p>
     *         Sha1 hashes have been deprecated.
     *         </p>
     * @see DocumentHashType
     */
    public DocumentHashType hashType() {
        return DocumentHashType.fromValue(hashType);
    }

    /**
     * <p>
     * The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.
     * </p>
     * <note>
     * <p>
     * Sha1 hashes have been deprecated.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hashType} will
     * return {@link DocumentHashType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hashTypeAsString}.
     * </p>
     * 
     * @return The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.</p> <note>
     *         <p>
     *         Sha1 hashes have been deprecated.
     *         </p>
     * @see DocumentHashType
     */
    public String hashTypeAsString() {
        return hashType;
    }

    /**
     * <p>
     * The name of the Systems Manager document.
     * </p>
     * 
     * @return The name of the Systems Manager document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The version of the artifact associated with the document.
     * </p>
     * 
     * @return The version of the artifact associated with the document.
     */
    public String versionName() {
        return versionName;
    }

    /**
     * <p>
     * The AWS user account that created the document.
     * </p>
     * 
     * @return The AWS user account that created the document.
     */
    public String owner() {
        return owner;
    }

    /**
     * <p>
     * The date when the document was created.
     * </p>
     * 
     * @return The date when the document was created.
     */
    public Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * The status of the Systems Manager document.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DocumentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the Systems Manager document.
     * @see DocumentStatus
     */
    public DocumentStatus status() {
        return DocumentStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the Systems Manager document.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DocumentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the Systems Manager document.
     * @see DocumentStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A message returned by AWS Systems Manager that explains the <code>Status</code> value. For example, a
     * <code>Failed</code> status might be explained by the <code>StatusInformation</code> message,
     * "The specified S3 bucket does not exist. Verify that the URL of the S3 bucket is correct."
     * </p>
     * 
     * @return A message returned by AWS Systems Manager that explains the <code>Status</code> value. For example, a
     *         <code>Failed</code> status might be explained by the <code>StatusInformation</code> message,
     *         "The specified S3 bucket does not exist. Verify that the URL of the S3 bucket is correct."
     */
    public String statusInformation() {
        return statusInformation;
    }

    /**
     * <p>
     * The document version.
     * </p>
     * 
     * @return The document version.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * A description of the document.
     * </p>
     * 
     * @return A description of the document.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the Parameters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A description of the parameters for a document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A description of the parameters for a document.
     */
    public List<DocumentParameter> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The list of OS platforms compatible with this Systems Manager document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPlatformTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of OS platforms compatible with this Systems Manager document.
     */
    public List<PlatformType> platformTypes() {
        return PlatformTypeListCopier.copyStringToEnum(platformTypes);
    }

    /**
     * Returns true if the PlatformTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasPlatformTypes() {
        return platformTypes != null && !(platformTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of OS platforms compatible with this Systems Manager document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPlatformTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of OS platforms compatible with this Systems Manager document.
     */
    public List<String> platformTypesAsStrings() {
        return platformTypes;
    }

    /**
     * <p>
     * The type of document.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The type of document.
     * @see DocumentType
     */
    public DocumentType documentType() {
        return DocumentType.fromValue(documentType);
    }

    /**
     * <p>
     * The type of document.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The type of document.
     * @see DocumentType
     */
    public String documentTypeAsString() {
        return documentType;
    }

    /**
     * <p>
     * The schema version.
     * </p>
     * 
     * @return The schema version.
     */
    public String schemaVersion() {
        return schemaVersion;
    }

    /**
     * <p>
     * The latest version of the document.
     * </p>
     * 
     * @return The latest version of the document.
     */
    public String latestVersion() {
        return latestVersion;
    }

    /**
     * <p>
     * The default version.
     * </p>
     * 
     * @return The default version.
     */
    public String defaultVersion() {
        return defaultVersion;
    }

    /**
     * <p>
     * The document format, either JSON or YAML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return The document format, either JSON or YAML.
     * @see DocumentFormat
     */
    public DocumentFormat documentFormat() {
        return DocumentFormat.fromValue(documentFormat);
    }

    /**
     * <p>
     * The document format, either JSON or YAML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return The document format, either JSON or YAML.
     * @see DocumentFormat
     */
    public String documentFormatAsString() {
        return documentFormat;
    }

    /**
     * <p>
     * The target type which defines the kinds of resources the document can run on. For example, /AWS::EC2::Instance.
     * For a list of valid resource types, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">AWS
     * Resource Types Reference</a> in the <i>AWS CloudFormation User Guide</i>.
     * </p>
     * 
     * @return The target type which defines the kinds of resources the document can run on. For example,
     *         /AWS::EC2::Instance. For a list of valid resource types, see <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">AWS
     *         Resource Types Reference</a> in the <i>AWS CloudFormation User Guide</i>.
     */
    public String targetType() {
        return targetType;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags, or metadata, that have been applied to the document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tags, or metadata, that have been applied to the document.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * Returns true if the AttachmentsInformation property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAttachmentsInformation() {
        return attachmentsInformation != null && !(attachmentsInformation instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Details about the document attachments, including names, locations, sizes, etc.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttachmentsInformation()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Details about the document attachments, including names, locations, sizes, etc.
     */
    public List<AttachmentInformation> attachmentsInformation() {
        return attachmentsInformation;
    }

    /**
     * Returns true if the Requires property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRequires() {
        return requires != null && !(requires instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code> document
     * requires an <code>ApplicationConfigurationSchema</code> document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRequires()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
     *         document requires an <code>ApplicationConfigurationSchema</code> document.
     */
    public List<DocumentRequires> requires() {
        return requires;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sha1());
        hashCode = 31 * hashCode + Objects.hashCode(hash());
        hashCode = 31 * hashCode + Objects.hashCode(hashTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(versionName());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusInformation());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(platformTypesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(documentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(latestVersion());
        hashCode = 31 * hashCode + Objects.hashCode(defaultVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetType());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(attachmentsInformation());
        hashCode = 31 * hashCode + Objects.hashCode(requires());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentDescription)) {
            return false;
        }
        DocumentDescription other = (DocumentDescription) obj;
        return Objects.equals(sha1(), other.sha1()) && Objects.equals(hash(), other.hash())
                && Objects.equals(hashTypeAsString(), other.hashTypeAsString()) && Objects.equals(name(), other.name())
                && Objects.equals(versionName(), other.versionName()) && Objects.equals(owner(), other.owner())
                && Objects.equals(createdDate(), other.createdDate()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusInformation(), other.statusInformation())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(description(), other.description()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(platformTypesAsStrings(), other.platformTypesAsStrings())
                && Objects.equals(documentTypeAsString(), other.documentTypeAsString())
                && Objects.equals(schemaVersion(), other.schemaVersion())
                && Objects.equals(latestVersion(), other.latestVersion())
                && Objects.equals(defaultVersion(), other.defaultVersion())
                && Objects.equals(documentFormatAsString(), other.documentFormatAsString())
                && Objects.equals(targetType(), other.targetType()) && Objects.equals(tags(), other.tags())
                && Objects.equals(attachmentsInformation(), other.attachmentsInformation())
                && Objects.equals(requires(), other.requires());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DocumentDescription").add("Sha1", sha1()).add("Hash", hash())
                .add("HashType", hashTypeAsString()).add("Name", name()).add("VersionName", versionName()).add("Owner", owner())
                .add("CreatedDate", createdDate()).add("Status", statusAsString()).add("StatusInformation", statusInformation())
                .add("DocumentVersion", documentVersion()).add("Description", description()).add("Parameters", parameters())
                .add("PlatformTypes", platformTypesAsStrings()).add("DocumentType", documentTypeAsString())
                .add("SchemaVersion", schemaVersion()).add("LatestVersion", latestVersion())
                .add("DefaultVersion", defaultVersion()).add("DocumentFormat", documentFormatAsString())
                .add("TargetType", targetType()).add("Tags", tags()).add("AttachmentsInformation", attachmentsInformation())
                .add("Requires", requires()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Sha1":
            return Optional.ofNullable(clazz.cast(sha1()));
        case "Hash":
            return Optional.ofNullable(clazz.cast(hash()));
        case "HashType":
            return Optional.ofNullable(clazz.cast(hashTypeAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "VersionName":
            return Optional.ofNullable(clazz.cast(versionName()));
        case "Owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "CreatedDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusInformation":
            return Optional.ofNullable(clazz.cast(statusInformation()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "PlatformTypes":
            return Optional.ofNullable(clazz.cast(platformTypesAsStrings()));
        case "DocumentType":
            return Optional.ofNullable(clazz.cast(documentTypeAsString()));
        case "SchemaVersion":
            return Optional.ofNullable(clazz.cast(schemaVersion()));
        case "LatestVersion":
            return Optional.ofNullable(clazz.cast(latestVersion()));
        case "DefaultVersion":
            return Optional.ofNullable(clazz.cast(defaultVersion()));
        case "DocumentFormat":
            return Optional.ofNullable(clazz.cast(documentFormatAsString()));
        case "TargetType":
            return Optional.ofNullable(clazz.cast(targetType()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "AttachmentsInformation":
            return Optional.ofNullable(clazz.cast(attachmentsInformation()));
        case "Requires":
            return Optional.ofNullable(clazz.cast(requires()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DocumentDescription, T> g) {
        return obj -> g.apply((DocumentDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentDescription> {
        /**
         * <p>
         * The SHA1 hash of the document, which you can use for verification.
         * </p>
         * 
         * @param sha1
         *        The SHA1 hash of the document, which you can use for verification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sha1(String sha1);

        /**
         * <p>
         * The Sha256 or Sha1 hash created by the system when the document was created.
         * </p>
         * <note>
         * <p>
         * Sha1 hashes have been deprecated.
         * </p>
         * </note>
         * 
         * @param hash
         *        The Sha256 or Sha1 hash created by the system when the document was created. </p> <note>
         *        <p>
         *        Sha1 hashes have been deprecated.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hash(String hash);

        /**
         * <p>
         * The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.
         * </p>
         * <note>
         * <p>
         * Sha1 hashes have been deprecated.
         * </p>
         * </note>
         * 
         * @param hashType
         *        The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.</p>
         *        <note>
         *        <p>
         *        Sha1 hashes have been deprecated.
         *        </p>
         * @see DocumentHashType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentHashType
         */
        Builder hashType(String hashType);

        /**
         * <p>
         * The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.
         * </p>
         * <note>
         * <p>
         * Sha1 hashes have been deprecated.
         * </p>
         * </note>
         * 
         * @param hashType
         *        The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.</p>
         *        <note>
         *        <p>
         *        Sha1 hashes have been deprecated.
         *        </p>
         * @see DocumentHashType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentHashType
         */
        Builder hashType(DocumentHashType hashType);

        /**
         * <p>
         * The name of the Systems Manager document.
         * </p>
         * 
         * @param name
         *        The name of the Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The version of the artifact associated with the document.
         * </p>
         * 
         * @param versionName
         *        The version of the artifact associated with the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionName(String versionName);

        /**
         * <p>
         * The AWS user account that created the document.
         * </p>
         * 
         * @param owner
         *        The AWS user account that created the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * The date when the document was created.
         * </p>
         * 
         * @param createdDate
         *        The date when the document was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * The status of the Systems Manager document.
         * </p>
         * 
         * @param status
         *        The status of the Systems Manager document.
         * @see DocumentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the Systems Manager document.
         * </p>
         * 
         * @param status
         *        The status of the Systems Manager document.
         * @see DocumentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentStatus
         */
        Builder status(DocumentStatus status);

        /**
         * <p>
         * A message returned by AWS Systems Manager that explains the <code>Status</code> value. For example, a
         * <code>Failed</code> status might be explained by the <code>StatusInformation</code> message,
         * "The specified S3 bucket does not exist. Verify that the URL of the S3 bucket is correct."
         * </p>
         * 
         * @param statusInformation
         *        A message returned by AWS Systems Manager that explains the <code>Status</code> value. For example, a
         *        <code>Failed</code> status might be explained by the <code>StatusInformation</code> message,
         *        "The specified S3 bucket does not exist. Verify that the URL of the S3 bucket is correct."
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusInformation(String statusInformation);

        /**
         * <p>
         * The document version.
         * </p>
         * 
         * @param documentVersion
         *        The document version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * A description of the document.
         * </p>
         * 
         * @param description
         *        A description of the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A description of the parameters for a document.
         * </p>
         * 
         * @param parameters
         *        A description of the parameters for a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<DocumentParameter> parameters);

        /**
         * <p>
         * A description of the parameters for a document.
         * </p>
         * 
         * @param parameters
         *        A description of the parameters for a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(DocumentParameter... parameters);

        /**
         * <p>
         * A description of the parameters for a document.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DocumentParameter>.Builder} avoiding the
         * need to create one manually via {@link List<DocumentParameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DocumentParameter>.Builder#build()} is called immediately
         * and its result is passed to {@link #parameters(List<DocumentParameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<DocumentParameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<DocumentParameter>)
         */
        Builder parameters(Consumer<DocumentParameter.Builder>... parameters);

        /**
         * <p>
         * The list of OS platforms compatible with this Systems Manager document.
         * </p>
         * 
         * @param platformTypes
         *        The list of OS platforms compatible with this Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypesWithStrings(Collection<String> platformTypes);

        /**
         * <p>
         * The list of OS platforms compatible with this Systems Manager document.
         * </p>
         * 
         * @param platformTypes
         *        The list of OS platforms compatible with this Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypesWithStrings(String... platformTypes);

        /**
         * <p>
         * The list of OS platforms compatible with this Systems Manager document.
         * </p>
         * 
         * @param platformTypes
         *        The list of OS platforms compatible with this Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(Collection<PlatformType> platformTypes);

        /**
         * <p>
         * The list of OS platforms compatible with this Systems Manager document.
         * </p>
         * 
         * @param platformTypes
         *        The list of OS platforms compatible with this Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(PlatformType... platformTypes);

        /**
         * <p>
         * The type of document.
         * </p>
         * 
         * @param documentType
         *        The type of document.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(String documentType);

        /**
         * <p>
         * The type of document.
         * </p>
         * 
         * @param documentType
         *        The type of document.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(DocumentType documentType);

        /**
         * <p>
         * The schema version.
         * </p>
         * 
         * @param schemaVersion
         *        The schema version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The latest version of the document.
         * </p>
         * 
         * @param latestVersion
         *        The latest version of the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestVersion(String latestVersion);

        /**
         * <p>
         * The default version.
         * </p>
         * 
         * @param defaultVersion
         *        The default version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultVersion(String defaultVersion);

        /**
         * <p>
         * The document format, either JSON or YAML.
         * </p>
         * 
         * @param documentFormat
         *        The document format, either JSON or YAML.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(String documentFormat);

        /**
         * <p>
         * The document format, either JSON or YAML.
         * </p>
         * 
         * @param documentFormat
         *        The document format, either JSON or YAML.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(DocumentFormat documentFormat);

        /**
         * <p>
         * The target type which defines the kinds of resources the document can run on. For example,
         * /AWS::EC2::Instance. For a list of valid resource types, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">AWS
         * Resource Types Reference</a> in the <i>AWS CloudFormation User Guide</i>.
         * </p>
         * 
         * @param targetType
         *        The target type which defines the kinds of resources the document can run on. For example,
         *        /AWS::EC2::Instance. For a list of valid resource types, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         *        >AWS Resource Types Reference</a> in the <i>AWS CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetType(String targetType);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Details about the document attachments, including names, locations, sizes, etc.
         * </p>
         * 
         * @param attachmentsInformation
         *        Details about the document attachments, including names, locations, sizes, etc.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentsInformation(Collection<AttachmentInformation> attachmentsInformation);

        /**
         * <p>
         * Details about the document attachments, including names, locations, sizes, etc.
         * </p>
         * 
         * @param attachmentsInformation
         *        Details about the document attachments, including names, locations, sizes, etc.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentsInformation(AttachmentInformation... attachmentsInformation);

        /**
         * <p>
         * Details about the document attachments, including names, locations, sizes, etc.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttachmentInformation>.Builder} avoiding
         * the need to create one manually via {@link List<AttachmentInformation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttachmentInformation>.Builder#build()} is called
         * immediately and its result is passed to {@link #attachmentsInformation(List<AttachmentInformation>)}.
         * 
         * @param attachmentsInformation
         *        a consumer that will call methods on {@link List<AttachmentInformation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachmentsInformation(List<AttachmentInformation>)
         */
        Builder attachmentsInformation(Consumer<AttachmentInformation.Builder>... attachmentsInformation);

        /**
         * <p>
         * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         * document requires an <code>ApplicationConfigurationSchema</code> document.
         * </p>
         * 
         * @param requires
         *        A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         *        document requires an <code>ApplicationConfigurationSchema</code> document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requires(Collection<DocumentRequires> requires);

        /**
         * <p>
         * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         * document requires an <code>ApplicationConfigurationSchema</code> document.
         * </p>
         * 
         * @param requires
         *        A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         *        document requires an <code>ApplicationConfigurationSchema</code> document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requires(DocumentRequires... requires);

        /**
         * <p>
         * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         * document requires an <code>ApplicationConfigurationSchema</code> document.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DocumentRequires>.Builder} avoiding the
         * need to create one manually via {@link List<DocumentRequires>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DocumentRequires>.Builder#build()} is called immediately and
         * its result is passed to {@link #requires(List<DocumentRequires>)}.
         * 
         * @param requires
         *        a consumer that will call methods on {@link List<DocumentRequires>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #requires(List<DocumentRequires>)
         */
        Builder requires(Consumer<DocumentRequires.Builder>... requires);
    }

    static final class BuilderImpl implements Builder {
        private String sha1;

        private String hash;

        private String hashType;

        private String name;

        private String versionName;

        private String owner;

        private Instant createdDate;

        private String status;

        private String statusInformation;

        private String documentVersion;

        private String description;

        private List<DocumentParameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private List<String> platformTypes = DefaultSdkAutoConstructList.getInstance();

        private String documentType;

        private String schemaVersion;

        private String latestVersion;

        private String defaultVersion;

        private String documentFormat;

        private String targetType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<AttachmentInformation> attachmentsInformation = DefaultSdkAutoConstructList.getInstance();

        private List<DocumentRequires> requires = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentDescription model) {
            sha1(model.sha1);
            hash(model.hash);
            hashType(model.hashType);
            name(model.name);
            versionName(model.versionName);
            owner(model.owner);
            createdDate(model.createdDate);
            status(model.status);
            statusInformation(model.statusInformation);
            documentVersion(model.documentVersion);
            description(model.description);
            parameters(model.parameters);
            platformTypesWithStrings(model.platformTypes);
            documentType(model.documentType);
            schemaVersion(model.schemaVersion);
            latestVersion(model.latestVersion);
            defaultVersion(model.defaultVersion);
            documentFormat(model.documentFormat);
            targetType(model.targetType);
            tags(model.tags);
            attachmentsInformation(model.attachmentsInformation);
            requires(model.requires);
        }

        public final String getSha1() {
            return sha1;
        }

        @Override
        public final Builder sha1(String sha1) {
            this.sha1 = sha1;
            return this;
        }

        public final void setSha1(String sha1) {
            this.sha1 = sha1;
        }

        public final String getHash() {
            return hash;
        }

        @Override
        public final Builder hash(String hash) {
            this.hash = hash;
            return this;
        }

        public final void setHash(String hash) {
            this.hash = hash;
        }

        public final String getHashTypeAsString() {
            return hashType;
        }

        @Override
        public final Builder hashType(String hashType) {
            this.hashType = hashType;
            return this;
        }

        @Override
        public final Builder hashType(DocumentHashType hashType) {
            this.hashType(hashType == null ? null : hashType.toString());
            return this;
        }

        public final void setHashType(String hashType) {
            this.hashType = hashType;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getVersionName() {
            return versionName;
        }

        @Override
        public final Builder versionName(String versionName) {
            this.versionName = versionName;
            return this;
        }

        public final void setVersionName(String versionName) {
            this.versionName = versionName;
        }

        public final String getOwner() {
            return owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DocumentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusInformation() {
            return statusInformation;
        }

        @Override
        public final Builder statusInformation(String statusInformation) {
            this.statusInformation = statusInformation;
            return this;
        }

        public final void setStatusInformation(String statusInformation) {
            this.statusInformation = statusInformation;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<DocumentParameter.Builder> getParameters() {
            return parameters != null ? parameters.stream().map(DocumentParameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<DocumentParameter> parameters) {
            this.parameters = DocumentParameterListCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(DocumentParameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<DocumentParameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> DocumentParameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setParameters(Collection<DocumentParameter.BuilderImpl> parameters) {
            this.parameters = DocumentParameterListCopier.copyFromBuilder(parameters);
        }

        public final Collection<String> getPlatformTypesAsStrings() {
            return platformTypes;
        }

        @Override
        public final Builder platformTypesWithStrings(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformTypesWithStrings(String... platformTypes) {
            platformTypesWithStrings(Arrays.asList(platformTypes));
            return this;
        }

        @Override
        public final Builder platformTypes(Collection<PlatformType> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copyEnumToString(platformTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformTypes(PlatformType... platformTypes) {
            platformTypes(Arrays.asList(platformTypes));
            return this;
        }

        public final void setPlatformTypesWithStrings(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
        }

        public final String getDocumentTypeAsString() {
            return documentType;
        }

        @Override
        public final Builder documentType(String documentType) {
            this.documentType = documentType;
            return this;
        }

        @Override
        public final Builder documentType(DocumentType documentType) {
            this.documentType(documentType == null ? null : documentType.toString());
            return this;
        }

        public final void setDocumentType(String documentType) {
            this.documentType = documentType;
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        public final String getLatestVersion() {
            return latestVersion;
        }

        @Override
        public final Builder latestVersion(String latestVersion) {
            this.latestVersion = latestVersion;
            return this;
        }

        public final void setLatestVersion(String latestVersion) {
            this.latestVersion = latestVersion;
        }

        public final String getDefaultVersion() {
            return defaultVersion;
        }

        @Override
        public final Builder defaultVersion(String defaultVersion) {
            this.defaultVersion = defaultVersion;
            return this;
        }

        public final void setDefaultVersion(String defaultVersion) {
            this.defaultVersion = defaultVersion;
        }

        public final String getDocumentFormatAsString() {
            return documentFormat;
        }

        @Override
        public final Builder documentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
            return this;
        }

        @Override
        public final Builder documentFormat(DocumentFormat documentFormat) {
            this.documentFormat(documentFormat == null ? null : documentFormat.toString());
            return this;
        }

        public final void setDocumentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
        }

        public final String getTargetType() {
            return targetType;
        }

        @Override
        public final Builder targetType(String targetType) {
            this.targetType = targetType;
            return this;
        }

        public final void setTargetType(String targetType) {
            this.targetType = targetType;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final Collection<AttachmentInformation.Builder> getAttachmentsInformation() {
            return attachmentsInformation != null ? attachmentsInformation.stream().map(AttachmentInformation::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder attachmentsInformation(Collection<AttachmentInformation> attachmentsInformation) {
            this.attachmentsInformation = AttachmentInformationListCopier.copy(attachmentsInformation);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachmentsInformation(AttachmentInformation... attachmentsInformation) {
            attachmentsInformation(Arrays.asList(attachmentsInformation));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachmentsInformation(Consumer<AttachmentInformation.Builder>... attachmentsInformation) {
            attachmentsInformation(Stream.of(attachmentsInformation)
                    .map(c -> AttachmentInformation.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAttachmentsInformation(Collection<AttachmentInformation.BuilderImpl> attachmentsInformation) {
            this.attachmentsInformation = AttachmentInformationListCopier.copyFromBuilder(attachmentsInformation);
        }

        public final Collection<DocumentRequires.Builder> getRequires() {
            return requires != null ? requires.stream().map(DocumentRequires::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder requires(Collection<DocumentRequires> requires) {
            this.requires = DocumentRequiresListCopier.copy(requires);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requires(DocumentRequires... requires) {
            requires(Arrays.asList(requires));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requires(Consumer<DocumentRequires.Builder>... requires) {
            requires(Stream.of(requires).map(c -> DocumentRequires.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRequires(Collection<DocumentRequires.BuilderImpl> requires) {
            this.requires = DocumentRequiresListCopier.copyFromBuilder(requires);
        }

        @Override
        public DocumentDescription build() {
            return new DocumentDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
