/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.services.signin.internal;

import java.math.BigInteger;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.security.AlgorithmParameters;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPrivateKeySpec;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.InvalidParameterSpecException;
import java.util.Arrays;
import java.util.Base64;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.utils.Pair;

@SdkInternalApi
public final class EcKeyLoader {
    private static final String SECP_256_R1_STD_NAME = "secp256r1";
    private static final byte DER_SEQUENCE_TAG = 48;
    private static final byte DER_INTEGER_TAG = 2;
    private static final byte DER_OCTET_STRING_TAG = 4;
    private static final byte DER_BIT_STRING_TAG = 3;
    private static final byte DER_OPTIONAL_SEQ_PARAM_0 = -96;
    private static final byte DER_OPTIONAL_SEQ_PARAM_1 = -95;
    private static final byte DER_OBJECT_IDENTIFIER_TAG = 6;
    private static final int SEC1_VERSION = 1;
    private static byte[] SECP_256_R1_OID_BYTES = new byte[]{42, -122, 72, -50, 61, 3, 1, 7};

    private EcKeyLoader() {
    }

    public static Pair<ECPrivateKey, ECPublicKey> loadSec1Pem(String pem) {
        try {
            byte[] sec1Der = EcKeyLoader.pemToDer(pem);
            ParsedEcKey parsed = EcKeyLoader.parseSec1(sec1Der);
            if (parsed.curveOid == null) {
                throw new IllegalArgumentException("Missing EC Curve OID");
            }
            ECParameterSpec params = EcKeyLoader.curveFromOid(parsed.curveOid);
            ECPrivateKey privateKey = (ECPrivateKey)KeyFactory.getInstance("EC").generatePrivate(new ECPrivateKeySpec(parsed.privateScalar, params));
            if (parsed.publicBytes == null) {
                throw new IllegalArgumentException("Invalid certificate - public key is required.");
            }
            ECPublicKey publicKey = EcKeyLoader.derivePublicFromBytes(parsed.publicBytes, privateKey.getParams());
            return Pair.of((Object)privateKey, (Object)publicKey);
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException | InvalidParameterSpecException e) {
            throw new RuntimeException(e);
        }
    }

    private static ECParameterSpec curveFromOid(byte[] oid) throws NoSuchAlgorithmException, InvalidParameterSpecException {
        if (Arrays.equals(SECP_256_R1_OID_BYTES, oid)) {
            AlgorithmParameters parameters = null;
            parameters = AlgorithmParameters.getInstance("EC");
            parameters.init(new ECGenParameterSpec(SECP_256_R1_STD_NAME));
            return parameters.getParameterSpec(ECParameterSpec.class);
        }
        throw new IllegalArgumentException("Unsupported curve OID: " + Arrays.toString(oid));
    }

    private static ECPublicKey derivePublicFromBytes(byte[] raw, ECParameterSpec params) throws NoSuchAlgorithmException, InvalidKeySpecException {
        if (raw[0] != 4) {
            throw new IllegalArgumentException("Expected uncompressed point");
        }
        int len = (raw.length - 1) / 2;
        BigInteger x = new BigInteger(1, Arrays.copyOfRange(raw, 1, 1 + len));
        BigInteger y = new BigInteger(1, Arrays.copyOfRange(raw, 1 + len, 1 + 2 * len));
        ECPoint w = new ECPoint(x, y);
        ECPublicKeySpec spec = new ECPublicKeySpec(w, params);
        return (ECPublicKey)KeyFactory.getInstance("EC").generatePublic(spec);
    }

    private static ParsedEcKey parseSec1(byte[] der) {
        ParsedEcKey result = new ParsedEcKey();
        ByteBuffer buffer = ByteBuffer.wrap(der);
        try {
            if (buffer.get() != 48) {
                throw new IllegalArgumentException("Invalid SEC1 Private Key: Not a SEQUENCE");
            }
            EcKeyLoader.readLength(buffer);
            if (buffer.get() != 2) {
                throw new IllegalArgumentException("Invalid SEC1 Private Key: Expected INTEGER");
            }
            int len = EcKeyLoader.readLength(buffer);
            if (len != 1 || buffer.get() != 1) {
                throw new IllegalArgumentException("Invalid SEC1 Private Key: invalid version");
            }
            if (buffer.get() != 4) {
                throw new IllegalArgumentException("Invalid SEC1 Private Key: Expected OCTET STRING");
            }
            len = EcKeyLoader.readLength(buffer);
            byte[] privateKeyBytes = new byte[len];
            buffer.get(privateKeyBytes);
            result.privateScalar = new BigInteger(1, privateKeyBytes);
            while (buffer.hasRemaining()) {
                byte tag = buffer.get();
                len = EcKeyLoader.readLength(buffer);
                if (tag == -96) {
                    if (buffer.get() != 6) {
                        throw new IllegalArgumentException("Invalid SEC1 Private Key: Expected OID");
                    }
                    int oidLen = EcKeyLoader.readLength(buffer);
                    byte[] oid = new byte[oidLen];
                    buffer.get(oid);
                    result.curveOid = oid;
                    continue;
                }
                if (tag == -95) {
                    byte bitTag = buffer.get();
                    if (bitTag != 3) {
                        throw new IllegalArgumentException("Invalid SEC1 Private Key: Expected BIT STRING");
                    }
                    int bitLen = EcKeyLoader.readLength(buffer);
                    byte[] bitString = new byte[bitLen];
                    buffer.get(bitString);
                    result.publicBytes = Arrays.copyOfRange(bitString, 1, bitString.length);
                    continue;
                }
                buffer.position(buffer.position() + len);
            }
        }
        catch (BufferUnderflowException e) {
            throw new IllegalArgumentException("Invalid SEC1 Private Key: failed to parse.", e);
        }
        return result;
    }

    public static byte[] pemToDer(String pem) {
        StringBuilder sb = new StringBuilder();
        for (String line : pem.split("\\r?\\n")) {
            if (line.startsWith("-----")) continue;
            sb.append(line.trim());
        }
        return Base64.getDecoder().decode(sb.toString());
    }

    private static int readLength(ByteBuffer buffer) {
        int b = buffer.get() & 0xFF;
        if ((b & 0x80) == 0) {
            return b;
        }
        int num = b & 0x7F;
        if (num == 0) {
            throw new IllegalArgumentException("Indefinite lengths not supported");
        }
        if (num > 4) {
            throw new IllegalArgumentException("Too many bytes in length");
        }
        int val = 0;
        for (int i = 0; i < num; ++i) {
            int nb = buffer.get() & 0xFF;
            if (nb < 0) {
                throw new IllegalArgumentException("Unexpected EOF in length bytes");
            }
            val = val << 8 | nb & 0xFF;
        }
        return val;
    }

    private static class ParsedEcKey {
        BigInteger privateScalar;
        byte[] curveOid;
        byte[] publicBytes;

        private ParsedEcKey() {
        }
    }
}

