/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sfn.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateStateMachineRequest extends SfnRequest implements
        ToCopyableBuilder<CreateStateMachineRequest.Builder, CreateStateMachineRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CreateStateMachineRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DEFINITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("definition").getter(getter(CreateStateMachineRequest::definition)).setter(setter(Builder::definition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("definition").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(CreateStateMachineRequest::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(CreateStateMachineRequest::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<LoggingConfiguration> LOGGING_CONFIGURATION_FIELD = SdkField
            .<LoggingConfiguration> builder(MarshallingType.SDK_POJO).memberName("loggingConfiguration")
            .getter(getter(CreateStateMachineRequest::loggingConfiguration)).setter(setter(Builder::loggingConfiguration))
            .constructor(LoggingConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("loggingConfiguration").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(CreateStateMachineRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TracingConfiguration> TRACING_CONFIGURATION_FIELD = SdkField
            .<TracingConfiguration> builder(MarshallingType.SDK_POJO).memberName("tracingConfiguration")
            .getter(getter(CreateStateMachineRequest::tracingConfiguration)).setter(setter(Builder::tracingConfiguration))
            .constructor(TracingConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tracingConfiguration").build())
            .build();

    private static final SdkField<Boolean> PUBLISH_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("publish").getter(getter(CreateStateMachineRequest::publish)).setter(setter(Builder::publish))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publish").build()).build();

    private static final SdkField<String> VERSION_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("versionDescription").getter(getter(CreateStateMachineRequest::versionDescription))
            .setter(setter(Builder::versionDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versionDescription").build())
            .build();

    private static final SdkField<EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfiguration> builder(MarshallingType.SDK_POJO).memberName("encryptionConfiguration")
            .getter(getter(CreateStateMachineRequest::encryptionConfiguration)).setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DEFINITION_FIELD,
            ROLE_ARN_FIELD, TYPE_FIELD, LOGGING_CONFIGURATION_FIELD, TAGS_FIELD, TRACING_CONFIGURATION_FIELD, PUBLISH_FIELD,
            VERSION_DESCRIPTION_FIELD, ENCRYPTION_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String name;

    private final String definition;

    private final String roleArn;

    private final String type;

    private final LoggingConfiguration loggingConfiguration;

    private final List<Tag> tags;

    private final TracingConfiguration tracingConfiguration;

    private final Boolean publish;

    private final String versionDescription;

    private final EncryptionConfiguration encryptionConfiguration;

    private CreateStateMachineRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.definition = builder.definition;
        this.roleArn = builder.roleArn;
        this.type = builder.type;
        this.loggingConfiguration = builder.loggingConfiguration;
        this.tags = builder.tags;
        this.tracingConfiguration = builder.tracingConfiguration;
        this.publish = builder.publish;
        this.versionDescription = builder.versionDescription;
        this.encryptionConfiguration = builder.encryptionConfiguration;
    }

    /**
     * <p>
     * The name of the state machine.
     * </p>
     * <p>
     * A name must <i>not</i> contain:
     * </p>
     * <ul>
     * <li>
     * <p>
     * white space
     * </p>
     * </li>
     * <li>
     * <p>
     * brackets <code>&lt; &gt; { } [ ]</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * wildcard characters <code>? *</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * special characters <code>" # % \ ^ | ~ ` $ &amp; , ; : /</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * control characters (<code>U+0000-001F</code>, <code>U+007F-009F</code>, <code>U+FFFE-FFFF</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * surrogates (<code>U+D800-DFFF</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * invalid characters (<code> U+10FFFF</code>)
     * </p>
     * </li>
     * </ul>
     * <p>
     * To enable logging with CloudWatch Logs, the name should only contain 0-9, A-Z, a-z, - and _.
     * </p>
     * 
     * @return The name of the state machine. </p>
     *         <p>
     *         A name must <i>not</i> contain:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         white space
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         brackets <code>&lt; &gt; { } [ ]</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         wildcard characters <code>? *</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         special characters <code>" # % \ ^ | ~ ` $ &amp; , ; : /</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         control characters (<code>U+0000-001F</code>, <code>U+007F-009F</code>, <code>U+FFFE-FFFF</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         surrogates (<code>U+D800-DFFF</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         invalid characters (<code> U+10FFFF</code>)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To enable logging with CloudWatch Logs, the name should only contain 0-9, A-Z, a-z, - and _.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon States Language definition of the state machine. See <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html">Amazon States
     * Language</a>.
     * </p>
     * 
     * @return The Amazon States Language definition of the state machine. See <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html">Amazon
     *         States Language</a>.
     */
    public final String definition() {
        return definition;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role to use for this state machine.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role to use for this state machine.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * Determines whether a Standard or Express state machine is created. The default is <code>STANDARD</code>. You
     * cannot update the <code>type</code> of a state machine once it has been created.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link StateMachineType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Determines whether a Standard or Express state machine is created. The default is <code>STANDARD</code>.
     *         You cannot update the <code>type</code> of a state machine once it has been created.
     * @see StateMachineType
     */
    public final StateMachineType type() {
        return StateMachineType.fromValue(type);
    }

    /**
     * <p>
     * Determines whether a Standard or Express state machine is created. The default is <code>STANDARD</code>. You
     * cannot update the <code>type</code> of a state machine once it has been created.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link StateMachineType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Determines whether a Standard or Express state machine is created. The default is <code>STANDARD</code>.
     *         You cannot update the <code>type</code> of a state machine once it has been created.
     * @see StateMachineType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Defines what execution history events are logged and where they are logged.
     * </p>
     * <note>
     * <p>
     * By default, the <code>level</code> is set to <code>OFF</code>. For more information see <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html">Log Levels</a> in the Step
     * Functions User Guide.
     * </p>
     * </note>
     * 
     * @return Defines what execution history events are logged and where they are logged.</p> <note>
     *         <p>
     *         By default, the <code>level</code> is set to <code>OFF</code>. For more information see <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html">Log Levels</a> in
     *         the Step Functions User Guide.
     *         </p>
     */
    public final LoggingConfiguration loggingConfiguration() {
        return loggingConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags to be added when creating a state machine.
     * </p>
     * <p>
     * An array of key-value pairs. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>, and <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using IAM
     * Tags</a>.
     * </p>
     * <p>
     * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Tags to be added when creating a state machine.</p>
     *         <p>
     *         An array of key-value pairs. For more information, see <a
     *         href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost
     *         Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>, and <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using IAM
     *         Tags</a>.
     *         </p>
     *         <p>
     *         Tags may only contain Unicode letters, digits, white space, or these symbols:
     *         <code>_ . : / = + - @</code>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Selects whether X-Ray tracing is enabled.
     * </p>
     * 
     * @return Selects whether X-Ray tracing is enabled.
     */
    public final TracingConfiguration tracingConfiguration() {
        return tracingConfiguration;
    }

    /**
     * <p>
     * Set to <code>true</code> to publish the first version of the state machine during creation. The default is
     * <code>false</code>.
     * </p>
     * 
     * @return Set to <code>true</code> to publish the first version of the state machine during creation. The default
     *         is <code>false</code>.
     */
    public final Boolean publish() {
        return publish;
    }

    /**
     * <p>
     * Sets description about the state machine version. You can only set the description if the <code>publish</code>
     * parameter is set to <code>true</code>. Otherwise, if you set <code>versionDescription</code>, but
     * <code>publish</code> to <code>false</code>, this API action throws <code>ValidationException</code>.
     * </p>
     * 
     * @return Sets description about the state machine version. You can only set the description if the
     *         <code>publish</code> parameter is set to <code>true</code>. Otherwise, if you set
     *         <code>versionDescription</code>, but <code>publish</code> to <code>false</code>, this API action throws
     *         <code>ValidationException</code>.
     */
    public final String versionDescription() {
        return versionDescription;
    }

    /**
     * <p>
     * Settings to configure server-side encryption.
     * </p>
     * 
     * @return Settings to configure server-side encryption.
     */
    public final EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(definition());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(loggingConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(tracingConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(publish());
        hashCode = 31 * hashCode + Objects.hashCode(versionDescription());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateStateMachineRequest)) {
            return false;
        }
        CreateStateMachineRequest other = (CreateStateMachineRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(definition(), other.definition())
                && Objects.equals(roleArn(), other.roleArn()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(loggingConfiguration(), other.loggingConfiguration()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(tracingConfiguration(), other.tracingConfiguration())
                && Objects.equals(publish(), other.publish()) && Objects.equals(versionDescription(), other.versionDescription())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateStateMachineRequest").add("Name", name())
                .add("Definition", definition() == null ? null : "*** Sensitive Data Redacted ***").add("RoleArn", roleArn())
                .add("Type", typeAsString()).add("LoggingConfiguration", loggingConfiguration())
                .add("Tags", hasTags() ? tags() : null).add("TracingConfiguration", tracingConfiguration())
                .add("Publish", publish())
                .add("VersionDescription", versionDescription() == null ? null : "*** Sensitive Data Redacted ***")
                .add("EncryptionConfiguration", encryptionConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "definition":
            return Optional.ofNullable(clazz.cast(definition()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "loggingConfiguration":
            return Optional.ofNullable(clazz.cast(loggingConfiguration()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "tracingConfiguration":
            return Optional.ofNullable(clazz.cast(tracingConfiguration()));
        case "publish":
            return Optional.ofNullable(clazz.cast(publish()));
        case "versionDescription":
            return Optional.ofNullable(clazz.cast(versionDescription()));
        case "encryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("definition", DEFINITION_FIELD);
        map.put("roleArn", ROLE_ARN_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("loggingConfiguration", LOGGING_CONFIGURATION_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("tracingConfiguration", TRACING_CONFIGURATION_FIELD);
        map.put("publish", PUBLISH_FIELD);
        map.put("versionDescription", VERSION_DESCRIPTION_FIELD);
        map.put("encryptionConfiguration", ENCRYPTION_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateStateMachineRequest, T> g) {
        return obj -> g.apply((CreateStateMachineRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SfnRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateStateMachineRequest> {
        /**
         * <p>
         * The name of the state machine.
         * </p>
         * <p>
         * A name must <i>not</i> contain:
         * </p>
         * <ul>
         * <li>
         * <p>
         * white space
         * </p>
         * </li>
         * <li>
         * <p>
         * brackets <code>&lt; &gt; { } [ ]</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * wildcard characters <code>? *</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * special characters <code>" # % \ ^ | ~ ` $ &amp; , ; : /</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * control characters (<code>U+0000-001F</code>, <code>U+007F-009F</code>, <code>U+FFFE-FFFF</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * surrogates (<code>U+D800-DFFF</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * invalid characters (<code> U+10FFFF</code>)
         * </p>
         * </li>
         * </ul>
         * <p>
         * To enable logging with CloudWatch Logs, the name should only contain 0-9, A-Z, a-z, - and _.
         * </p>
         * 
         * @param name
         *        The name of the state machine. </p>
         *        <p>
         *        A name must <i>not</i> contain:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        white space
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        brackets <code>&lt; &gt; { } [ ]</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        wildcard characters <code>? *</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        special characters <code>" # % \ ^ | ~ ` $ &amp; , ; : /</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        control characters (<code>U+0000-001F</code>, <code>U+007F-009F</code>, <code>U+FFFE-FFFF</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        surrogates (<code>U+D800-DFFF</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        invalid characters (<code> U+10FFFF</code>)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To enable logging with CloudWatch Logs, the name should only contain 0-9, A-Z, a-z, - and _.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon States Language definition of the state machine. See <a
         * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html">Amazon
         * States Language</a>.
         * </p>
         * 
         * @param definition
         *        The Amazon States Language definition of the state machine. See <a
         *        href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html"
         *        >Amazon States Language</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder definition(String definition);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role to use for this state machine.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the IAM role to use for this state machine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * Determines whether a Standard or Express state machine is created. The default is <code>STANDARD</code>. You
         * cannot update the <code>type</code> of a state machine once it has been created.
         * </p>
         * 
         * @param type
         *        Determines whether a Standard or Express state machine is created. The default is
         *        <code>STANDARD</code>. You cannot update the <code>type</code> of a state machine once it has been
         *        created.
         * @see StateMachineType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateMachineType
         */
        Builder type(String type);

        /**
         * <p>
         * Determines whether a Standard or Express state machine is created. The default is <code>STANDARD</code>. You
         * cannot update the <code>type</code> of a state machine once it has been created.
         * </p>
         * 
         * @param type
         *        Determines whether a Standard or Express state machine is created. The default is
         *        <code>STANDARD</code>. You cannot update the <code>type</code> of a state machine once it has been
         *        created.
         * @see StateMachineType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateMachineType
         */
        Builder type(StateMachineType type);

        /**
         * <p>
         * Defines what execution history events are logged and where they are logged.
         * </p>
         * <note>
         * <p>
         * By default, the <code>level</code> is set to <code>OFF</code>. For more information see <a
         * href="https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html">Log Levels</a> in the
         * Step Functions User Guide.
         * </p>
         * </note>
         * 
         * @param loggingConfiguration
         *        Defines what execution history events are logged and where they are logged.</p> <note>
         *        <p>
         *        By default, the <code>level</code> is set to <code>OFF</code>. For more information see <a
         *        href="https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html">Log Levels</a>
         *        in the Step Functions User Guide.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingConfiguration(LoggingConfiguration loggingConfiguration);

        /**
         * <p>
         * Defines what execution history events are logged and where they are logged.
         * </p>
         * <note>
         * <p>
         * By default, the <code>level</code> is set to <code>OFF</code>. For more information see <a
         * href="https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html">Log Levels</a> in the
         * Step Functions User Guide.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link LoggingConfiguration.Builder}
         * avoiding the need to create one manually via {@link LoggingConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LoggingConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #loggingConfiguration(LoggingConfiguration)}.
         * 
         * @param loggingConfiguration
         *        a consumer that will call methods on {@link LoggingConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loggingConfiguration(LoggingConfiguration)
         */
        default Builder loggingConfiguration(Consumer<LoggingConfiguration.Builder> loggingConfiguration) {
            return loggingConfiguration(LoggingConfiguration.builder().applyMutation(loggingConfiguration).build());
        }

        /**
         * <p>
         * Tags to be added when creating a state machine.
         * </p>
         * <p>
         * An array of key-value pairs. For more information, see <a
         * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost
         * Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>, and <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using IAM
         * Tags</a>.
         * </p>
         * <p>
         * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
         * </p>
         * 
         * @param tags
         *        Tags to be added when creating a state machine.</p>
         *        <p>
         *        An array of key-value pairs. For more information, see <a
         *        href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost
         *        Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>, and <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using
         *        IAM Tags</a>.
         *        </p>
         *        <p>
         *        Tags may only contain Unicode letters, digits, white space, or these symbols:
         *        <code>_ . : / = + - @</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Tags to be added when creating a state machine.
         * </p>
         * <p>
         * An array of key-value pairs. For more information, see <a
         * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost
         * Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>, and <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using IAM
         * Tags</a>.
         * </p>
         * <p>
         * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
         * </p>
         * 
         * @param tags
         *        Tags to be added when creating a state machine.</p>
         *        <p>
         *        An array of key-value pairs. For more information, see <a
         *        href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost
         *        Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>, and <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using
         *        IAM Tags</a>.
         *        </p>
         *        <p>
         *        Tags may only contain Unicode letters, digits, white space, or these symbols:
         *        <code>_ . : / = + - @</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Tags to be added when creating a state machine.
         * </p>
         * <p>
         * An array of key-value pairs. For more information, see <a
         * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost
         * Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>, and <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using IAM
         * Tags</a>.
         * </p>
         * <p>
         * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sfn.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.sfn.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.sfn.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.sfn.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Selects whether X-Ray tracing is enabled.
         * </p>
         * 
         * @param tracingConfiguration
         *        Selects whether X-Ray tracing is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tracingConfiguration(TracingConfiguration tracingConfiguration);

        /**
         * <p>
         * Selects whether X-Ray tracing is enabled.
         * </p>
         * This is a convenience method that creates an instance of the {@link TracingConfiguration.Builder} avoiding
         * the need to create one manually via {@link TracingConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TracingConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #tracingConfiguration(TracingConfiguration)}.
         * 
         * @param tracingConfiguration
         *        a consumer that will call methods on {@link TracingConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tracingConfiguration(TracingConfiguration)
         */
        default Builder tracingConfiguration(Consumer<TracingConfiguration.Builder> tracingConfiguration) {
            return tracingConfiguration(TracingConfiguration.builder().applyMutation(tracingConfiguration).build());
        }

        /**
         * <p>
         * Set to <code>true</code> to publish the first version of the state machine during creation. The default is
         * <code>false</code>.
         * </p>
         * 
         * @param publish
         *        Set to <code>true</code> to publish the first version of the state machine during creation. The
         *        default is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publish(Boolean publish);

        /**
         * <p>
         * Sets description about the state machine version. You can only set the description if the
         * <code>publish</code> parameter is set to <code>true</code>. Otherwise, if you set
         * <code>versionDescription</code>, but <code>publish</code> to <code>false</code>, this API action throws
         * <code>ValidationException</code>.
         * </p>
         * 
         * @param versionDescription
         *        Sets description about the state machine version. You can only set the description if the
         *        <code>publish</code> parameter is set to <code>true</code>. Otherwise, if you set
         *        <code>versionDescription</code>, but <code>publish</code> to <code>false</code>, this API action
         *        throws <code>ValidationException</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionDescription(String versionDescription);

        /**
         * <p>
         * Settings to configure server-side encryption.
         * </p>
         * 
         * @param encryptionConfiguration
         *        Settings to configure server-side encryption.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * Settings to configure server-side encryption.
         * </p>
         * This is a convenience method that creates an instance of the {@link EncryptionConfiguration.Builder} avoiding
         * the need to create one manually via {@link EncryptionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SfnRequest.BuilderImpl implements Builder {
        private String name;

        private String definition;

        private String roleArn;

        private String type;

        private LoggingConfiguration loggingConfiguration;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private TracingConfiguration tracingConfiguration;

        private Boolean publish;

        private String versionDescription;

        private EncryptionConfiguration encryptionConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateStateMachineRequest model) {
            super(model);
            name(model.name);
            definition(model.definition);
            roleArn(model.roleArn);
            type(model.type);
            loggingConfiguration(model.loggingConfiguration);
            tags(model.tags);
            tracingConfiguration(model.tracingConfiguration);
            publish(model.publish);
            versionDescription(model.versionDescription);
            encryptionConfiguration(model.encryptionConfiguration);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDefinition() {
            return definition;
        }

        public final void setDefinition(String definition) {
            this.definition = definition;
        }

        @Override
        public final Builder definition(String definition) {
            this.definition = definition;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(StateMachineType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final LoggingConfiguration.Builder getLoggingConfiguration() {
            return loggingConfiguration != null ? loggingConfiguration.toBuilder() : null;
        }

        public final void setLoggingConfiguration(LoggingConfiguration.BuilderImpl loggingConfiguration) {
            this.loggingConfiguration = loggingConfiguration != null ? loggingConfiguration.build() : null;
        }

        @Override
        public final Builder loggingConfiguration(LoggingConfiguration loggingConfiguration) {
            this.loggingConfiguration = loggingConfiguration;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final TracingConfiguration.Builder getTracingConfiguration() {
            return tracingConfiguration != null ? tracingConfiguration.toBuilder() : null;
        }

        public final void setTracingConfiguration(TracingConfiguration.BuilderImpl tracingConfiguration) {
            this.tracingConfiguration = tracingConfiguration != null ? tracingConfiguration.build() : null;
        }

        @Override
        public final Builder tracingConfiguration(TracingConfiguration tracingConfiguration) {
            this.tracingConfiguration = tracingConfiguration;
            return this;
        }

        public final Boolean getPublish() {
            return publish;
        }

        public final void setPublish(Boolean publish) {
            this.publish = publish;
        }

        @Override
        public final Builder publish(Boolean publish) {
            this.publish = publish;
            return this;
        }

        public final String getVersionDescription() {
            return versionDescription;
        }

        public final void setVersionDescription(String versionDescription) {
            this.versionDescription = versionDescription;
        }

        @Override
        public final Builder versionDescription(String versionDescription) {
            this.versionDescription = versionDescription;
            return this;
        }

        public final EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public final Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateStateMachineRequest build() {
            return new CreateStateMachineRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
