/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sesv2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that contains additional information about the verification status for the identity.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VerificationInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<VerificationInfo.Builder, VerificationInfo> {
    private static final SdkField<Instant> LAST_CHECKED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastCheckedTimestamp").getter(getter(VerificationInfo::lastCheckedTimestamp))
            .setter(setter(Builder::lastCheckedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastCheckedTimestamp").build())
            .build();

    private static final SdkField<Instant> LAST_SUCCESS_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastSuccessTimestamp").getter(getter(VerificationInfo::lastSuccessTimestamp))
            .setter(setter(Builder::lastSuccessTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastSuccessTimestamp").build())
            .build();

    private static final SdkField<String> ERROR_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorType").getter(getter(VerificationInfo::errorTypeAsString)).setter(setter(Builder::errorType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorType").build()).build();

    private static final SdkField<SOARecord> SOA_RECORD_FIELD = SdkField.<SOARecord> builder(MarshallingType.SDK_POJO)
            .memberName("SOARecord").getter(getter(VerificationInfo::soaRecord)).setter(setter(Builder::soaRecord))
            .constructor(SOARecord::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SOARecord").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LAST_CHECKED_TIMESTAMP_FIELD,
            LAST_SUCCESS_TIMESTAMP_FIELD, ERROR_TYPE_FIELD, SOA_RECORD_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant lastCheckedTimestamp;

    private final Instant lastSuccessTimestamp;

    private final String errorType;

    private final SOARecord soaRecord;

    private VerificationInfo(BuilderImpl builder) {
        this.lastCheckedTimestamp = builder.lastCheckedTimestamp;
        this.lastSuccessTimestamp = builder.lastSuccessTimestamp;
        this.errorType = builder.errorType;
        this.soaRecord = builder.soaRecord;
    }

    /**
     * <p>
     * The last time a verification attempt was made for this identity.
     * </p>
     * 
     * @return The last time a verification attempt was made for this identity.
     */
    public final Instant lastCheckedTimestamp() {
        return lastCheckedTimestamp;
    }

    /**
     * <p>
     * The last time a successful verification was made for this identity.
     * </p>
     * 
     * @return The last time a successful verification was made for this identity.
     */
    public final Instant lastSuccessTimestamp() {
        return lastSuccessTimestamp;
    }

    /**
     * <p>
     * Provides the reason for the failure describing why Amazon SES was not able to successfully verify the identity.
     * Below are the possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INVALID_VALUE</code> – Amazon SES was able to find the record, but the value contained within the record
     * was invalid. Ensure you have published the correct values for the record.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TYPE_NOT_FOUND</code> – The queried hostname exists but does not have the requested type of DNS record.
     * Ensure that you have published the correct type of DNS record.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HOST_NOT_FOUND</code> – The queried hostname does not exist or was not reachable at the time of the
     * request. Ensure that you have published the required DNS record(s).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SERVICE_ERROR</code> – A temporary issue is preventing Amazon SES from determining the verification status
     * of the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DNS_SERVER_ERROR</code> – The DNS server encountered an issue and was unable to complete the request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_ACCESS_DENIED</code> – The verification failed because the user does not have the required
     * permissions to replicate the DKIM key from the primary region. Ensure you have the necessary permissions in both
     * primary and replica regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_PRIMARY_NOT_FOUND</code> – The verification failed because no corresponding identity was found
     * in the specified primary region. Ensure the identity exists in the primary region before attempting replication.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED</code> – The verification failed because the identity in the
     * primary region is configured with Bring Your Own DKIM (BYODKIM). DKIM key replication is only supported for
     * identities using Easy DKIM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED</code> – The verification failed because the specified primary
     * identity is a replica of another identity, and multi-level replication is not supported; the primary identity
     * must be a non-replica identity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_PRIMARY_INVALID_REGION</code> – The verification failed due to an invalid primary region
     * specified. Ensure you provide a valid Amazon Web Services region where Amazon SES is available and different from
     * the replica region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorType} will
     * return {@link VerificationError#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorTypeAsString}.
     * </p>
     * 
     * @return Provides the reason for the failure describing why Amazon SES was not able to successfully verify the
     *         identity. Below are the possible values: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INVALID_VALUE</code> – Amazon SES was able to find the record, but the value contained within the
     *         record was invalid. Ensure you have published the correct values for the record.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TYPE_NOT_FOUND</code> – The queried hostname exists but does not have the requested type of DNS
     *         record. Ensure that you have published the correct type of DNS record.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>HOST_NOT_FOUND</code> – The queried hostname does not exist or was not reachable at the time of the
     *         request. Ensure that you have published the required DNS record(s).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SERVICE_ERROR</code> – A temporary issue is preventing Amazon SES from determining the verification
     *         status of the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DNS_SERVER_ERROR</code> – The DNS server encountered an issue and was unable to complete the
     *         request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_ACCESS_DENIED</code> – The verification failed because the user does not have the
     *         required permissions to replicate the DKIM key from the primary region. Ensure you have the necessary
     *         permissions in both primary and replica regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_PRIMARY_NOT_FOUND</code> – The verification failed because no corresponding identity
     *         was found in the specified primary region. Ensure the identity exists in the primary region before
     *         attempting replication.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED</code> – The verification failed because the identity in
     *         the primary region is configured with Bring Your Own DKIM (BYODKIM). DKIM key replication is only
     *         supported for identities using Easy DKIM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED</code> – The verification failed because the specified
     *         primary identity is a replica of another identity, and multi-level replication is not supported; the
     *         primary identity must be a non-replica identity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_PRIMARY_INVALID_REGION</code> – The verification failed due to an invalid primary
     *         region specified. Ensure you provide a valid Amazon Web Services region where Amazon SES is available and
     *         different from the replica region.
     *         </p>
     *         </li>
     * @see VerificationError
     */
    public final VerificationError errorType() {
        return VerificationError.fromValue(errorType);
    }

    /**
     * <p>
     * Provides the reason for the failure describing why Amazon SES was not able to successfully verify the identity.
     * Below are the possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INVALID_VALUE</code> – Amazon SES was able to find the record, but the value contained within the record
     * was invalid. Ensure you have published the correct values for the record.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TYPE_NOT_FOUND</code> – The queried hostname exists but does not have the requested type of DNS record.
     * Ensure that you have published the correct type of DNS record.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HOST_NOT_FOUND</code> – The queried hostname does not exist or was not reachable at the time of the
     * request. Ensure that you have published the required DNS record(s).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SERVICE_ERROR</code> – A temporary issue is preventing Amazon SES from determining the verification status
     * of the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DNS_SERVER_ERROR</code> – The DNS server encountered an issue and was unable to complete the request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_ACCESS_DENIED</code> – The verification failed because the user does not have the required
     * permissions to replicate the DKIM key from the primary region. Ensure you have the necessary permissions in both
     * primary and replica regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_PRIMARY_NOT_FOUND</code> – The verification failed because no corresponding identity was found
     * in the specified primary region. Ensure the identity exists in the primary region before attempting replication.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED</code> – The verification failed because the identity in the
     * primary region is configured with Bring Your Own DKIM (BYODKIM). DKIM key replication is only supported for
     * identities using Easy DKIM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED</code> – The verification failed because the specified primary
     * identity is a replica of another identity, and multi-level replication is not supported; the primary identity
     * must be a non-replica identity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REPLICATION_PRIMARY_INVALID_REGION</code> – The verification failed due to an invalid primary region
     * specified. Ensure you provide a valid Amazon Web Services region where Amazon SES is available and different from
     * the replica region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorType} will
     * return {@link VerificationError#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorTypeAsString}.
     * </p>
     * 
     * @return Provides the reason for the failure describing why Amazon SES was not able to successfully verify the
     *         identity. Below are the possible values: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INVALID_VALUE</code> – Amazon SES was able to find the record, but the value contained within the
     *         record was invalid. Ensure you have published the correct values for the record.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TYPE_NOT_FOUND</code> – The queried hostname exists but does not have the requested type of DNS
     *         record. Ensure that you have published the correct type of DNS record.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>HOST_NOT_FOUND</code> – The queried hostname does not exist or was not reachable at the time of the
     *         request. Ensure that you have published the required DNS record(s).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SERVICE_ERROR</code> – A temporary issue is preventing Amazon SES from determining the verification
     *         status of the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DNS_SERVER_ERROR</code> – The DNS server encountered an issue and was unable to complete the
     *         request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_ACCESS_DENIED</code> – The verification failed because the user does not have the
     *         required permissions to replicate the DKIM key from the primary region. Ensure you have the necessary
     *         permissions in both primary and replica regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_PRIMARY_NOT_FOUND</code> – The verification failed because no corresponding identity
     *         was found in the specified primary region. Ensure the identity exists in the primary region before
     *         attempting replication.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED</code> – The verification failed because the identity in
     *         the primary region is configured with Bring Your Own DKIM (BYODKIM). DKIM key replication is only
     *         supported for identities using Easy DKIM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED</code> – The verification failed because the specified
     *         primary identity is a replica of another identity, and multi-level replication is not supported; the
     *         primary identity must be a non-replica identity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REPLICATION_PRIMARY_INVALID_REGION</code> – The verification failed due to an invalid primary
     *         region specified. Ensure you provide a valid Amazon Web Services region where Amazon SES is available and
     *         different from the replica region.
     *         </p>
     *         </li>
     * @see VerificationError
     */
    public final String errorTypeAsString() {
        return errorType;
    }

    /**
     * <p>
     * An object that contains information about the start of authority (SOA) record associated with the identity.
     * </p>
     * 
     * @return An object that contains information about the start of authority (SOA) record associated with the
     *         identity.
     */
    public final SOARecord soaRecord() {
        return soaRecord;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(lastCheckedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(errorTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(soaRecord());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VerificationInfo)) {
            return false;
        }
        VerificationInfo other = (VerificationInfo) obj;
        return Objects.equals(lastCheckedTimestamp(), other.lastCheckedTimestamp())
                && Objects.equals(lastSuccessTimestamp(), other.lastSuccessTimestamp())
                && Objects.equals(errorTypeAsString(), other.errorTypeAsString())
                && Objects.equals(soaRecord(), other.soaRecord());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VerificationInfo").add("LastCheckedTimestamp", lastCheckedTimestamp())
                .add("LastSuccessTimestamp", lastSuccessTimestamp()).add("ErrorType", errorTypeAsString())
                .add("SOARecord", soaRecord()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LastCheckedTimestamp":
            return Optional.ofNullable(clazz.cast(lastCheckedTimestamp()));
        case "LastSuccessTimestamp":
            return Optional.ofNullable(clazz.cast(lastSuccessTimestamp()));
        case "ErrorType":
            return Optional.ofNullable(clazz.cast(errorTypeAsString()));
        case "SOARecord":
            return Optional.ofNullable(clazz.cast(soaRecord()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("LastCheckedTimestamp", LAST_CHECKED_TIMESTAMP_FIELD);
        map.put("LastSuccessTimestamp", LAST_SUCCESS_TIMESTAMP_FIELD);
        map.put("ErrorType", ERROR_TYPE_FIELD);
        map.put("SOARecord", SOA_RECORD_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<VerificationInfo, T> g) {
        return obj -> g.apply((VerificationInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VerificationInfo> {
        /**
         * <p>
         * The last time a verification attempt was made for this identity.
         * </p>
         * 
         * @param lastCheckedTimestamp
         *        The last time a verification attempt was made for this identity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastCheckedTimestamp(Instant lastCheckedTimestamp);

        /**
         * <p>
         * The last time a successful verification was made for this identity.
         * </p>
         * 
         * @param lastSuccessTimestamp
         *        The last time a successful verification was made for this identity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessTimestamp(Instant lastSuccessTimestamp);

        /**
         * <p>
         * Provides the reason for the failure describing why Amazon SES was not able to successfully verify the
         * identity. Below are the possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INVALID_VALUE</code> – Amazon SES was able to find the record, but the value contained within the
         * record was invalid. Ensure you have published the correct values for the record.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TYPE_NOT_FOUND</code> – The queried hostname exists but does not have the requested type of DNS record.
         * Ensure that you have published the correct type of DNS record.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>HOST_NOT_FOUND</code> – The queried hostname does not exist or was not reachable at the time of the
         * request. Ensure that you have published the required DNS record(s).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SERVICE_ERROR</code> – A temporary issue is preventing Amazon SES from determining the verification
         * status of the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DNS_SERVER_ERROR</code> – The DNS server encountered an issue and was unable to complete the request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_ACCESS_DENIED</code> – The verification failed because the user does not have the required
         * permissions to replicate the DKIM key from the primary region. Ensure you have the necessary permissions in
         * both primary and replica regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_PRIMARY_NOT_FOUND</code> – The verification failed because no corresponding identity was
         * found in the specified primary region. Ensure the identity exists in the primary region before attempting
         * replication.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED</code> – The verification failed because the identity in the
         * primary region is configured with Bring Your Own DKIM (BYODKIM). DKIM key replication is only supported for
         * identities using Easy DKIM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED</code> – The verification failed because the specified
         * primary identity is a replica of another identity, and multi-level replication is not supported; the primary
         * identity must be a non-replica identity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_PRIMARY_INVALID_REGION</code> – The verification failed due to an invalid primary region
         * specified. Ensure you provide a valid Amazon Web Services region where Amazon SES is available and different
         * from the replica region.
         * </p>
         * </li>
         * </ul>
         * 
         * @param errorType
         *        Provides the reason for the failure describing why Amazon SES was not able to successfully verify the
         *        identity. Below are the possible values: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INVALID_VALUE</code> – Amazon SES was able to find the record, but the value contained within
         *        the record was invalid. Ensure you have published the correct values for the record.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TYPE_NOT_FOUND</code> – The queried hostname exists but does not have the requested type of DNS
         *        record. Ensure that you have published the correct type of DNS record.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>HOST_NOT_FOUND</code> – The queried hostname does not exist or was not reachable at the time of
         *        the request. Ensure that you have published the required DNS record(s).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SERVICE_ERROR</code> – A temporary issue is preventing Amazon SES from determining the
         *        verification status of the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DNS_SERVER_ERROR</code> – The DNS server encountered an issue and was unable to complete the
         *        request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_ACCESS_DENIED</code> – The verification failed because the user does not have the
         *        required permissions to replicate the DKIM key from the primary region. Ensure you have the necessary
         *        permissions in both primary and replica regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_PRIMARY_NOT_FOUND</code> – The verification failed because no corresponding identity
         *        was found in the specified primary region. Ensure the identity exists in the primary region before
         *        attempting replication.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED</code> – The verification failed because the identity
         *        in the primary region is configured with Bring Your Own DKIM (BYODKIM). DKIM key replication is only
         *        supported for identities using Easy DKIM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED</code> – The verification failed because the
         *        specified primary identity is a replica of another identity, and multi-level replication is not
         *        supported; the primary identity must be a non-replica identity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_PRIMARY_INVALID_REGION</code> – The verification failed due to an invalid primary
         *        region specified. Ensure you provide a valid Amazon Web Services region where Amazon SES is available
         *        and different from the replica region.
         *        </p>
         *        </li>
         * @see VerificationError
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerificationError
         */
        Builder errorType(String errorType);

        /**
         * <p>
         * Provides the reason for the failure describing why Amazon SES was not able to successfully verify the
         * identity. Below are the possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INVALID_VALUE</code> – Amazon SES was able to find the record, but the value contained within the
         * record was invalid. Ensure you have published the correct values for the record.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TYPE_NOT_FOUND</code> – The queried hostname exists but does not have the requested type of DNS record.
         * Ensure that you have published the correct type of DNS record.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>HOST_NOT_FOUND</code> – The queried hostname does not exist or was not reachable at the time of the
         * request. Ensure that you have published the required DNS record(s).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SERVICE_ERROR</code> – A temporary issue is preventing Amazon SES from determining the verification
         * status of the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DNS_SERVER_ERROR</code> – The DNS server encountered an issue and was unable to complete the request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_ACCESS_DENIED</code> – The verification failed because the user does not have the required
         * permissions to replicate the DKIM key from the primary region. Ensure you have the necessary permissions in
         * both primary and replica regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_PRIMARY_NOT_FOUND</code> – The verification failed because no corresponding identity was
         * found in the specified primary region. Ensure the identity exists in the primary region before attempting
         * replication.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED</code> – The verification failed because the identity in the
         * primary region is configured with Bring Your Own DKIM (BYODKIM). DKIM key replication is only supported for
         * identities using Easy DKIM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED</code> – The verification failed because the specified
         * primary identity is a replica of another identity, and multi-level replication is not supported; the primary
         * identity must be a non-replica identity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REPLICATION_PRIMARY_INVALID_REGION</code> – The verification failed due to an invalid primary region
         * specified. Ensure you provide a valid Amazon Web Services region where Amazon SES is available and different
         * from the replica region.
         * </p>
         * </li>
         * </ul>
         * 
         * @param errorType
         *        Provides the reason for the failure describing why Amazon SES was not able to successfully verify the
         *        identity. Below are the possible values: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INVALID_VALUE</code> – Amazon SES was able to find the record, but the value contained within
         *        the record was invalid. Ensure you have published the correct values for the record.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TYPE_NOT_FOUND</code> – The queried hostname exists but does not have the requested type of DNS
         *        record. Ensure that you have published the correct type of DNS record.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>HOST_NOT_FOUND</code> – The queried hostname does not exist or was not reachable at the time of
         *        the request. Ensure that you have published the required DNS record(s).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SERVICE_ERROR</code> – A temporary issue is preventing Amazon SES from determining the
         *        verification status of the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DNS_SERVER_ERROR</code> – The DNS server encountered an issue and was unable to complete the
         *        request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_ACCESS_DENIED</code> – The verification failed because the user does not have the
         *        required permissions to replicate the DKIM key from the primary region. Ensure you have the necessary
         *        permissions in both primary and replica regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_PRIMARY_NOT_FOUND</code> – The verification failed because no corresponding identity
         *        was found in the specified primary region. Ensure the identity exists in the primary region before
         *        attempting replication.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED</code> – The verification failed because the identity
         *        in the primary region is configured with Bring Your Own DKIM (BYODKIM). DKIM key replication is only
         *        supported for identities using Easy DKIM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED</code> – The verification failed because the
         *        specified primary identity is a replica of another identity, and multi-level replication is not
         *        supported; the primary identity must be a non-replica identity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REPLICATION_PRIMARY_INVALID_REGION</code> – The verification failed due to an invalid primary
         *        region specified. Ensure you provide a valid Amazon Web Services region where Amazon SES is available
         *        and different from the replica region.
         *        </p>
         *        </li>
         * @see VerificationError
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerificationError
         */
        Builder errorType(VerificationError errorType);

        /**
         * <p>
         * An object that contains information about the start of authority (SOA) record associated with the identity.
         * </p>
         * 
         * @param soaRecord
         *        An object that contains information about the start of authority (SOA) record associated with the
         *        identity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder soaRecord(SOARecord soaRecord);

        /**
         * <p>
         * An object that contains information about the start of authority (SOA) record associated with the identity.
         * </p>
         * This is a convenience method that creates an instance of the {@link SOARecord.Builder} avoiding the need to
         * create one manually via {@link SOARecord#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SOARecord.Builder#build()} is called immediately and its result
         * is passed to {@link #soaRecord(SOARecord)}.
         * 
         * @param soaRecord
         *        a consumer that will call methods on {@link SOARecord.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #soaRecord(SOARecord)
         */
        default Builder soaRecord(Consumer<SOARecord.Builder> soaRecord) {
            return soaRecord(SOARecord.builder().applyMutation(soaRecord).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Instant lastCheckedTimestamp;

        private Instant lastSuccessTimestamp;

        private String errorType;

        private SOARecord soaRecord;

        private BuilderImpl() {
        }

        private BuilderImpl(VerificationInfo model) {
            lastCheckedTimestamp(model.lastCheckedTimestamp);
            lastSuccessTimestamp(model.lastSuccessTimestamp);
            errorType(model.errorType);
            soaRecord(model.soaRecord);
        }

        public final Instant getLastCheckedTimestamp() {
            return lastCheckedTimestamp;
        }

        public final void setLastCheckedTimestamp(Instant lastCheckedTimestamp) {
            this.lastCheckedTimestamp = lastCheckedTimestamp;
        }

        @Override
        public final Builder lastCheckedTimestamp(Instant lastCheckedTimestamp) {
            this.lastCheckedTimestamp = lastCheckedTimestamp;
            return this;
        }

        public final Instant getLastSuccessTimestamp() {
            return lastSuccessTimestamp;
        }

        public final void setLastSuccessTimestamp(Instant lastSuccessTimestamp) {
            this.lastSuccessTimestamp = lastSuccessTimestamp;
        }

        @Override
        public final Builder lastSuccessTimestamp(Instant lastSuccessTimestamp) {
            this.lastSuccessTimestamp = lastSuccessTimestamp;
            return this;
        }

        public final String getErrorType() {
            return errorType;
        }

        public final void setErrorType(String errorType) {
            this.errorType = errorType;
        }

        @Override
        public final Builder errorType(String errorType) {
            this.errorType = errorType;
            return this;
        }

        @Override
        public final Builder errorType(VerificationError errorType) {
            this.errorType(errorType == null ? null : errorType.toString());
            return this;
        }

        public final SOARecord.Builder getSoaRecord() {
            return soaRecord != null ? soaRecord.toBuilder() : null;
        }

        public final void setSoaRecord(SOARecord.BuilderImpl soaRecord) {
            this.soaRecord = soaRecord != null ? soaRecord.build() : null;
        }

        @Override
        public final Builder soaRecord(SOARecord soaRecord) {
            this.soaRecord = soaRecord;
            return this;
        }

        @Override
        public VerificationInfo build() {
            return new VerificationInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
