/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sesv2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send a single formatted email using Amazon SES. For more information, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-email-formatted.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SendEmailRequest extends SesV2Request implements ToCopyableBuilder<SendEmailRequest.Builder, SendEmailRequest> {
    private static final SdkField<String> FROM_EMAIL_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FromEmailAddress").getter(getter(SendEmailRequest::fromEmailAddress))
            .setter(setter(Builder::fromEmailAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FromEmailAddress").build()).build();

    private static final SdkField<String> FROM_EMAIL_ADDRESS_IDENTITY_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FromEmailAddressIdentityArn")
            .getter(getter(SendEmailRequest::fromEmailAddressIdentityArn))
            .setter(setter(Builder::fromEmailAddressIdentityArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FromEmailAddressIdentityArn")
                    .build()).build();

    private static final SdkField<Destination> DESTINATION_FIELD = SdkField.<Destination> builder(MarshallingType.SDK_POJO)
            .memberName("Destination").getter(getter(SendEmailRequest::destination)).setter(setter(Builder::destination))
            .constructor(Destination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destination").build()).build();

    private static final SdkField<List<String>> REPLY_TO_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ReplyToAddresses")
            .getter(getter(SendEmailRequest::replyToAddresses))
            .setter(setter(Builder::replyToAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplyToAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FEEDBACK_FORWARDING_EMAIL_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FeedbackForwardingEmailAddress")
            .getter(getter(SendEmailRequest::feedbackForwardingEmailAddress))
            .setter(setter(Builder::feedbackForwardingEmailAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeedbackForwardingEmailAddress")
                    .build()).build();

    private static final SdkField<String> FEEDBACK_FORWARDING_EMAIL_ADDRESS_IDENTITY_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FeedbackForwardingEmailAddressIdentityArn")
            .getter(getter(SendEmailRequest::feedbackForwardingEmailAddressIdentityArn))
            .setter(setter(Builder::feedbackForwardingEmailAddressIdentityArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("FeedbackForwardingEmailAddressIdentityArn").build()).build();

    private static final SdkField<EmailContent> CONTENT_FIELD = SdkField.<EmailContent> builder(MarshallingType.SDK_POJO)
            .memberName("Content").getter(getter(SendEmailRequest::content)).setter(setter(Builder::content))
            .constructor(EmailContent::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Content").build()).build();

    private static final SdkField<List<MessageTag>> EMAIL_TAGS_FIELD = SdkField
            .<List<MessageTag>> builder(MarshallingType.LIST)
            .memberName("EmailTags")
            .getter(getter(SendEmailRequest::emailTags))
            .setter(setter(Builder::emailTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmailTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MessageTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONFIGURATION_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationSetName").getter(getter(SendEmailRequest::configurationSetName))
            .setter(setter(Builder::configurationSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSetName").build())
            .build();

    private static final SdkField<String> ENDPOINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointId").getter(getter(SendEmailRequest::endpointId)).setter(setter(Builder::endpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointId").build()).build();

    private static final SdkField<String> TENANT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TenantName").getter(getter(SendEmailRequest::tenantName)).setter(setter(Builder::tenantName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TenantName").build()).build();

    private static final SdkField<ListManagementOptions> LIST_MANAGEMENT_OPTIONS_FIELD = SdkField
            .<ListManagementOptions> builder(MarshallingType.SDK_POJO).memberName("ListManagementOptions")
            .getter(getter(SendEmailRequest::listManagementOptions)).setter(setter(Builder::listManagementOptions))
            .constructor(ListManagementOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ListManagementOptions").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FROM_EMAIL_ADDRESS_FIELD,
            FROM_EMAIL_ADDRESS_IDENTITY_ARN_FIELD, DESTINATION_FIELD, REPLY_TO_ADDRESSES_FIELD,
            FEEDBACK_FORWARDING_EMAIL_ADDRESS_FIELD, FEEDBACK_FORWARDING_EMAIL_ADDRESS_IDENTITY_ARN_FIELD, CONTENT_FIELD,
            EMAIL_TAGS_FIELD, CONFIGURATION_SET_NAME_FIELD, ENDPOINT_ID_FIELD, TENANT_NAME_FIELD, LIST_MANAGEMENT_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String fromEmailAddress;

    private final String fromEmailAddressIdentityArn;

    private final Destination destination;

    private final List<String> replyToAddresses;

    private final String feedbackForwardingEmailAddress;

    private final String feedbackForwardingEmailAddressIdentityArn;

    private final EmailContent content;

    private final List<MessageTag> emailTags;

    private final String configurationSetName;

    private final String endpointId;

    private final String tenantName;

    private final ListManagementOptions listManagementOptions;

    private SendEmailRequest(BuilderImpl builder) {
        super(builder);
        this.fromEmailAddress = builder.fromEmailAddress;
        this.fromEmailAddressIdentityArn = builder.fromEmailAddressIdentityArn;
        this.destination = builder.destination;
        this.replyToAddresses = builder.replyToAddresses;
        this.feedbackForwardingEmailAddress = builder.feedbackForwardingEmailAddress;
        this.feedbackForwardingEmailAddressIdentityArn = builder.feedbackForwardingEmailAddressIdentityArn;
        this.content = builder.content;
        this.emailTags = builder.emailTags;
        this.configurationSetName = builder.configurationSetName;
        this.endpointId = builder.endpointId;
        this.tenantName = builder.tenantName;
        this.listManagementOptions = builder.listManagementOptions;
    }

    /**
     * <p>
     * The email address to use as the "From" address for the email. The address that you specify has to be verified.
     * </p>
     * 
     * @return The email address to use as the "From" address for the email. The address that you specify has to be
     *         verified.
     */
    public final String fromEmailAddress() {
        return fromEmailAddress;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the
     * <code>FromEmailAddress</code> parameter.
     * </p>
     * <p>
     * For example, if the owner of example.com (which has ARN arn:aws:ses:us-east-1:123456789012:identity/example.com)
     * attaches a policy to it that authorizes you to use sender@example.com, then you would specify the
     * <code>FromEmailAddressIdentityArn</code> to be arn:aws:ses:us-east-1:123456789012:identity/example.com, and the
     * <code>FromEmailAddress</code> to be sender@example.com.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * <p>
     * For Raw emails, the <code>FromEmailAddressIdentityArn</code> value overrides the X-SES-SOURCE-ARN and
     * X-SES-FROM-ARN headers specified in raw email message content.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>FromEmailAddress</code> parameter.</p>
     *         <p>
     *         For example, if the owner of example.com (which has ARN
     *         arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you to
     *         use sender@example.com, then you would specify the <code>FromEmailAddressIdentityArn</code> to be
     *         arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FromEmailAddress</code> to be
     *         sender@example.com.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     *         </p>
     *         <p>
     *         For Raw emails, the <code>FromEmailAddressIdentityArn</code> value overrides the X-SES-SOURCE-ARN and
     *         X-SES-FROM-ARN headers specified in raw email message content.
     */
    public final String fromEmailAddressIdentityArn() {
        return fromEmailAddressIdentityArn;
    }

    /**
     * <p>
     * An object that contains the recipients of the email message.
     * </p>
     * 
     * @return An object that contains the recipients of the email message.
     */
    public final Destination destination() {
        return destination;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReplyToAddresses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReplyToAddresses() {
        return replyToAddresses != null && !(replyToAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The "Reply-to" email addresses for the message. When the recipient replies to the message, each Reply-to address
     * receives the reply.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReplyToAddresses} method.
     * </p>
     * 
     * @return The "Reply-to" email addresses for the message. When the recipient replies to the message, each Reply-to
     *         address receives the reply.
     */
    public final List<String> replyToAddresses() {
        return replyToAddresses;
    }

    /**
     * <p>
     * The address that you want bounce and complaint notifications to be sent to.
     * </p>
     * 
     * @return The address that you want bounce and complaint notifications to be sent to.
     */
    public final String feedbackForwardingEmailAddress() {
        return feedbackForwardingEmailAddress;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the
     * <code>FeedbackForwardingEmailAddress</code> parameter.
     * </p>
     * <p>
     * For example, if the owner of example.com (which has ARN arn:aws:ses:us-east-1:123456789012:identity/example.com)
     * attaches a policy to it that authorizes you to use feedback@example.com, then you would specify the
     * <code>FeedbackForwardingEmailAddressIdentityArn</code> to be
     * arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FeedbackForwardingEmailAddress</code> to
     * be feedback@example.com.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>FeedbackForwardingEmailAddress</code> parameter.</p>
     *         <p>
     *         For example, if the owner of example.com (which has ARN
     *         arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you to
     *         use feedback@example.com, then you would specify the
     *         <code>FeedbackForwardingEmailAddressIdentityArn</code> to be
     *         arn:aws:ses:us-east-1:123456789012:identity/example.com, and the
     *         <code>FeedbackForwardingEmailAddress</code> to be feedback@example.com.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public final String feedbackForwardingEmailAddressIdentityArn() {
        return feedbackForwardingEmailAddressIdentityArn;
    }

    /**
     * <p>
     * An object that contains the body of the message. You can send either a Simple message, Raw message, or a
     * Templated message.
     * </p>
     * 
     * @return An object that contains the body of the message. You can send either a Simple message, Raw message, or a
     *         Templated message.
     */
    public final EmailContent content() {
        return content;
    }

    /**
     * For responses, this returns true if the service returned a value for the EmailTags property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEmailTags() {
        return emailTags != null && !(emailTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags, in the form of name/value pairs, to apply to an email that you send using the
     * <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so that you
     * can publish email sending events.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEmailTags} method.
     * </p>
     * 
     * @return A list of tags, in the form of name/value pairs, to apply to an email that you send using the
     *         <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so
     *         that you can publish email sending events.
     */
    public final List<MessageTag> emailTags() {
        return emailTags;
    }

    /**
     * <p>
     * The name of the configuration set to use when sending the email.
     * </p>
     * 
     * @return The name of the configuration set to use when sending the email.
     */
    public final String configurationSetName() {
        return configurationSetName;
    }

    /**
     * <p>
     * The ID of the multi-region endpoint (global-endpoint).
     * </p>
     * 
     * @return The ID of the multi-region endpoint (global-endpoint).
     */
    public final String endpointId() {
        return endpointId;
    }

    /**
     * <p>
     * The name of the tenant through which this email will be sent.
     * </p>
     * <note>
     * <p>
     * The email sending operation will only succeed if all referenced resources (identities, configuration sets, and
     * templates) are associated with this tenant.
     * </p>
     * </note>
     * 
     * @return The name of the tenant through which this email will be sent.</p> <note>
     *         <p>
     *         The email sending operation will only succeed if all referenced resources (identities, configuration
     *         sets, and templates) are associated with this tenant.
     *         </p>
     */
    public final String tenantName() {
        return tenantName;
    }

    /**
     * <p>
     * An object used to specify a list or topic to which an email belongs, which will be used when a contact chooses to
     * unsubscribe.
     * </p>
     * 
     * @return An object used to specify a list or topic to which an email belongs, which will be used when a contact
     *         chooses to unsubscribe.
     */
    public final ListManagementOptions listManagementOptions() {
        return listManagementOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(fromEmailAddress());
        hashCode = 31 * hashCode + Objects.hashCode(fromEmailAddressIdentityArn());
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        hashCode = 31 * hashCode + Objects.hashCode(hasReplyToAddresses() ? replyToAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(feedbackForwardingEmailAddress());
        hashCode = 31 * hashCode + Objects.hashCode(feedbackForwardingEmailAddressIdentityArn());
        hashCode = 31 * hashCode + Objects.hashCode(content());
        hashCode = 31 * hashCode + Objects.hashCode(hasEmailTags() ? emailTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        hashCode = 31 * hashCode + Objects.hashCode(endpointId());
        hashCode = 31 * hashCode + Objects.hashCode(tenantName());
        hashCode = 31 * hashCode + Objects.hashCode(listManagementOptions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendEmailRequest)) {
            return false;
        }
        SendEmailRequest other = (SendEmailRequest) obj;
        return Objects.equals(fromEmailAddress(), other.fromEmailAddress())
                && Objects.equals(fromEmailAddressIdentityArn(), other.fromEmailAddressIdentityArn())
                && Objects.equals(destination(), other.destination()) && hasReplyToAddresses() == other.hasReplyToAddresses()
                && Objects.equals(replyToAddresses(), other.replyToAddresses())
                && Objects.equals(feedbackForwardingEmailAddress(), other.feedbackForwardingEmailAddress())
                && Objects.equals(feedbackForwardingEmailAddressIdentityArn(), other.feedbackForwardingEmailAddressIdentityArn())
                && Objects.equals(content(), other.content()) && hasEmailTags() == other.hasEmailTags()
                && Objects.equals(emailTags(), other.emailTags())
                && Objects.equals(configurationSetName(), other.configurationSetName())
                && Objects.equals(endpointId(), other.endpointId()) && Objects.equals(tenantName(), other.tenantName())
                && Objects.equals(listManagementOptions(), other.listManagementOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SendEmailRequest").add("FromEmailAddress", fromEmailAddress())
                .add("FromEmailAddressIdentityArn", fromEmailAddressIdentityArn()).add("Destination", destination())
                .add("ReplyToAddresses", hasReplyToAddresses() ? replyToAddresses() : null)
                .add("FeedbackForwardingEmailAddress", feedbackForwardingEmailAddress())
                .add("FeedbackForwardingEmailAddressIdentityArn", feedbackForwardingEmailAddressIdentityArn())
                .add("Content", content()).add("EmailTags", hasEmailTags() ? emailTags() : null)
                .add("ConfigurationSetName", configurationSetName()).add("EndpointId", endpointId())
                .add("TenantName", tenantName()).add("ListManagementOptions", listManagementOptions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FromEmailAddress":
            return Optional.ofNullable(clazz.cast(fromEmailAddress()));
        case "FromEmailAddressIdentityArn":
            return Optional.ofNullable(clazz.cast(fromEmailAddressIdentityArn()));
        case "Destination":
            return Optional.ofNullable(clazz.cast(destination()));
        case "ReplyToAddresses":
            return Optional.ofNullable(clazz.cast(replyToAddresses()));
        case "FeedbackForwardingEmailAddress":
            return Optional.ofNullable(clazz.cast(feedbackForwardingEmailAddress()));
        case "FeedbackForwardingEmailAddressIdentityArn":
            return Optional.ofNullable(clazz.cast(feedbackForwardingEmailAddressIdentityArn()));
        case "Content":
            return Optional.ofNullable(clazz.cast(content()));
        case "EmailTags":
            return Optional.ofNullable(clazz.cast(emailTags()));
        case "ConfigurationSetName":
            return Optional.ofNullable(clazz.cast(configurationSetName()));
        case "EndpointId":
            return Optional.ofNullable(clazz.cast(endpointId()));
        case "TenantName":
            return Optional.ofNullable(clazz.cast(tenantName()));
        case "ListManagementOptions":
            return Optional.ofNullable(clazz.cast(listManagementOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FromEmailAddress", FROM_EMAIL_ADDRESS_FIELD);
        map.put("FromEmailAddressIdentityArn", FROM_EMAIL_ADDRESS_IDENTITY_ARN_FIELD);
        map.put("Destination", DESTINATION_FIELD);
        map.put("ReplyToAddresses", REPLY_TO_ADDRESSES_FIELD);
        map.put("FeedbackForwardingEmailAddress", FEEDBACK_FORWARDING_EMAIL_ADDRESS_FIELD);
        map.put("FeedbackForwardingEmailAddressIdentityArn", FEEDBACK_FORWARDING_EMAIL_ADDRESS_IDENTITY_ARN_FIELD);
        map.put("Content", CONTENT_FIELD);
        map.put("EmailTags", EMAIL_TAGS_FIELD);
        map.put("ConfigurationSetName", CONFIGURATION_SET_NAME_FIELD);
        map.put("EndpointId", ENDPOINT_ID_FIELD);
        map.put("TenantName", TENANT_NAME_FIELD);
        map.put("ListManagementOptions", LIST_MANAGEMENT_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SendEmailRequest, T> g) {
        return obj -> g.apply((SendEmailRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SesV2Request.Builder, SdkPojo, CopyableBuilder<Builder, SendEmailRequest> {
        /**
         * <p>
         * The email address to use as the "From" address for the email. The address that you specify has to be
         * verified.
         * </p>
         * 
         * @param fromEmailAddress
         *        The email address to use as the "From" address for the email. The address that you specify has to be
         *        verified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromEmailAddress(String fromEmailAddress);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>FromEmailAddress</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of example.com (which has ARN
         * arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you to use
         * sender@example.com, then you would specify the <code>FromEmailAddressIdentityArn</code> to be
         * arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FromEmailAddress</code> to be
         * sender@example.com.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * <p>
         * For Raw emails, the <code>FromEmailAddressIdentityArn</code> value overrides the X-SES-SOURCE-ARN and
         * X-SES-FROM-ARN headers specified in raw email message content.
         * </p>
         * 
         * @param fromEmailAddressIdentityArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>FromEmailAddress</code> parameter.</p>
         *        <p>
         *        For example, if the owner of example.com (which has ARN
         *        arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you
         *        to use sender@example.com, then you would specify the <code>FromEmailAddressIdentityArn</code> to be
         *        arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FromEmailAddress</code> to be
         *        sender@example.com.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         *        </p>
         *        <p>
         *        For Raw emails, the <code>FromEmailAddressIdentityArn</code> value overrides the X-SES-SOURCE-ARN and
         *        X-SES-FROM-ARN headers specified in raw email message content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromEmailAddressIdentityArn(String fromEmailAddressIdentityArn);

        /**
         * <p>
         * An object that contains the recipients of the email message.
         * </p>
         * 
         * @param destination
         *        An object that contains the recipients of the email message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(Destination destination);

        /**
         * <p>
         * An object that contains the recipients of the email message.
         * </p>
         * This is a convenience method that creates an instance of the {@link Destination.Builder} avoiding the need to
         * create one manually via {@link Destination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Destination.Builder#build()} is called immediately and its result
         * is passed to {@link #destination(Destination)}.
         * 
         * @param destination
         *        a consumer that will call methods on {@link Destination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destination(Destination)
         */
        default Builder destination(Consumer<Destination.Builder> destination) {
            return destination(Destination.builder().applyMutation(destination).build());
        }

        /**
         * <p>
         * The "Reply-to" email addresses for the message. When the recipient replies to the message, each Reply-to
         * address receives the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The "Reply-to" email addresses for the message. When the recipient replies to the message, each
         *        Reply-to address receives the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(Collection<String> replyToAddresses);

        /**
         * <p>
         * The "Reply-to" email addresses for the message. When the recipient replies to the message, each Reply-to
         * address receives the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The "Reply-to" email addresses for the message. When the recipient replies to the message, each
         *        Reply-to address receives the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(String... replyToAddresses);

        /**
         * <p>
         * The address that you want bounce and complaint notifications to be sent to.
         * </p>
         * 
         * @param feedbackForwardingEmailAddress
         *        The address that you want bounce and complaint notifications to be sent to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder feedbackForwardingEmailAddress(String feedbackForwardingEmailAddress);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>FeedbackForwardingEmailAddress</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of example.com (which has ARN
         * arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you to use
         * feedback@example.com, then you would specify the <code>FeedbackForwardingEmailAddressIdentityArn</code> to be
         * arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FeedbackForwardingEmailAddress</code>
         * to be feedback@example.com.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param feedbackForwardingEmailAddressIdentityArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>FeedbackForwardingEmailAddress</code> parameter.</p>
         *        <p>
         *        For example, if the owner of example.com (which has ARN
         *        arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you
         *        to use feedback@example.com, then you would specify the
         *        <code>FeedbackForwardingEmailAddressIdentityArn</code> to be
         *        arn:aws:ses:us-east-1:123456789012:identity/example.com, and the
         *        <code>FeedbackForwardingEmailAddress</code> to be feedback@example.com.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder feedbackForwardingEmailAddressIdentityArn(String feedbackForwardingEmailAddressIdentityArn);

        /**
         * <p>
         * An object that contains the body of the message. You can send either a Simple message, Raw message, or a
         * Templated message.
         * </p>
         * 
         * @param content
         *        An object that contains the body of the message. You can send either a Simple message, Raw message, or
         *        a Templated message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(EmailContent content);

        /**
         * <p>
         * An object that contains the body of the message. You can send either a Simple message, Raw message, or a
         * Templated message.
         * </p>
         * This is a convenience method that creates an instance of the {@link EmailContent.Builder} avoiding the need
         * to create one manually via {@link EmailContent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EmailContent.Builder#build()} is called immediately and its
         * result is passed to {@link #content(EmailContent)}.
         * 
         * @param content
         *        a consumer that will call methods on {@link EmailContent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #content(EmailContent)
         */
        default Builder content(Consumer<EmailContent.Builder> content) {
            return content(EmailContent.builder().applyMutation(content).build());
        }

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         * <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so that
         * you can publish email sending events.
         * </p>
         * 
         * @param emailTags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         *        <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so
         *        that you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailTags(Collection<MessageTag> emailTags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         * <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so that
         * you can publish email sending events.
         * </p>
         * 
         * @param emailTags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         *        <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so
         *        that you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailTags(MessageTag... emailTags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         * <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so that
         * you can publish email sending events.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sesv2.model.MessageTag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.sesv2.model.MessageTag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sesv2.model.MessageTag.Builder#build()} is called immediately and its
         * result is passed to {@link #emailTags(List<MessageTag>)}.
         * 
         * @param emailTags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sesv2.model.MessageTag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #emailTags(java.util.Collection<MessageTag>)
         */
        Builder emailTags(Consumer<MessageTag.Builder>... emailTags);

        /**
         * <p>
         * The name of the configuration set to use when sending the email.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set to use when sending the email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);

        /**
         * <p>
         * The ID of the multi-region endpoint (global-endpoint).
         * </p>
         * 
         * @param endpointId
         *        The ID of the multi-region endpoint (global-endpoint).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointId(String endpointId);

        /**
         * <p>
         * The name of the tenant through which this email will be sent.
         * </p>
         * <note>
         * <p>
         * The email sending operation will only succeed if all referenced resources (identities, configuration sets,
         * and templates) are associated with this tenant.
         * </p>
         * </note>
         * 
         * @param tenantName
         *        The name of the tenant through which this email will be sent.</p> <note>
         *        <p>
         *        The email sending operation will only succeed if all referenced resources (identities, configuration
         *        sets, and templates) are associated with this tenant.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tenantName(String tenantName);

        /**
         * <p>
         * An object used to specify a list or topic to which an email belongs, which will be used when a contact
         * chooses to unsubscribe.
         * </p>
         * 
         * @param listManagementOptions
         *        An object used to specify a list or topic to which an email belongs, which will be used when a contact
         *        chooses to unsubscribe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder listManagementOptions(ListManagementOptions listManagementOptions);

        /**
         * <p>
         * An object used to specify a list or topic to which an email belongs, which will be used when a contact
         * chooses to unsubscribe.
         * </p>
         * This is a convenience method that creates an instance of the {@link ListManagementOptions.Builder} avoiding
         * the need to create one manually via {@link ListManagementOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ListManagementOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #listManagementOptions(ListManagementOptions)}.
         * 
         * @param listManagementOptions
         *        a consumer that will call methods on {@link ListManagementOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #listManagementOptions(ListManagementOptions)
         */
        default Builder listManagementOptions(Consumer<ListManagementOptions.Builder> listManagementOptions) {
            return listManagementOptions(ListManagementOptions.builder().applyMutation(listManagementOptions).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesV2Request.BuilderImpl implements Builder {
        private String fromEmailAddress;

        private String fromEmailAddressIdentityArn;

        private Destination destination;

        private List<String> replyToAddresses = DefaultSdkAutoConstructList.getInstance();

        private String feedbackForwardingEmailAddress;

        private String feedbackForwardingEmailAddressIdentityArn;

        private EmailContent content;

        private List<MessageTag> emailTags = DefaultSdkAutoConstructList.getInstance();

        private String configurationSetName;

        private String endpointId;

        private String tenantName;

        private ListManagementOptions listManagementOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(SendEmailRequest model) {
            super(model);
            fromEmailAddress(model.fromEmailAddress);
            fromEmailAddressIdentityArn(model.fromEmailAddressIdentityArn);
            destination(model.destination);
            replyToAddresses(model.replyToAddresses);
            feedbackForwardingEmailAddress(model.feedbackForwardingEmailAddress);
            feedbackForwardingEmailAddressIdentityArn(model.feedbackForwardingEmailAddressIdentityArn);
            content(model.content);
            emailTags(model.emailTags);
            configurationSetName(model.configurationSetName);
            endpointId(model.endpointId);
            tenantName(model.tenantName);
            listManagementOptions(model.listManagementOptions);
        }

        public final String getFromEmailAddress() {
            return fromEmailAddress;
        }

        public final void setFromEmailAddress(String fromEmailAddress) {
            this.fromEmailAddress = fromEmailAddress;
        }

        @Override
        public final Builder fromEmailAddress(String fromEmailAddress) {
            this.fromEmailAddress = fromEmailAddress;
            return this;
        }

        public final String getFromEmailAddressIdentityArn() {
            return fromEmailAddressIdentityArn;
        }

        public final void setFromEmailAddressIdentityArn(String fromEmailAddressIdentityArn) {
            this.fromEmailAddressIdentityArn = fromEmailAddressIdentityArn;
        }

        @Override
        public final Builder fromEmailAddressIdentityArn(String fromEmailAddressIdentityArn) {
            this.fromEmailAddressIdentityArn = fromEmailAddressIdentityArn;
            return this;
        }

        public final Destination.Builder getDestination() {
            return destination != null ? destination.toBuilder() : null;
        }

        public final void setDestination(Destination.BuilderImpl destination) {
            this.destination = destination != null ? destination.build() : null;
        }

        @Override
        public final Builder destination(Destination destination) {
            this.destination = destination;
            return this;
        }

        public final Collection<String> getReplyToAddresses() {
            if (replyToAddresses instanceof SdkAutoConstructList) {
                return null;
            }
            return replyToAddresses;
        }

        public final void setReplyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = EmailAddressListCopier.copy(replyToAddresses);
        }

        @Override
        public final Builder replyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = EmailAddressListCopier.copy(replyToAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder replyToAddresses(String... replyToAddresses) {
            replyToAddresses(Arrays.asList(replyToAddresses));
            return this;
        }

        public final String getFeedbackForwardingEmailAddress() {
            return feedbackForwardingEmailAddress;
        }

        public final void setFeedbackForwardingEmailAddress(String feedbackForwardingEmailAddress) {
            this.feedbackForwardingEmailAddress = feedbackForwardingEmailAddress;
        }

        @Override
        public final Builder feedbackForwardingEmailAddress(String feedbackForwardingEmailAddress) {
            this.feedbackForwardingEmailAddress = feedbackForwardingEmailAddress;
            return this;
        }

        public final String getFeedbackForwardingEmailAddressIdentityArn() {
            return feedbackForwardingEmailAddressIdentityArn;
        }

        public final void setFeedbackForwardingEmailAddressIdentityArn(String feedbackForwardingEmailAddressIdentityArn) {
            this.feedbackForwardingEmailAddressIdentityArn = feedbackForwardingEmailAddressIdentityArn;
        }

        @Override
        public final Builder feedbackForwardingEmailAddressIdentityArn(String feedbackForwardingEmailAddressIdentityArn) {
            this.feedbackForwardingEmailAddressIdentityArn = feedbackForwardingEmailAddressIdentityArn;
            return this;
        }

        public final EmailContent.Builder getContent() {
            return content != null ? content.toBuilder() : null;
        }

        public final void setContent(EmailContent.BuilderImpl content) {
            this.content = content != null ? content.build() : null;
        }

        @Override
        public final Builder content(EmailContent content) {
            this.content = content;
            return this;
        }

        public final List<MessageTag.Builder> getEmailTags() {
            List<MessageTag.Builder> result = MessageTagListCopier.copyToBuilder(this.emailTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEmailTags(Collection<MessageTag.BuilderImpl> emailTags) {
            this.emailTags = MessageTagListCopier.copyFromBuilder(emailTags);
        }

        @Override
        public final Builder emailTags(Collection<MessageTag> emailTags) {
            this.emailTags = MessageTagListCopier.copy(emailTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder emailTags(MessageTag... emailTags) {
            emailTags(Arrays.asList(emailTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder emailTags(Consumer<MessageTag.Builder>... emailTags) {
            emailTags(Stream.of(emailTags).map(c -> MessageTag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final String getEndpointId() {
            return endpointId;
        }

        public final void setEndpointId(String endpointId) {
            this.endpointId = endpointId;
        }

        @Override
        public final Builder endpointId(String endpointId) {
            this.endpointId = endpointId;
            return this;
        }

        public final String getTenantName() {
            return tenantName;
        }

        public final void setTenantName(String tenantName) {
            this.tenantName = tenantName;
        }

        @Override
        public final Builder tenantName(String tenantName) {
            this.tenantName = tenantName;
            return this;
        }

        public final ListManagementOptions.Builder getListManagementOptions() {
            return listManagementOptions != null ? listManagementOptions.toBuilder() : null;
        }

        public final void setListManagementOptions(ListManagementOptions.BuilderImpl listManagementOptions) {
            this.listManagementOptions = listManagementOptions != null ? listManagementOptions.build() : null;
        }

        @Override
        public final Builder listManagementOptions(ListManagementOptions listManagementOptions) {
            this.listManagementOptions = listManagementOptions;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SendEmailRequest build() {
            return new SendEmailRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
