/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sesv2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a single metric data query to include in a batch.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchGetMetricDataQuery implements SdkPojo, Serializable,
        ToCopyableBuilder<BatchGetMetricDataQuery.Builder, BatchGetMetricDataQuery> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(BatchGetMetricDataQuery::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(BatchGetMetricDataQuery::namespaceAsString))
            .setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<String> METRIC_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Metric")
            .getter(getter(BatchGetMetricDataQuery::metricAsString)).setter(setter(Builder::metric))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metric").build()).build();

    private static final SdkField<Map<String, String>> DIMENSIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Dimensions")
            .getter(getter(BatchGetMetricDataQuery::dimensionsAsStrings))
            .setter(setter(Builder::dimensionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> START_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartDate").getter(getter(BatchGetMetricDataQuery::startDate)).setter(setter(Builder::startDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartDate").build()).build();

    private static final SdkField<Instant> END_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndDate").getter(getter(BatchGetMetricDataQuery::endDate)).setter(setter(Builder::endDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAMESPACE_FIELD,
            METRIC_FIELD, DIMENSIONS_FIELD, START_DATE_FIELD, END_DATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String namespace;

    private final String metric;

    private final Map<String, String> dimensions;

    private final Instant startDate;

    private final Instant endDate;

    private BatchGetMetricDataQuery(BuilderImpl builder) {
        this.id = builder.id;
        this.namespace = builder.namespace;
        this.metric = builder.metric;
        this.dimensions = builder.dimensions;
        this.startDate = builder.startDate;
        this.endDate = builder.endDate;
    }

    /**
     * <p>
     * The query identifier.
     * </p>
     * 
     * @return The query identifier.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The query namespace - e.g. <code>VDM</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #namespace} will
     * return {@link MetricNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #namespaceAsString}.
     * </p>
     * 
     * @return The query namespace - e.g. <code>VDM</code>
     * @see MetricNamespace
     */
    public final MetricNamespace namespace() {
        return MetricNamespace.fromValue(namespace);
    }

    /**
     * <p>
     * The query namespace - e.g. <code>VDM</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #namespace} will
     * return {@link MetricNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #namespaceAsString}.
     * </p>
     * 
     * @return The query namespace - e.g. <code>VDM</code>
     * @see MetricNamespace
     */
    public final String namespaceAsString() {
        return namespace;
    }

    /**
     * <p>
     * The queried metric. This can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SEND</code> – Emails sent eligible for tracking in the VDM dashboard. This excludes emails sent to the
     * mailbox simulator and emails addressed to more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COMPLAINT</code> – Complaints received for your account. This excludes complaints from the mailbox
     * simulator, those originating from your account-level suppression list (if enabled), and those for emails
     * addressed to more than one recipient
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PERMANENT_BOUNCE</code> – Permanent bounces - i.e. feedback received for emails sent to non-existent
     * mailboxes. Excludes bounces from the mailbox simulator, those originating from your account-level suppression
     * list (if enabled), and those for emails addressed to more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TRANSIENT_BOUNCE</code> – Transient bounces - i.e. feedback received for delivery failures excluding issues
     * with non-existent mailboxes. Excludes bounces from the mailbox simulator, and those for emails addressed to more
     * than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OPEN</code> – Unique open events for emails including open trackers. Excludes opens for emails addressed to
     * more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CLICK</code> – Unique click events for emails including wrapped links. Excludes clicks for emails addressed
     * to more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELIVERY</code> – Successful deliveries for email sending attempts. Excludes deliveries to the mailbox
     * simulator and for emails addressed to more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELIVERY_OPEN</code> – Successful deliveries for email sending attempts. Excludes deliveries to the mailbox
     * simulator, for emails addressed to more than one recipient, and emails without open trackers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELIVERY_CLICK</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
     * mailbox simulator, for emails addressed to more than one recipient, and emails without click trackers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELIVERY_COMPLAINT</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
     * mailbox simulator, for emails addressed to more than one recipient, and emails addressed to recipients hosted by
     * ISPs with which Amazon SES does not have a feedback loop agreement.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metric} will
     * return {@link Metric#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricAsString}.
     * </p>
     * 
     * @return The queried metric. This can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SEND</code> – Emails sent eligible for tracking in the VDM dashboard. This excludes emails sent to
     *         the mailbox simulator and emails addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COMPLAINT</code> – Complaints received for your account. This excludes complaints from the mailbox
     *         simulator, those originating from your account-level suppression list (if enabled), and those for emails
     *         addressed to more than one recipient
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PERMANENT_BOUNCE</code> – Permanent bounces - i.e. feedback received for emails sent to
     *         non-existent mailboxes. Excludes bounces from the mailbox simulator, those originating from your
     *         account-level suppression list (if enabled), and those for emails addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TRANSIENT_BOUNCE</code> – Transient bounces - i.e. feedback received for delivery failures
     *         excluding issues with non-existent mailboxes. Excludes bounces from the mailbox simulator, and those for
     *         emails addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OPEN</code> – Unique open events for emails including open trackers. Excludes opens for emails
     *         addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CLICK</code> – Unique click events for emails including wrapped links. Excludes clicks for emails
     *         addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELIVERY</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
     *         mailbox simulator and for emails addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELIVERY_OPEN</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
     *         mailbox simulator, for emails addressed to more than one recipient, and emails without open trackers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELIVERY_CLICK</code> – Successful deliveries for email sending attempts. Excludes deliveries to
     *         the mailbox simulator, for emails addressed to more than one recipient, and emails without click
     *         trackers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELIVERY_COMPLAINT</code> – Successful deliveries for email sending attempts. Excludes deliveries
     *         to the mailbox simulator, for emails addressed to more than one recipient, and emails addressed to
     *         recipients hosted by ISPs with which Amazon SES does not have a feedback loop agreement.
     *         </p>
     *         </li>
     * @see Metric
     */
    public final Metric metric() {
        return Metric.fromValue(metric);
    }

    /**
     * <p>
     * The queried metric. This can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SEND</code> – Emails sent eligible for tracking in the VDM dashboard. This excludes emails sent to the
     * mailbox simulator and emails addressed to more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COMPLAINT</code> – Complaints received for your account. This excludes complaints from the mailbox
     * simulator, those originating from your account-level suppression list (if enabled), and those for emails
     * addressed to more than one recipient
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PERMANENT_BOUNCE</code> – Permanent bounces - i.e. feedback received for emails sent to non-existent
     * mailboxes. Excludes bounces from the mailbox simulator, those originating from your account-level suppression
     * list (if enabled), and those for emails addressed to more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TRANSIENT_BOUNCE</code> – Transient bounces - i.e. feedback received for delivery failures excluding issues
     * with non-existent mailboxes. Excludes bounces from the mailbox simulator, and those for emails addressed to more
     * than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OPEN</code> – Unique open events for emails including open trackers. Excludes opens for emails addressed to
     * more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CLICK</code> – Unique click events for emails including wrapped links. Excludes clicks for emails addressed
     * to more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELIVERY</code> – Successful deliveries for email sending attempts. Excludes deliveries to the mailbox
     * simulator and for emails addressed to more than one recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELIVERY_OPEN</code> – Successful deliveries for email sending attempts. Excludes deliveries to the mailbox
     * simulator, for emails addressed to more than one recipient, and emails without open trackers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELIVERY_CLICK</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
     * mailbox simulator, for emails addressed to more than one recipient, and emails without click trackers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELIVERY_COMPLAINT</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
     * mailbox simulator, for emails addressed to more than one recipient, and emails addressed to recipients hosted by
     * ISPs with which Amazon SES does not have a feedback loop agreement.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metric} will
     * return {@link Metric#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricAsString}.
     * </p>
     * 
     * @return The queried metric. This can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SEND</code> – Emails sent eligible for tracking in the VDM dashboard. This excludes emails sent to
     *         the mailbox simulator and emails addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COMPLAINT</code> – Complaints received for your account. This excludes complaints from the mailbox
     *         simulator, those originating from your account-level suppression list (if enabled), and those for emails
     *         addressed to more than one recipient
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PERMANENT_BOUNCE</code> – Permanent bounces - i.e. feedback received for emails sent to
     *         non-existent mailboxes. Excludes bounces from the mailbox simulator, those originating from your
     *         account-level suppression list (if enabled), and those for emails addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TRANSIENT_BOUNCE</code> – Transient bounces - i.e. feedback received for delivery failures
     *         excluding issues with non-existent mailboxes. Excludes bounces from the mailbox simulator, and those for
     *         emails addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OPEN</code> – Unique open events for emails including open trackers. Excludes opens for emails
     *         addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CLICK</code> – Unique click events for emails including wrapped links. Excludes clicks for emails
     *         addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELIVERY</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
     *         mailbox simulator and for emails addressed to more than one recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELIVERY_OPEN</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
     *         mailbox simulator, for emails addressed to more than one recipient, and emails without open trackers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELIVERY_CLICK</code> – Successful deliveries for email sending attempts. Excludes deliveries to
     *         the mailbox simulator, for emails addressed to more than one recipient, and emails without click
     *         trackers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELIVERY_COMPLAINT</code> – Successful deliveries for email sending attempts. Excludes deliveries
     *         to the mailbox simulator, for emails addressed to more than one recipient, and emails addressed to
     *         recipients hosted by ISPs with which Amazon SES does not have a feedback loop agreement.
     *         </p>
     *         </li>
     * @see Metric
     */
    public final String metricAsString() {
        return metric;
    }

    /**
     * <p>
     * An object that contains mapping between <code>MetricDimensionName</code> and <code>MetricDimensionValue</code> to
     * filter metrics by.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return An object that contains mapping between <code>MetricDimensionName</code> and
     *         <code>MetricDimensionValue</code> to filter metrics by.
     */
    public final Map<MetricDimensionName, String> dimensions() {
        return DimensionsCopier.copyStringToEnum(dimensions);
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An object that contains mapping between <code>MetricDimensionName</code> and <code>MetricDimensionValue</code> to
     * filter metrics by.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return An object that contains mapping between <code>MetricDimensionName</code> and
     *         <code>MetricDimensionValue</code> to filter metrics by.
     */
    public final Map<String, String> dimensionsAsStrings() {
        return dimensions;
    }

    /**
     * <p>
     * Represents the start date for the query interval.
     * </p>
     * 
     * @return Represents the start date for the query interval.
     */
    public final Instant startDate() {
        return startDate;
    }

    /**
     * <p>
     * Represents the end date for the query interval.
     * </p>
     * 
     * @return Represents the end date for the query interval.
     */
    public final Instant endDate() {
        return endDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(namespaceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(metricAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startDate());
        hashCode = 31 * hashCode + Objects.hashCode(endDate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchGetMetricDataQuery)) {
            return false;
        }
        BatchGetMetricDataQuery other = (BatchGetMetricDataQuery) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(namespaceAsString(), other.namespaceAsString())
                && Objects.equals(metricAsString(), other.metricAsString()) && hasDimensions() == other.hasDimensions()
                && Objects.equals(dimensionsAsStrings(), other.dimensionsAsStrings())
                && Objects.equals(startDate(), other.startDate()) && Objects.equals(endDate(), other.endDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BatchGetMetricDataQuery").add("Id", id()).add("Namespace", namespaceAsString())
                .add("Metric", metricAsString()).add("Dimensions", hasDimensions() ? dimensionsAsStrings() : null)
                .add("StartDate", startDate()).add("EndDate", endDate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespaceAsString()));
        case "Metric":
            return Optional.ofNullable(clazz.cast(metricAsString()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensionsAsStrings()));
        case "StartDate":
            return Optional.ofNullable(clazz.cast(startDate()));
        case "EndDate":
            return Optional.ofNullable(clazz.cast(endDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BatchGetMetricDataQuery, T> g) {
        return obj -> g.apply((BatchGetMetricDataQuery) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BatchGetMetricDataQuery> {
        /**
         * <p>
         * The query identifier.
         * </p>
         * 
         * @param id
         *        The query identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The query namespace - e.g. <code>VDM</code>
         * </p>
         * 
         * @param namespace
         *        The query namespace - e.g. <code>VDM</code>
         * @see MetricNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricNamespace
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The query namespace - e.g. <code>VDM</code>
         * </p>
         * 
         * @param namespace
         *        The query namespace - e.g. <code>VDM</code>
         * @see MetricNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricNamespace
         */
        Builder namespace(MetricNamespace namespace);

        /**
         * <p>
         * The queried metric. This can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SEND</code> – Emails sent eligible for tracking in the VDM dashboard. This excludes emails sent to the
         * mailbox simulator and emails addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COMPLAINT</code> – Complaints received for your account. This excludes complaints from the mailbox
         * simulator, those originating from your account-level suppression list (if enabled), and those for emails
         * addressed to more than one recipient
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PERMANENT_BOUNCE</code> – Permanent bounces - i.e. feedback received for emails sent to non-existent
         * mailboxes. Excludes bounces from the mailbox simulator, those originating from your account-level suppression
         * list (if enabled), and those for emails addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TRANSIENT_BOUNCE</code> – Transient bounces - i.e. feedback received for delivery failures excluding
         * issues with non-existent mailboxes. Excludes bounces from the mailbox simulator, and those for emails
         * addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OPEN</code> – Unique open events for emails including open trackers. Excludes opens for emails
         * addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CLICK</code> – Unique click events for emails including wrapped links. Excludes clicks for emails
         * addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELIVERY</code> – Successful deliveries for email sending attempts. Excludes deliveries to the mailbox
         * simulator and for emails addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELIVERY_OPEN</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
         * mailbox simulator, for emails addressed to more than one recipient, and emails without open trackers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELIVERY_CLICK</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
         * mailbox simulator, for emails addressed to more than one recipient, and emails without click trackers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELIVERY_COMPLAINT</code> – Successful deliveries for email sending attempts. Excludes deliveries to
         * the mailbox simulator, for emails addressed to more than one recipient, and emails addressed to recipients
         * hosted by ISPs with which Amazon SES does not have a feedback loop agreement.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metric
         *        The queried metric. This can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SEND</code> – Emails sent eligible for tracking in the VDM dashboard. This excludes emails sent
         *        to the mailbox simulator and emails addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COMPLAINT</code> – Complaints received for your account. This excludes complaints from the
         *        mailbox simulator, those originating from your account-level suppression list (if enabled), and those
         *        for emails addressed to more than one recipient
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PERMANENT_BOUNCE</code> – Permanent bounces - i.e. feedback received for emails sent to
         *        non-existent mailboxes. Excludes bounces from the mailbox simulator, those originating from your
         *        account-level suppression list (if enabled), and those for emails addressed to more than one
         *        recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TRANSIENT_BOUNCE</code> – Transient bounces - i.e. feedback received for delivery failures
         *        excluding issues with non-existent mailboxes. Excludes bounces from the mailbox simulator, and those
         *        for emails addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OPEN</code> – Unique open events for emails including open trackers. Excludes opens for emails
         *        addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CLICK</code> – Unique click events for emails including wrapped links. Excludes clicks for
         *        emails addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELIVERY</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
         *        mailbox simulator and for emails addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELIVERY_OPEN</code> – Successful deliveries for email sending attempts. Excludes deliveries to
         *        the mailbox simulator, for emails addressed to more than one recipient, and emails without open
         *        trackers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELIVERY_CLICK</code> – Successful deliveries for email sending attempts. Excludes deliveries to
         *        the mailbox simulator, for emails addressed to more than one recipient, and emails without click
         *        trackers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELIVERY_COMPLAINT</code> – Successful deliveries for email sending attempts. Excludes
         *        deliveries to the mailbox simulator, for emails addressed to more than one recipient, and emails
         *        addressed to recipients hosted by ISPs with which Amazon SES does not have a feedback loop agreement.
         *        </p>
         *        </li>
         * @see Metric
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Metric
         */
        Builder metric(String metric);

        /**
         * <p>
         * The queried metric. This can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SEND</code> – Emails sent eligible for tracking in the VDM dashboard. This excludes emails sent to the
         * mailbox simulator and emails addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COMPLAINT</code> – Complaints received for your account. This excludes complaints from the mailbox
         * simulator, those originating from your account-level suppression list (if enabled), and those for emails
         * addressed to more than one recipient
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PERMANENT_BOUNCE</code> – Permanent bounces - i.e. feedback received for emails sent to non-existent
         * mailboxes. Excludes bounces from the mailbox simulator, those originating from your account-level suppression
         * list (if enabled), and those for emails addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TRANSIENT_BOUNCE</code> – Transient bounces - i.e. feedback received for delivery failures excluding
         * issues with non-existent mailboxes. Excludes bounces from the mailbox simulator, and those for emails
         * addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OPEN</code> – Unique open events for emails including open trackers. Excludes opens for emails
         * addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CLICK</code> – Unique click events for emails including wrapped links. Excludes clicks for emails
         * addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELIVERY</code> – Successful deliveries for email sending attempts. Excludes deliveries to the mailbox
         * simulator and for emails addressed to more than one recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELIVERY_OPEN</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
         * mailbox simulator, for emails addressed to more than one recipient, and emails without open trackers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELIVERY_CLICK</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
         * mailbox simulator, for emails addressed to more than one recipient, and emails without click trackers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELIVERY_COMPLAINT</code> – Successful deliveries for email sending attempts. Excludes deliveries to
         * the mailbox simulator, for emails addressed to more than one recipient, and emails addressed to recipients
         * hosted by ISPs with which Amazon SES does not have a feedback loop agreement.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metric
         *        The queried metric. This can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SEND</code> – Emails sent eligible for tracking in the VDM dashboard. This excludes emails sent
         *        to the mailbox simulator and emails addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COMPLAINT</code> – Complaints received for your account. This excludes complaints from the
         *        mailbox simulator, those originating from your account-level suppression list (if enabled), and those
         *        for emails addressed to more than one recipient
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PERMANENT_BOUNCE</code> – Permanent bounces - i.e. feedback received for emails sent to
         *        non-existent mailboxes. Excludes bounces from the mailbox simulator, those originating from your
         *        account-level suppression list (if enabled), and those for emails addressed to more than one
         *        recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TRANSIENT_BOUNCE</code> – Transient bounces - i.e. feedback received for delivery failures
         *        excluding issues with non-existent mailboxes. Excludes bounces from the mailbox simulator, and those
         *        for emails addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OPEN</code> – Unique open events for emails including open trackers. Excludes opens for emails
         *        addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CLICK</code> – Unique click events for emails including wrapped links. Excludes clicks for
         *        emails addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELIVERY</code> – Successful deliveries for email sending attempts. Excludes deliveries to the
         *        mailbox simulator and for emails addressed to more than one recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELIVERY_OPEN</code> – Successful deliveries for email sending attempts. Excludes deliveries to
         *        the mailbox simulator, for emails addressed to more than one recipient, and emails without open
         *        trackers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELIVERY_CLICK</code> – Successful deliveries for email sending attempts. Excludes deliveries to
         *        the mailbox simulator, for emails addressed to more than one recipient, and emails without click
         *        trackers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELIVERY_COMPLAINT</code> – Successful deliveries for email sending attempts. Excludes
         *        deliveries to the mailbox simulator, for emails addressed to more than one recipient, and emails
         *        addressed to recipients hosted by ISPs with which Amazon SES does not have a feedback loop agreement.
         *        </p>
         *        </li>
         * @see Metric
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Metric
         */
        Builder metric(Metric metric);

        /**
         * <p>
         * An object that contains mapping between <code>MetricDimensionName</code> and
         * <code>MetricDimensionValue</code> to filter metrics by.
         * </p>
         * 
         * @param dimensions
         *        An object that contains mapping between <code>MetricDimensionName</code> and
         *        <code>MetricDimensionValue</code> to filter metrics by.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensionsWithStrings(Map<String, String> dimensions);

        /**
         * <p>
         * An object that contains mapping between <code>MetricDimensionName</code> and
         * <code>MetricDimensionValue</code> to filter metrics by.
         * </p>
         * 
         * @param dimensions
         *        An object that contains mapping between <code>MetricDimensionName</code> and
         *        <code>MetricDimensionValue</code> to filter metrics by.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Map<MetricDimensionName, String> dimensions);

        /**
         * <p>
         * Represents the start date for the query interval.
         * </p>
         * 
         * @param startDate
         *        Represents the start date for the query interval.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startDate(Instant startDate);

        /**
         * <p>
         * Represents the end date for the query interval.
         * </p>
         * 
         * @param endDate
         *        Represents the end date for the query interval.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endDate(Instant endDate);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String namespace;

        private String metric;

        private Map<String, String> dimensions = DefaultSdkAutoConstructMap.getInstance();

        private Instant startDate;

        private Instant endDate;

        private BuilderImpl() {
        }

        private BuilderImpl(BatchGetMetricDataQuery model) {
            id(model.id);
            namespace(model.namespace);
            metric(model.metric);
            dimensionsWithStrings(model.dimensions);
            startDate(model.startDate);
            endDate(model.endDate);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        @Override
        public final Builder namespace(MetricNamespace namespace) {
            this.namespace(namespace == null ? null : namespace.toString());
            return this;
        }

        public final String getMetric() {
            return metric;
        }

        public final void setMetric(String metric) {
            this.metric = metric;
        }

        @Override
        public final Builder metric(String metric) {
            this.metric = metric;
            return this;
        }

        @Override
        public final Builder metric(Metric metric) {
            this.metric(metric == null ? null : metric.toString());
            return this;
        }

        public final Map<String, String> getDimensions() {
            if (dimensions instanceof SdkAutoConstructMap) {
                return null;
            }
            return dimensions;
        }

        public final void setDimensions(Map<String, String> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
        }

        @Override
        public final Builder dimensionsWithStrings(Map<String, String> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        public final Builder dimensions(Map<MetricDimensionName, String> dimensions) {
            this.dimensions = DimensionsCopier.copyEnumToString(dimensions);
            return this;
        }

        public final Instant getStartDate() {
            return startDate;
        }

        public final void setStartDate(Instant startDate) {
            this.startDate = startDate;
        }

        @Override
        public final Builder startDate(Instant startDate) {
            this.startDate = startDate;
            return this;
        }

        public final Instant getEndDate() {
            return endDate;
        }

        public final void setEndDate(Instant endDate) {
            this.endDate = endDate;
        }

        @Override
        public final Builder endDate(Instant endDate) {
            this.endDate = endDate;
            return this;
        }

        @Override
        public BatchGetMetricDataQuery build() {
            return new BatchGetMetricDataQuery(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
