/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * When included in a receipt rule, this action publishes a notification to Amazon Simple Notification Service (Amazon
 * SNS). This action includes a complete copy of the email content in the Amazon SNS notifications. Amazon SNS
 * notifications for all other actions simply provide information about the email. They do not include the email content
 * itself.
 * </p>
 * <p>
 * If you own the Amazon SNS topic, you don't need to do anything to give Amazon SES permission to publish emails to it.
 * However, if you don't own the Amazon SNS topic, you need to attach a policy to the topic to give Amazon SES
 * permissions to access it. For information about giving permissions, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-permissions.html">Amazon SES Developer Guide</a>.
 * </p>
 * <important>
 * <p>
 * You can only publish emails that are 150 KB or less (including the header) to Amazon SNS. Larger emails bounce. If
 * you anticipate emails larger than 150 KB, use the S3 action instead.
 * </p>
 * </important>
 * <p>
 * For information about using a receipt rule to publish an Amazon SNS notification, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-action-sns.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SNSAction implements SdkPojo, Serializable, ToCopyableBuilder<SNSAction.Builder, SNSAction> {
    private static final SdkField<String> TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TopicArn").getter(getter(SNSAction::topicArn)).setter(setter(Builder::topicArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TopicArn").build()).build();

    private static final SdkField<String> ENCODING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Encoding").getter(getter(SNSAction::encodingAsString)).setter(setter(Builder::encoding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encoding").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TOPIC_ARN_FIELD,
            ENCODING_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("TopicArn", TOPIC_ARN_FIELD);
                    put("Encoding", ENCODING_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String topicArn;

    private final String encoding;

    private SNSAction(BuilderImpl builder) {
        this.topicArn = builder.topicArn;
        this.encoding = builder.encoding;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon SNS topic to notify. You can find the ARN of a topic by using the <a
     * href="https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html">ListTopics</a> operation in Amazon SNS.
     * </p>
     * <p>
     * For more information about Amazon SNS topics, see the <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon SNS topic to notify. You can find the ARN of a topic by
     *         using the <a href="https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html">ListTopics</a>
     *         operation in Amazon SNS.</p>
     *         <p>
     *         For more information about Amazon SNS topics, see the <a
     *         href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
     */
    public final String topicArn() {
        return topicArn;
    }

    /**
     * <p>
     * The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
     * preserve all special characters when a message was encoded with a different encoding format. Base64 preserves all
     * special characters. The default value is UTF-8.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encoding} will
     * return {@link SNSActionEncoding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #encodingAsString}.
     * </p>
     * 
     * @return The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
     *         preserve all special characters when a message was encoded with a different encoding format. Base64
     *         preserves all special characters. The default value is UTF-8.
     * @see SNSActionEncoding
     */
    public final SNSActionEncoding encoding() {
        return SNSActionEncoding.fromValue(encoding);
    }

    /**
     * <p>
     * The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
     * preserve all special characters when a message was encoded with a different encoding format. Base64 preserves all
     * special characters. The default value is UTF-8.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encoding} will
     * return {@link SNSActionEncoding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #encodingAsString}.
     * </p>
     * 
     * @return The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
     *         preserve all special characters when a message was encoded with a different encoding format. Base64
     *         preserves all special characters. The default value is UTF-8.
     * @see SNSActionEncoding
     */
    public final String encodingAsString() {
        return encoding;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(topicArn());
        hashCode = 31 * hashCode + Objects.hashCode(encodingAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SNSAction)) {
            return false;
        }
        SNSAction other = (SNSAction) obj;
        return Objects.equals(topicArn(), other.topicArn()) && Objects.equals(encodingAsString(), other.encodingAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SNSAction").add("TopicArn", topicArn()).add("Encoding", encodingAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TopicArn":
            return Optional.ofNullable(clazz.cast(topicArn()));
        case "Encoding":
            return Optional.ofNullable(clazz.cast(encodingAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<SNSAction, T> g) {
        return obj -> g.apply((SNSAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SNSAction> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon SNS topic to notify. You can find the ARN of a topic by using
         * the <a href="https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html">ListTopics</a> operation in
         * Amazon SNS.
         * </p>
         * <p>
         * For more information about Amazon SNS topics, see the <a
         * href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
         * </p>
         * 
         * @param topicArn
         *        The Amazon Resource Name (ARN) of the Amazon SNS topic to notify. You can find the ARN of a topic by
         *        using the <a href="https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html">ListTopics</a>
         *        operation in Amazon SNS.</p>
         *        <p>
         *        For more information about Amazon SNS topics, see the <a
         *        href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topicArn(String topicArn);

        /**
         * <p>
         * The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
         * preserve all special characters when a message was encoded with a different encoding format. Base64 preserves
         * all special characters. The default value is UTF-8.
         * </p>
         * 
         * @param encoding
         *        The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may
         *        not preserve all special characters when a message was encoded with a different encoding format.
         *        Base64 preserves all special characters. The default value is UTF-8.
         * @see SNSActionEncoding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SNSActionEncoding
         */
        Builder encoding(String encoding);

        /**
         * <p>
         * The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
         * preserve all special characters when a message was encoded with a different encoding format. Base64 preserves
         * all special characters. The default value is UTF-8.
         * </p>
         * 
         * @param encoding
         *        The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may
         *        not preserve all special characters when a message was encoded with a different encoding format.
         *        Base64 preserves all special characters. The default value is UTF-8.
         * @see SNSActionEncoding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SNSActionEncoding
         */
        Builder encoding(SNSActionEncoding encoding);
    }

    static final class BuilderImpl implements Builder {
        private String topicArn;

        private String encoding;

        private BuilderImpl() {
        }

        private BuilderImpl(SNSAction model) {
            topicArn(model.topicArn);
            encoding(model.encoding);
        }

        public final String getTopicArn() {
            return topicArn;
        }

        public final void setTopicArn(String topicArn) {
            this.topicArn = topicArn;
        }

        @Override
        public final Builder topicArn(String topicArn) {
            this.topicArn = topicArn;
            return this;
        }

        public final String getEncoding() {
            return encoding;
        }

        public final void setEncoding(String encoding) {
            this.encoding = encoding;
        }

        @Override
        public final Builder encoding(String encoding) {
            this.encoding = encoding;
            return this;
        }

        @Override
        public final Builder encoding(SNSActionEncoding encoding) {
            this.encoding(encoding == null ? null : encoding.toString());
            return this;
        }

        @Override
        public SNSAction build() {
            return new SNSAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
