/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.serverlessapplicationrepository.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Put policy request
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutApplicationPolicyRequest extends ServerlessApplicationRepositoryRequest implements
        ToCopyableBuilder<PutApplicationPolicyRequest.Builder, PutApplicationPolicyRequest> {
    private final String applicationId;

    private final List<ApplicationPolicyStatement> statements;

    private PutApplicationPolicyRequest(BuilderImpl builder) {
        super(builder);
        this.applicationId = builder.applicationId;
        this.statements = builder.statements;
    }

    /**
     * The id of the application to put policy for
     * 
     * @return The id of the application to put policy for
     */
    public String applicationId() {
        return applicationId;
    }

    /**
     * Array of policy statements applied to the application.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Array of policy statements applied to the application.
     */
    public List<ApplicationPolicyStatement> statements() {
        return statements;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(applicationId());
        hashCode = 31 * hashCode + Objects.hashCode(statements());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutApplicationPolicyRequest)) {
            return false;
        }
        PutApplicationPolicyRequest other = (PutApplicationPolicyRequest) obj;
        return Objects.equals(applicationId(), other.applicationId()) && Objects.equals(statements(), other.statements());
    }

    @Override
    public String toString() {
        return ToString.builder("PutApplicationPolicyRequest").add("ApplicationId", applicationId())
                .add("Statements", statements()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationId":
            return Optional.ofNullable(clazz.cast(applicationId()));
        case "Statements":
            return Optional.ofNullable(clazz.cast(statements()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends ServerlessApplicationRepositoryRequest.Builder,
            CopyableBuilder<Builder, PutApplicationPolicyRequest> {
        /**
         * The id of the application to put policy for
         * 
         * @param applicationId
         *        The id of the application to put policy for
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationId(String applicationId);

        /**
         * Array of policy statements applied to the application.
         * 
         * @param statements
         *        Array of policy statements applied to the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statements(Collection<ApplicationPolicyStatement> statements);

        /**
         * Array of policy statements applied to the application.
         * 
         * @param statements
         *        Array of policy statements applied to the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statements(ApplicationPolicyStatement... statements);

        /**
         * Array of policy statements applied to the application. This is a convenience that creates an instance of the
         * {@link List<ApplicationPolicyStatement>.Builder} avoiding the need to create one manually via {@link List
         * <ApplicationPolicyStatement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ApplicationPolicyStatement>.Builder#build()} is called
         * immediately and its result is passed to {@link #statements(List<ApplicationPolicyStatement>)}.
         * 
         * @param statements
         *        a consumer that will call methods on {@link List<ApplicationPolicyStatement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statements(List<ApplicationPolicyStatement>)
         */
        Builder statements(Consumer<ApplicationPolicyStatement.Builder>... statements);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ServerlessApplicationRepositoryRequest.BuilderImpl implements Builder {
        private String applicationId;

        private List<ApplicationPolicyStatement> statements = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PutApplicationPolicyRequest model) {
            super(model);
            applicationId(model.applicationId);
            statements(model.statements);
        }

        public final String getApplicationId() {
            return applicationId;
        }

        @Override
        public final Builder applicationId(String applicationId) {
            this.applicationId = applicationId;
            return this;
        }

        public final void setApplicationId(String applicationId) {
            this.applicationId = applicationId;
        }

        public final Collection<ApplicationPolicyStatement.Builder> getStatements() {
            return statements != null ? statements.stream().map(ApplicationPolicyStatement::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder statements(Collection<ApplicationPolicyStatement> statements) {
            this.statements = ListOfApplicationPolicyStatementCopier.copy(statements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statements(ApplicationPolicyStatement... statements) {
            statements(Arrays.asList(statements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statements(Consumer<ApplicationPolicyStatement.Builder>... statements) {
            statements(Stream.of(statements).map(c -> ApplicationPolicyStatement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setStatements(Collection<ApplicationPolicyStatement.BuilderImpl> statements) {
            this.statements = ListOfApplicationPolicyStatementCopier.copyFromBuilder(statements);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutApplicationPolicyRequest build() {
            return new PutApplicationPolicyRequest(this);
        }
    }
}
